/* lib.d file lg.c */
/* This file contains all the functions which  operate on labelled graphs
and their vertices and do not refer to the fine structure kept in the magic 
pocket of a vertex. A function referring to the fine structure, and so 
specific to one particular type of labelled graph, is found in whichever of 
the files wdiff.web, wtree.web and acceptor.web is appropriate. 
*/

/*Here are the files to be included: */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "defs.h"
#include "list.h"
#include "input.h"
#include "word.h"
#include "lg.h"
static void vtxhashlist_stretch PARMS ((vtxhashlist * hlp,int));
void lg_erase_indices PARMS((lg * lgp));
void lgp_print_dp PARMS ((dp p));
int largest_lg_first PARMS ((dp dtpp1, dp dtpp2));
void lgp_cpy_dp PARMS ((dp oldp,dp newp));
lg ** lgp_create();

mp_fntab GENERIC_fntab =
{
	0,
	0,
	0,
	0,
	0,
	0
};

/*
We start with the functions operating on graphs as a whole.
*/

/*
Initialize a labelled graph.
*/
void
lg_init(lgp,fntab,type,degree)
	lg * lgp;
	mp_fntab * fntab;
	int type;
	int degree;
{
	lgp->fntab = fntab;
	lgp->degree = degree;
	lgp->type=type;
	lgp->num_verts=0;
	lgp->basepoint = lg_vertex_create(lgp);
	lgp->locked = FALSE;
	set_back_ptr(lgp->basepoint,BACK_FROM_BASE);
	list_init(&(lgp->discards),VINDEX,FIFO);
}



/* We return to the system all space which has been allocated to a
labelled graph. We remember not to leave a dangling pointer
|lgp->basepoint|.
*/
void
lg_clear(lgp)
	lg *lgp;
{
	bfs_traverser tlg;
	vindex v;
	if (lgp->basepoint != UNDEFINED) {
			/*|kill_and_devour()| puts the basepoint
			equal to |UNDEFINED|*/
		bfs_init(&tlg,lgp);
		while (bfs_next(&tlg,&v))
			(void)list_insert(&(lgp->discards),(dp)&v);
		bfs_clear(&tlg);
	}
	while(list_delget_first(&(lgp->discards),(dp)&v)) {
		lg_vertex_kill(lgp,v);
		if (v == lgp->basepoint)
			lgp->basepoint = UNDEFINED;
	}
	list_clear(&((lgp)->discards));
	assert(lgp->num_verts == 0);
} 

boolean
lg_eq(lg1p,lg2p)
	lg * lg1p;
	lg * lg2p;
{
	boolean ans=TRUE;
	int num_verts=lg1p->num_verts;
	int degree=lg1p->degree;

	if (num_verts!=lg2p->num_verts || degree!=lg2p->degree)
		ans=FALSE;
	else {
		list vertices1;
		list vertices2;
		bfs_traverser  bfst;
		int i=1;
		vindex v1=UNDEFINED;
		vindex v2=UNDEFINED;
		gen g;
		list_init(&vertices1,VINDEX,FIFO);
		list_init(&vertices2,VINDEX,FIFO);
		bfs_init(&bfst,lg1p);
		while (bfs_next(&bfst,&v1)){
			set_index(v1,i++);
			list_insert(&vertices1,(dp)&v1);
		}
		bfs_clear(&bfst);
		i=1;
		bfs_init(&bfst,lg2p);
		while (bfs_next(&bfst,&v2)){
			set_index(v2,i++);
			list_insert(&vertices2,(dp)&v2);
		}
		bfs_clear(&bfst);
		while ((list_delget_first(&vertices1,(dp)&v1))
				&&(list_delget_first(&vertices2,(dp)&v2))) {
			if (get_category(v1)!=get_category(v2))
					ans=FALSE;
			else
				for (g=1;g<=degree;g++){
				if (get(v1,g)==UNDEFINED&&get(v2,g)!=UNDEFINED)
						ans=FALSE;
			else if (get(v1,g)!=UNDEFINED&&get(v2,g)==UNDEFINED)
						ans=FALSE;
			else if (get(v1,g)!=UNDEFINED&&get(v2,g)!=UNDEFINED)
				if (get_index(get(v1,g))!=get_index(get(v2,g)))
							ans=FALSE;
				}
			if (ans==FALSE)
				break; /* out of the while loop */
		}	
		list_clear(&vertices1);
		list_clear(&vertices2);
	}
	return ans;
}
		
	
void
lg_cpy(lg1p,lg2p)
	lg * lg1p;
	lg * lg2p;
{
	int num_verts=lg1p->num_verts;
	int degree=lg1p->degree;
	bfs_traverser  bfst;
	int i=1;
	gen g;
	vindex * vertices1;
	vindex * vertices2;
	vindex v=UNDEFINED;
	lg_clear(lg2p);
	lg_init(lg2p,lg1p->fntab,lg1p->type,degree);
	vertices1=vzalloc2(vindex,num_verts + 1);
	vertices2=vzalloc2(vindex,num_verts + 1);
	bfs_init(&bfst,lg1p);
	while (bfs_next(&bfst,&v)){
		set_index(v,i);
		vertices1[i++]=v;
	}
	bfs_clear(&bfst);
	vertices2[1]=basept(lg2p);
	for (i=2;i<=num_verts;i++)
		vertices2[i]=lg_vertex_create(lg2p);
	for (i=1;i<=num_verts;i++){
		set_category(vertices2[i],get_category(vertices1[i]));
		for (g=1;g<=lg1p->degree;g++)
			if (get(vertices1[i],g)!=UNDEFINED)
		set(vertices2[i],g,vertices2[get_index(get(vertices1[i],g))]);
	}
	Free_dp((dp)vertices1); vertices1=0;
	Free_dp((dp)vertices2); vertices2=0;
}
		
lg *
lg_read(fntab,type,file)
	mp_fntab * fntab;
	int type;
	FILE * file;
{
	lg * lgp=0;
	int num_verts;
	int start=1;
	int degree;
	boolean fsa=FALSE;
	int index=1;
	int nbor_index=0;
	int i;
	vindex v;
	vindex * vertices;
	int letter;
	char * label;
	int variables=1;
	boolean eos=TRUE;
	boolean compressed=FALSE;
	
	gen g=1;	

	label=vzalloc2(char,9);
	if (type == BASIC_FSA||type==OPT_FSA)
		fsa=TRUE;
	while (getc(file)!='\{');
	read_next_string(label,8,file);
	if (strcmp(label,"states  ")!=0)
		bad_data();
	else
		read_next_int(&num_verts,file);
	read_next_string(label,8,file);
	if (strcmp(label,"symbols ")!=0)
		bad_data();
	else
		read_next_int(&degree,file);
	lgp=vzalloc1(lg);
	lg_init(lgp,fntab,type,degree);
	if (user_gen_name==0){
		if (gen_array_size<degree+1)
			gen_array_size=degree+1;
				/* this will certainly be big enough */
		user_gen_name=vzalloc2(word,gen_array_size);
		for (i=0;i<gen_array_size;i++)
			word_init(user_gen_name+i);
	}
	while (read_next_string(label,8,file)&&strcmp(label,"%       ")!=0){
		if (strcmp(label,"min     ")==0)
			lgp->type=OPT_FSA;
		else if (strcmp(label,"variable")==0)
			read_next_int(&variables,file);
		else if (strcmp(label,"no-eos  ")==0 ||strcmp(label,"no_eos  ")==0)
			eos=FALSE;
		else if (strcmp(label,"base-alp")==0 ||strcmp(label,"base_alp")==0){
			while (getc(file)!='\{') 
				;
			while (read_next_int(&g,file))
				if (num_gens<g){
					read_next_word(user_gen_name+g,file);
					num_gens++;
				}
			while (getc(file)!='\}')
				;
	/* if |num_gens| is already set we don't need to read these
words in */
		}
		else if (strcmp(label,"alphabet")==0){
			while (getc(file)!='\{')
					;
			while (read_next_int(&g,file))
				if (variables==1&&num_gens<g){
					read_next_word(user_gen_name+g,file);
					num_gens++;
				}
			while (getc(file)!='\}')
				;
		/* if |num_gens| is already set we don't need to read these
words in */
			
		}
		else if (strcmp(label,"start   ")==0){
			while (getc(file)!='\{')
				;
			while (read_next_int(&i,file))
				start=i;
			while (getc(file)!='\}')
				;
		}
	}

	/* at this stage |num_gens| is equal to the number of symbols in the
base alphabet, one of these may be the padding symbol */
	if (eos || variables==2)
		num_gens--;
	vertices = vzalloc2(vindex, (start>num_verts) ? start+1 : num_verts+1);
	vertices[start]=basept(lgp);
	for (i=1;i<=num_verts;++i){
		if (i!=start)
			vertices[i]=lg_vertex_create(lgp);
	}
	read_next_string(label,8,file);
	if (strcmp(label,"ctable  ")==0)
		compressed=TRUE;
	while (index<=num_verts){
		v = vertices[index];
		if (fsa){
			(void)read_next_letter(&letter,file);
			if (letter=='N'){
				if ((i=getc(file))!='g'){
					ungetc(i,file);
					set_category(v,NONACCEPTSTATE);
				}
				else {
					read_next_int(&i,file);
					set_category(v,2+i);
				}
			}
			else if (letter=='A')
				set_category(v,ACCEPTSTATE);
			else
				fprintf(stderr,"\t# Invalid category\n");
		}
		if (compressed){
			while (read_next_int(&g,file)){
				(void)read_next_int(&nbor_index,file);
				v->adj[g]=vertices[nbor_index];
			}
			while (getc(file)!='\;')
				;
		}
		else {
			for (g=1;g<=degree;++g) {
				(void)read_next_int(&nbor_index,file);
				v->adj[g]=vertices[nbor_index];
			}
		}
		index++;
	} 
	while (getc(file)!='\}')
		;
	Free_dp((dp)label); label=0;
	Free_dp((dp)vertices); vertices=0;
	return lgp;
}
	
		

/* Print the adjancency matrix of a labelled graph
Since labelled graphs are directed the adjacency matrix need not be
symmetric. Row i lists the neighbours of vertex i reached by an edge going
out of i. 
*/
void
lg_print(wfile,lgp)
	FILE * wfile;
	lg * lgp;
{
	boolean fsa=TRUE;
	boolean product_strings=FALSE;
	boolean eos=TRUE;
	list to_print;
	vindex u = UNDEFINED;
	int i = 1;
	int j=1;
	int count=1;
	gen g,h;
	bfs_traverser bfst;
	assert(lgp != NULL);
	bfs_init(&bfst,lgp);
	while (bfs_next(&bfst,&u))
		set_index(u,i++);
	bfs_clear(&bfst);
	list_init(&to_print,VINDEX,FIFO);
	switch (lgp->type){
		case BASIC_WTREE: 
		case KMP_WTREE:
		case MF_WTREE:
		case CON_WTREE:
			fprintf(wfile,"wordtree \{\n");
			fsa=FALSE;
			break;
		case BASIC_FSA:
		case OPT_FSA:
			fprintf(wfile,"fsa \{\n");
			break;
		default:
			break;  
	}
	if (fsa)
		fprintf(wfile,"\tstates %d\n",lgp->num_verts);
	else
		fprintf(wfile,"\tvertices %d\n",lgp->num_verts);
	fprintf(wfile,"\tsymbols %d\n",lgp->degree);
	fprintf(wfile,"\tbfs\n");
	if (lgp->type==OPT_FSA)
		fprintf(wfile,"\tmin\n");
	if (fsa){
		switch ((lgp->degree)-num_gens){
			case 0:
				fprintf(wfile,"\tvariables 1\n\tno_eos\n");
				eos=FALSE;
				break;
			case 1:
				fprintf(wfile,"\tvariables 1\n");
				if (num_gens==0)
					fprintf(wfile,"\t# actually, I'm not sure about this\n");
				break;
			default:
				fprintf(wfile,"\tvariables 2\n");
				product_strings=TRUE;
				if (lgp->degree==(num_gens+2)*(num_gens)){
					fprintf(wfile,"\tno_eos\n");
					eos=FALSE;
				}
				else if (lgp->degree!=(num_gens+1)*(num_gens+1))
					fprintf(wfile,"\t# UNEXPECTED DEGREE\n");
				break;
		}
	}
	if (product_strings==FALSE){
		fprintf(wfile,"\talphabet \{");
		for (g=1;g<=lgp->degree;g++){
			fprintf(wfile,"%d = ",g);
			gen_print(wfile,g);
			fprintf(wfile," ");
			if (g%10==0)
				fprintf(wfile,"\n");
		}
		fprintf(wfile," \}\n");
	}
	else {
		fprintf(wfile,"\tbase_alphabet \{");
		for (g=1;g<=num_gens+1;g++){
			fprintf(wfile,"%d = ",g);
			gen_print(wfile,g);
			fprintf(wfile," ");
			if (g%10==0)
				fprintf(wfile,"\n");
		}
		fprintf(wfile," \}\n");
		fprintf(wfile,"\talphabet \{");
		for (g=1;g<=num_gens+1;g++){
			for (h=1;h<=num_gens+1;h++){
				if (eos==FALSE&&g==num_gens+1&&h==num_gens+1)
					break;
				fprintf(wfile,"%d = ",j);
				fprintf(wfile,"\(");
				gen_print(wfile,g);
				fprintf(wfile,",");
				gen_print(wfile,h);
				fprintf(wfile,"\) ");
				if (j%5==0)
					fprintf(wfile,"\n");
				j++;
			}
		}
		fprintf(wfile," \}\n");
	}
	fprintf(wfile,"\tstart \{ 1 \}\n");
	if (num_gens+1<(lgp->degree))
		fprintf(wfile,"\n%%\nctable\n");
	else
		fprintf(wfile,"\n%%\natable\n");
	(void)list_insert(&to_print,(dp)&basept(lgp));
	count++;
 	while (list_delget_first(&to_print,(dp) &u)){ 
 		for(g=1;g<=lgp->degree;g++)  {
 			vindex t = get(u,g);
 			if (t != UNDEFINED && (get_index(t)==count)){ 
				/* we don't want to print a vertex out twice */
 				(void)list_insert(&to_print,(dp)&t);
				count++;
			}
 		}
	if (num_gens+1<(lgp->degree))
		vertex_cprint(wfile,u,lgp);
	else
		vertex_print(wfile,u,lgp);
	}
	fprintf(wfile,"\}\n");
	list_clear(&to_print);
}

void
lg_erase_indices(lgp)
	lg * lgp;
{
	vindex u;
	list index_eraser;
	list_init(&index_eraser,VINDEX,FIFO);
	u = basept(lgp);
	set_index(u,0);
	list_insert(&index_eraser,(dp)&u);
	while (list_delget_first(&index_eraser,(dp) &u)) {
		gen g;
		for(g=1;g<=lgp->degree;g++)  {
			vindex t = get(u,g);
			if (t != UNDEFINED && get_index(t) != 0) {
				set_index(t,0);
				list_insert(&index_eraser,(dp)&t);
			}
		}
	}
	list_clear(&index_eraser);
}

void
fsa_language_enumerate(wfile,FSA,n,order,nsyms)
	FILE * wfile;
	lg * FSA;
	int n;
	int order;
	int nsyms;
{
	word w;
	word w1;
	word w2;
	gen g=INVALID_GEN;
	gen h = INVALID_GEN;
	int l=0;
	vindex * states;
	boolean double_strings=FALSE;
	if (nsyms<FSA->degree){
		double_strings=TRUE;
		word_init(&w1);
		word_init(&w2);
	}
	word_init(&w); 
	states = vzalloc2(vindex,n+1);
	states[0]=basept(FSA);
	if (get_category(states[l])==ACCEPTSTATE) {
		if (double_strings==FALSE)
			word_print(wfile,&w);
		else {
			word_print(wfile,&w1);
			fprintf(wfile,"  ");
			word_print(wfile,&w2);
		}
		fprintf(wfile,"\n");
	}
	if (FSA->degree>0)
		g=1;
	if (n>0){
		while (l>0||g!=INVALID_GEN) {
			if (l<n && g!=INVALID_GEN ) { /* move forward */
				states[l+1] = get(states[l],g);
				word_put_last(&w,g);
				if (double_strings){
					word_put_last(&w1,1+((g-1)/nsyms));
					word_put_last(&w2,(g-1)%nsyms+1);
				}
				if (order==CIRCULAR && g<FSA->degree)
					g = g+1;
				else
					g = 1;
		if (states[l+1]!=UNDEFINED&&get_category(states[l+1])==ACCEPTSTATE) {
					if (double_strings==FALSE)
						word_print(wfile,&w);
					else {
						word_print(wfile,&w1);
						fprintf(wfile,"  ");
						word_print(wfile,&w2);
					}
					fprintf(wfile,"\n");
				}
				else if (states[l+1]==UNDEFINED)
					g=INVALID_GEN;
				l++;
			} 
			else {  /* backtrack */
				if (double_strings){
					word_del_last(&w1);
					word_del_last(&w2);
				}
				word_delget_last(&w,&g);
				l--;
			/* find the next generator, if there is one */
				if ((order==CIRCULAR&&(word_get_last(&w,&h))&&h==g)
						||(order==LINEAR&&g==FSA->degree))
					g=INVALID_GEN;
				else if (order==CIRCULAR&&g==FSA->degree){
					if (l!=0)
						g=1;
					else
						g=INVALID_GEN;
				}
				else
					g=g++;
			}
		}
	}
	Free_dp((dp)states); states=0;
	word_clear(&w);
	if (double_strings){
		word_clear(&w1);
		word_clear(&w2);
	}
}	

/*
If the word |*wp| cannot be found in the tree |*lgp| it is added in. The
address of the vertex at the end of the path it traces out is returned.
*/
vindex
wtree_word_trace(lgp,wp)
	lg *lgp;
	word *wp;
{
	vindex v = basept(lgp);
	vindex t = UNDEFINED;
	word copy;
	gen g ;
	word_init(&copy);
	word_cpy(wp,&copy);
	while(word_get_first(&copy,&g)) {
		t = get(v,g);
		if (t == UNDEFINED)
			break;
		v = t;
		(void)word_del_first(&copy);
	}
	while(word_delget_first(&copy,&g)) {
		t = lg_vertex_create(lgp);
		set(v,g,t);
		set(t,inv(g),v);
		set_back_ptr(t,g);
		v = t;
	}
	word_clear(&copy);
	return v;
}


void
bfs_init(bfst,lgp)
	bfs_traverser *bfst;
	lg *lgp;
{
	assert(lgp->locked == FALSE);
	bfst->lgp = lgp;
	lgp->locked = TRUE;
	bfst->verts = vzalloc2(vindex,get_num_verts(lgp));
	bfst->start_fringe = 0;
	bfst->end = 0;
	bfst->verts[0] = lgp->basepoint;
	lgp->basepoint->bits.visit = 1;
}

/* Because the vertex most recently presented by |bfs_next()| has been
recorded, we can split the process of putting vertices on the queue and
updating the ``visited'' field (which is in fact called
|bits.visit|) from the process of presenting the next vertex.
The advantage of the splitting is,
that when |bfs_next()| presents a vertex, we can see
how the breadth first search is progressing locally, and this information
is useful.
*/
boolean
bfs_next(bfst,vp)
	bfs_traverser *bfst;
	vindex *vp;
{
	boolean ans = TRUE;
	gen g = INVALID_GEN;
	if (bfst->start_fringe > bfst->end) {
		*vp = UNDEFINED;
		ans = FALSE;
	}
	else {
		for(g=1;g<=bfst->lgp->degree;g++) {
			vindex t = get(bfst->verts[bfst->start_fringe],g);
			if (t != UNDEFINED && t->bits.visit == 0) {
				t->bits.visit = 1;
				(bfst->end)++;
				bfst->verts[bfst->end] = t;
				assert(bfst->end < bfst->lgp->num_verts);
			}
		}
		*vp = bfst->verts[bfst->start_fringe] ;
		(bfst->start_fringe)++;
	}
	return ans;
}

void
bfs_clear(tp)
	bfs_traverser *tp;
{
	int i = 0;
	vindex * bv = tp->verts;
	assert(tp->lgp->locked);
	tp->lgp->locked = FALSE;
	tp->lgp = (lg *)0;
	for (i=0; i<= tp->end; i++)
		bv[i]->bits.visit = 0;
	Free_dp((dp)tp->verts);
	tp->verts = 0;
	tp->start_fringe = -1;
	tp->end = -1;
}

/* \Pre |v| is a vertex in a tree. |wp| points to an initialized word.
\Post |wp| points to the word which corresponds to |v| (i.e., if we trace
|*wp| out, starting from the basepoint, we get to |v|).
*/
void
vindex2word(v,wp)
	vindex v;
	word * wp;
{
	gen g;
	
	assert(wp);
	assert(v);
	word_reset(wp);
	while (!is_basepoint(v) && (g=backg(v))!=INVALID_GEN) {
		assert(g != BACK_FROM_BASE);
		word_put_first(wp,g);
		v = get(v,inv(g));
		assert(v != UNDEFINED);
	}
}

/* The word |*wp| is applied to the basepoint of the graph |*lgp|. If the word 
cannot be followed in the labelled graph, the function returns |UNDEFINED|. 
Otherwise it returns to the address of the vertex where we end up.
*/
vindex
lg_word_find(lgp,wp)
	lg *lgp;
	word *wp;
{
	vindex upto=basept(lgp);
	gen g;
	word_traverser wt; 
	assert(wp);
	word_traverser_init(&wt,wp);
	while (word_next(&wt,&g)) {
		upto = get(upto,g);
		if (upto == UNDEFINED) break;
	}
	word_traverser_clear(&wt);
	return upto;
}




/* The function table for a list of labelled graphs. The labelled
graphs themselves are not stored, only their addresses. They are
stored according to the number of vertices, with largest vertex first.
*/
elt_fntab
LGP_fntab = {
	largest_lg_first,
	lgp_cpy_dp,
	(V2DP)lgp_create,
	Free_dp,
	lgp_print_dp,
	0,
	0
};


bfs_debug(lgp)
	lg * lgp;
{
	bfs_traverser tlgp;
	vindex v;
	bfs_init(&tlgp,lgp);
	while (bfs_next(&tlgp,&v))
		assert(v != UNDEFINED);
	bfs_clear(&tlgp);
}



vindex
lg_vertex_create(lgp)
	lg *lgp;
{
	vindex  v;
	v = vzalloc1(vertex);
	v->fntab = lgp->fntab;
	v->degree = get_degree(lgp);
	v->adj = vzalloc2(vindex,(v->degree)+1);
	if ((lgp->fntab->mp_create) !=0)
		v->mp = (*(lgp->fntab->mp_create))();
	set_back_ptr(v,INVALID_GEN);
	set_category(v,NONACCEPTSTATE);
	/* this is only relevant to fsa's but it seems best to have this default
setting */
	(lgp->num_verts)++;
#if (UNDEFINED != 0)
	{
		int i;
		for (i=1; i <= (v->degree) ; i++)
			v->adj[i] = UNDEFINED;
	}
#endif
	v->adj[0] = v;
	return v;
}

void
lg_vertex_kill(lgp,vtd)
	vindex vtd; /*vertex to delete*/
	lg *lgp;
{
	gen g;
/*first de-allocate space for magic pocket, according to type of labelled
		graph*/
	assert( (vtd->mp)? (lgp->fntab->mp_kill != 0) : TRUE);
	/*if the magic pocket exists, there must be a way of killing it*/
	/*tobedone but it may not be necessary to go into the magic pocket*/
	if (vtd->mp) {
		lgp->fntab->mp_kill(vtd);
	}
	for(g=1;g<=lgp->degree;g++) {
		vindex t = get(vtd,g);
		if (t != UNDEFINED) {
			unset(vtd,g);
			if (lgp->type!= BASIC_FSA&&lgp->type!=OPT_FSA) 
				unset(t,inv(g));
		}
	}
	Free_dp((dp)(vtd->adj));
	vtd->adj=0;
	Free_dp((dp)vtd);
	vtd = 0;
	(lgp->num_verts)--;
}




void
lg_reset_basepoint(lgp,v)
	lg * lgp;
	vindex v;
{
	lgp->basepoint=v;
	set_back_ptr(v,BACK_FROM_BASE);
}

void
lg_kill(lgp)
	lg *lgp;
{
	lg_clear(lgp);
	Free_dp((dp)lgp);
}



/* Is the labelled graph non-trivial? It is trivial if it consists only
of the basepoint, with no arrows.
*/
boolean
non_trivial(lgp)
	lg *lgp;
{
	boolean ans = FALSE;
	gen g;
	if (get_num_verts(lgp) > 1)
		ans = TRUE;
	else {
		vindex v = basept(lgp);
		for(g=1;g<=(lgp->degree);g++) {
			if (get(v,g) != UNDEFINED) {
				ans = TRUE;
				break;
			}
		}
	}
	return ans;
}


void
lg_set_indices(lgp)
	lg * lgp;
{
	bfs_traverser bfst;
	vindex v;
	int index=1;
	bfs_init(&bfst,lgp);
	while (bfs_next(&bfst,&v))
		set_index(v,index++);
	bfs_clear(&bfst);
}

int
dist(v)
	vindex v;
{
	int count=0;
	while (!is_basepoint(v)){
		v = get(v,inv(backg(v)));
		count++;
	}
	return count;
}
			

void
vertex_print_dp(wfile,v)
	FILE * wfile;
	dp v; /* really, |v| is a |vertex *|, but lists require |dp| arguments*/
{
	gen g;
	vindex vv = (vindex)v;
	fprintf(wfile,"[%d]: ",get_index(vv));
	for (g=1; g<=(vv->degree); g++) {
		vindex t = get(vv,g);
		if (t != UNDEFINED) {
			fprintf(wfile," =");
			if (num_gens!=0)
/* the graph is associated with a group, and the generators as input by the
user, should label the edges */
				gen_print(wfile,g);
			fprintf(wfile,"=>%d",get_index(t));
		}
	}
	fprintf(wfile,"\n");
}


void
vertex_print(wfile,v,lgp)
	FILE * wfile;
	vindex v;
	lg * lgp;
{
	gen g;
	fprintf(wfile," %3d ",get_index(v));
	if (lgp->type==BASIC_FSA||lgp->type==OPT_FSA) {
		if((v->category)==ACCEPTSTATE)	
			fprintf(wfile," A\t");
		else if((v->category)==NONACCEPTSTATE)	
			fprintf(wfile," N\t");
		else
			fprintf(wfile," Ng%d\t ",(v->category)-2);
	}
	for (g=1; g<=lgp->degree; g++) {
		vindex t = get(v,g);
		if (t != UNDEFINED) 
			fprintf(wfile," %3d",get_index(t));
		else
			fprintf(wfile,"   0");
		if (g%15==0)
			fprintf(wfile,"\n         \t");
	}
	fprintf(wfile,";\n");
}

void
vertex_cprint(wfile,v,lgp)
	FILE * wfile;
	vindex v;
	lg * lgp;
{
	gen g;
	int entries=0;
	fprintf(wfile," %3d ",get_index(v));
	if (lgp->type==BASIC_FSA||lgp->type==OPT_FSA) {
		if((v->category)==ACCEPTSTATE)	
			fprintf(wfile," A\t");
		else if((v->category)==NONACCEPTSTATE)	
			fprintf(wfile," N\t");
		else
			fprintf(wfile," Ng%d\t ",(v->category)-2);
	}
	for (g=1; g<=lgp->degree; g++) {
		vindex t = get(v,g);
		if (t !=UNDEFINED){
			if (entries!=0){
				fprintf(wfile,",");
				if (entries%8==0)
					fprintf(wfile,"\n         \t");
			}
			fprintf(wfile," %d > %d",g,get_index(t));
			entries++;
		}
	}
	fprintf(wfile,";\n");
}


elt_fntab
VINDEX_fntab =
	{
		vindex_sgn_dp, /*compare function */
		vindex_cpy_dp, /*copy function*/
		(V2DP)vindex_create, 
		Free_dp, /*kill function*/
		vindex_print_dp, /*print for humans*/
		get_vindex_hpindex_dp,
		set_vindex_hpindex_dp
	};

int vindex_sgn_dp(dp1,dp2)
	dp dp1;
	dp dp2;
{
	vindex  v1, v2;
	auto int (*sgn) PARMS ((vindex,vindex));
	int ans=0;
	v1 =  *(vindex*)dp1;
	v2 =  *(vindex*)dp2;
	sgn = (v1->fntab)->mp_sgn;
	if (sgn)
		ans = (*sgn)(v1,v2);
	return ans;
}
	

/*Copies contents of |oldp| into |newp|.
Only the address is copied, not the whole structure of a vertex.
\Pre |oldp| and |newp| point to |vindex|'s.
\Post |*oldp| is unaltered and |*newp| becomes equal to |*oldp|.
*/
void
vindex_cpy_dp(oldp,newp)
	dp oldp;
	dp newp;
{
	vindex *oldv = (vindex*)oldp;
	vindex *newv = (vindex*)newp;
	assert(oldp);
	assert(newp);
	*newv = *oldv;
}

/* \Returns a pointer to the right amount of space to store a |vindex|.
This space is filled with zeroes.
*/
vindex *
vindex_create()
{
	return vzalloc1(vindex);
}


/*Print a |vindex|. This function is used for lists of |vindex|'s.
As a consequence, the argument is given as a |dp|, but we must think
of it as a |vindex *|.
*/
void
vindex_print_dp(wfile,dtp)
	FILE * wfile;
	dp dtp;
{
	vindex v = *((vindex *)dtp);
	fprintf(wfile,"0x%x",v);
}

int get_vindex_hpindex_dp(dtp)
	dp dtp; /* actually a pointer to a vindex */
{
	int ans=0;
	auto int (*ghi) PARMS ((vindex));
	vindex  v = *(vindex*)dtp;
	ghi = (v->fntab)->mp_gethpindex;
	if (ghi)
		ans = (*ghi)(v);
	return ans;
}

void set_vindex_hpindex_dp(dtp,i)
	dp dtp;
	int i;
{
	auto void (*shi) PARMS ((vindex, int));
	vindex v = *(vindex *)dtp;
	shi = (v->fntab)->mp_sethpindex;
	if (shi)
		(*shi)(v,i);
}



boolean
is_basepoint(v)
	vindex v;
{
	return (backg(v) == BACK_FROM_BASE);
}


void
vtxhashlist_init(hlp,size)
	vtxhashlist * hlp;
	int size;
{
	hlp->array=vzalloc2(vtxlink*,size);	
	hlp->maxhash=size-1;
}

void
vtxhashlist_insert(hlp,vp,hash)
	vtxhashlist * hlp;
	vindex *vp;
	int hash;
{
	vtxlink * newp;
	vtxlink * lp;
	int size=(hlp->maxhash) +1;
	auto int (*sgn) PARMS((vindex,vindex));
	sgn = ((*vp)->fntab)->mp_sgn;
	while (hash>=size)
		size *=2;
	vtxhashlist_stretch(hlp,size);
	lp=(hlp->array)[hash];
	newp=vzalloc1(vtxlink);
	newp->v = *vp;
	if (lp==0||(*sgn)(*vp,lp->v)>0){
	/* |*vp| has to be in the first place */
		newp->next=lp;
		(hlp->array)[hash]=newp;
	}
	else { 
		while (lp->next!=0&&(*sgn)(*vp,lp->next->v)<0)
			lp=lp->next;
	/* now |*vp| fits in after lp */
		newp->next=lp->next;
		lp->next=newp;
	}
}
	

boolean
vtxhashlist_find(hlp,in,out,hash)
	vtxhashlist * hlp;
	vindex *in;
	vindex *out;
	int hash;
{
	boolean ans=FALSE;
	if (hash<=(hlp->maxhash)&&((hlp->array)[hash]!=0)){
		auto int (*sgn) PARMS((vindex,vindex));
		vtxlink * lp=(hlp->array)[hash];
		sgn = ((*in)->fntab)->mp_sgn;
		while (lp!=0&&(*sgn)(*in,lp->v)<0)
			lp=lp->next;
		if (lp==0||(*sgn)(*in,lp->v)>0)
			*out=0;
		else{
			ans=TRUE;
			*out=lp->v;
		}
	}
	return ans;
} 
	
void vtxhashlist_clear(hlp)
	vtxhashlist * hlp;
{
	int i=0;
	vtxlink * lp;
	vtxlink * spare;
	for (i=0;i<=hlp->maxhash;i++)
		if ((lp=(hlp->array)[i])!=0)
			while (lp){
				spare=lp;
				lp=lp->next;
				Free_dp((dp)spare); spare=0;
			}
	Free_dp((dp)(hlp->array));
	hlp->array=0;
}

	
static void 
vtxhashlist_stretch(hlp,size)
	vtxhashlist * hlp;
	int size;
{
	vtxlink ** newarray;
	int i;
	newarray=vzalloc2(vtxlink*,size);
	for (i=0;i<=hlp->maxhash;i++)
		newarray[i]=(hlp->array)[i];
	Free_dp((dp)(hlp->array)); hlp->array=0;
	hlp->array=newarray;
	hlp->maxhash=size-1;
}
	
	
	


int
largest_lg_first(dtpp1,dtpp2)
	dp dtpp1, dtpp2; /*these are really pointers to |lg *| */
{
	lg ** lgpp1 = (lg * *)dtpp1;
	lg ** lgpp2 = (lg * *)dtpp2;
	lg * lgp1 = (*lgpp1);
	lg * lgp2 = (*lgpp2);
	int ans = 0;
	if (lgp1->num_verts > lgp2->num_verts)
		ans = 1;
	else if (lgp1->num_verts < lgp2->num_verts)
		ans = -1;
	return ans;
}


void
lgp_cpy_dp(oldp,newp)
	dp oldp, newp;
{
	lg ** oldlgpp = (lg **)oldp;
	lg ** newlgpp = (lg **)newp;
	*newlgpp = *oldlgpp; /*make the addresses of the |lg|'s equal*/
}


lg **
lgp_create()
{
	return vzalloc1(lg *);
}



void
lgp_print_dp(wfile,p)
	FILE * wfile;
	dp p;
{
	lg ** lgpp = (lg **)p;
	lg_print(wfile,*lgpp);
}

