/* -*- C -*-
 *
 * Program:	ximap
 * File:        search.c -- performing a search on a mailbox.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
 
#include <netdb.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <memory.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>

#ifdef MOTIF
#include <X11/Xm/Text.h>
#else
#include <X11/Xaw/Text.h>
#include <X11/Xaw/AsciiText.h>
#endif

#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Label.h>

#include "structures.h"

#include "modal.h"
#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "util.h"
#include "search.h"
#include "mailbox.h"
#include "browserutil.h"
#include "textutil.h"
#include "ximap.h"

#include "Bitmaps/empty.bit"

static void swinBeep();

static void swinQuitAct();
static void swinBrowseAct();
static void swinSearchAct();

static void gotoSearchFrom();
static void gotoSearchTo();
static void gotoSearchSubject();
static void gotoSearchText();
static void gotoSearchCc();
static void gotoSearchBody();
static void gotoSearchKeyword();
static void gotoSearchBcc();
static void gotoSearchSince();
static void gotoSearchOn();
static void gotoSearchBefore();

static XtActionsRec search_actions[] = 
{
  {"swinBeep", swinBeep},
  {"swinQuit", swinQuitAct},
  {"swinBrowse", swinBrowseAct},
  {"swinSearch", swinSearchAct},
  {"gotoSearchFrom", gotoSearchFrom},
  {"gotoSearchTo", gotoSearchTo},
  {"gotoSearchSubject", gotoSearchSubject},
  {"gotoSearchText", gotoSearchText },
  {"gotoSearchCc", gotoSearchCc },
  {"gotoSearchBody", gotoSearchBody},
  {"gotoSearchKeyword", gotoSearchKeyword},
  {"gotoSearchBcc", gotoSearchBcc},
  {"gotoSearchSince", gotoSearchSince},
  {"gotoSearchOn", gotoSearchOn},
  {"gotoSearchBefore", gotoSearchBefore},
};

static char toTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchTo()               \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchBody()";

static char bodyTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchBody()               \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchText()";

static char textTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchText()               \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchKeyword()";

static char keywordTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchKeyword()               \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchSubject()";

static char subjectTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchSubject()          \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchFrom()";

static char fromTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchFrom()            \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchCc()";

static char ccTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchCc()          \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchBcc() ";

static char bccTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchBcc()          \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchSince() ";

static char sinceTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchSince()          \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchBefore() ";

static char beforeTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchBefore()          \n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchOn() ";

static char onTranslations[] = 
 "<Btn1Down>: select-start() gotoSearchOn()\n\
  Meta<Key>S: swinSearch()\n\
  Meta<Key>B: swinBrowse()\n\
  Ctrl<Key>Q: swinQuit()\n\
  <Key>Tab:   gotoSearchTo()";

static void createStringsPanel(/*  SEARCHWINDOW*  */);
static void createDatesPanel(/*  SEARCHWINDOW*  */);
static void createFlagsPanel(/*  SEARCHWINDOW*  */);
static void searchQuit(/*  Widget, SEARCHWINDOW*, XEvent*  */);

static void doSearch(/*  Widget, SEARCHWINDOW*, XEvent*  */);
/* static void doFilter(); */
static void doBrowse();
static void cleanup();

static char* makeSearchFilter(/*  SEARCHWINDOW*  */);
static void clear_search_criteria(/*  Widget, SEARCHWINDOW*, XEvent*  */);

ButtonStruct searchButtons[] = 
{
  { "Quit",    searchQuit , NULL,  NEVER},
  { "Search",  doSearch   , NULL,  NEVER},
  { "Browse",  doBrowse   , NULL,  NEVER},
  { "Clear",  clear_search_criteria , NULL,  NEVER},
  NULL,
};

Pixmap emptyPixmap;

extern XtAppContext app_context;

void addSearchActions( ac )
     XtAppContext ac;
{
    XtAppAddActions( ac, search_actions, XtNumber(search_actions));
}

void createSearchPanel(ms)
     MailBox* ms;
{	
    Widget SearchButtonBox;
    
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    SEARCHWINDOW *sw;
    
    BrowserEntry *be = getActiveBrowser(ms);
    
    if( be->searchWin == NULL )
    {
	sw = be->searchWin = (SEARCHWINDOW*)XtMalloc(sizeof(SEARCHWINDOW));
	
	emptyPixmap= 
	    XCreateBitmapFromData(XtDisplay(ms->mailWindow),
				  RootWindowOfScreen(XtScreen(ms->mailWindow)),
				  empty_bits,
				  empty_width, empty_height);
	
	sw->ms = ms;
	sw->be = be;
	sw->last_focus = NULL;
      	
	XtSetArg(warg[n], XtNtitle, "Find messages..."); n++;
	sw->shell = XtCreatePopupShell("search_shell",
				       topLevelShellWidgetClass,
				       ms->mailWindow,
				       warg,
				       n); n = 0;
	
	sw->panes =  XtCreateManagedWidget("search_panes",
					   panedWidgetClass,
					   sw->shell,
					   warg,
					   n); n = 0;
	
	XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	SearchButtonBox =  XtCreateManagedWidget("search_buttons",
						 boxWidgetClass, 
						 sw->panes,
						 warg,
						 n); n = 0;
	
	sw->states = createButtons(SearchButtonBox, 
				   be->browser,
				   searchButtons);
	
	createStringsPanel( sw );
	createDatesPanel( sw );
	createFlagsPanel( sw );
    }
    else
    {
	sw = be->searchWin;
    }
    
    XtPopup( sw->shell, XtGrabNone );
    XtSetKeyboardFocus( sw->shell, sw->toText );
    XawTextDisplayCaret(sw->toText , TRUE);
    sw->last_focus = sw->toText;
}

  static void 
createStringsPanel( sw )
     SEARCHWINDOW *sw;
{
  Arg warg[ARGLISTSIZE];
  int n = 0;
  
  XFontStruct *font;

  Widget keywordLabel, toLabel, textLabel, bodyLabel, subjectLabel;
  Widget fromLabel, ccLabel, bccLabel;

  XtTranslations totr, fromtr, cctr;
  XtTranslations subjecttr, keywordtr, bodytr, texttr, bcctr;
    
  totr = XtParseTranslationTable( toTranslations );
  bodytr = XtParseTranslationTable( bodyTranslations );
  texttr = XtParseTranslationTable( textTranslations );
  keywordtr = XtParseTranslationTable( keywordTranslations );
  subjecttr = XtParseTranslationTable( subjectTranslations );
  fromtr = XtParseTranslationTable( fromTranslations );
  cctr = XtParseTranslationTable( ccTranslations );
  bcctr = XtParseTranslationTable( bccTranslations );

  sw->form =  XtCreateManagedWidget("search_strings",
				    formWidgetClass,
				    sw->panes,
				    warg,
				    n); n = 0;
  
  XtSetArg(warg[n], XtNlabel, "     to:"); n++;
  toLabel = XtCreateManagedWidget("to_label", labelWidgetClass,
				  sw->form,
				  warg,
				  n); n = 0;

#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, toLabel); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->toText = XtCreateManagedWidget("to_text",
				     xmTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, toLabel); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->toText = XtCreateManagedWidget("to_text",
				     asciiTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->toText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->toText, warg, n); n = 0;
  XtOverrideTranslations( sw->toText , totr );
#endif
  
  XtSetArg(warg[n], XtNlabel, "   body:"); n++;
  XtSetArg(warg[n], XtNfromVert, sw->toText); n++;
  bodyLabel = XtCreateManagedWidget("body_label", labelWidgetClass,
				    sw->form,
				    warg,
				    n); n = 0;
#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, bodyLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->toText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->bodyText = XtCreateManagedWidget("body_text",
				       xmTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
#else

  XtSetArg(warg[n], XtNfromHoriz, bodyLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->toText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->bodyText = XtCreateManagedWidget("body_text",
				       asciiTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->bodyText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->bodyText, warg, n); n = 0;
  XtOverrideTranslations( sw->bodyText , bodytr );
#endif

  XtSetArg(warg[n], XtNlabel, "   text:"); n++;
  XtSetArg(warg[n], XtNfromVert, sw->bodyText); n++;
  textLabel = XtCreateManagedWidget("text_label", labelWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, textLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->bodyText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->textText = XtCreateManagedWidget("text_text",
				       xmTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;

  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->textText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->textText, warg, n); n = 0;
  XtOverrideTranslations( sw->textText , texttr );
#else
  XtSetArg(warg[n], XtNfromHoriz, textLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->bodyText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->textText = XtCreateManagedWidget("text_text",
				       asciiTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;

  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->textText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->textText, warg, n); n = 0;
  XtOverrideTranslations( sw->textText , texttr );
#endif

  XtSetArg(warg[n], XtNfromVert, sw->textText); n++;
  XtSetArg(warg[n], XtNlabel, "keyword:"); n++;
  keywordLabel = XtCreateManagedWidget("keyword_label", labelWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;

#ifdef MOTIF
  XtSetArg(warg[n], XtNfromHoriz, keywordLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->textText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->keywordText = XtCreateManagedWidget("keyword_text",
					  xmTextWidgetClass,
					  sw->form,
					  warg,
					  n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, keywordLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->textText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->keywordText = XtCreateManagedWidget("keyword_text",
					  asciiTextWidgetClass,
					  sw->form,
					  warg,
					  n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->keywordText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->keywordText, warg, n); n = 0;
  XtOverrideTranslations( sw->keywordText , keywordtr );
#endif

  XtSetArg(warg[n], XtNfromVert, sw->keywordText); n++;
  XtSetArg(warg[n], XtNlabel, "subject:"); n++;
  subjectLabel = XtCreateManagedWidget("subject_label", labelWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;

#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, subjectLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->keywordText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->subjectText = XtCreateManagedWidget("subject_text",
					  xmTextWidgetClass,
					  sw->form,
					  warg,
					  n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, subjectLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->keywordText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->subjectText = XtCreateManagedWidget("subject_text",
					  asciiTextWidgetClass,
					  sw->form,
					  warg,
					  n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->subjectText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->subjectText, warg, n); n = 0;
  XtOverrideTranslations( sw->subjectText , subjecttr );
#endif

  XtSetArg(warg[n], XtNfromVert, sw->subjectText); n++;
  XtSetArg(warg[n], XtNlabel, "   from:"); n++;
  fromLabel = XtCreateManagedWidget("from_label", labelWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, fromLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->subjectText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->fromText = XtCreateManagedWidget("from_text",
				       xmTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, fromLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->subjectText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->fromText = XtCreateManagedWidget("from_text",
				       asciiTextWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->fromText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->fromText, warg, n); n = 0;
  XtOverrideTranslations( sw->fromText , fromtr );
#endif

  XtSetArg(warg[n], XtNfromVert, sw->fromText); n++;
  XtSetArg(warg[n], XtNlabel, "     cc:"); n++;
  ccLabel = XtCreateManagedWidget("cc_label", labelWidgetClass,
				       sw->form,
				       warg,
				       n); n = 0;
#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, ccLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->fromText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->ccText = XtCreateManagedWidget("cc_text",
				     xmTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, ccLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->fromText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->ccText = XtCreateManagedWidget("cc_text",
				     asciiTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->ccText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->ccText, warg, n); n = 0;
  XtOverrideTranslations( sw->ccText , cctr );
#endif


  XtSetArg(warg[n], XtNfromVert, sw->ccText); n++;
  XtSetArg(warg[n], XtNlabel, "    bcc:"); n++;
  bccLabel = XtCreateManagedWidget("bcc_label", labelWidgetClass,
				   sw->form,
				   warg,
				   n); n = 0;
#ifdef MOTIF  
  XtSetArg(warg[n], XtNfromHoriz, bccLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->ccText); n++;
  XtSetArg(warg[n], XmNvalue, ""); n++;
  sw->bccText = XtCreateManagedWidget("bcc_text",
				      xmTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
#else
  XtSetArg(warg[n], XtNfromHoriz, bccLabel); n++;
  XtSetArg(warg[n], XtNfromVert, sw->ccText); n++;
  XtSetArg(warg[n], XtNstring, ""); n++;
  sw->bccText = XtCreateManagedWidget("bcc_text",
				     asciiTextWidgetClass,
				     sw->form,
				     warg,
				     n); n = 0;
  XtSetArg(warg[n], XtNfont, &font); n++ ;
  XtGetValues(sw->bccText, warg, n); n = 0;
  XtSetArg(warg[n], XtNheight,
	   (font->max_bounds.ascent + font->max_bounds.descent) + 6);n++ ;
  XtSetArg(warg[n], XtNwidth,
	   (font->max_bounds.width * 60)); n++;
  XtSetValues(sw->bccText, warg, n); n = 0;
  XtOverrideTranslations( sw->bccText , bcctr );
#endif
}

  static void 
createFlagsPanel( sw )
     SEARCHWINDOW *sw;
{
  Widget form;

  Arg warg[ARGLISTSIZE];
  int n = 0;

  form =  XtCreateManagedWidget("search_form",
				formWidgetClass,
				sw->panes,
				warg,
				n); n = 0;

  sw->flags.form =  XtCreateManagedWidget("search_flags",
					  formWidgetClass,
					  form,
					  warg,
					  n); n = 0;

  XtSetArg(warg[n], XtNlabel, "        "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("empty", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "yes "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("yes_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "no  "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("no_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "both"); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("both_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "  Seen  "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("seen_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;  

  XtSetArg(warg[n], XtNradioData, XB_SEEN_YES); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.seenToggle =  XtCreateManagedWidget("seen_yes", 
						toggleWidgetClass,           
						sw->flags.form,                
						warg, n); n = 0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.seenToggle); n++;
  XtSetArg(warg[n], XtNradioData, XB_SEEN_NO); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.unseenToggle =  XtCreateManagedWidget("seen_no",
						  toggleWidgetClass,
						  sw->flags.form,
						  warg, n); n = 0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.seenToggle); n++;
  XtSetArg(warg[n], XtNradioData, XB_SEEN_EITHER); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtCreateManagedWidget("seen_both", 
			toggleWidgetClass,           
			sw->flags.form,                
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "Deleted "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("deleted_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;
  
  XtSetArg(warg[n], XtNradioData, XB_DELETED_YES ); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.deletedToggle =  XtCreateManagedWidget("deleted_yes", 
						   toggleWidgetClass,         
						   sw->flags.form,         
						   warg, n); n =0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.deletedToggle); n++;
  XtSetArg(warg[n], XtNradioData, XB_DELETED_NO ); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.undeletedToggle  =  XtCreateManagedWidget("deleted_no", 
						      toggleWidgetClass,      
						      sw->flags.form,         
						      warg, n); n = 0;
  
  XtSetArg(warg[n], XtNradioGroup, sw->flags.deletedToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_DELETED_EITHER ); n++;
  XtCreateManagedWidget("deleted_both", 
			toggleWidgetClass,           
			sw->flags.form,                
			warg, n ); n = 0;

  XtSetArg(warg[n], XtNlabel, " Recent "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("recent_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n ); n = 0;

  XtSetArg(warg[n], XtNradioData, XB_RECENT_YES ); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.recentToggle =  XtCreateManagedWidget("recent_yes", 
						  toggleWidgetClass,
						  sw->flags.form,    
						  warg, n); n = 0;
  
  XtSetArg(warg[n], XtNradioGroup, sw->flags.recentToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_RECENT_NO ); n++;
  sw->flags.unrecentToggle =  XtCreateManagedWidget("recent_no", 
						    toggleWidgetClass,        
						    sw->flags.form,       
						    warg, n); n = 0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.recentToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_RECENT_EITHER  ); n++;
  XtCreateManagedWidget("recent_both", 
			toggleWidgetClass,           
			sw->flags.form,                
			warg, n); n = 0;

  XtSetArg(warg[n], XtNlabel, "Answered"); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("answered_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;

  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_ANSWERED_YES  ); n++;
  sw->flags.answeredToggle =  XtCreateManagedWidget("answered_yes", 
						    toggleWidgetClass,       
						    sw->flags.form,        
						    warg, n); n = 0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.answeredToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_ANSWERED_NO ); n++;
  sw->flags.unansweredToggle =  XtCreateManagedWidget("answered_no", 
				      toggleWidgetClass,           
				      sw->flags.form,                
						    warg, n); n = 0;

  
  XtSetArg(warg[n], XtNradioGroup, sw->flags.answeredToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_ANSWERED_EITHER ); n++;
  XtCreateManagedWidget("answered_both", 
			toggleWidgetClass,           
			sw->flags.form,                
			warg, n); n = 0;


  XtSetArg(warg[n], XtNlabel, "Flagged "); n++;
  XtSetArg(warg[n], XtNborderWidth, 0); n++;
  XtCreateManagedWidget("flagged_label", 
			labelWidgetClass,  
			sw->flags.form,      
			warg, n); n = 0;
  
  XtSetArg(warg[n], XtNradioData, XB_FLAGGED_YES ); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  sw->flags.flaggedToggle =  XtCreateManagedWidget("flagged_yes", 
						   toggleWidgetClass, 
						   sw->flags.form,       
						   warg, n); n = 0;

  XtSetArg(warg[n], XtNradioGroup, sw->flags.flaggedToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_FLAGGED_NO ); n++;
  sw->flags.unflaggedToggle =  XtCreateManagedWidget("flagged_no", 
						     toggleWidgetClass,       
						     sw->flags.form,          
						     warg, n); n = 0;

  
  XtSetArg(warg[n], XtNradioGroup, sw->flags.flaggedToggle); n++;
  XtSetArg(warg[n], XtNbitmap, emptyPixmap); n++;
  XtSetArg(warg[n], XtNradioData, XB_FLAGGED_EITHER ); n++;
  XtCreateManagedWidget("flagged_both", 
			toggleWidgetClass,           
			sw->flags.form,                
			warg, n); n = 0;
}

static void createDatesPanel( sw )
     SEARCHWINDOW *sw;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;

    XtTranslations sincetr, beforetr, ontr;

    sincetr = XtParseTranslationTable( sinceTranslations );
    beforetr = XtParseTranslationTable( beforeTranslations );
    ontr = XtParseTranslationTable( onTranslations );
    
    sw->dates.form =  XtCreateManagedWidget("search_dates",
					    formWidgetClass,
					    sw->panes,
					    warg,
					    n); n = 0;
    
    XtCreateManagedWidget("since_label",
			  labelWidgetClass,
			  sw->dates.form,
			  warg, n); n = 0;

    XtCreateManagedWidget("before_label",
			  labelWidgetClass,
			  sw->dates.form,
			  warg, n); n = 0;

    XtCreateManagedWidget("on_label",
			  labelWidgetClass,
			  sw->dates.form,
			  warg, n); n = 0;

    XtSetArg(warg[n], XtNstring, ""); n++;
    sw->dates.sinceText = XtCreateManagedWidget("since_text",
					       asciiTextWidgetClass,
					       sw->dates.form,
					       warg,
					       n); n = 0;
    XtOverrideTranslations( sw->dates.sinceText, sincetr );

    XtSetArg(warg[n], XtNstring, ""); n++;
    sw->dates.beforeText = XtCreateManagedWidget("before_text",
					       asciiTextWidgetClass,
					       sw->dates.form,
					       warg,
					       n); n = 0;
    XtOverrideTranslations( sw->dates.beforeText, beforetr );

    XtSetArg(warg[n], XtNstring, ""); n++;
    sw->dates.onText = XtCreateManagedWidget("on_text",
					     asciiTextWidgetClass,
					     sw->dates.form,
					     warg,
					     n); n = 0;
    XtOverrideTranslations( sw->dates.onText, ontr );
}

  static void 
searchQuit( w, bw, e )
     Widget w;
     Widget bw;
     XEvent* e;
{
    BrowserEntry *be;
    MailBox* ms = getStatusFromBrowser(bw);
    be = getActiveBrowser(ms);

    if ( be->searchWin )
    {
	XtPopdown(be->searchWin->shell);
    }
}

static void clear_search_criteria(w, bw, event)
     Widget w;
     Widget bw;
     XEvent *event;
{
    Arg warg[1];
    BrowserEntry *be;
    SEARCHWINDOW *sw;
    MailBox* ms = getStatusFromBrowser(bw);
    be = getActiveBrowser(ms);
    sw = be->searchWin;

    if(sw)
    {

#ifdef SEARCH_CLEAR_CONFIRM
	if ( XbConfirm(sw->shell,
		       "Do you really want to\nclear the search\ncriteria?", 
		       TRUE) == TRUE )
#endif

	{
	    XtSetArg(warg[0], XtNstring, "");
	    
	    XtSetValues(sw->bodyText, warg, 1);
	    XtSetValues(sw->dates.onText, warg, 1);
	    XtSetValues(sw->dates.sinceText, warg, 1);
	    XtSetValues(sw->dates.beforeText, warg, 1);
	    XtSetValues(sw->textText, warg, 1);
	    XtSetValues(sw->fromText, warg, 1);
	    XtSetValues(sw->bccText, warg, 1);
	    XtSetValues(sw->ccText, warg, 1);
	    XtSetValues(sw->toText, warg, 1);
	    XtSetValues(sw->subjectText, warg, 1);
	    XtSetValues(sw->keywordText, warg, 1);
	    
	    XawToggleUnsetCurrent(sw->flags.seenToggle);
	    XawToggleUnsetCurrent(sw->flags.deletedToggle);
	    XawToggleUnsetCurrent(sw->flags.recentToggle);
	    XawToggleUnsetCurrent(sw->flags.answeredToggle);
	    XawToggleUnsetCurrent(sw->flags.flaggedToggle);
	}
    }
}
		      
static char* makeSearchFilter( sw )
     SEARCHWINDOW *sw;
{
    static char* add_select_text(/* Widget, char**, char* */);
    static char* add_toggle_data(/* Widget, char**, char* */);

    char filter[TMPLEN];
    char *ftemp = filter;
    
    char *temp = NULL;
    
    char *putString(/* char*, char* */);
    
    memset(filter, 0, TMPLEN);

    ftemp = putString(ftemp, ALL);

    if((add_select_text(sw->dates.onText, &ftemp, "ON")) == NULL)
    {
	add_select_text(sw->dates.sinceText, &ftemp, "SINCE");
	add_select_text(sw->dates.beforeText, &ftemp, "BEFORE");
    }
    add_select_text(sw->bodyText,   &ftemp, "BODY");
    add_select_text(sw->textText,   &ftemp, "TEXT");
    add_select_text(sw->fromText,   &ftemp, "FROM");
    add_select_text(sw->bccText,    &ftemp, "BCC");
    add_select_text(sw->ccText,     &ftemp, "CC");
    add_select_text(sw->toText,     &ftemp, "TO");
    add_select_text(sw->subjectText,&ftemp, "SUBJECT");
    add_select_text(sw->keywordText,&ftemp, "KEYWORD");

    if((add_toggle_data(sw->flags.seenToggle, &ftemp, "SEEN" )) == NULL)
	add_toggle_data(sw->flags.unseenToggle,&ftemp, "UNSEEN");
    if((add_toggle_data(sw->flags.deletedToggle, &ftemp, "DELETED")) == NULL)
	add_toggle_data(sw->flags.undeletedToggle, &ftemp, "UNDELETED");
    if((add_toggle_data(sw->flags.recentToggle, &ftemp, "RECENT" )) == NULL)
	add_toggle_data(sw->flags.unrecentToggle, &ftemp, "OLD" );
    if((add_toggle_data(sw->flags.answeredToggle, &ftemp, "ANSWERED")) == NULL)
	add_toggle_data(sw->flags.unansweredToggle, &ftemp, "UNANSWERED" );
    if((add_toggle_data(sw->flags.flaggedToggle, &ftemp, "FLAGGED" )) == NULL)
	add_toggle_data(sw->flags.unflaggedToggle, &ftemp, "UNFLAGGED" );

    temp = cpystr(filter);
    return(temp);
}  



  char*
putString(dest, src)
     char *dest;
     char *src;
{
  while (*dest++ = *src++)
    ; /* EMPTY */
  --dest;

  return(dest);
}
     

static char*
add_select_text(w, string, key)
     Widget w;
     char **string;
     char *key;
{
    char *temp = cpystr(XbTextCopyBuffer(w));

    if (temp && *temp)
    {
	sprintf(*string," %s \"%s\"", key, temp);
	*string += strlen(*string);

	XtFree(temp);			/* Bill Yeager - 25 Nov 92 */
	return *string;
    }
    else
    {
	return NULL;
    }
}

static char*
add_toggle_data(w, string, key)
     Widget w;
     char **string;
     char *key;
{
    Boolean toggle_state;
    Arg warg[1];
    
    XtSetArg(warg[0], XtNstate, &toggle_state);
    XtGetValues(w, warg, 1);

    if(toggle_state == TRUE)
    {
	sprintf(*string," %s", key);
	*string += strlen(*string);

	return *string;
    }
    else
    {
	return NULL;
    }
}

static void cleanup(ms,be)
     MailBox      *ms;
     BrowserEntry *be;
{
    XbListSelection *ret;
    int i;
    char temp[TMPLEN];

    sprintf(temp,"Found %d messages", ms->num_found);
    mm_log(temp, NIL);
    
    if (ms->num_found > 0)
    {
	be = getActiveBrowser(ms);
	if(be)
	{
	    i = ms->num_found;
	    ret = ximap_newlistmap(ms->num_found);
	    
	    while (i--)
		ret->entries[i].index = ms->found[i]-1;
	    
	    XbBrowserHighlight(be->browser, ret);
	}
	
	check_menus(NULL, be, NULL);
	ms->num_found = 0;
	XbListFreeSelection(ret);	/* Bill Yeager - 25 Nov 92 */
    }
}

static void doSearch(w, bw, e)
     Widget w;
     Widget bw;
     XEvent* e;
{
    char *filter;
    BrowserEntry *be;
    SEARCHWINDOW *sw;
    MailBox* ms = getStatusFromBrowser(bw);
    be = getActiveBrowser(ms);
    sw = be->searchWin;
    
    filter = makeSearchFilter( sw );

    if (ISINACTIVE(ms->status))
    {
	ms->status |= SEARCHSELECT;
	ms->status |= SEARCHING;

	if ( filter != NULL )
	{
	    mail_search( ms->mailbox, filter );
	    cleanup(ms, be);
	}
    }

    ms->status &= ~SEARCHING;
    ms->status &= ~SEARCHSELECT;
    searchQuit( w, bw, e );
}

static void doBrowse(w, bw, e)
     Widget w;
     Widget bw;
     XEvent* e;
{
    SEARCHWINDOW *sw;
    char *filter;
    BrowserEntry *be;
    MailBox* ms = getStatusFromBrowser(bw);
    be = getActiveBrowser(ms);
    sw = be->searchWin;

    filter = makeSearchFilter(sw);
    
    if (ISINACTIVE(ms->status))
    {
	ms->status |= SEARCHBROWSE;
	ms->status |= SEARCHING;
	
	if ( filter != NULL )
	{
	    mail_search( sw->ms->mailbox, filter );
	    cleanup(ms, be);
	}
	
	ms->status &= ~SEARCHING;
	ms->status &= ~SEARCHBROWSE;
	
	BrowseSelectedHeaders(w, be, e);
	XbBrowserUnHighlight( be->browser );
    }
    else
    {
	ms->status &= ~SEARCHING;
	ms->status &= ~SEARCHBROWSE;
    }
    
    searchQuit( w, bw, e );
}


/*************************************************

  Actions

*************************************************/

static void swinSearchAct(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);

    doSearch( w, swin->be->browser, e );
}

static void swinBrowseAct(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    
    doBrowse( w, swin->be->browser, e );
}

static void swinQuitAct(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    
    searchQuit( w, swin->be->browser, e );
}

static void swinBeep(w, e)
     Widget w;
     XEvent *e;
{
    /* SEARCHWINDOW * swin = getSearchWinFromWidget(w); */
}

static void gotoSearchFrom(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->fromText );
    XawTextDisplayCaret(swin->fromText , TRUE);
    swin->last_focus = swin->fromText ;
}

static void gotoSearchTo(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->toText );
    XawTextDisplayCaret(swin->toText , TRUE);
    swin->last_focus = swin->toText ;
}

static void gotoSearchSubject(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->subjectText );
    XawTextDisplayCaret(swin->subjectText , TRUE);
    swin->last_focus = swin->subjectText ;
}

static void gotoSearchText(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->textText );
    XawTextDisplayCaret(swin->textText , TRUE);
    swin->last_focus = swin->textText ;
}

static void gotoSearchCc(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->ccText );
    XawTextDisplayCaret(swin->ccText , TRUE);
    swin->last_focus = swin->ccText ;
}

static void gotoSearchBody(w, e)
     Widget w;
     XEvent *e;
{
     SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->bodyText );
    XawTextDisplayCaret(swin->bodyText , TRUE);
    swin->last_focus = swin->bodyText ;
}

static void gotoSearchKeyword(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->keywordText );
    XawTextDisplayCaret(swin->keywordText , TRUE);
    swin->last_focus = swin->keywordText ;
}

static void gotoSearchBcc(w, e)
     Widget w;
     XEvent *e;
{
     SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->bccText );
    XawTextDisplayCaret(swin->bccText , TRUE);
    swin->last_focus = swin->bccText ;
}

static void gotoSearchSince(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->dates.sinceText );
    XawTextDisplayCaret(swin->dates.sinceText , TRUE);
    swin->last_focus = swin->dates.sinceText ;
}

static void gotoSearchOn(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->dates.onText );
    XawTextDisplayCaret(swin->dates.onText , TRUE);
    swin->last_focus = swin->dates.onText ;
}

static void gotoSearchBefore(w, e)
     Widget w;
     XEvent *e;
{
    SEARCHWINDOW * swin = getSearchWinFromWidget(w);
    XawTextDisplayCaret( swin->last_focus , FALSE );
    XtSetKeyboardFocus( swin->shell, swin->dates.beforeText );
    XawTextDisplayCaret(swin->dates.beforeText , TRUE);
    swin->last_focus = swin->dates.beforeText ;
}


