/* -*- C -*-
 *
 * Program:	ximap
 * File:        mailbox.c -- functions for dealing with the mailbox itself (as
 *                           opposed to any particular view or display of the mailbox)
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <Client/osdep.h>
#include <Client/mail.h>
#include <Client/misc.h>
 
#include <netdb.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>

#include <stdio.h>
#include <memory.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>

#include <Xb/Browser.h>

#include "structures.h"
#include "modal.h"

#include "message.h"
#include "globals.h"
#include "buttons.h"
#include "resources.h"
#include "util.h"
#include "browserutil.h"
#include "addrutil.h"
#include "ximap.h"
#include "mailbox.h"
#include "readwin.h"
#include "compose.h"
#include "search.h"

void mailboxBeforeActions(
#if defined(__cplusplus) || defined(__STDC__)
			  MailBox *ms
#endif
);

void mailboxAfterActions(
#if defined(__cplusplus) || defined(__STDC__)
			  MailBox *ms
#endif
);

void check_menus();

void mbx_MarkDeleted();
void mbx_MarkUndeleted();
void mbx_MarkSeen();
void mbx_MarkUnseen();
void mbx_MarkRecent();
void mbx_MarkOld();
void mbx_MarkFlagged();
void mbx_MarkUnflagged();
void mbx_MarkAnswered();
void mbx_MarkUnanswered();

void copyMessages();
void printMessages();
void moveMessages();
void expungeMessages();
void readMessages();
void findMessages();

void readOne();
void clearKeyword();
void setKeyword();

void flagUp();
void flagDown();
void setFlag();
void clearFlag();
void setupKeywords();

void ComposeNew();
void ReplyToSender();
void ReplyToAll();
void checkCallback();

void ForwardMessages();
void RemailMessage();
void ToggleHeaderMode();
void toggleZoom();
void BrowseSelectedHeaders();
void clear_browser();

void setBrowserSize();
static void setFilterHeaders();

#define NOSELECTION ((unsigned int) 1 << 1)

static ButtonStruct  mbx_clear_menu[30];
static ButtonStruct  mbx_set_menu[30];

ButtonStruct  mbx_keywords_menu[] =
{
  { "clear",NULL, mbx_clear_menu, NOSELECTION},
  { "set",  NULL, mbx_set_menu, NOSELECTION},
  NULL,
};

ButtonStruct mbx_flags_menu[] = 
{

  { "seen",      mbx_MarkSeen     ,  NULL, NOSELECTION },
  { "unseen",    mbx_MarkUnseen   ,  NULL, NOSELECTION },
  { "recent",    mbx_MarkRecent ,    NULL, NOSELECTION },
  { "old",       mbx_MarkOld,        NULL, NOSELECTION },
  { "answered",  mbx_MarkAnswered ,  NULL, NOSELECTION },
  { "unanswered",mbx_MarkUnanswered, NULL, NOSELECTION },
  { "flagged",   mbx_MarkFlagged,    NULL, NOSELECTION },
  { "unflagged", mbx_MarkUnflagged,  NULL, NOSELECTION },
  NULL,
};

ButtonStruct mbx_reply_menu[] =
{
  { "sender",ReplyToSender, NULL,  NOSELECTION  },
  { "all",   ReplyToAll,    NULL,  NOSELECTION },
  NULL,
};


ButtonStruct mbx_response_menu[] = 
{
  { "Reply",   NULL,            mbx_reply_menu,  NOSELECTION   },
  { "Remail",  RemailMessage,   NULL,  NOSELECTION  },  
  { "Forward", ForwardMessages, NULL, NOSELECTION  },
  NULL,
};

ButtonStruct mbx_copy_menu[] = 
{
  { "Copy", copyMessages, NULL,  NOSELECTION },
  { "Move", moveMessages, NULL, NOSELECTION },
  NULL,
};

ButtonStruct mbx_delete_menu[] = 
{
  { "Delete",   mbx_MarkDeleted,     NULL, NOSELECTION  },
  { "UnDelete", mbx_MarkUndeleted,   NULL, NOSELECTION  },
  { "Expunge",  expungeMessages,     NULL,   NEVER  },
  NULL,
};

ButtonStruct mainButtons[] = 
{
  { "Quit",    browserQuit,  NULL, NEVER  },
  { "Check",   checkCallback,  NULL,  NEVER  },
  { "Read",    readMessages, NULL,  NOSELECTION  },
  { "Delete",  NULL, mbx_delete_menu,  NEVER  },
  { "Copy",    NULL,   mbx_copy_menu,  NOSELECTION   },
  { "Compose", ComposeNew, NULL,  NEVER  },
  { "Reply", NULL, mbx_response_menu , NOSELECTION },
  { "Find", findMessages, NULL,  NEVER  },
  { "Flags", NULL, mbx_flags_menu,  NOSELECTION  },     
  { "Keywords",NULL, mbx_keywords_menu,  NOSELECTION  },
  { "Print", printMessages, NULL,  NEVER  },
  { "BrowseSelection", BrowseSelectedHeaders, NULL, NOSELECTION  },
  { "Zoom", toggleZoom, NULL,  NOSELECTION  },
  { "Clear", clear_browser, NULL, NOSELECTION },
  NULL,
};

ButtonStruct browserButtons[] = 
{
  { "Quit",    browserQuit,  NULL, NEVER  },
  { "Read",    readMessages, NULL,  NOSELECTION  },
  { "Delete",  NULL, mbx_delete_menu,  NEVER  },
  { "Copy",    NULL,   mbx_copy_menu,  NOSELECTION   },
  { "Compose", ComposeNew, NULL,  NEVER  },
  { "Reply",   NULL, mbx_response_menu , NOSELECTION },
  { "Find",    findMessages, NULL,  NEVER  },
  { "Flags",   NULL, mbx_flags_menu,  NOSELECTION  },     
  { "Keywords",NULL, mbx_keywords_menu,  NOSELECTION  },
  { "Print",   printMessages, NULL,  NEVER  },
  { "BrowseSelection", BrowseSelectedHeaders, NULL, NOSELECTION  },
  NULL,
};

static void HighlightInitial( /* MAILSTATUS* */ );
void checkScrollPosition( /* Widget, void*, void* */ );
void checkJumpPosition( /* Widget, void*, void* */ );
static void CreateMailboxFlag( /* MAILSTATUS* */ );

void mainUndeleteAct();
void mainDeleteAct();
void mainExpungeAct();
void mainReadAct();
void mainCopyAct();
void mainMoveAct();
void mainQuitAct();
void mainComposeAct();
void mainForwardAct();
void mainRemailAct();
void mainReplySenderAct();
void mainReplyAllAct();
void mainZoomAct();
void mainFindAct();
void mainBrowseAct();
void mainPrintAct();
void mainFlagAct();
void mainUnflagAct();
void mainClearAct();

static XtActionsRec mailbox_actions[] = {
  { "main_undelete",     mainUndeleteAct },
  { "main_delete",       mainDeleteAct },
  { "main_unflag",       mainUnflagAct },
  { "main_flag",         mainFlagAct },
  { "main_expunge",      mainExpungeAct },
  { "main_read",         mainReadAct },
  { "main_copy",         mainCopyAct },
  { "main_move",         mainMoveAct },
  { "main_quit",         mainQuitAct },
  { "main_compose",      mainComposeAct },
  { "main_forward",      mainForwardAct },
  { "main_remail",       mainRemailAct },
  { "main_replysender",  mainReplySenderAct },
  { "main_replyall",     mainReplyAllAct },
  { "main_zoom",         mainZoomAct },
  { "main_search",       mainFindAct },
  { "main_browse",       mainBrowseAct },
  { "main_print",        mainPrintAct },
  { "main_clear",        mainClearAct },
};

static void updateStatus();

void makeKeywordsMenu(/* MAILSTATUS *ms */);

extern Widget toplevel;
extern Widget telemetry;
extern XtAppContext app_context;
extern AppResources res;
extern MAILSESSION id;
extern int updateMailbox;

int imp_log = FALSE;
/*************************************************


  Initialization routines


*************************************************/

void addMailboxActions(ac)
     XtAppContext ac;
{
    XtAppAddActions(ac, mailbox_actions, XtNumber(mailbox_actions));
}

Widget createMailbox(ms)
     MailBox* ms;
{
    XbListSelectionPtr temp = NULL;

    char title[TMPLEN];
    Arg warg[16];
    int n = 0;
    void makeDefaultMapping();
    void createHeaders();

    checkMailBox( NULL, ms,  NULL);
    
    createHeaders( ms );
    makeDefaultMapping(ms, FALSE);

    setupKeywords(ms);

    if(ms->defaultMapping)
    {
	ms->browserList = ximap_newBrowserEntry();
	ms->browserList->ms = ms;

	if( ms->nmsgs != 1)
	    sprintf(title,"%s -- %d msgs",ms->mailboxName, ms->nmsgs);
	else 
	    sprintf(title,"%s -- 1 msg",ms->mailboxName);
	    
	XtSetArg(warg[n], XtNtitle, title); n++;
	XtSetArg(warg[n], XtNiconName, ms->mailboxName); n++;
	XtSetArg(warg[n], XtNallowShellResize, TRUE); n++;
	ms->mailWindow = ms->browserList->shell = 
	    XtCreatePopupShell("mailbox_shell", 
			       topLevelShellWidgetClass,
			       toplevel, warg, n); n = 0;
	
	ms->browserList->panes = XtCreateManagedWidget("panes",
						       panedWidgetClass,
						       ms->browserList->shell,
						       warg, n); n = 0;
	
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	XtSetArg(warg[n], XtNresizeToPreferred, TRUE); n++; 
	ms->browserList->buttons = XtCreateManagedWidget("buttons", 
							 boxWidgetClass,
							 ms->browserList->panes,
							 warg, n); n = 0;

        ms->browserList->order = copyMapping( ms->defaultMapping );
        ms->browserList->size = ms->nmsgs;
	ms->browserList->active = TRUE;
	ms->browserList->main = TRUE;

	/* Do NOT free temp ... It's done in List.c */
	temp = XbListCopySelection(ms->defaultMapping);

	XtSetArg(warg[n], XtNinitialViewHt, ms->view); n++;
	XtSetArg(warg[n], XtNlistPosition, ms->first_header); n++;
	XtSetArg(warg[n], XtNlist, temp); n++;
	ms->browserList->browser = XtCreateManagedWidget("mailbox_browser",
		      			                 XbbrowserWidgetClass,
					                 ms->browserList->panes,
					                 warg, n); n = 0;
	
	XtAddCallback(ms->browserList->browser, XtNsingleClick,
		      check_menus,ms->browserList);
	XtAddCallback(ms->browserList->browser, XtNdoubleClick,
		      readOne,ms->browserList);
	XtAddCallback(ms->browserList->browser, XtNscrollProc, 
		      checkScrollPosition,ms->browserList);
	XtAddCallback(ms->browserList->browser, XtNjumpProc, 
		      checkJumpPosition, ms->browserList);
	
	ms->browserList->states = createButtons(ms->browserList->buttons, 
						ms->browserList,
						mainButtons);

	ms->background_proc = XtAppAddWorkProc(app_context,      
					       BackgroundFetch,  
					       ms);

	if ((res.flag_no_mailboxes == FALSE)
	    && (res.flag_all_mailboxes == TRUE
	        || strstr(ms->mailboxName, "INBOX")))
	{
	    CreateMailboxFlag( ms );
	}
	else
	{
	    ms->flagShell = NULL;
	}

	HighlightInitial( ms );
	setBrowserSize(ms->browserList);
	XtPopup(ms->browserList->shell, XtGrabNone);
	setFilterHeaders(ms, ms->browserList, res.filter_headers);
	XtSetKeyboardFocus(ms->browserList->shell, ms->browserList->browser);
	
	if ( ms->new > 0 )
	    flagUp( ms );
	else if (ms->flagShell)
	    flagDown( ms );
	    
	if ( ms->flagShell )
	    XtPopup(ms->flagShell, XtGrabNone);
    }
    else
    {
	return NULL;
    }
    
    if( ms->new )
	checkButtonStates( ms->browserList->states, DEFAULTSTATE );
    else
	checkButtonStates( ms->browserList->states, NOSELECTION );

    return (ms->browserList->browser);		   
}

void createHeaders( ms )
     MailBox*  ms;
{
    MessageNode*  node;
    
    char header[TMPLEN];
    int count = 0;
    Boolean highlight = FALSE;

    char *userName = getUsername();
    
    ms->virtual_size = ms->nmsgs;
    ms->nzoomed = 0;
    ms->new = ms->mailbox->recent;
    ms->list_size = ms->nmsgs + 20;
    ms->view = res.initial_view;


    if (ms->virtual_size > 0)
    {
	ms->messages = 
	    (MessageNode**)XtMalloc(ms->list_size*sizeof(MessageNode*));
	ms->last_message = ms->messages + ms->nmsgs - 1;
	ms->found = (int*) XtMalloc ((ms->nmsgs+1) * sizeof(int));
	
	highlight = res.highlight_new;

	ms->actual_size = (ms->new > ms->view)?(ms->new):(ms->view);
	ms->first_header = ((ms->virtual_size)-(ms->actual_size));

	if ( ms->first_header < 0 )
	    ms->first_header = 0;

	if ( highlight == TRUE )
	{
	    ms->start_highlight = ms->first_header ;
	    if (ms->actual_size == ms->view)
		ms->start_highlight = ((ms->virtual_size)-(ms->new));
	}
	
	for ( count = 0; count < ms->virtual_size; count++ )
	{
	    node = ximap_newnode();
	    node->msgno = count + 1;
	    if ( count < ms->first_header )
	    {
		node->elt = NULL;
		node->env = NULL;
		node->body =  NULL;
		header[0] = ' ';
		header[1] = '\0';
	    }
	    else
	    {
		memset( header, 0, TMPLEN );
		node->env = mail_fetchstructure(  ms->mailbox, count+1, &node->body);
		node->elt = mail_elt( ms->mailbox, count+1);      
		headerline( header, ms->mailbox, node);
	    }

	    if(node->env && node->env->to)
	    {
		int to, cc;
		to = onAddressList( userName, node->env->to );
		cc = onAddressList( userName, node->env->cc );
		if( to || cc )
		    node->toMe = TRUE;
	    }
	    node->header_line = cpystr(header);
	    ms->messages[count] = node;
	}

	while (count < ms->list_size)
	    ms->messages[count++] = NULL;

    }
    else
    {
	ms->messages = ms->last_message = NULL;
	ms->actual_size = ms->virtual_size = 0;
	ms->first_header = 0;
    }
}

void makeDefaultMapping( ms, disposeOldmap )
     MailBox*  ms;
     Boolean disposeOldmap;
{
    int count, bug = 0;
    register MessageNode **node;

    /*
     * Dispose only if caller requests to do so */
    if (disposeOldmap)
       XbListFreeSelection(ms->defaultMapping);

    if (ms->virtual_size > 0)
    {
        char tmp[512];

	ms->defaultMapping = ximap_newlistmap(0);
	ms->defaultMapping = ximap_newlistmap(ms->virtual_size);

/*	sprintf(tmp, "\nMakeDefaultMapping @ 0x%x, size %d(%d) bug %d",
 *		ms->defaultMapping, ms->defaultMapping->size,
 *		ms->virtual_size, bug);
 *
 *	always_log(tmp);
 */
	for ( count = 0, node = ms->messages; 
	     count < ms->virtual_size; count++, node++ )
	{
	    ms->defaultMapping->entries[count].entry = 
		cpystr(ms->messages[count]->header_line);

	    ms->defaultMapping->entries[count].index = count;
	    ms->defaultMapping->entries[count].bold = ms->messages[count]->toMe;
	    (*node)->inzoom = FALSE;
	}
    }
    else
    {
        char tmp[512];

	ms->defaultMapping = ximap_newlistmap(0);
	ms->defaultMapping = ximap_newlistmap(ms->virtual_size);

/*	sprintf(tmp, "\nMakeDefaultMapping @ 0x%x, size %d",
 *		ms->defaultMapping, ms->defaultMapping->size);
 *	always_log(tmp);
 */
    }
}

/* Called if a new message(s) has arrived while we are zoomed.
 * Need to add the new one(s) to the zoom map */
void makeUpdatedZoomMapping( ms, be, plus, disposeOldmap )
     MailBox *ms;
     BrowserEntry *be;
     int plus;
     Boolean disposeOldmap;
{
    register MessageNode* *node;
    int count = 0;
    int nzoomed = ms->nzoomed + plus;
    char tmp[512];

    sprintf(tmp, "\n MakeUpdatedZoomMapping, nzoomed %d", nzoomed);
    ximap_log(tmp);

    /*
     * Dispose only if caller requests to do so */
    if (disposeOldmap)
       XbListFreeSelection(ms->zoomMapping);

    if (nzoomed > 0)
    {
        ms->zoomMapping = ximap_newlistmap(nzoomed);
	ms->nzoomed = nzoomed;		/* Bill Yeager - 30 Nov 92 */
	for (count = 0, node = ms->messages; 
	     node <= ms->last_message; 
	     node+=1 )
	{
	    int item = (*node)->msgno-1;

	    /* (zoomed) OR ((NOT Zoomed) and NEW) */
	    if( (*node)->elt && (((*node)->inzoom == TRUE) ||
			         ( !((*node)->inzoom == TRUE) 
				  && (*node)->elt->recent
				  && !(*node)->elt->seen )) )
	    {
		ms->zoomMapping->entries[count].entry = cpystr((*node)->header_line);
		ms->zoomMapping->entries[count].index = (*node)->msgno-1;
		ms->zoomMapping->entries[count].bold = (*node)->toMe;
		(*node)->inzoom = TRUE;
		count++;
	    }
	}
    }
    else
    {
	ms->zoomMapping = ximap_newlistmap(0);
    }

    ximap_log("\n SORTANT: MakeUpdatedZoomMapping");
}



void makeZoomMapping( ms, be, disposeOldmap )
     MailBox *ms;
     BrowserEntry *be;
     Boolean disposeOldmap;
{
    register MessageNode* *node;
    int count = 0;
    int nzoomed = ms->nzoomed;
    char tmp[512];

    sprintf(tmp, "\n MakeZoomMapping, nzoomed %d", nzoomed);
    ximap_log(tmp);

    /*
     * Dispose only if caller requests to do so */
    if (disposeOldmap)
       XbListFreeSelection(ms->zoomMapping);

    if (nzoomed > 0)
    {
        ms->zoomMapping = ximap_newlistmap(nzoomed);
	
	for (count = 0, node = ms->messages; 
	     node <= ms->last_message; 
	     node+=1 )
	{
	    int item = (*node)->msgno-1;

	    if( (*node)->elt && XbBrowserIsHighlighted(be->browser, item) )

	    {
		ms->zoomMapping->entries[count].entry = cpystr((*node)->header_line);
		ms->zoomMapping->entries[count].index = (*node)->msgno-1;
		ms->zoomMapping->entries[count].bold = (*node)->toMe;
		(*node)->inzoom = TRUE;
		count++;
	    }
	}
    }
    else
    {
	ms->zoomMapping = ximap_newlistmap(0);
    }
    ximap_log("\n SORTANT: MakeZoomMapping");
}

static void HighlightInitial(ms)
     MailBox* ms;
{
    int i = 0;
    int start = ms->start_highlight;
    int number = ms->virtual_size - ms->start_highlight;

    XbListSelectionPtr map = ximap_newlistmap(number);
    
    if ( res.highlight_new == TRUE )
    {
	if( ms->nmsgs &&  (i = number)> 0 )
	{
	    while ( i-- )
	    {
		map->entries[i].index = start + i;
		map->entries[i].entry = NULL;
	    }
	    
	    XbBrowserHighlight( ms->browserList->browser, map);
	}
    }

    XbListFreeSelection(map);
}

void setupKeywords(ms)
     MailBox*  ms;
{
    char **keywords = ms->mailbox->user_flags;
    char **temp = keywords;
    
    int  num_keys = 0;
    
    while ( num_keys < NUSERFLAGS && *temp++ != NULL )
	++num_keys;
    
    temp = keywords;
    
    if (*temp)
    {
	mbx_clear_menu[num_keys].name = NULL;
	mbx_set_menu[num_keys].name = NULL;

	while(num_keys--)
	{
	    mbx_clear_menu[num_keys].name = cpystr(temp[num_keys]);
	    mbx_clear_menu[num_keys].function = clearKeyword;
	    mbx_set_menu[num_keys].name = cpystr(temp[num_keys]);
	    mbx_set_menu[num_keys].function = setKeyword;
	    mbx_clear_menu[num_keys].state = NEVER;
	    mbx_set_menu[num_keys].state = NEVER;
	    mbx_clear_menu[num_keys].menu = NULL;
	    mbx_set_menu[num_keys].menu = NULL;
	}
    }
    else
    {
	mbx_clear_menu[0].name = NULL;
	mbx_clear_menu[0].function = NULL;
	mbx_set_menu[0].name = NULL;
	mbx_set_menu[0].function = NULL;
	mbx_clear_menu[0].menu = NULL;
	mbx_set_menu[0].menu = NULL;

	mbx_keywords_menu[0].state = ~NEVER;
	mbx_keywords_menu[1].state = ~NEVER;
    }

}


/*************************************************


  Browser stuff


*************************************************/


void fetchNodeInfo( ms, node )
     MailBox*     ms;
     MessageNode* node;
{
    char header[TMPLEN];
    char *old = NULL;
    XbListEntryPtr item = (XbListEntryPtr)XtMalloc(sizeof(XbListEntry));

    node->env = mail_fetchstructure(  ms->mailbox, node->msgno, &node->body );  
    node->elt = mail_elt( ms->mailbox, node->msgno );     
    
    if(node->env && node->env->to)
    {
	if(   onAddressList( res.mail_name, node->env->to )
	   || onAddressList( res.mail_name, node->env->cc ))
	    node->toMe = TRUE;
    }

    headerline( header, ms->mailbox, node);
    old = node->header_line;
    node->header_line = cpystr(header);
    
    item->entry = cpystr(node->header_line);
    item->bold = node->toMe;
    item->index = node->msgno-1;

    XbBrowserReplaceString(ms->browserList->browser, item);
    XtFree(old);
    ms->actual_size++;
}

void AddBrowser( bw, selection )
     Widget  bw;
     XbListSelectionPtr selection;
{
    MailBox* ms;
    
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    BrowserEntry *be = NULL;
    BrowserEntry *temp;
    int view = selection->size;
    
    ms = getStatusFromBrowser(bw);

    temp = ms->browserList;

    fetchMapping( ms, selection );

    if( view > ms->view )
	view = ms->view;
    
    if(temp)
	while( temp->next )
	    temp = temp->next;
    
    be = ximap_newBrowserEntry();
    be->ms = ms;
    be->currentMap = selection;
    be->order = copyMapping( selection );

    if(temp)
    {
	be->next = temp->next;
	temp->next = be;
    }
    else
    {
	be->next = NULL;
	ms->browserList = be;
    }
	
    be->prev = temp;
    be->main = FALSE;
    be->size = selection->size;
    
    XtSetArg(warg[n], XtNallowShellResize, TRUE); n++;
    be->shell = XtCreatePopupShell("browser_shell", 
				   topLevelShellWidgetClass,
				   ms->mailWindow, warg, n); n = 0;
	
    be->panes = XtCreateManagedWidget("panes",
				      panedWidgetClass,
				      be->shell,
				      warg, n); n = 0;

    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNresizeToPreferred, TRUE); n++;
    be->buttons = XtCreateManagedWidget("buttons", 
				 boxWidgetClass,
				 be->panes,
				 warg, n); n = 0;

    
    XtSetArg(warg[n], XtNinitialViewHt, view); n++;
    XtSetArg(warg[n], XtNlist, selection); n++;
    XtSetArg(warg[n], XtNinitialHighlight, TRUE); n++;
    be->browser = XtCreateManagedWidget("browser",
				 XbbrowserWidgetClass,
				 be->panes,
				 warg, n); n = 0;
    
    XtAddCallback(be->browser, XtNsingleClick, check_menus, be);
    XtAddCallback(be->browser, XtNdoubleClick,readOne,be);

    be->states = createButtons(be->buttons, 
			       be,
			       browserButtons);

    setBrowserSize(be);

    XtPopup( be->shell, XtGrabNone );

    XbBrowserHighlight( be->browser, selection );
    checkButtonStates(be->states, DEFAULTSTATE );

    XtSetKeyboardFocus(be->shell, be->browser);
}
/*************************************************


  Callbacks


*************************************************/


 /*
  * Clear ALL highlighted messaages */

void clear_browser(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
  XbBrowserClear(be->browser);
}
  
 /*
  * Zoom/Unzoom */

void toggleZoom(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    Arg warg[1];
    int view = 0;
    int position = 0;

    MailBox* ms = be->ms;
    XbListSelectionPtr dummy= NULL;
    int nsel;
    char tmp[512];

    /* Find if anything is selected  */
    nsel = ms->nzoomed = countSelected(be->browser);
    if (nsel <= 0 && !ms->zoom)
      return;		/* WJY -- Rien a faire -- Nothing is selected */

  /*  imp_log =  TRUE;                    /* turn on printing */
    sprintf(tmp, "\n\nZOOM:");
    ximap_log(tmp);

    setActiveBrowser(ms, be->browser);

    if( ms->zoom == FALSE )
    {					/* Not yet zoomed */
        /* changeItemsWithHighlight will dipose of the old mapping
	 * by the means of calls to XbBrowserChangetimes ... */
	makeZoomMapping(ms, be, FALSE);
	be->order = copyMapping( ms->zoomMapping );
	be->size = ms->nzoomed;
	/* During the following  call the old copy of ms->defaultMapping
	 * is disposed of by XbBrowserChangeItems. */
	ms->defaultMapping = NULL;
	changeItemsWithHighlight(be->browser, 
				 ms->zoomMapping, 
				 ms->zoomMapping, 
				 XMS_HLTREPLACE );
	XbBrowserSetPosition(be->browser, position);
	ms->zoom = TRUE;
	XtSetArg(warg[0], XtNlabel, "UnZoom");
    }
    else
    {					/* already zoomed */
        makeDefaultMapping(ms, FALSE);
	be->order = copyMapping( ms->defaultMapping );
	be->size = ms->nmsgs;

	ms->zoom = FALSE;
	dummy = ximap_newlistmap(0);
	/* As a result of the following call ms->zoomMapping will
         * be freed automatically. So, we set the pointer to a NULL.
	 * Note that a pointer to this mapping is stashed away in
	 * the be->browser structure and that value is released and
	 * replaced by the ms->defaultMapping!
         *                    Bill Yeager - 24 Nov. 92 */
	ms->zoomMapping = NULL;
	changeItemsWithHighlight(be->browser, 
				 ms->defaultMapping, 
				 dummy, 
				 XMS_HLTRETAIN );
	
	view = XbBrowserGetView(be->browser);
	if( ms->nmsgs > view )
	  position = ms->nmsgs - view;
	
	XbBrowserSetPosition(be->browser, position);
	XtSetArg(warg[0], XtNlabel, "Zoom");
    }
    XtSetValues( w, warg, 1 );
    check_menus(w, be, NULL);
    if (dummy) {
      sprintf(tmp, "\n Free: dummy 0x%x!!", dummy);
      ximap_log(tmp);

      XbListFreeSelection(dummy);
    }

    ximap_log("\nSORTANT ZOOM!!\n");
    imp_log = FALSE;
}
  
void expungeMessages(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
  char temp[TMPLEN];
  char name[TMPLEN];
  Arg warg[ARGLISTSIZE];
  int n = 0;
  int i = 0;
  MailBox* ms = be->ms;
  XbListSelection * ret;
  BrowserEntry * tmpbe;
  if(ms->mailbox)
    {
      setActiveBrowser(ms, be->browser);
      if (ISINACTIVE(ms->status) && !ms->mailbox->lock)
	{
	  
	  mailboxBeforeActions(ms, EXPUNGING); /* start critical region */
	  
	  ms->num_expunged = 0;
	  strcpy( name, ms->mailboxName );
	  
	  mail_expunge( ms->mailbox);
	  if ( ms->num_expunged > 0 )
	    {
	      if( ms->nmsgs < 0)
		ms->nmsgs = 0;
	      redoList(ms);
	      if (ms->num_expunged > 1)
		sprintf (temp,"%s: Expunged %d messages", name, 
			 ms->num_expunged );
	      else 
		sprintf (temp,"%s: Expunged 1 message", name);
	      mm_log(temp, NIL);
	      sprintf (temp,"%s -- %d msgs", name, ms->nmsgs);
	      XtSetArg(warg[n], XtNtitle, temp); n++;
	      XtSetValues(ms->mailWindow, warg, n); n = 0;	
	      ms->num_expunged = 0;
	    }
	  
	  mailboxAfterActions(ms, EXPUNGING);

	}
    }
}


void copyMessages(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    XbListSelectionPtr selection = XbBrowserCurrentHighlight(be->browser);

    if (selection->size)
    {
	NodeList*     nodelist;
	MailBox*      ms = be->ms;
	setActiveBrowser(ms, be->browser);

	nodelist = listMapToNodeList(selection, ms);
	doCopy(ms, nodelist);
    }

    XbListFreeSelection(selection);
}

void moveMessages(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    XbListSelectionPtr    selection = XbBrowserCurrentHighlight(be->browser);

    if(selection->size)
    {
	NodeList*   nodelist;
	MailBox*    ms = be->ms;
	setActiveBrowser(ms, be->browser);

	nodelist = listMapToNodeList(selection, ms);
	doMove( ms, nodelist );
    }

    XbListFreeSelection(selection);
}

void browserQuit( w, be, ev )
     Widget w;
     BrowserEntry *be;
     XEvent *ev;
{
    BrowserEntry *b = be;
    BrowserEntry *last = NULL;
    MailBox* ms = be->ms;
    Widget temp = w;

    be = ms->browserList;

    setActiveBrowser(ms, be->browser);

    while( be && be != b )
    {
	last = be;
	be = be->next;
    }
    
    if( be && be->main == TRUE )
    {
	if ( XbConfirm(ms->mailWindow,
		       "Do you really want to\n close the mailbox?", 
		       FALSE) == TRUE )
	{
	    if (ms->background_proc)
		XtRemoveWorkProc(ms->background_proc);
	    
	    XtDestroyWidget(ms->mailWindow);
	    ms->mailWindow = NULL;
	    
	    closeMailBox( w, ms, ev );
	}
	return;
    }
    else if( be )
    {
	if(last)
	    last->next = be->next;
	else
	    ms->browserList = be->next;

	ximap_freeBrowserEntry( be );
    }
    
    while( !XtIsShell( temp ))
	temp = XtParent(temp);
	 
    XtDestroyWidget(temp);
}

void checkCallback(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent *event;
{
    MailBox* ms = be->ms;
    
    setActiveBrowser(ms, be->browser);
    checkMailBox(w, ms, event);
}


void BrowseSelectedHeaders( w, be, ev )
     Widget w;
     BrowserEntry *be;
     XEvent *ev;
{
    /* Remember NOT to call any 'free' on this ... */
    XbListSelectionPtr map = XbBrowserCurrentHighlight(be->browser);

    if(map->size)
    {
	MailBox*      ms = be->ms;

	setActiveBrowser(ms, be->browser);
	AddBrowser(be->browser, map);
    }
}


void readOne( w, be, map)
     Widget  w;
     BrowserEntry *be;
     XbListSelectionPtr map;
{
    NodeList*      nodelist;
    MailBox*       ms = be->ms;
    
    if(map && map->size == 1)
    {
	setActiveBrowser(ms, be->browser);
	nodelist = listMapToNodeList(map, ms);
	doRead(ms, nodelist);
    }
}

static void setFilterHeaders( ms, be, filter )
     MailBox*     ms;
     BrowserEntry *be;
     Boolean     filter;
{
    if ( filter == FALSE )
	ms->filter_heads = FALSE;
    else
	ms->filter_heads = TRUE;

    check_menus(be->browser, be, NULL);    
}

void ToggleHeaderMode( w, be, event)
     Widget w;  
     BrowserEntry *be;
     XEvent *event;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;
    char *label = NULL;
    
    MailBox* ms = be->ms;

    setActiveBrowser(ms, be->browser);

    if ( ms->filter_heads == TRUE )
    {
	ms->filter_heads = FALSE;
	
	label = cpystr("Header Mode: literal");
    }
    else
    {
	ms->filter_heads = TRUE;
	label = cpystr("Header Mode: filtered");
    }
    
    XtSetArg(warg[n], XtNlabel, label); n++;
    XtSetValues(w, warg, n); n = 0; 
    check_menus(w, be, NULL);
}

void findMessages( w, be, e )
     Widget w;
     BrowserEntry *be;
     XEvent* e;
{
    MailBox* ms = be->ms;
    setActiveBrowser(ms,be->browser);
    createSearchPanel( ms );
}

void ComposeNew( w, be, e)
     Widget w;
     BrowserEntry *be;
     XEvent* e;
{
    MailBox* ms = be->ms;
    setActiveBrowser(ms,be->browser);
    doComposeNew(ms);
}     

void setKeyword( w, be, call)
     Widget w;
     BrowserEntry *be;
     void *call;
{
    char *temp = NULL;
    
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    XtSetArg(warg[n], XtNlabel, &temp); n++;
    XtGetValues( w, warg, n); n = 0;
    
    if (temp != NULL)
    {
	setFlag(be, temp);
    }
}


void clearKeyword( w, be, call)
     Widget w;
     BrowserEntry *be;
     void *call;
{
    char *temp = NULL;
    
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    XtSetArg(warg[n], XtNlabel, &temp); n++;
    XtGetValues( w, warg, n); n = 0;
    
    if (temp != NULL)
    {
	clearFlag(be, temp);
    }
}


void mbx_MarkDeleted(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    setFlag( be, DELETED_FLAG );
}

 
void mbx_MarkUndeleted(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    clearFlag( be, DELETED_FLAG );
}


void mbx_MarkSeen(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    setFlag( be, SEEN_FLAG );
}

 
void mbx_MarkUnseen(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    clearFlag( be, SEEN_FLAG );
}


void mbx_MarkRecent(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    setFlag( be, RECENT_FLAG );
}

 
void mbx_MarkOld(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    clearFlag( be, RECENT_FLAG );
}


void mbx_MarkAnswered(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    setFlag( be, ANSWERED_FLAG );
}

 
void mbx_MarkUnanswered(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    clearFlag(be, ANSWERED_FLAG );
}


void mbx_MarkFlagged(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    setFlag( be, FLAGGED_FLAG );
}

 
void mbx_MarkUnflagged(w, be, event)
     Widget w;
     BrowserEntry *be;
     XEvent* event;
{
    clearFlag( be, FLAGGED_FLAG );
}


void checkScrollPosition(w, be, first)
     Widget w;
     BrowserEntry *be;
     int first; 
{
    int currentView = 0;
    int start = 0;
    int end = 0;

    MailBox* ms = be->ms;

    if( !ms->zoom )
    {
	setActiveBrowser(ms, be->browser);
	if ( ms->actual_size == ms->virtual_size)
	{
	    XtRemoveCallback(be->browser, 
			     XtNscrollProc, 
			     checkScrollPosition, 
			     be);
	    return;
	}
	
	currentView = XbBrowserGetView(be->browser);
	start = first - currentView / 2;
	end = first  + currentView * 3 / 2;
	
	if (first > ms->nmsgs - currentView)
	    first = ms->nmsgs - currentView;
	
	if ( start < 0 ) 
	    start = 0;
	
	if ( end > ms->nmsgs - 1)
	    end = ms->nmsgs - 1;
	
	changeHeaders(ms, start, end);
    }	
	
    if ( ms->actual_size == ms->virtual_size)
	XtRemoveCallback(be->browser, 
			 XtNscrollProc, 
			 checkScrollPosition, 
			 be);
}

void checkJumpPosition(w, be, first)
    Widget w;
    BrowserEntry *be;
    int first;
{
    int currentView = 0;
    int start = 0;
    int end = 0;
    MailBox* ms = be->ms;

    if( !ms->zoom )
    {
	setActiveBrowser(ms, be->browser);
	if ( ms->actual_size == ms->virtual_size)
	{
	    XtRemoveCallback(be->browser,
			     XtNjumpProc,
			     checkJumpPosition,
			     be);
	    return;
	}
	
	currentView = XbBrowserGetView(be->browser);
	start = first - currentView / 2;
	end = first + currentView * 3 / 2;
	
	if (first > ms->nmsgs - currentView)
	    first = ms->nmsgs - currentView;
	
	if ( start < 0 ) 
	    start = 0;
	
	if ( end > ms->nmsgs - 1)
	    end = ms->nmsgs - 1;
	
	changeHeaders(ms, start, end);
    }

    if ( ms->actual_size == ms->virtual_size)
	XtRemoveCallback(be->browser,
			 XtNjumpProc,
			 checkJumpPosition,
			 be);
}


void readMessages( w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    MailBox*  ms = be->ms;
    NodeList* nodelist = NULL;
    XbListSelectionPtr selection = XbBrowserCurrentHighlight( be->browser );

    if(selection->size)
    {
	setActiveBrowser(ms, be->browser);

	nodelist = listMapToNodeList(selection, ms);
	doRead( ms, nodelist);
    }
    XbListFreeSelection(selection);
}

void ForwardMessages( w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    NodeList*     nodelist;
    MailBox*      ms = be->ms;
    XbListSelectionPtr   selection = XbBrowserCurrentHighlight(be->browser);

    if(selection->size)
    {
	setActiveBrowser(ms, be->browser);

	nodelist = listMapToNodeList(selection, ms);
	doForward( ms, nodelist );
    }
    XbListFreeSelection(selection);
}

void printMessages( w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    NodeList*               nodelist;
    XbListSelectionPtr      selection = NULL;
    MailBox*                ms = be->ms;

    if ( XbConfirm(ms->mailWindow,
		   "Print selected messages...", 
		   TRUE) == TRUE )
    {
	selection = XbBrowserCurrentHighlight(be->browser);
	if(selection->size)
	{
	    nodelist = listMapToNodeList(selection, ms);
	    doPrint( ms, nodelist );
	}
	XbListFreeSelection(selection);
    }
}

void RemailMessage(w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    NodeList* nodelist;
    MailBox* ms = be->ms;
    XbListSelectionPtr selection = XbBrowserCurrentHighlight( be->browser );
    
    if(selection->size)
    {
	setActiveBrowser(ms, be->browser);

	nodelist = listMapToNodeList(selection, ms);
	doRemail( ms, nodelist );
    }
    XbListFreeSelection(selection);
}

void ReplyToSender( w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    NodeList* nodelist;
    MailBox*  ms = be->ms;
    XbListSelectionPtr  selection = XbBrowserCurrentHighlight(be->browser);

    if(selection->size)
    {
	setActiveBrowser(ms, be->browser);
    
	nodelist = listMapToNodeList(selection, ms);
	    
	nodelist->nodes[0]->status &= ~INRESPONSE;
	nodelist->nodes[0]->status |= REPLYSENDER;
	doReply( NULL, ms, nodelist );
	nodelist->nodes[0]->status &= ~INRESPONSE;
    }
    XbListFreeSelection(selection);
}

void ReplyToAll( w, be, ca)
     Widget w;
     BrowserEntry *be;
     XEvent *ca;
{ 
    MailBox*  ms = be->ms;
    NodeList* nodelist;
    XbListSelectionPtr  selection = XbBrowserCurrentHighlight(be->browser);

    if(selection->size)
    {
	setActiveBrowser(ms, be->browser);
	
	nodelist = listMapToNodeList(selection, ms);
	    
	nodelist->nodes[0]->status &= ~INRESPONSE;
	nodelist->nodes[0]->status |= REPLYALL;
	doReply( NULL, ms, nodelist );
	nodelist->nodes[0]->status &= ~INRESPONSE;
    }
    XbListFreeSelection(selection);
}

/*************************************************


  Flag routines


*************************************************/

static void CreateMailboxFlag(ms)
     MailBox* ms;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    Widget Paned;
    
    char *name = NULL;
    char *temp = ms->mailboxName;
    
    static void lowerFlag(/* Widget, MAILSTATUS*, XEvent* */);
    
    if (   res.mail_empty != NULL 
	&& res.mail_full != NULL )
    {
	name = XtMalloc( strlen(ms->mailboxName) + 2 );
	while(*temp++ != '}' 
	      && *temp != NULL)
	    ; /* EMPTY */
	--temp;
	if (*temp++ = '}')
	{
	    strcpy(name, temp);
	}
	else
	{
	    strcpy(name, ms->mailboxName);
	}
	
	XtSetArg(warg[n], XtNtitle, "Flag"); n++;
	ms->flagShell = XtCreatePopupShell("flag_shell", 
					   topLevelShellWidgetClass, 
					   ms->mailWindow, 
					   warg, 
					   n); n = 0;
	
	XtSetArg(warg[n], XtNorientation, XtorientVertical); n++;
	Paned = XtCreateManagedWidget("flag_box", 
				    panedWidgetClass, 
				    ms->flagShell, 
				    warg, 
				    n); n = 0;
	
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	XtSetArg(warg[n], XtNbitmap, res.mail_empty); n++;
	ms->mailboxFlag = XtCreateManagedWidget("flag_bitmap", 
						commandWidgetClass, 
						Paned, warg, n); n = 0;
      
	XtSetArg(warg[n], XtNborderWidth, 0); n++;
	XtSetArg(warg[n], XtNlabel, name); n++;
	XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
	XtCreateManagedWidget("flag_name", 
			      labelWidgetClass, 
			      Paned, warg, n); n = 0;
	
	XtAddCallback(ms->mailboxFlag, XtNcallback, lowerFlag, ms);
    }
    else
    {
	ms->flagShell = NULL;
	ms->mailboxFlag = NULL;
    }
}

void flagUp(ms)
     MailBox* ms;
{
    Arg warg[1];
    int n = 0;
    int i = 0;
    
    if ( ms->mailboxFlag )
    {
	XtSetArg(warg[n], XtNbitmap, res.mail_full); n++;
	XtSetValues(ms->mailboxFlag, warg, n); n = 0;
    }

    if (res.ring_bell_for_new == TRUE
	&& res.number_of_bells > 0)
    {
	for (i = 0; i < res.number_of_bells; i++)
	{
	    XBell(XtDisplay(toplevel), 100);
	}
    }
}

void flagDown(ms)
     MailBox* ms;
{
    Arg warg[ARGLISTSIZE];
    int n = 0;
    
    if ( ms->mailboxFlag)
    {
	XtSetArg(warg[n], XtNbitmap, res.mail_empty); n++;
	XtSetValues(ms->mailboxFlag, warg, n); n = 0;
    }
}

static void lowerFlag(w, ms, e)
     Widget w;
     MailBox* ms;
     XEvent *e;
{
    flagDown( ms );
}


/*************************************************


  Utility routines


*************************************************/


void check_menus( w, be, event )
     Widget w;
     BrowserEntry* be;
     XEvent *event;
{
    BrowserEntry *temp = be;
    XbListSelectionPtr       selection = NULL;
    MailBox*       ms = be->ms;
    char tmp[512];

    selection = XbBrowserCurrentHighlight( be->browser );

    sprintf(tmp,"\n Check_menus: allocated @ 0x%x", selection);
    ximap_log(tmp);

    be = ms->browserList;
    while( be !=temp )
	be = be->next;

    if( selection && selection->size > 0 )
	checkButtonStates( be->states, DEFAULTSTATE );
    else
	checkButtonStates( be->states, NOSELECTION );

    XbListFreeSelection(selection);

    sprintf(tmp, "\n SORTANT: check_menus, Freed @x%x", selection);
    ximap_log(tmp);

}

void closeMailBox(w, ms, event)
     Widget w;
     MailBox* ms;
     XEvent* event;
{
    if (ms->mailbox)
    {
	mail_close(ms->mailbox);
	
	if(ms->timeout)
	    XtRemoveTimeOut(ms->timeout);
	
	ms->mailbox = NULL;
	while(ms->last_message &&  ms->messages
	      && (ms->last_message >=  ms->messages))
	{
	    if(*ms->last_message)
	    {
		free_node(*ms->last_message);
	    }
	    ms->last_message--;
	}
	
	XtFree((char*)ms->messages);
	XtFree(ms->found);
	XbListFreeSelection(ms->defaultMapping);
	XbListFreeSelection(ms->zoomMapping);
	killStatus(ms);
    } 
    else
    {
	mm_log("There is no open mailbox", WARN);
    }
}

void toggleDebug( w, ms, e )
     Widget w;
     MailBox* ms;
     XEvent* e;
{
  return;
}

void setFlag(be, flag)
     BrowserEntry *be;
     char       *flag;
{
    MailBox*  ms = be->ms;
    NodeList* nodelist;
    XbListSelectionPtr selection = XbBrowserCurrentHighlight( be->browser );
    
    setActiveBrowser(ms, be->browser);
    
    if (flag && selection->size)
    {
	nodelist = listMapToNodeList(selection, ms);
	doSetFlag(ms, nodelist, flag);
    }
    XbListFreeSelection(selection);
}

void clearFlag(be, flag)
     BrowserEntry *be;
     char       *flag;
{
    MailBox*    ms = be->ms;
    NodeList*   nodelist;
    XbListSelectionPtr    selection = XbBrowserCurrentHighlight( be->browser );
    
    setActiveBrowser(ms, be->browser);
    
    if (flag && selection->size)
    {
	nodelist = listMapToNodeList(selection, ms);
	doClearFlag(ms, nodelist, flag);
    }
    XbListFreeSelection(selection);
}

static void updateStatus(ms, start, end)
     MailBox* ms;
     int start, end;
{
    MessageNode*   node;
    char header[TMPLEN];
    int count = 0;
    
    if (ms->list_size <= end + 2
	|| ms->messages == NULL )
    {
	ms->list_size = end + 10;
	if( ms->messages)
	{
	    ms->messages = 
		(MessageNode**)XtRealloc(ms->messages,
					ms->list_size*sizeof(MessageNode*));
	}
	else
	{
	    ms->messages = 
		(MessageNode**)XtMalloc(ms->list_size*sizeof(MessageNode*));
	}

    }

    if (ms->messages == NULL)
    {
	mm_log("ERROR -- CANNOT PERFORM REALLOC", ERROR);
	ms->last_message = NULL;
	return;
    }
    else
    {
	ms->last_message = ms->messages + ms->nmsgs - 1;
    }

    for( count = start; count <= end; count++ )
    {
	memset(header, 0, TMPLEN );
	node = ximap_newnode();

	ms->messages[count] = node;
    
	node->env = mail_fetchstructure(  ms->mailbox, count+1, &node->body);  
	node->elt = mail_elt( ms->mailbox, count+1);      
    
	if(node->env && node->env->to)
	{
	    if(   onAddressList( res.mail_name, node->env->to )
	       || onAddressList( res.mail_name, node->env->cc ))
		node->toMe = TRUE;
	}

	node->msgno = node->elt->msgno;
	headerline( header, ms->mailbox, node);
    
	node->header_line = cpystr(header);
	ms->virtual_size++;
	ms->actual_size++;
    }
    
    ms->found = (int*) XtRealloc(ms->found, (count + 1)*sizeof(int));
}

/*
 * Ask the server if there are new messages.  
 *   If we are in the process of logging in (i.e., this
 *   is the first check) then don't do any of the fancy updating.
 *   
 *   Else update mailbox, mailbox title after the check (if there 
 *   were new messages)
 */
void checkMailBox(w, ms, event)
     Widget w;
     MailBox* ms;
     XEvent *event;
{
    char temp[TMPLEN];
    char title[TMPLEN];
    Arg warg[ARGLISTSIZE];
    int n = 0;
    int delta = 0;
    int oldsize = 0;
    
    if ((ISINACTIVE(ms->status) 
	 || (ms->status & TIMEDCHECK))
	&& ms->mailbox
	&& !ms->mailbox->lock)

    {
        update_telemetry_label(" Checking ...", TRUE);

	mailboxBeforeActions(ms, CHECKMAILBOX);
	mail_check(ms->mailbox);
	if (!(ms->status & LOGGINGIN))
	{
	    oldsize = ms->virtual_size;
	    delta = ms->nmsgs - oldsize;
	    if ( delta > 0 )
	    {
	        char tmp[512];

		updateStatus( ms, oldsize, ms->nmsgs - 1);
		updateBrowser(ms, delta);
		sprintf(title,"%s -- %d msgs",ms->mailboxName, ms->nmsgs);
		XtSetArg(warg[n], XtNtitle, title); n++;
		XtSetValues(ms->mailWindow, warg, n); n = 0;
		
		if(! (ms->status & UNSOLICITEDCHECK))
		    flagUp( ms );

		check_menus(NULL, ms->browserList, NULL);
		if(delta == 1)
		    sprintf (temp,
			     "%s: 1 new message", ms->mailboxName);
		else
		    sprintf (temp,
			     "%s: %d new messages",
			     ms->mailboxName, delta);

		mm_log(temp, NIL);
	    }
	    else if (ms->status & DEBUGGING)
	    {
		sprintf (temp,
			 "%s: There are no new messages",
			 ms->mailboxName, delta);
		mm_dlog(temp);

	    }
	}
	mailboxAfterActions(ms,CHECKMAILBOX);
    }
}

int countSelected(be)
     Widget be;
{
  if (be)
    return (XbBrowserCountHighlighted(be));
  else
    return 0;
	    
}

void mailboxBeforeActions(ms, status)
     MailBox *ms;
     unsigned long status;
{
    ms->status |= status;
    if(ms->background_proc)
	XtRemoveWorkProc(ms->background_proc);
    
}


void mailboxAfterActions(ms, status)
     MailBox *ms;
     unsigned long status;
{
    ms->status &= ~status;
    if(ms->background_proc)
    {
	ms->background_proc = XtAppAddWorkProc(app_context,      
					       BackgroundFetch,  
					       ms);
    }
}

/*************************************************


  Actions


*************************************************/


void mainUnflagAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    mbx_MarkUnflagged(w, be, e);
}
void mainFlagAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    mbx_MarkFlagged(w, be, e);
}

void mainUndeleteAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    mbx_MarkUndeleted(w, be, e);
}

void mainDeleteAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    mbx_MarkDeleted(w, be, e);
}

void mainExpungeAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    expungeMessages(w, be, e);
}

void mainReadAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    readMessages(w, be, e);
}

void mainForwardAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    ForwardMessages(w, be, e);
}

void mainPrintAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    printMessages(w, be, e);
}

void mainRemailAct(w, e)
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    RemailMessage(w, be, e);
}

void mainReplySenderAct(w, e)
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    ReplyToSender(w, be, e);
}

void mainReplyAllAct(w, e)
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    ReplyToAll(w, be, e);
}

void mainCopyAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    copyMessages(w, be, e);
}

void mainMoveAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    moveMessages(w, be, e);
}

void mainQuitAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    browserQuit(w, be, e);
}

void mainComposeAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    doComposeNew(be->ms);
}

void mainZoomAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    toggleZoom(w, be, e);
}

void mainFindAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    findMessages(w, be, e);
}

void mainClearAct( w, e )
      Widget   w;
      XEvent*  e;
{
  BrowserEntry *be = getBEFromBrowser(w);
  clear_browser(w, be, e);
}

void mainBrowseAct( w, e )
     Widget   w;
     XEvent*  e;
{
    BrowserEntry *be = getBEFromBrowser(w);
    BrowseSelectedHeaders(w, be, e);
}

/* 
 * Called when the size of a mailbox has changed unexpectedly.
 *
 */

void checkMailboxes()
{
    STREAMARRAY *sa = id.mailboxes;

    while(sa)
    {
	if(sa->status)
	    if(sa->status->changedSize)
	    {
		sa->status->status |=  UNSOLICITEDCHECK;
		checkMailBox(NULL,sa->status,NULL);
		sa->status->changedSize = 0;
		sa->status->status &=  ~UNSOLICITEDCHECK;
	    }

	sa = sa->next;
    }

    updateMailbox = 0;
}
