/*
 *	
 *  Author(s):       Kevin J. Brock
 *                   Symbolic Systems Resources Group
 *		    Knowledge Systems Laboratory
 *		    Departments of Computer Science and Medicine
 *		    Stanford University
 *		    Stanford, CA 94305
 *
 *  Date:            28 March, 1991
 *
 *  Copyright () 1991 by the Leland Stanford Junior University
 *  
 *  "This program may be distributed without restriction for non-commercial
 *  use. Any sale or use of this program or adaptations thereof for commercial
 *  purposes is prohibited except under license from the Stanford Office of
 *  Technology Licensing."
 *
 ***********************************************************/
#include <string.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/XawInit.h>
#include <X11/Xmu/Misc.h>

#include <X11/Xaw/Scrollbar.h>

#include "List.h"
#include "BrowserP.h"

static void clearBrowser();

static char defaultTranslations[] =  
  "<Key>U:       Clear()";

static XtActionsRec actions[] = {
      {"Clear",     clearBrowser},
      {NULL,NULL}
};

static XtResource Xbbrowser_resources[]= {
{ XtNlistPosition, XtCListPosition, 
      XtRInt, sizeof(int),
      XtOffset(XbBrowserWidget, Xbbrowser.initialPosition), 
      XtRImmediate, 0 },
{ XtNinitialHighlight, XtCInitialHighlight, 
      XtRInt, sizeof(int),
      XtOffset(XbBrowserWidget, Xbbrowser.initialHighlight), 
      XtRString, "-1" },
{ XtNinitialViewHt, XtCInitialViewHt, 
      XtRInt, sizeof(int),
      XtOffset(XbBrowserWidget, Xbbrowser.view), 
      XtRString, "20" },
{ XtNminimumView, XtCMinimumView, 
      XtRInt, sizeof(int),
      XtOffset(XbBrowserWidget, Xbbrowser.minView), 
      XtRString, "5" },
{ XtNlist, XtCList, 
      XtRPointer, sizeof( void* ),
      XtOffset(XbBrowserWidget, Xbbrowser.items), 
      XtRImmediate, (void*) NULL },
{ XtNsingleClick, XtCCallback, 
      XtRCallback, sizeof(caddr_t),
      XtOffset(XbBrowserWidget, Xbbrowser.singleClick),
      XtRCallback, NULL},
{ XtNdoubleClick, XtCCallback, 
      XtRCallback, sizeof(caddr_t),
      XtOffset(XbBrowserWidget, Xbbrowser.doubleClick), 
      XtRCallback, NULL},
{ XtNjumpProc, XtCCallback, 
      XtRCallback, sizeof(caddr_t),
      XtOffset(XbBrowserWidget, Xbbrowser.jumpProc), 
      XtRCallback, NULL},
{ XtNscrollProc, XtCCallback, 
      XtRCallback, sizeof(caddr_t),
      XtOffset(XbBrowserWidget, Xbbrowser.scrollProc), 
      XtRCallback, NULL},
{ XtNforceBars, XtCBoolean, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(XbBrowserWidget, viewport.forcebars), 
      XtRImmediate, (caddr_t)True},
{ XtNallowHoriz, XtCBoolean, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(XbBrowserWidget, viewport.allowhoriz), 
      XtRImmediate, (caddr_t)False},
{ XtNallowVert, XtCBoolean, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(XbBrowserWidget, viewport.allowvert), 
      XtRImmediate, (caddr_t)True},
{ XtNuseBottom, XtCBoolean, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(XbBrowserWidget, viewport.usebottom), 
      XtRImmediate, (caddr_t)False},
{ XtNuseRight, XtCBoolean, 
      XtRBoolean, sizeof(Boolean),
      XtOffset(XbBrowserWidget, viewport.useright), 
      XtRImmediate, (caddr_t)False},
};
 
static void Initialize();
static void Realize();
static void ConstraintInitialize();
static void Destroy();
static Boolean SetValues();

/* Initialize sub-functions */

static void browserCreateList();

/* Miscellaneous */

static void browserSetView( /* Widget browser, int numberVisible */ );
static void browserSetPosition( /* Widget browser, int firstVisible */ );

static void browserJumpProc( );
static void browserScrollProc( );
static void browserSingleClick();
static void browserDoubleClick();

static void makeReverseMap(/* Widget */);
static XbListSelectionPtr makeDisplayList( );

static XtGeometryResult PreferredGeometry(), GeometryManager();

#define superclass (&viewportClassRec)
    
XbBrowserClassRec XbbrowserClassRec = {
  {
    /* superclass         */    (WidgetClass) superclass,
    /* class_name         */    "XbBrowser",
    /* size               */    sizeof(XbBrowserRec),
    /* class_initialize   */	NULL,
    /* class_part_initialize*/	NULL,
    /* Class init'ed      */	FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */	NULL,
    /* realize            */    Realize,
    /* actions            */    actions,
    /* num_actions        */    XtNumber(actions),
    /* resources          */    Xbbrowser_resources,
    /* resource_count     */	XtNumber(Xbbrowser_resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE, 
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/ 	TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    Destroy,
    /* resize             */    XtInheritResize,
    /* expose             */    XtInheritExpose,
    /* set_values         */    SetValues,
    /* set_values_hook    */	NULL,
    /* set_values_almost  */	XtInheritSetValuesAlmost,  
    /* get_values_hook    */	NULL,			
    /* accept_focus       */    NULL,
    /* intrinsics version */	XtVersion,
    /* callback offsets   */    NULL,
    /* tm_table		  */    defaultTranslations,
    /* query_geometry	  */    PreferredGeometry,
    /* display_accelerator*/    NULL,
    /* extension	  */    NULL
    },{
      /* composite class fields */
      /* geometry_manager   */   GeometryManager,
      /* change_managed     */   XtInheritChangeManaged,
      /* insert_child       */   XtInheritInsertChild,
      /* delete_child       */   XtInheritDeleteChild,
      /* extension          */   NULL
      },{
	/* constraint class fields */
        /* subresourses       */   NULL,
        /* subresource_count  */   0,
        /* constraint_size    */   sizeof(XbBrowserConstraintsRec),
        /* initialize         */   NULL,
        /* destroy            */   NULL,
        /* set_values         */   NULL,
        /* extension          */   NULL
	},{
	  /* Form part */           XtInheritLayout
	  },{
	    /* Viewport extension */       0
	    },
              {
	      /* XbBrowser extension */      0
	      }
};

WidgetClass XbbrowserWidgetClass = (WidgetClass) &XbbrowserClassRec;

/*      Function Name: Initialize
 *      Description: Initializes the mailbox widget
 *      Arguments: request - the widget requested by the argument list.
 *                 new     - the new widget with both resource and non
 *                           resource values.
 *      Returns: none.
 */

/* ARGSUSED */
static void Initialize(request, new)
     Widget request;
     Widget new;
{
    XbBrowserWidget bw = (XbBrowserWidget) new;
    XbBrowserWidget bwr = (XbBrowserWidget) request;

    bw->Xbbrowser.reverseMap = NULL;
    bw->Xbbrowser.maxItem = -1;
    bw->Xbbrowser.maxReverse = -1;

    bw->Xbbrowser.minViewDim = 1;

    bw->Xbbrowser.list = NULL;

    bw->viewport.forcebars = TRUE;
    bw->viewport.allowvert = TRUE;

    makeReverseMap( new);
    browserCreateList( new );
    browserSetView( bw, bw->Xbbrowser.view );
}


/* ARGSUSED */
static void Realize(w, msk, att)
     Widget               w;
     XtValueMask          *msk;
     XSetWindowAttributes *att;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    
    (*superclass->core_class.realize)(w, msk, att);
    
    if ( bw->viewport.vert_bar )
    {
	XtAddCallback(bw->viewport.vert_bar, XtNscrollProc, 
		      (void*)browserScrollProc, w);
	XtAddCallback(bw->viewport.vert_bar, XtNjumpProc, 
		      (void*)browserJumpProc, w);
    }
    
    browserSetPosition( w, bw->Xbbrowser.initialPosition );
}

/* ARGSUSED */
static Boolean SetValues(current, request, new)
     Widget current;
     Widget request;
     Widget new;
{
  XbBrowserWidget bcur = (XbBrowserWidget) current;
  XbBrowserWidget bnew = (XbBrowserWidget) new;
  XbBrowserWidget breq = (XbBrowserWidget) request;

  return(FALSE);
}

/* ARGSUSED */
static void Destroy( w )
     Widget w;
{
    int i = 0;
    XbBrowserWidget browser = (XbBrowserWidget) w;
    XbListSelectionPtr items = browser->Xbbrowser.items;

    XtFree(browser->Xbbrowser.reverseMap);
    XbListFreeSelection(items);
}

/* ARGSUSED */
static void ConstraintInitialize(request, new)
     Widget request;
     Widget new;
{
    XbBrowserWidget breq = (XbBrowserWidget) request;
    XbBrowserWidget bnew = (XbBrowserWidget) new;
}

static void browserCreateList( bw )
     XbBrowserWidget bw;
{
    Arg warg[3];

    XbListSelectionPtr list = makeDisplayList(bw);
    
    XtSetArg(warg[0], XtNlist, list);
    XtSetArg(warg[1], XtNdefaultColumns, 1);
    XtSetArg(warg[2], XtNforceColumns, TRUE);
    bw->Xbbrowser.list = XtCreateManagedWidget("list", XblistWidgetClass,
					       (Widget) bw, 
					       warg, 3);
    
    XtAddCallback( bw->Xbbrowser.list, XtNcallback, browserSingleClick, bw );
    XtAddCallback( bw->Xbbrowser.list, XtNdoubleClick, browserDoubleClick, bw ); 
}

static void browserSetView(bw, numberVisible)
     XbBrowserWidget bw;
     int     numberVisible;
{
    Arg warg[3];
    int n = 0;
    
    Dimension elementHeight;
    Dimension height;
    Dimension width;
    Dimension min;

    XtSetArg(warg[n], XtNwidth, &width); n++;
    XtSetArg(warg[n], XtNrowHeight, &elementHeight); n++;
    XtGetValues( bw->Xbbrowser.list, warg, n); n = 0;
    
    height = elementHeight * (Dimension)numberVisible;
    bw->Xbbrowser.minViewDim = elementHeight * bw->Xbbrowser.minView;
    
    bw->core.height = height;
    bw->core.width = width;
}

static void browserSetPosition(w, firstVisible)
     Widget w;
     int     firstVisible;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    
    register Widget list = bw->viewport.child;
    register Widget clip = bw->viewport.clip;
    
    Position x = 0, y = 0;

    Arg warg[2];

    int n = 0;
    int size;
    int realFirst;

    float thumb = 0.0;

    if( bw->Xbbrowser.reverseMap 
       && list
       && firstVisible >= 0 )
    {
	x = list->core.x;

	XtSetArg(warg[n], XtNnumberStrings, &size); n++;
	XtGetValues(list, warg, n); n = 0;
	
	if( firstVisible <= bw->Xbbrowser.maxReverse )
	    realFirst = bw->Xbbrowser.reverseMap[firstVisible];
	else
	    return;
	
	if(realFirst < 0)
	    realFirst = 0;
	
	thumb = 1.0 * realFirst / size;
	if(thumb < 0)
	    thumb = 0;

	if( bw->viewport.vert_bar )
	{
	    if ( thumb >= 0)
	    {
		y = -(int)( thumb * list->core.height );

		/* make sure we never move past left/top borders */
		if (x >= 0) x = 0;
		if (y >= 0) y = 0;
		
		XtMoveWidget(list, x, y);

		XawScrollbarSetThumb(bw->viewport.vert_bar, 
				     thumb, 
				     ((float)clip->core.height )/list->core.height);
	    }
	}
    }
}

void XbBrowserHighlight(w, highlight)
     Widget  w;
     XbListSelectionPtr highlight;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    int             number = 0;

    ximap_log("\n XbBrowserHighlight");

    if(   highlight->size > 0
       && bw->Xbbrowser.reverseMap 
       && bw->Xbbrowser.initialHighlight )
    {
	XbListSelectionPtr hgl = 
	    (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
	hgl->size = 0;
	hgl->entries = (XbListEntryPtr) XtMalloc (highlight->size*sizeof( XbListEntry ));
	
	while( number < highlight->size )
	{
	    hgl->entries[hgl->size].bold = 0;
	    hgl->entries[hgl->size].invert = 0;
	    hgl->entries[hgl->size].index = -1;
	    hgl->entries[hgl->size].entry = NULL;

	    if( highlight->entries[number].index <= bw->Xbbrowser.maxReverse
	       && bw->Xbbrowser.reverseMap[highlight->entries[number].index] >= 0)
	    {
		hgl->entries[hgl->size].index = 
		    bw->Xbbrowser.reverseMap[highlight->entries[number].index];
		hgl->size++;
	    }
	    number++;
	}
	
	if( hgl->size > 0 )
	    XbListHighlight( bw->Xbbrowser.list, hgl );
	
	XbListFreeSelection(hgl);
    }
    ximap_log("\n SORTANT: XbBrowserHighlight");
}

static void clearBrowser(w, e, par, num)
     Widget w;
     XEvent * e;
     String * par;
     Cardinal *num;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    XbListUnhighlight( bw->Xbbrowser.list );
}

/*
 * clear highlighted entries */
void XbBrowserClear(w)
     Widget w;
{
   XbBrowserWidget bw = (XbBrowserWidget) w;
   XbListUnhighlight( bw->Xbbrowser.list );
}

void XbBrowserUnHighlight(w)
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    XbListUnhighlight( bw->Xbbrowser.list );
}

static void browserDoubleClick( w, brwsr, client)
     Widget w;
     Widget brwsr;
     XbListSelectionPtr client;
{
    XbBrowserWidget bw = (XbBrowserWidget) brwsr;

    int i = 0;
    int number = 0;

    XbListSelectionPtr ret = NULL;

    if( bw->Xbbrowser.doubleClick )
    {
	i = client->size;
	ret = (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
	ret->size = i;
	ret->entries = (XbListEntryPtr)XtMalloc(i*sizeof(XbListEntry));

	if( bw->Xbbrowser.reverseMap )
	{
	    while( i-- )
	    {
		number = client->entries[i].index;
		ret->entries[i].index = bw->Xbbrowser.items->entries[number].index;
		ret->entries[i].entry = 
		    XtMalloc(1+strlen(bw->Xbbrowser.items->entries[number].entry));
		strcpy(ret->entries[i].entry,bw->Xbbrowser.items->entries[number].entry);
		ret->entries[i].bold  = bw->Xbbrowser.items->entries[number].bold;
	    }
	}
	else
	{
	    XtFree( ret );
	    ret = NULL;
	}

	XtCallCallbackList( brwsr, bw->Xbbrowser.doubleClick, ret );
    }

    XbListFreeSelection(ret);
}

static void browserSingleClick( w, brwsr, client)
     Widget w;
     Widget brwsr;
     XbListSelectionPtr client;
{
    XbBrowserWidget bw = (XbBrowserWidget) brwsr;
    XbListSelectionPtr ret = NULL;

    int i = 0;
    int number = 0;

    if( bw->Xbbrowser.singleClick)
    {
	i = client->size;
	
	ret = (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
	ret->size = i;
	ret->entries = (XbListEntryPtr)XtMalloc(i*sizeof(XbListEntry));

	if( bw->Xbbrowser.reverseMap )
	{
	    while( i-- )
	    {
		number = client->entries[i].index;
		ret->entries[i].index = bw->Xbbrowser.items->entries[number].index;
		ret->entries[i].entry = 
		    XtMalloc(1+strlen(bw->Xbbrowser.items->entries[number].entry));
		strcpy(ret->entries[i].entry,bw->Xbbrowser.items->entries[number].entry);
		ret->entries[i].bold  = bw->Xbbrowser.items->entries[number].bold;
	    }
	    
	}
	else
	{
	    XtFree( ret );
	    ret = NULL;
	}

	XtCallCallbackList( brwsr, bw->Xbbrowser.singleClick, ret );
    }
    XbListFreeSelection(ret);
}


static void browserScrollProc( w, brwsr, p)
     Widget w;
     Widget brwsr;
     void   *p;
{
    XbBrowserWidget bw = (XbBrowserWidget) brwsr;

    Arg warg[16];
    int n = 0;

    int pos = (int) p;
    int view = 0;
    int first = 0;
    float thumb = 0.0;
    float shown = 0.0;

    Dimension length = 0;

    int i = 0;
    int ret = -1;

    if( bw->Xbbrowser.scrollProc
       && bw->viewport.vert_bar )
    {
	XtSetArg(warg[n], XtNshown, &shown); n++;
	XtSetArg(warg[n], XtNtopOfThumb, &thumb); n++;
	XtSetArg(warg[n], XtNlength, &length); n++;
	XtGetValues( bw->viewport.vert_bar, warg, n); n = 0;

	first = thumb * (bw->Xbbrowser.maxItem + 1) - 1;
	view  = shown * (bw->Xbbrowser.maxItem + 1);

	if ( pos < 0 )
	    pos = -pos;

	ret = first - (view + 1) * (pos*1.0/length) + 1;

	while( i < ret )
	    i++;

	if( i > ret )
	    ret = i - 1;
	else 
	    ret = i;

	if ( ret < 0 )
	    ret = 0;

	if ( ret > bw->Xbbrowser.maxItem )
	    ret = bw->Xbbrowser.maxItem;

	XtCallCallbackList( brwsr, bw->Xbbrowser.scrollProc, ret );
    }
}

static void browserJumpProc(w, brwsr, p)
     Widget w;
     Widget brwsr;
     void *p; /* pointer position length of the scrollbar (float*) */
{
    XbBrowserWidget bw = (XbBrowserWidget) brwsr;

    float percent =  *(float*) p;

    int ret = 0;
    int i = 0, j = 0;

    if ( bw->Xbbrowser.jumpProc
	&& bw->viewport.vert_bar )
    {
	j = (bw->Xbbrowser.maxItem + 1) * percent + 1;

	while( i < j )
	    i++;

	if( i > ret )
	    ret = i - 1;
	else 
	    ret = i;

	if (ret < 0)
	    ret = 0;
	    
	if (ret > bw->Xbbrowser.maxItem)
	    ret = bw->Xbbrowser.maxItem;

	XtCallCallbackList( brwsr, bw->Xbbrowser.jumpProc, ret );
    }
}

static void makeReverseMap( w )
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    XbListSelectionPtr maptmp = bw->Xbbrowser.items;

    int i = 0;
    int number = maptmp->size;

    if( maptmp)
    {
	bw->Xbbrowser.maxItem = -1;
	bw->Xbbrowser.maxReverse = -1;
	
	if( bw->Xbbrowser.reverseMap )
	    XtFree(bw->Xbbrowser.reverseMap);
	bw->Xbbrowser.reverseMap = NULL;

	bw->Xbbrowser.maxItem = number - 1; /* the max index for the browser */
	
	while( number-- )
	{
	    if(maptmp->entries[number].index > bw->Xbbrowser.maxReverse)   
		bw->Xbbrowser.maxReverse = maptmp->entries[number].index;
	}
	
	i = bw->Xbbrowser.maxReverse;
	if( i >= 0 )
	{
	    int *oldM = bw->Xbbrowser.reverseMap;
	    bw->Xbbrowser.reverseMap = (int*) XtMalloc ((i+1) * sizeof(int));
	    bw->Xbbrowser.reverseMap[i] = -1;
	    while( i-- )
		bw->Xbbrowser.reverseMap[i] = -1;
	    
	    if( maptmp->size )
	    {
		i = maptmp->size - 1;
		number = bw->Xbbrowser.items->entries[i].index;
		bw->Xbbrowser.reverseMap[number] = i ;
		while( i-- )
		{
		    number = bw->Xbbrowser.items->entries[i].index;
		    bw->Xbbrowser.reverseMap[number] = i ;
		}
	    }
	    if(oldM)
		XtFree(oldM);
	}
    }
}

static XbListSelectionPtr makeDisplayList( w )
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;

    int i = 0;
    XbListSelectionPtr retval = NULL;

    retval = (XbListSelectionPtr)XtMalloc(sizeof(XbListSelection));
    if( bw->Xbbrowser.reverseMap )
    {
	retval->size = bw->Xbbrowser.items->size;
	retval->entries = (XbListEntryPtr)XtMalloc(retval->size*sizeof(XbListEntry));

	i = retval->size;
	while(i--)
	{
	    retval->entries[i].entry = 
		XtMalloc(1 + strlen(bw->Xbbrowser.items->entries[i].entry));
	    strcpy(retval->entries[i].entry,bw->Xbbrowser.items->entries[i].entry);
	    retval->entries[i].index = i;
	    retval->entries[i].bold = bw->Xbbrowser.items->entries[i].bold;
	    retval->entries[i].invert = bw->Xbbrowser.items->entries[i].invert;
	}
    }
    else
    {
	retval->size = 0;
	retval->entries = NULL;
    }

    return( retval );
}

char* XbBrowserReplaceString(w, item)
     Widget w;
     XbListEntryPtr item;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    int realPos;
    
    if( bw->Xbbrowser.reverseMap )
    {
	char *oldString = NULL;

	if( item->index <= bw->Xbbrowser.maxReverse )
	    realPos = bw->Xbbrowser.reverseMap[item->index];
	else
	    return(NULL);

	if( realPos < 0 )
	    return( NULL );

	oldString = bw->Xbbrowser.items->entries[realPos].entry;

	bw->Xbbrowser.items->entries[realPos].entry = XtMalloc(1+strlen(item->entry));
	strcpy(bw->Xbbrowser.items->entries[realPos].entry,item->entry);
	bw->Xbbrowser.items->entries[realPos].index = item->index;
	bw->Xbbrowser.items->entries[realPos].bold = item->bold;

	item->index = realPos;

	XbListReplaceString(bw->Xbbrowser.list, item, TRUE);

	XtFree(oldString);
	return(NULL);
    }
    else
    {
	return((char*) NULL);
    }
}

XbListSelectionPtr XbBrowserGetItems(w)
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    return( bw->Xbbrowser.items );
}

XbListSelectionPtr XbBrowserChangeItems(w, items)
     Widget  w;
     XbListSelectionPtr items;
{
    int i  = 0;
    XbBrowserWidget bw = (XbBrowserWidget) w;

    XbListSelectionPtr new = NULL;

    XbListSelectionPtr old = bw->Xbbrowser.items;

    ximap_log("\n XbBrowserChangeItems");

    bw->Xbbrowser.items = items;

    makeReverseMap( bw );
    new = makeDisplayList(w);

    XbListChange( bw->Xbbrowser.list, new, 0, TRUE );
    XbListFreeSelection(old);

    ximap_log("\n SORTANT: XbBrowserChangeItems");

    return( NULL );
}

XbListSelectionPtr XbBrowserCurrentHighlight(w)
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    int number = 0;
    int i = 0;
    XbListSelectionPtr result;

    ximap_log("\n XbBrowserCurrentHighlight");

    result = XbListShowCurrent(bw->Xbbrowser.list);

    i = result->size;
    while(i--)
    {
	if( result->entries[i].index >= 0 )
	{
	    number = result->entries[i].index;
	    result->entries[i].index = bw->Xbbrowser.items->entries[number].index;
	}
    }
    return(result);
}

void XbBrowserSetPosition( w, first )
     Widget w;
     int    first;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    browserSetPosition( w, first );
}

int XbBrowserGetPosition( w )
     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;

    Arg warg[1];
    float thumb = 0.0;
    
    register int i = 0;
    int ret = 0;

    if( bw->viewport.vert_bar)
    {
	XtSetArg(warg[0], XtNtopOfThumb, &thumb);
	XtGetValues( bw->viewport.vert_bar, warg, 1);

	ret = thumb * (bw->Xbbrowser.maxItem+1);
	while( i < ret )
	    i++;

	return( i - 1 );
    }

    return( 0 );
}

int XbBrowserGetView( w )

     Widget w;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;

    Arg warg[1];
    float shown = 0.0;
    float actual = 0.0;
    int ret = 0;

    if( bw->viewport.vert_bar)
    {
	XtSetArg(warg[0], XtNshown, &shown);
	XtGetValues( bw->viewport.vert_bar, warg, 1);

	actual = shown * (bw->Xbbrowser.maxItem + 1);
	ret = shown * (bw->Xbbrowser.maxItem + 1);

	if( ret < (actual-.5) )
	    ret++;

	return( ret );
    }

    return( 0 );
}

extern XbListEntry*  XbBrowserGetItem(w, n)
     Widget w;
     int n;
{
    XbBrowserWidget bw = (XbBrowserWidget) w;
    XbListEntry* ret;

    ret = XbListGetItem(bw->Xbbrowser.list, n);

    return(ret);
}

static XtGeometryResult PreferredGeometry(w, constraints, reply)
    Widget w;
    XtWidgetGeometry *constraints, *reply;
{
    Arg warg[8];
    int n = 0;

    XbBrowserWidget bw= (XbBrowserWidget)w;
    Dimension currHeight, prefHeight;
    Dimension elementHeight;
    Dimension minHeight;

    if(!(constraints->request_mode & CWHeight))
	return(XtGeometryYes);

    XtSetArg(warg[n], XtNrowHeight, &elementHeight); n++;
    XtGetValues( bw->Xbbrowser.list, warg, n); n = 0;

    if(bw->Xbbrowser.view > bw->Xbbrowser.items->size)
	prefHeight = elementHeight *  bw->Xbbrowser.items->size;
    else
	prefHeight = elementHeight * bw->Xbbrowser.view;

    currHeight = bw->core.height;

    if(constraints->request_mode & XtCWQueryOnly)
    {
	reply->request_mode |= CWHeight;
	if(prefHeight > currHeight)
	    reply->height = prefHeight;
	else
	    reply->height = currHeight;
	
	return XtGeometryAlmost;
    }
    else if (bw->viewport.child != NULL)
    {
	Arg warg[3];
	int n = 0;
    
	if(prefHeight == currHeight 
	   && prefHeight != constraints->height)
	    return(XtGeometryNo);

	if(prefHeight == constraints->height)
	    return(XtGeometryYes);

	reply->request_mode |= CWHeight;
	reply->height = prefHeight;
	return XtGeometryAlmost;
    }

    return XtGeometryYes;
}


static XtGeometryResult GeometryManager(child, request, reply)
    Widget child;
    XtWidgetGeometry *request, *reply;
{
    XbBrowserWidget bw = (XbBrowserWidget)child->core.parent;
    Dimension h;
    XtGeometryResult retval = (*superclass->composite_class.geometry_manager)(child,request,reply);
    
    if(request->request_mode & CWHeight)
    {
	h = bw->Xbbrowser.minViewDim;
	if(reply->request_mode & CWHeight)
	{
	    if(reply->height < h)
	    {
		reply->height = h;
		retval = XtGeometryAlmost;
	    }
	}
	else
	{
	    reply->request_mode |= CWHeight;
	    reply->height = h;
	    retval = XtGeometryAlmost;
	}
    }

    return(retval);
}

void XbBrowserSortBrowser(w, position)
     Widget w;
     int position;
{
    /* We should already have a default sort array which is
       the identity array. */

    /* create new sort array.*/

    /* We'll sort the Browser by manipulating entries in the
       sort array, and not actually moving anything in the 
       browser's list.  Resolving the number of an item can be done by indexing 
       the sort array by the index of the item in the List, and using the
       result as an index into the Browser's list of items. */

    /* Call sort routine with sort array. */

    return;
}

static void SortAlphaPosition(w, sort, pos)
     Widget  w;
     int    *sort;
     int     pos;
{
    

    return;
}

int XbBrowserCountHighlighted (w)
     Widget w;
{
     XbBrowserWidget bw = (XbBrowserWidget)w;
     int count = XbListCountHighlighted(bw->Xbbrowser.list);

     return count;
}

Boolean XbBrowserIsHighlighted(w, item)
     Widget w;
     int item;
{
     XbBrowserWidget bw = (XbBrowserWidget)w;

     Boolean eclairant = IsHighlighted(bw->Xbbrowser.list, item);
     return eclairant;
}

  
