/* ========================================================================
 * Copyright 1988-2008 University of Washington
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * 
 * ========================================================================
 */

/*
 * Program:	mix format conversion utility
 *
 * Author:	Mark Crispin
 *		UW Technology
 *		University of Washington
 *		Seattle, WA  98195
 *		Internet: MRC@Washington.EDU
 *
 * Date:	6 March 2007
 * Last Edited:	1 May 2008
 */


#include <stdio.h>
#include <errno.h>
extern int errno;		/* just in case */
#include <pwd.h>
#include "mail.h"
#include "osdep.h"
#include "misc.h"
#include "linkage.h"

/* Globals */

char *version = "3";		/* program version */
int debugp = NIL;		/* flag saying debug */
int verbosep = NIL;		/* flag saying verbose */
int critical = NIL;		/* flag saying in critical code */
FILE *f = NIL;


/* Function prototypes */

int main (int argc,char *argv[]);
long write_message (MAILSTREAM *source,unsigned long i,FILE *index,FILE **data,
		    unsigned long *curfile,char *dest,time_t now);
FILE *create_data (char *dest,unsigned long curfile);

/* Definitions from mix.c */

/* MIX files */

#define MIXNAME ".mix"		/* prefix for all MIX file names */
#define MIXMETA "meta"		/* suffix for metadata */
#define MIXINDEX "index"	/* suffix for index */
#define MIXSTATUS "status"	/* suffix for status */


#define MIXDATAROLL 1048576	/* 1MB */


/* MIX file formats */

				/* sequence format (all but msg files) */
#define SEQFMT "S%08lx\015\012"
				/* metadata file format */
#define MTAFMT "V%08lx\015\012L%08lx\015\012N%08lx\015\012"
				/* index file record format */
#define IXRFMT ":%08lx:%04d%02d%02d%02d%02d%02d%c%02d%02d:%08lx:%08lx:%08lx:%08lx:%08lx:\015\012"
				/* status file record format */
#define STRFMT ":%08lx:%08lx:%04x:%08lx:\015\012"
				/* message file header format */
#define MSRFMT "%s%08lx:%04d%02d%02d%02d%02d%02d%c%02d%02d:%08lx:\015\012"
#define MSGTOK ":msg:"

/* Main program */

int main (int argc,char *argv[])
{
  FILE *meta,*index,*status;
  time_t now = time (0);
  MESSAGECACHE *elt;
  char *s,dest[MAILTMPLEN],tmp[MAILTMPLEN];
  short moreswitchp;
  unsigned long i;
  long dirprot = (long) mail_parameters (NIL,GET_DIRPROTECTION,NIL);
  long mbxprot = (long) mail_parameters (NIL,GET_MBXPROTECTION,NIL);
  MAILSTREAM *source = NIL;
  int retcode = 1;
  char *src = NIL;
  char *dst = NIL;
  char *pgm = argc ? argv[0] : "mixcvt";
#include "linkage.c"
  for (i = moreswitchp = 1; i < argc; i++) {
    s = argv[i];		/* pick up argument */
				/* parse switches */
    if (moreswitchp && (*s == '-')) {
      if (!strcmp (s,"-debug") || !strcmp (s,"-d")) debugp = T;
      else if (!strcmp (s,"-verbose") || !strcmp (s,"-v")) verbosep = T;
      else if ((!strcmp (s,"-user") || !strcmp (s,"-u")) && (++i < argc)) {
	struct passwd *pw = getpwnam (s = argv[i]);
	if (!pw) {
	  printf ("unknown user id: %s\n",argv[i]);
	  exit (retcode);
	}
	else if (setuid (pw->pw_uid)) {
	  perror ("unable to change user id");
	  exit (retcode);
	}
      }
				/* -- means no more switches, so mailbox
				   name can start with "-" */
      else if ((s[1] == '-') && !s[2]) moreswitchp = NIL;
      else {
	printf ("unknown switch: %s\n",s);
	exit (retcode);
      }
    }
    else if (!src) src = s;	/* second non-switch is source */
    else if (!dst) dst = s;	/* third non-switch is destination */
    else {
      printf ("unknown argument: %s\n",s);
      exit (retcode);
    }
  }

  if (!(src && dst && (strlen (dst) < 500) && mailboxfile (dest,dst) &&
	(*dest || mailboxfile (dest,"~/INBOX"))))
    printf ("%s version %s.%s\n\nusage: %s [-d] [-v] source destination\n",
	    pgm,CCLIENTVERSION,version,pgm);
  else if (!(source = mail_open (NIL,src,
				 OP_READONLY | (debugp ? OP_DEBUG : NIL))));
  else if (mkdir (dest,(mode_t) dirprot))
    fprintf (stderr,"Unable to create directory %.800s: %.80s\n",
	     dest,strerror (errno));
  else if (!(meta = fopen (strcat (strcpy (tmp,strcat (dest,"/" MIXNAME)),
				   MIXMETA),"w")))
    fprintf (stderr,"Unable to create metadata %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (fchmod (fileno (meta),(mode_t) mbxprot))
    fprintf (stderr,"Unable to protect metadata %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (!(index = fopen (strcat (strcpy (tmp,dest),MIXINDEX),"w")))
    fprintf (stderr,"Unable to create index %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (fchmod (fileno (index),(mode_t) mbxprot))
    fprintf (stderr,"Unable to protect index %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (!(status = fopen (strcat (strcpy (tmp,dest),MIXSTATUS),"w")))
    fprintf (stderr,"Unable to create status %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (fchmod (fileno (status),(mode_t) mbxprot))
    fprintf (stderr,"Unable to protect status %.800s: %.80s\n",
	     tmp,strerror (errno));
  else {

    FILE *data = NIL;
    unsigned long curfile = 0;
    if (source->nmsgs) {
      sprintf (tmp,"1:%lu",source->nmsgs);
      mail_fetch_fast (source,tmp,NIL);
    }
				/* try to lock up destination */
    flock (fileno (meta),LOCK_EX);
    fprintf (index,SEQFMT,now);	/* write index and data files */
    for (i = 1; write_message (source,i,index,&data,&curfile,dest,now); ++i);
    if (ferror (index))
      fprintf (stderr,"Unable to write index: %.80s\n",strerror (errno));
    else if (data && ferror (data))
      fprintf (stderr,"Unable to write data: %.80s\n",strerror (errno));
    else if (i <= source->nmsgs)
      fprintf (stderr,"Failed at message %lu\n",i);
    else if (fclose (index))
      fprintf (stderr,"Unable to close index: %.80s\n",strerror (errno));
    else if (fclose (data))
      fprintf (stderr,"Unable to close data: %.80s\n",strerror (errno));
    else {			/* write status */
      fprintf (status,SEQFMT,now);
      for (i = 1; i <= source->nmsgs; ++i) {
	elt = mail_elt (source,i);
	/* Someday change it to use elt->private.mod instead of now */
	fprintf (status,STRFMT,mail_uid (source,i),elt->user_flags,
		 (fSEEN * elt->seen) + (fDELETED * elt->deleted) +
		 (fFLAGGED * elt->flagged) + (fANSWERED * elt->answered) +
		 (fDRAFT * elt->draft) + (elt->recent ? NIL : fOLD),now);
      }
      if (ferror (status))
	fprintf (stderr,"Unable to write status: %.80s\n",strerror (errno));
      else if (fclose (status))
	fprintf (stderr,"Unable to close status: %.80s\n",strerror (errno));
      else {			/* write metadata */
	fprintf (meta,SEQFMT,now);
	fprintf (meta,MTAFMT,source->uid_validity,source->uid_last,curfile);
	if (source->user_flags[0]) {
	  for (i = 0; (i < NUSERFLAGS) && source->user_flags[i]; ++i)
	    fprintf (meta,"%c%s",i ? ' ' : 'K',source->user_flags[i]);
	  fputs ("\015\012",meta);
	}
	if (ferror (meta))
	  fprintf (stderr,"Unable to write metadata: %.80s\n",
		   strerror (errno));
	else if (fclose (meta))
	  fprintf (stderr,"Unable to close meta: %.80s\n",strerror (errno));
	else retcode = NIL;		/* all looks good */
      }
    }
  }
				/* close source */
  if (source) mail_close (source);
  exit (retcode);
  return retcode;		/* stupid compilers */
}

/* Write message to index and data file
 * Accepts: MAIL stream of source
 *	    message number
 *	    index file
 *	    pointer to data file
 *	    pointer to current data file number
 *	    destination name string
 *	    file number for new message (1 + maximum we will use)
 * Returns: T if success, NIL if failure
 */

long write_message (MAILSTREAM *source,unsigned long i,FILE *index,FILE **data,
		    unsigned long *curfile,char *dest,time_t now)
{
  MESSAGECACHE *elt;
  char *s;
  unsigned long msgpos,hdroff,hdrlen,txtlen;
  if (i > source->nmsgs) {	/* end of mailbox? */
				/* create a data file if source empty */
    if (!source->nmsgs) *data = create_data (dest,*curfile = now);
    return NIL;
  }
  elt = mail_elt (source,i);	/* get elt for this message */
  if (!*data ||			/* need to make data file here? */
      (((msgpos = ftell (*data)) + elt->rfc822_size) >= MIXDATAROLL)) {
    if (*data && fclose (*data)) {
      fprintf (stderr,"Unable to close/roll data: %.80s\n",strerror (errno));
      return NIL;
    }
    if (*data = create_data (dest,*curfile = now - i)) msgpos = 0;
    else return NIL;
  }
  fprintf (*data,MSRFMT,MSGTOK,mail_uid (source,i),
	   elt->year + BASEYEAR,elt->month,elt->day,
	   elt->hours,elt->minutes,elt->seconds,
	   elt->zoccident ? '-' : '+',elt->zhours,elt->zminutes,
	   elt->rfc822_size);
  if (!ferror (*data)) {
    hdroff = ftell (*data) - msgpos;
    s = mail_fetch_header (source,i,NIL,NIL,&hdrlen,FT_PEEK);
    if (!hdrlen || fwrite (s,hdrlen,1,*data)) {
      s = mail_fetch_text (source,i,NIL,&txtlen,FT_PEEK);
      if (elt->rfc822_size != (hdrlen + txtlen))
	fprintf (stderr,"Data length error %lu != %lu + %lu\n",
		 elt->rfc822_size,hdrlen,txtlen);
      else if (!txtlen || fwrite (s,txtlen,1,*data)) {
	fprintf (index,IXRFMT,mail_uid (source,i),
		 elt->year + BASEYEAR,elt->month,elt->day,
		 elt->hours,elt->minutes,elt->seconds,
		 elt->zoccident ? '-' : '+',elt->zhours,elt->zminutes,
		 elt->rfc822_size,*curfile,msgpos,hdroff,hdrlen);
	if (!ferror (index)) return LONGT;
      }
    }
  }
  return NIL;
}

/* Create data file
 * Accepts: base string for data file name
 *	    current file number
 * Returns: FILE structure if success, NIL otherwise
 */

FILE *create_data (char *dest,unsigned long curfile)
{
  FILE *data;
  char tmp[MAILTMPLEN];
  long mbxprot = (long) mail_parameters (NIL,GET_MBXPROTECTION,NIL);
				/* build data file name and number */
  sprintf (tmp,"%s%08lx",dest,curfile);
  if (!(data = fopen (tmp,"w")))
    fprintf (stderr,"Unable to create data %.800s: %.80s\n",
	     tmp,strerror (errno));
  else if (fchmod (fileno (data),(mode_t) mbxprot)) {
    fprintf (stderr,"Unable to protect data %.800s: %.80s\n",
	     tmp,strerror (errno));
    fclose (data);
    data = NIL;
  }
  return data;
}

/* Co-routines from MAIL library */


/* Message matches a search
 * Accepts: MAIL stream
 *	    message number
 */

void mm_searched (MAILSTREAM *stream,unsigned long msgno)
{
				/* dummy routine */
}


/* Message exists (i.e. there are that many messages in the mailbox)
 * Accepts: MAIL stream
 *	    message number
 */

void mm_exists (MAILSTREAM *stream,unsigned long number)
{
				/* dummy routine */
}


/* Message expunged
 * Accepts: MAIL stream
 *	    message number
 */

void mm_expunged (MAILSTREAM *stream,unsigned long number)
{
				/* dummy routine */
}


/* Message flags update seen
 * Accepts: MAIL stream
 *	    message number
 */

void mm_flags (MAILSTREAM *stream,unsigned long number)
{
				/* dummy routine */
}

/* Mailbox found
 * Accepts: MAIL stream
 *	    hierarchy delimiter
 *	    mailbox name
 *	    mailbox attributes
 */

void mm_list (MAILSTREAM *stream,int delimiter,char *name,long attributes)
{
				/* dummy routine */
}


/* Subscribe mailbox found
 * Accepts: MAIL stream
 *	    hierarchy delimiter
 *	    mailbox name
 *	    mailbox attributes
 */

void mm_lsub (MAILSTREAM *stream,int delimiter,char *name,long attributes)
{
				/* dummy routine */
}


/* Mailbox status
 * Accepts: MAIL stream
 *	    mailbox name
 *	    mailbox status
 */

void mm_status (MAILSTREAM *stream,char *mailbox,MAILSTATUS *status)
{
				/* dummy routine */
}

/* Notification event
 * Accepts: MAIL stream
 *	    string to log
 *	    error flag
 */

void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
}


/* Log an event for the user to see
 * Accepts: string to log
 *	    error flag
 */

void mm_log (char *string,long errflg)
{
  switch (errflg) {  
  case BYE:
  case NIL:			/* no error */
    if (verbosep) fprintf (stderr,"[%s]\n",string);
    break;
  case PARSE:			/* parsing problem */
  case WARN:			/* warning */
    fprintf (stderr,"warning: %s\n",string);
    break;
  case ERROR:			/* error */
  default:
    fprintf (stderr,"%s\n",string);
    break;
  }
}


/* Log an event to debugging telemetry
 * Accepts: string to log
 */

void mm_dlog (char *string)
{
  fprintf (stderr,"%s\n",string);
}

/* Get user name and password for this host
 * Accepts: parse of network mailbox name
 *	    where to return user name
 *	    where to return password
 *	    trial count
 */

void mm_login (NETMBX *mb,char *username,char *password,long trial)
{
  char *s,tmp[MAILTMPLEN];
  if (*mb->user) {
    sprintf (tmp,"{%s/%s/user=%s} password: ",mb->host,mb->service,
	     strcpy (username,mb->user));
    s = tmp;
  }
  else {
    printf ("{%s/%s} username: ",mb->host,mb->service);
    fgets (username,NETMAXUSER-1,stdin);
    username[NETMAXUSER-1] = '\0';
    if (s = strchr (username,'\n')) *s = '\0';
    s = "password: ";
  }
  strcpy (password,getpass (s));
}


/* About to enter critical code
 * Accepts: stream
 */

void mm_critical (MAILSTREAM *stream)
{
  critical = T;			/* note in critical code */
}


/* About to exit critical code
 * Accepts: stream
 */

void mm_nocritical (MAILSTREAM *stream)
{
  critical = NIL;		/* note not in critical code */
}


/* Disk error found
 * Accepts: stream
 *	    system error code
 *	    flag indicating that mailbox may be clobbered
 * Returns: T if user wants to abort
 */

long mm_diskerror (MAILSTREAM *stream,long errcode,long serious)
{
  return T;
}


/* Log a fatal error event
 * Accepts: string to log
 */

void mm_fatal (char *string)
{
  fprintf (stderr,"FATAL: %s\n",string);
}
