//
//  XTHtmlCharEntityExpanderTests.m
//  TadsTerp
//
//  Created by Rune Berg on 05/04/14.
//  Copyright (c) 2014 Rune Berg. All rights reserved.
//

#import <XCTest/XCTest.h>
#import "XTHtmlCharEntityExpander.h"


@interface XTHtmlCharEntityExpanderTests : XCTestCase

@property XTHtmlCharEntityExpander *expander;

@end


@implementation XTHtmlCharEntityExpanderTests

- (void)setUp
{
    [super setUp];
    self.expander = [XTHtmlCharEntityExpander new];
}

- (void)tearDown
{
    [super tearDown];
}

- (void)test_nil
{
	XCTAssertNil([self.expander expand:nil]);
}

- (void)test_emptyString
{
	NSString *string = @"";
	XCTAssertEqualObjects(string, [self.expander expand:string]);
}

- (void)test_stringWithoutCharEntity
{
	NSString *string = @"a";
	XCTAssertEqualObjects(string, [self.expander expand:string]);
	
	string = @"abc";
	XCTAssertEqualObjects(string, [self.expander expand:string]);
}

//TODO strings with incomplete char entities

- (void)test_singleNamedCharEntity
{
	NSString *string = @"&gt;";
	XCTAssertEqualObjects(@">", [self.expander expand:string]);
}

- (void)test_mixOfNamedCharEntitiesAndText
{
	NSString *string = @"abc&gt;de&lt;f";
	XCTAssertEqualObjects(@"abc>de<f", [self.expander expand:string]);

	string = @"&gt;&lt;&amp;";
	XCTAssertEqualObjects(@"><&", [self.expander expand:string]);
}

- (void)test_unknownNamedCharEntity
{
	NSString *string = @"&foobar;";
	XCTAssertEqualObjects(@"&foobar;", [self.expander expand:string]);
}


- (void)test_singleHexCharEntity
{
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#x26;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#x026;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#x0026;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#x000026;"]);

	XCTAssertEqualObjects(@"<", [self.expander expand:@"&#x3C;"]);
	XCTAssertEqualObjects(@"<", [self.expander expand:@"&#x3c;"]);
}

- (void)test_singleDecimalCharEntity
{
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#38;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#038;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#0038;"]);
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&#000038;"]);
	
	XCTAssertEqualObjects(@"<", [self.expander expand:@"&#60;"]);
}

- (void)test_singleSpecialChars
{
	XCTAssertEqualObjects(@"\x15", [self.expander expand:@"&#21;"]);
	XCTAssertEqualObjects(@"\x15", [self.expander expand:@"&#x15;"]);
}

- (void)test_illegalCharInState
{
	XCTAssertEqualObjects(@"&+", [self.expander expand:@"&+"]);
	XCTAssertEqualObjects(@"&#+", [self.expander expand:@"&#+"]);
	XCTAssertEqualObjects(@"&am+", [self.expander expand:@"&am+"]);
	XCTAssertEqualObjects(@"&#x+", [self.expander expand:@"&#x+"]);
	XCTAssertEqualObjects(@"&#x8+", [self.expander expand:@"&#x8+"]);
	XCTAssertEqualObjects(@"&#8+", [self.expander expand:@"&#8+"]);
}

- (void)test_unfinishedEntity
{
	XCTAssertEqualObjects(@"&", [self.expander expand:@"&"]);
	XCTAssertEqualObjects(@"&#", [self.expander expand:@"&#"]);
	XCTAssertEqualObjects(@"&#x", [self.expander expand:@"&#x"]);
	XCTAssertEqualObjects(@"&#x1", [self.expander expand:@"&#x1"]);

	XCTAssertEqualObjects(@"&#2", [self.expander expand:@"&#2"]);

	XCTAssertEqualObjects(@"&&", [self.expander expand:@"&&"]);
	XCTAssertEqualObjects(@"&#&", [self.expander expand:@"&#&"]);
	XCTAssertEqualObjects(@"&#x&", [self.expander expand:@"&#x&"]);
	XCTAssertEqualObjects(@"&&&", [self.expander expand:@"&&&"]);

	XCTAssertEqualObjects(@"<&&", [self.expander expand:@"&lt&&"]);
	XCTAssertEqualObjects(@"&<&&", [self.expander expand:@"&&lt&&"]);
}

- (void)test_missingSemicolonAfterNamedCharEntity
{
	NSString *s;
	
	s = [self.expander expand:@"&quotxx;"];
	XCTAssertEqualObjects(@"\"xx;", s);
	
	s = [self.expander expand:@"&quotx;"];
	XCTAssertEqualObjects(@"\"x;", s);
	
	s = [self.expander expand:@"&quotx"];
	XCTAssertEqualObjects(@"\"x", s);
	
	s = [self.expander expand:@"&quot"];
	XCTAssertEqualObjects(@"\"", s);
	
	s = [self.expander expand:@"&poook"];
	XCTAssertEqualObjects(@"&poook", s);
	
	s = [self.expander expand:@"&gt&"];
	XCTAssertEqualObjects(@">&", s);

	s = [self.expander expand:@"&gtc"];
	XCTAssertEqualObjects(@">c", s);
	
	s = [self.expander expand:@"&gtcast"];
	XCTAssertEqualObjects(@">cast", s);
	
	s = [self.expander expand:@"&gtcast fireball"];
	XCTAssertEqualObjects(@">cast fireball", s);

	s = [self.expander expand:@"a&gtcast fireball"];
	XCTAssertEqualObjects(@"a>cast fireball", s);
	
	s = [self.expander expand:@"&lt;&gt; fireball"];
	XCTAssertEqualObjects(@"<> fireball", s);

	s = [self.expander expand:@"&lt;&gtcast F"];
	XCTAssertEqualObjects(@"<>cast F", s);
	
	s = [self.expander expand:@"&#21;&gtcast fireball"];
	XCTAssertEqualObjects(@"\x15>cast fireball", s);
	
	s = [self.expander expand:@"&#21;&#21;&#21;&gtcast fireball at goblin mage"];
	XCTAssertEqualObjects(@"\x15\x15\x15>cast fireball at goblin mage", s);

	s = [self.expander expand:@"a&gtcast"];
	XCTAssertEqualObjects(@"a>cast", s);
	
	s = [self.expander expand:@"&lt;&gtcast"];
	XCTAssertEqualObjects(@"<>cast", s);

	s = [self.expander expand:@"&#21;&gtcast"];
	XCTAssertEqualObjects(@"\x15>cast", s);
	
	s = [self.expander expand:@"&#21;&#21;&#21;&gtcast"];
	XCTAssertEqualObjects(@"\x15\x15\x15>cast", s);
}

- (void)test_enspSpecialHandling
{
	NSString *s;
	
	s = [self.expander expand:@"&#8194;"];
	XCTAssertEqualObjects(@"  ", s);
	
	s = [self.expander expand:@"a&#8194;b"];
	XCTAssertEqualObjects(@"a  b", s);
	
	s = [self.expander expand:@"&ensp;"];
	XCTAssertEqualObjects(@"  ", s);
	
	s = [self.expander expand:@"a&ensp;b"];
	XCTAssertEqualObjects(@"a  b", s);
}

//TODO write many more tests ...

@end
