/* Chatter: A TADS NPC Interaction Library
 * Version 1.0
 * Suzanne Britton, 1999, Public Domain
 * tril@igs.net
 */

#pragma C+

// See the manual for explanations of the public properties (those not
// listed under "implementation").

/********************* Function Pre-declarations *******************/

preparseValidateSay: function;
preparseRemoveComma: function;
preparseConvertAutoTopic: function;
preparseConvertRequest: function;
preparseConvertThem: function;
preparseConvertHer: function;

/*************************** New Superclasses ***************************/

class topic: thing
  /*** Properties to Override ***/

  known = nil
  whatis = "You'll have to discover that for yourself."

  /*** Implementation ***/

  location = nil

  doWhatis(actor) = {
    self.whatis;
  }
;

class knownTopic: topic
  known = true
;

class infoSource: thing

  /*** Handlers and Default Responses to Override ***/

  lookupTopics(topic, words) = {return nil;}
  lookupDefault = "There's no entry for that topic."

  /*** Information Methods ***/

  lookedUp(t) = {
    local found;

    found = find(self.lookedUpList, t);
    if (found == nil)
      return nil;
    else
      return self.lookedUpStatus[found];
  }

  /*** Implementation ***/

  lookedUpList = []
  lookedUpStatus = []

  lookupTopicsAndRecord(topic, words) = {
    local retVal;
    local found;

    retVal = self.lookupTopics(topic, words);
    if (retVal != nil) {
      found = find(self.lookedUpList, topic);
      if (found == nil) {
        self.lookedUpList += topic;
        self.lookedUpStatus += retVal;
      }
      else
        self.lookedUpStatus[found] = retVal;
    }
    return (retVal != nil);
  }

  verDoConsult(actor) = {}
  doConsult(actor) = {
    /* If we're in conversation mode, turn it off first: */
    if (global.talkingTo != nil) {
      talkOffVerb.performTalkOff;
      "\n";
    }

    "[You are now consulting <<self.thedesc>>. Type \"consult help\"
    for assistance, \"consult off\" to exit.]";
    global.consulting = self;
    abort;
  }

  verDoConsultOn(actor, io) = {self.verIoLookupIn(actor);}
  doConsultOn(actor, io) = {
    io = translateTopic(io, self, true, nil);
    if (!self.lookupTopicsAndRecord(io, objwords(2)))
      self.lookupDefault;
  }

  verIoLookupIn(actor) = {}
  ioLookupIn(actor, dobj) = {
    dobj = translateTopic(dobj, self, nil, nil);
    if (!self.lookupTopicsAndRecord(dobj, objwords(1)))
      self.lookupDefault;
  }
;

class yesno: object
  /*** Options to Override ***/

  actor = nil
  expireTime = 3
  noText = ""
  req = true
  resetAfterAnswer = true
  yesText = ""

  /*** Methods to call in your own code ***/

  expire = {
    if (global.lastYesnoAsked == self)
      global.lastYesnoAsked = nil;
    if (self.actor.lastYesnoAsked == self)
      self.actor.lastYesnoAsked = nil;
  }
  
  flag = {
    unnotify(self, &expire);
    global.lastYesnoAsked = self;
    self.actor.lastYesnoAsked = self;
    if (self.expireTime != nil)
      notify(self, &expire, expireTime+1);
  }

  /*** Implementation ***/

  /* process(val): Process this yesno question with the given answer (val is
   * true for "yes", nil for "no"). This is the bottleneck for all yes/no
   * responses, however phrased by the player.
   */
  process(val) = {
    if (!self.actor.isReachable(parserGetMe()))
      global.globalYesnoDefault;
    else if (!self.actor.canInteract(1)) {
      // do nothing
    }
    else if (!self.req)
      self.actor.yesNoDefault;
    else {
      if (self.resetAfterAnswer)
        self.expire;
      if (val)
        self.yesText;
      else
        self.noText;
    }
  }
;

/************************** adv.t Modifications **************************/

modify basicMe

  /*** Options & Defaults to Override ***/

  askDefault = "Predictably, there is no response."
  canSpeak = true
  mytopic = pcTopic

  /*** Implementation ***/

  noun = 'self'

  verIoShowTo(actor) = {}
  ioShowTo(actor, dobj) = {dobj.ldesc;}
;

modify global

  /*** Handlers, Options, & Defaults to Override ***/

  globalSayHandler(str) = {"\"<<str>>\"";}
  globalYesnoDefault = "There is no response."
  unknownMsg = "You don't know about that."

  preparseSequence = [
    &preparseValidateSay,
    &preparseRemoveComma,
    &preparseConvertAutoTopic,
    &preparseConvertRequest,
    &preparseConvertThem,
    &preparseConvertHer
  ]

  /*** Implementation ***/

  /* inConvModeTopic is set and reset during preparsing. It assists in the
   * disambiguation kludge as explained in askVerb.disambigDobj.
   */
  inConvModeTopic = nil

  /* Set within the hook parseNounPhrase so that it can call parseNounList
   * without going into an infinite loop. When parseNounPhrase detects that
   * it is being called (indirectly) by parseNounPhrase, it exits immediately.
   */
  inParseNounPhrase = nil

  /* The last yes/no question posed by the player to anyone. */
  lastYesnoAsked = nil

  /* The character with whom the player is in conversation with. nil if
   * conversation mode is not on.
   */
  talkingTo = nil

  /* The info source which the player is currently consulting. nil if consult
   * mode is not on.
   */
  consulting = nil
;

modify thing

  /*** Stuff to Override ***/

  mytopic = nil
  doubles = []

  /*** Implementation ***/

  /* modified to give a better adesc for a plural-type object */
  adesc = {
    if (self.isThem)
      "some <<self.sdesc>>";
    else
      pass adesc;
  }

  /* object pronoun for this object */
  objdesc = {
    if (self.isThem)
      "them";
    else if (self.isHim && !self.isHer)
      "him";
    else if (self.isHer && !self.isHim)
      "her";
    else
      "it";
  }

  /* subject pronoun for this object */
  subjdesc = {
    if (self.isThem)
      "they";
    else if (self.isHim && !self.isHer)
      "he";
    else if (self.isHer && !self.isHim)
      "she";
    else
      "it";
  }

  /*** Standard parser properties ***/

  replace actorAction(v, d, p, i) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
    exit;
  }

  verDoApologize(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  replace verDoAskAbout(actor, io) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  // These are the same as in adv.t, copied here for clarity
  replace verIoAskAbout(actor) = {}
  replace ioAskAbout(actor, dobj) = {
    dobj.doAskAbout(actor, self);
  }

  verIoAskFor(actor) = {}
  verDoAskFor(actor, io) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }
  ioAskFor(actor, dobj) = {dobj.doAskFor(actor, self);}

  verDoConsult(actor) = {"That's not an information source.";}
  verIoConsultOn(actor) = {}
  verDoConsultOn(actor, io) = {"That's not an information source.";}
  ioConsultOn(actor, dobj) = {dobj.doConsultOn(actor, self);}

  replace verIoGiveTo(actor) = {"There is no reaction to this.";}

  verDoGoodbye(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  verDoGreet(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  verDoHelp(actor) = {
    "You'll have to be more specific about how to do that.";
  }

  verIoLookupIn(actor) = {"That's not an information source.";}
  verDoLookupIn(actor, io) = {}

  verDoNo(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  verIoSayTo(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  replace verIoShowTo(actor) = {"There is no reaction to this.";}

  verDoSorry(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response to your apology.";
  }

  verDoTalk(actor) = {}
  doTalk(actor) = {
    /* If we're in consult mode, turn it off first: */
    if (global.consulting != nil) {
      consultOffVerb.performConsultOff;
      "\n";
    }

    "[You are now in conversation with <<self.thedesc>>. Type \"talk help\"
    for assistance, \"talk off\" to exit.]";
    global.talkingTo = self;
    abort;
  }

  verDoTalkAbout(actor, io) = {}
  verIoTalkAbout(actor) = {
    "You can ask about a topic, or tell about a topic. Please be more
    specific.";
  }

  replace verDoTellAbout(actor, io) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }
  // These are the same as in adv.t, copied here for clarity
  replace verIoTellAbout(actor) = {}
  replace ioTellAbout(actor, dobj) = {
    dobj.doTellAbout(actor, self);
  }

  verDoThank(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }

  verDoWhatis(actor) = {}
  doWhatis(actor) = {
    if (self.mytopic == nil)
      whatisVerb.noInfoMsg;
    else
      self.mytopic.doWhatis(actor);
  }

  verDoYes(actor) = {
    if (parserGetMe().canSpeak)
      "There is no response.";
  }
;

modify movableActor

  /*** Handlers to Override ***/

  askTopics(topic, words) = nil
  askForTopics(topic, words) = nil
  customAction(v,d,p,i) = nil
  customGive(obj) = nil
  customShow(obj) = nil
  sayHandler(str) = {self.tellDefault;}
  tellTopics(topic, words) = nil

  /*** Options to Override ***/

  askForRedirect = nil
  canInteract(verbType) = true
  giveRedirect = nil
  showRedirect = true
  tellRedirect = nil

  /*** Default Responses to Override ***/

  actionDefault = {self.tellDefault;}
  askDefault = "There is no response."
  askForDefault = {self.askDefault;}
  giveDefault = "\^<<self.subjdesc>> <<self.isThem ? 'show' : 'shows'>> no
                interest in this."
  goodbyeResponse = {self.tellDefault;}
  helloResponse = {self.tellDefault;}
  helpResponse = {self.actionDefault;}
  showDefault = "\^<<self.subjdesc>> <<self.isThem ? 'show' : 'shows'>> no
    interest in this."
  sorryResponse = {self.tellDefault;}
  tellDefault = {self.askDefault;}
  thankResponse = {self.tellDefault;}
  yesNoDefault =
    "\^<<self.subjdesc>> <<self.isThem ? "haven't" : "hasn't">> asked you
    a yes/no question."

  /*** Information Methods ***/

  askedAbout(t) = {
    local found;

    found = find(self.askedAboutList, t);
    if (found == nil)
      return nil;
    else
      return self.askedAboutStatus[found];
  }

  askedFor(t) = {
    local found;

    found = find(self.askedForList, t);
    if (found == nil)
      return nil;
    else
      return self.askedForStatus[found];
  }

  toldAbout(t) = {
    local found;

    found = find(self.toldAboutList, t);
    if (found == nil)
      return nil;
    else
      return self.toldAboutStatus[found];
  }


  /*** Implementation ***/

  // Custom Handlers and Properties

  lastYesnoAsked = nil
  askedAboutList = []
  askedAboutStatus = []
  askedForList = []
  askedForStatus = []
  toldAboutList = []
  toldAboutStatus = []

#define ALL_EXCEPT nil

  attemptTopicRedirect(topic, words, redirectList, tryTellAlso) = {
    if (topic == nil || redirectList == nil)
      return nil;
    if (datatype(redirectList) == DTY_TRUE ||
        (redirectList[1] == ALL_EXCEPT && find(redirectList, topic) == nil) ||
        (redirectList[1] != ALL_EXCEPT && find(redirectList, topic) != nil)) {
      if (tryTellAlso)
        return (self.tellTopicsAndRecord(topic, words) ||
                self.askTopicsAndRecord(topic, words));
      else
        return self.askTopicsAndRecord(topic, words);
    }
    else
      return nil;
  }

  attemptYesNo(val) = {
    if (self.lastYesnoAsked != nil)
      self.lastYesnoAsked.process(val);
    else
      self.yesNoDefault;
  }

  handleAskAbout(topic, words, translateSelfRefs, translateYou) = {
    topic = translateTopic(topic, self, translateSelfRefs, translateYou);
    if (topic == helpTopic)
      self.helpResponse;
    else if (!self.askTopicsAndRecord(topic, words))
      self.askDefault;
  }

  askTopicsAndRecord(topic, words) = {
    local retVal;
    local found;

    retVal = self.askTopics(topic, words);
    if (retVal != nil) {
      found = find(self.askedAboutList, topic);
      if (found == nil) {
        self.askedAboutList += topic;
        self.askedAboutStatus += retVal;
      }
      else
        self.askedAboutStatus[found] = retVal;
    }
    return (retVal != nil);
  }

  handleAskFor(topic, words, translateSelfRefs, translateYou) = {
    topic = translateTopic(topic, self, translateSelfRefs, translateYou);
    if (topic == helpTopic) {
      self.helpResponse;
      return;
    }
    if (self.askForTopicsAndRecord(topic, words))
      return;
    if (self.attemptTopicRedirect(topic, words, self.askForRedirect, nil))
      return;
    self.askForDefault;
  }

  askForTopicsAndRecord(topic, words) = {
    local retVal;
    local found;

    retVal = self.askForTopics(topic, words);
    if (retVal != nil) {
      found = find(self.askedForList, topic);
      if (found == nil) {
        self.askedForList += topic;
        self.askedForStatus += retVal;
      }
      else
        self.askedForStatus[found] = retVal;
    }
    return (retVal != nil);
  }

  handleTell(topic, words, translateSelfRefs, translateYou) = {
    topic = translateTopic(topic, self, translateSelfRefs, translateYou);
    if (self.tellTopicsAndRecord(topic, words))
      return;
    if (self.attemptTopicRedirect(topic, words, self.tellRedirect, nil))
      return;
    self.tellDefault;
  }

  tellTopicsAndRecord(topic, words) = {
    local retVal;
    local found;

    retVal = self.tellTopics(topic, words);
    if (retVal != nil) {
      found = find(self.toldAboutList, topic);
      if (found == nil) {
        self.toldAboutList += topic;
        self.toldAboutStatus += retVal;
      }
      else
        self.toldAboutStatus[found] = retVal;
    }
    return (retVal != nil);
  }

  // Standard Parser Methods

  replace actorAction(v,d,p,i) = {
    local tempTopic;

    if (!parserGetMe().canSpeak || !self.canInteract(1))
      exit;
    if (self.customAction(v,d,p,i)) {
      // do nothing
    }
    else if (v == helloVerb)
      self.helloResponse;
    else if (v == goodbyeVerb)
      self.goodbyeResponse;
    else if (v == thankVerb)
      self.thankResponse;
    else if (v == apologizeVerb)
      self.sorryResponse;
    else if (v == helpVerb && (d == nil || d == parserGetMe()))
      self.helpResponse;
    else if (v == yesVerb || v == noVerb)
      self.attemptYesNo((v == yesVerb));
    else if (v == whoamiVerb)
      self.handleAskAbout(pcTopic, ['me'], nil, nil);
    else if (v == whatisVerb)
      self.handleAskAbout(d, objwords(1), true, true);
    else if (v == tellVerb && d == parserGetMe())
      self.handleAskAbout(i, objwords(2), nil, true);
    else if (v == giveVerb && i == parserGetMe())
      self.handleAskFor(d, objwords(1), nil, true);
    else if (v == inspectVerb)
      self.ioShowTo(parserGetMe(), d);
    else
      self.actionDefault;
    exit;
  }

  replace doAskAbout(actor, io) = {
    if (!parserGetMe().canSpeak || !self.canInteract(1))
      return;
    self.handleAskAbout(io, objwords(2), true, nil);
  }

  verDoAskFor(actor, io) = {}
  doAskFor(actor, io) = {
    if (!parserGetMe().canSpeak || !self.canInteract(1))
      return;
    self.handleAskFor(io, objwords(2), true, nil);
  }

  verIoGiveTo(actor) = {}
  replace ioGiveTo(actor, dobj) = {
    local redirect;

    if (!self.canInteract(3))
      return;
    if (self.customGive(dobj))
      return;
    redirect = self.giveRedirect;
    if (datatype(redirect) == DTY_TRUE || datatype(redirect) == DTY_LIST &&
        ((redirect[1] == ALL_EXCEPT && find(redirect, dobj) == nil) ||
         (redirect[1] != ALL_EXCEPT && find(redirect, dobj) != nil)))
       self.ioShowTo(actor, dobj);
    else
      self.giveDefault;
  }

  verDoGoodbye(actor) = {}
  doGoodbye(actor) = {
    if (!parserGetMe().canSpeak || !self.canInteract(1))
      return;
    self.goodbyeResponse;
  }

  verDoGreet(actor) = {}
  doGreet(actor) = {
    if (!parserGetMe().canSpeak || !self.canInteract(1))
      return;
    self.helloResponse;
  }

  verDoNo(actor) = {}
  doNo(actor) = {execCommand(self, noVerb);}

  verIoSayTo(actor) = {}
  ioSayTo(actor, dobj) = {
    if (!parserGetMe().canSpeak || !self.canInteract(1))
      return;
    if (dobj.checkForYesNo(self)) {
      // do nothing
    }
    else if (dobj.checkForHello(self)) {
      // do nothing
    }
    else if (dobj.checkForThanks(self)) {
      // do nothing
    }
    else
      self.sayHandler(dobj.value);
  }

  verIoShowTo(actor) = {}
  replace ioShowTo(actor, dobj) = {
    if (!self.canInteract(2))
      return;
    if (self.customShow(dobj))
      return;
    if (self.attemptTopicRedirect(dobj.mytopic, objwords(1), self.showRedirect,
                                  true))
      return;
    self.showDefault;
  }

  verDoSorry(actor) = {}
  doSorry(actor) = {
    if (parserGetMe().canSpeak && self.canInteract(1))
      self.sorryResponse;
  }

  verDoTellAbout(actor, io) = {}
  doTellAbout(actor, io) = {
    if (parserGetMe().canSpeak && self.canInteract(1))
      self.handleTell(io, objwords(2), true, nil);
  }

  verDoThank(actor) = {}
  doThank(actor) = {
    if (parserGetMe().canSpeak && self.canInteract(1))
      self.thankResponse;
  }

  verDoYes(actor) = {}
  doYes(actor) = {execCommand(self, yesVerb);}
;

modify basicStrObj

  /*** Implementation ***/

  verDoSayTo(actor, io) = {}
  doSay(actor) = {
    local str;

    if (!parserGetMe().canSpeak)
      return;

    if (self.checkForHello(nil)) {
      // do nothing
    }
    else if (self.checkForThanks(nil)) {
      // do nothing
    }
    else if (self.checkForYesNo(nil)) {
      // do nothing
    }
    else
      global.globalSayHandler(self.value);
  }

  checkForHello(actor) = {
    local lowerStr = lower(self.value);

    if (lowerStr == 'hello' || lowerStr == 'hello.' ||
        lowerStr == 'hi' || lowerStr == 'hi.' ||
        lowerStr == 'greetings' || lowerStr == 'greetings.') {
      if (actor == nil)
        parserReplaceCommand('hello');
      else
        execCommand(parserGetMe(), helloVerb, actor);
      return true;
    }
    else if (lowerStr == 'bye' || lowerStr == 'bye.' ||
        lowerStr == 'goodbye' || lowerStr == 'goodbye.' ||
        lowerStr == 'good bye' || lowerStr == 'good bye.' ||
        lowerStr == 'farewell' || lowerStr == 'farewell.') {
      if (actor == nil)
        parserReplaceCommand('goodbye');
      else
        execCommand(parserGetMe(), goodbyeVerb, actor);
      return true;
    }
    else
      return nil;
  }

  checkForThanks(actor) = {
    local lowerStr = lower(self.value);

    if (lowerStr == 'thanks' || lowerStr == 'thanks.' ||
        lowerStr == 'thank you' || lowerStr == 'thank you.') {
      if (actor == nil)
        parserReplaceCommand('thanks');
      else
        execCommand(parserGetMe(), thankVerb, actor);
      return true;
    }
    else if (lowerStr == 'sorry' || lowerStr == 'sorry.' ||
             lowerStr == 'i\'m sorry' || lowerStr == 'i\'m sorry.') {
      if (actor == nil)
        parserReplaceCommand('sorry');
      else
        execCommand(parserGetMe(), apologizeVerb, actor);
      return true;
    }
    return nil;
  }

  checkForYesNo(actor) = {
    local lowerStr = lower(self.value);

    if (lowerStr == 'yes' || lowerStr == 'yes.' || lowerStr == 'ok' ||
        lowerStr == 'okay' || lowerStr == 'ok.' || lowerStr == 'okay.') {
      if (actor == nil)
        yesVerb.action(parserGetMe());
      else
        execCommand(parserGetMe(), yesVerb, actor);
      return true;
    }
    else if (lowerStr == 'no' || lowerStr == 'no.') {
      if (actor == nil)
        noVerb.action(parserGetMe());
      else
        execCommand(parserGetMe(), noVerb, actor);
      return true;
    }
    return nil;
  }
;

/**************************** Special Topics ****************************/

catchallNonTopic: knownTopic
  sdesc = "non-topic (you should never see this)"
;

helpTopic: knownTopic
  sdesc = "help"
  noun = 'help'
;

pcTopic: knownTopic
  sdesc = "yourself"
  thedesc = "yourself"
  noun = 'me' 'myself' 'self'
;

youTopic: knownTopic
  sdesc = "them"
  thedesc = "them"
  noun = 'you' 'yourself'
  plural = 'yourselves'
  whatis = "You'll have to address that question to someone."
;

itselfTopic: knownTopic
  sdesc = "itself"
  thedesc = "itself"
  noun = 'itself'
;

himselfTopic: knownTopic
  sdesc = "himself"
  thedesc = "himself"
  noun = 'himself'
  isHim = true
;

herselfTopic: knownTopic
  sdesc = "herself"
  thedesc = "herself"
  noun = 'herself'
  isHer = true
;

themselvesTopic: knownTopic
  sdesc = "themselves"
  thedesc = "themselves"
  noun = 'themself'
  plural = 'themselves'
  isThem = true
;

/************************ Other Special Objects *************************/

badNounPhraseObj: thing
  sdesc = "bad noun phrase (you should never see this)"
  location = nil
;

/************* New and Replaced Verbs & Verb Superclasses ***************/

modify deepverb
  doNPC = nil
  ioNPC = nil
;

class ioTopicVerb: deepverb
  validIoList(actor, prep, dobj) = (nil)
  validIo(actor, obj, seqno) = true
  ioDefault(actor, prep) = (nil)

  disambigIobj(actor, prep, dobj, verprop, wordlist, objlist, flaglist,
               numberWanted, isAmbiguous, silent) = {
    local i, len;
    local newlist = [];
    local unknownTopicsFound = nil;
    local wordLen;

    wordLen = length(wordlist[1]);
    if (length(wordlist) == 1 &&
        (wordlist[1] == 'his' || wordlist[1] == 'hers' ||
         wordlist[1] == 'my' || wordlist[1] == 'mine' ||
         (wordLen > 2 && substr(wordlist[1], wordLen - 1, 2) == '\'s'))) {
      "Please be more specific.";
      return DISAMBIG_ERROR;
    }

    len = length(objlist);
    for (i=1; i <= len; i++) {
      if (isclass(objlist[i], topic)) {
        if (objlist[i].known)
          newlist += objlist[i];
        else
          unknownTopicsFound = true;
      }
    }
    if (length(newlist) < 1) {
      if (unknownTopicsFound) {
        global.unknownMsg;
        return DISAMBIG_ERROR;
      }
      else
        newlist += catchallNonTopic;
    }
    return newlist;
  }
;

class doTopicVerb: deepverb
  validDoList(actor, prep, io) = (nil)
  validDo(actor, obj, seqno) = true
  doDefault(actor, prep, io) = (nil)

  // allowing multiple objects can cause erroneous sdescs (e.g.
  // catchallNonTopic.sdesc) to get printed. So we don't.
  rejectMultiDobj(prep) = {
    "You can't use multiple objects with that verb.";
    return true;
  }

  disambigDobj(actor, prep, io, verprop, wordlist, objlist, flaglist,
               numberWanted, isAmbiguous, silent) = {
    return ioTopicVerb.disambigIobj(actor, prep, io, verprop, wordlist,
           objlist, flaglist, numberWanted, isAmbiguous, silent);
  }
;

apologizeVerb: darkVerb
  verb = 'apologize' 'apologize to' 'sorry'
  sdesc = "apologize to"
  doAction = 'Sorry'
  doDefault(actor, prep, io) = (nil)
  doNPC = true
  action(actor) = {
    askdo;
  }
;

replace askVerb: ioTopicVerb, darkVerb
  verb = 'ask'
  sdesc = "ask"
  prepDefault = aboutPrep
  ioAction(aboutPrep) = 'AskAbout'
  ioAction(forPrep) = 'AskFor'
  doNPC = true

  /* This method is a hideous kludge. It is necessary because, in conversation
   * mode and consult mode, the program grabs a random piece of vocabulary from
   * the target and splices that into an "ask/tell/consult [target] about
   * [topic]" string. But what if another object in the room has that
   * vocabulary? When askVerb.disambigDobj sees that we are processing a
   * conversation mode topic, that we have an ambiguous list for dobj, and that
   * one of the items in the list is the target object, it automatically
   * discards all of the others.
   *
   * disambigDobj on tellVerb and consultVerb point to this, for the same
   * reason.
   */
  disambigDobj(actor, prep, io, verprop, wordlist, objlist, flaglist,
               numberWanted, isAmbiguous, silent) = {
    local i, len;
    local newList = [];
    local convTarget;

    if (global.inConvModeTopic && isAmbiguous) {
      convTarget = ((global.talkingTo != nil) ? global.talkingTo :
                                                global.consulting);
      len = length(objlist);
      for (i=1; i <= len; i++) {
        if (objlist[i] == convTarget) {
          newList += convTarget;
          break;
        }
      }
      if (length(newList) > 0)
        return newList;
    }
    return DISAMBIG_CONTINUE;
  }
;

consultVerb: ioTopicVerb
  verb = 'consult' 'consult with'
  sdesc = "consult"
  doAction = 'Consult'
  ioAction(onPrep) = 'ConsultOn'
  ioAction(aboutPrep) = 'ConsultOn'

  disambigDobj(actor, prep, io, verprop, wordlist, objlist, flaglist,
               numberWanted, isAmbiguous, silent) = {
    return askVerb.disambigDobj(actor, prep, io, verprop, wordlist, objlist,
                                flaglist, numberWanted, isAmbiguous, silent);
  }
;

consultOffVerb: sysverb
  verb = 'consult off' 'consultoff'
  sdesc = "consult off"
  action(actor) = {
    if (global.consulting == nil)
      "You aren't in consult mode at present.";
    else
      self.performConsultOff;
    abort;
  }

  performConsultOff = {
    "[Exiting consult mode.]";
    global.consulting = nil;
  }
;

consultHelpVerb: sysverb
  verb = 'consulthelp'
  sdesc = "consult help"
  action(actor) = {
    "Consult mode is an extension of conversation mode, as documented under
    \"talk help\". It works quite similarly. You can enter into consult mode by
    typing \"consult [source]\", where [source] is an information source such
    as a dictionary or encyclopedia.  Thereafter, you can simply type a topic
    to look up that topic in the selected source. This will last until you type
    \"consult off\" at the prompt, or until you type \"consult [another
    source]\", in which case the target will switch to that.\b

    If, while in consult mode, you type what does not appear to be a
    simple noun phrase, parsing will proceed as usual. So you need not exit
    consult mode to do other things:\b

    \ \ \ \ \ >CONSULT DICTIONARY\n
    \ \ \ \ \ [You are now consulting the dictionary. Type \"consult help\"\n
    \ \ \ \ \ for assistance, \"consult off\" to exit.]\b

    \ \ \ \ \ >ZEPHYR\n
    \ \ \ \ \ A zephyr is a light breeze.\b

    \ \ \ \ \ >JUMP\n
    \ \ \ \ \ You jump on the spot, fruitlessly.\b

    Note that consult mode will favor no-object verbs over topics. So, for
    instance, if you type \"east\", and there is an \"east passage\" topic, you
    will go east rather than looking up that topic. To override this
    behavior, you should explicitly type \"east passage\".\b

    You can only look up one topic at a time, and you can't mix topics and
    regular commands on the same line. For instance, this won't work:\b

    \ \ \ \ \ >ZEPHYR. TUQUE.\b

    If you type \"consult [source]\" while in conversation mode, the game will
    exit conversation mode before going into consult mode.";

    abort;
  }
;

modify giveVerb
  ioNPC = true
;

goodbyeVerb: darkVerb
  sdesc = "say goodbye to"
  verb = 'bye' 'good bye' 'goodbye' 'farewell'
  doAction = 'Goodbye'
  doDefault(actor, prep, io) = (nil)
  doNPC = true
  action(actor) = {
    askdo;
  }
;

replace helloVerb: darkVerb
  sdesc = "greet"
  verb = 'hello' 'hi' 'greetings' 'greet'
  doAction = 'Greet'
  doDefault(actor, prep, io) = (nil)
  doNPC = true
  action(actor) = {
    askdo;
  }
;

helpVerb: darkVerb
  verb = 'help' 'help out'
  sdesc = "help"
  action(actor) = {
    "You will probably want to link this to game instructions or hints.";
    abort;
  }
  doAction = 'Help'
;

lookupVerb: doTopicVerb
  verb = 'look up' 'read about'
  sdesc = "look up"
  prepDefault = inPrep
  ioAction(inPrep) = 'LookupIn'
;

noVerb: darkVerb
  verb = 'no'
  sdesc = "say no to"
  doNPC = true
  doAction = 'No'
  doDefault(actor, prep, io) = (nil)
  action(a) = {
    if (!parserGetMe().canSpeak)
      return;
    if (global.lastYesnoAsked != nil)
      global.lastYesnoAsked.process(nil);
    else
      global.globalYesnoDefault;
  }
;

modify sayVerb
  doDefault(actor, prep, io) = (nil)
  ioAction(toPrep) = 'SayTo'
  ioNPC = true
;

modify showVerb
  ioNPC = true
;

talkVerb: sysverb
  verb = 'talk' 'talk to' 'talk with'
  sdesc = "talk to"
  doAction = 'Talk'
  ioAction(aboutPrep) = 'TalkAbout'
  validIoList(actor, prep, dobj) = (nil)
  validIo(actor, obj, seqno) = {return (seqno == 1);}
  ioDefault(actor, prep) = (nil)
  doNPC = true
;

talkOffVerb: sysverb
  verb = 'talk off' 'talkoff'
  sdesc = "talk off"
  action(actor) = {
    if (global.talkingTo == nil)
      "You aren't in conversation mode at present.";
    else
      self.performTalkOff;
    abort;
  }

  performTalkOff = {
    "[Exiting conversation mode.]";
    global.talkingTo = nil;
  }
;

talkHelpVerb: sysverb
  verb = 'talkhelp'
  sdesc = "talk help"
  action(actor) = {
    "You can enter into conversation mode by typing \"talk to [character]\".
    Thereafter, you can simply type a topic to ask the selected character
    about that topic. You can also \(tell\) the character about a topic (this
    is sometimes a significant distinction) by typing \"tell [topic]\".
    This will last until you type \"talk off\" at the prompt, or you type
    \"talk to [another character]\", in which case the conversation target
    will switch to them.\b

    If, while in conversation mode, you type what does not appear to be a
    simple noun phrase, parsing will proceed as usual. So you need not exit
    conversation mode to do other things:\b

    \ \ \ \ \ >TALK TO MERLIN\n
    \ \ \ \ \ [You are now in conversation with Merlin. Type \"talk help\"\n
    \ \ \ \ \ for assistance, \"talk off\" to exit.]\b

    \ \ \ \ \ >EXCALIBUR\n
    \ \ \ \ \ \"Have you gotten it out of the stone yet?\" he replies.\b

    \ \ \ \ \ >JUMP\n
    \ \ \ \ \ You jump on the spot, fruitlessly.\b

    Note that conversation mode will favor no-object verbs over topics. So,
    for instance, if you type \"east\", and there is an \"east passage\" topic,
    you will go east rather than asking about that topic. To override this
    behavior, you can type \"ask east\" or \"tell east\" instead, or
    explicitly type \"east passage\".\b

    You can only ask about one topic at a time, and you can't mix topics
    and regular commands on the same line. For instance, this won't
    work:\b

    \ \ \ \ \ >EXCALIBUR. MAGIC WAND.\b

    If you type \"talk to [person]\" while in consult mode (see \"consult
    help\"), the game will exit consult mode before entering into
    conversation.";

    abort;
  }
;

replace tellVerb: ioTopicVerb, darkVerb
  verb = 'tell'
  sdesc = "tell"
  prepDefault = aboutPrep
  ioAction(aboutPrep) = 'TellAbout'
  doNPC = true

  disambigDobj(actor, prep, io, verprop, wordlist, objlist, flaglist,
               numberWanted, isAmbiguous, silent) = {
    return askVerb.disambigDobj(actor, prep, io, verprop, wordlist, objlist,
                                flaglist, numberWanted, isAmbiguous, silent);
  }
;

thankVerb: darkVerb
  verb = 'thank' 'thanks'
  sdesc = "thank"
  doAction = 'Thank'
  doDefault(actor, prep, io) = (nil)
  doNPC = true
  action(actor) = {
    askdo;
  }
;

whatisVerb: doTopicVerb
  verb = 'what is' 'what are' 'who is' 'who are'
  sdesc = "find out about"
  doAction = 'Whatis'
;

whoamiVerb: darkVerb
  verb = 'whoami' 'who ami' 'what ami'
  sdesc = "who am I"
  action(actor) = {
    pcTopic.doWhatis(actor);
  }
;

yesVerb: darkVerb
  verb = 'yes' 'okay' 'ok'
  sdesc = "say yes to"
  doAction = 'Yes'
  doNPC = true
  doDefault(actor, prep, io) = (nil)
  action(a) = {
    if (!parserGetMe().canSpeak)
      return;
    if (global.lastYesnoAsked != nil)
      global.lastYesnoAsked.process(true);
    else
      global.globalYesnoDefault;
  }
;

/**************************** New Prepositions ****************************/

amiPrep: Prep
  preposition = 'ami'
  sdesc = "am I"
;

arePrep: Prep
  preposition = 'are'
  sdesc = "are"
;

forPrep: Prep
  preposition = 'for'
  sdesc = "for"
;

isPrep: Prep
  preposition = 'is'
  sdesc = "is"
;

/*************************** New Compound Words ****************************/

compoundWord 'am' 'i' 'ami';
compoundWord 'consult' 'help' 'consulthelp';
compoundWord 'good' 'bye' 'goodbye';
compoundWord 'talk' 'help' 'talkhelp';
compoundWord 'thank' 'you' 'thanks';

/*************************** New Special Words *****************************/

modify specialWords
  nil,
  nil,
  nil,
  nil,
  nil,
  nil,
  nil,
  nil,
  'that',
  'they' = 'those',
  'he',
  'she',
  nil
;

/****************************** Parser Hooks *******************************/

/* Parser hook to ask "who" rather than "what" in dobj disambiguation
 * questions, where appropriate.
 */
parseAskobjActor: function (a, v, ...) {
  "<<v.doNPC ? 'Who' : 'What'>> do you want ";
  if (a != parserGetMe())
    "<<a.thedesc>> ";
  "to ";
  if (a != parserGetMe() && v == whatisVerb)
    "tell you about?";
  else
    "<<v.sdesc>>?";
}

/* Parser hook to ask "who" rather than "what" in iobj disambiguation
 * questions, where appropriate, and to figure out a good pronoun to use
 * for direct object in the question.
 */
parseAskobjIndirect: function(a, v, p, dobjInfo) {
  local firstObjFlags;
  local i, j, len, vlen, o;
  local dobjs, flags, words, vocab, prunedDobjs, prunedFlags;
  local numHim = 0, numHer = 0, numAndro = 0;
  local numPlural = 0;

  "<<v.ioNPC ? 'Who' : 'What'>> do you want ";
  if (a != parserGetMe())
    "<<a.thedesc>> ";
  "to ";
  if (v == lookupVerb) {
    "look it up in?";
    return;
  }
  "<<v.sdesc>> ";

  firstObjFlags = dobjInfo[1][3][1];
  if (length(dobjInfo) > 1 ||
      (firstObjFlags & PRSFLG_COUNT) != 0 ||
      (firstObjFlags & PRSFLG_ALL) != 0 ||
      (firstObjFlags & PRSFLG_EXCEPT) != 0)
    "those";
  else if ((firstObjFlags & PRSFLG_IT) != 0 ||
           (firstObjFlags & PRSFLG_NUM) != 0 ||
           (firstObjFlags & PRSFLG_STR) != 0)
    "it";
  else if ((firstObjFlags & PRSFLG_THEM) != 0)
    "them";
  else if ((firstObjFlags & PRSFLG_HIM) != 0)
    "him";
  else if ((firstObjFlags & PRSFLG_HER) != 0)
    "her";
  else {
    words = dobjInfo[1][1];
    dobjs = dobjInfo[1][2];
    flags = dobjInfo[1][3];
    len = length(dobjs);
    prunedDobjs = [];
    prunedFlags = [];

    /* We'll now prune the list of dobjs in every way we know how. If a
     * round of pruning eliminates every object in the current list, we
     * revert back to the original list (as of the beginning of that round)
     * and go on to the next round. */

    // Pass 1: prune all dobjs that don't pass validDo on this verb. This will
    // eliminate all dobjs not present in the room, generally. However, if
    // ANY object in the original list is flagged as PRSFLG_PLURAL, we
    // use "them" right away and exit the function.
    for (i=1; i <= len; i++) {
      if ((flags[i] & PRSFLG_PLURAL) != 0) {
        " them <<p.sdesc>>?";
        return;
      }
      if (v.validDo(a, dobjs[i], i)) {
        prunedDobjs += dobjs[i];
        prunedFlags += flags[i];
      }
    }
    if (length(prunedDobjs) > 0) {
      dobjs = prunedDobjs;
      flags = prunedFlags;
      len = length(prunedDobjs);
      prunedDobjs = [];
      prunedFlags = [];
    }

    // Pass 2: prune all dobjs that only matched a truncated word.
    for (i=1; i <= len; i++) {
      if ((flags[i] & PRSFLG_TRUNC) == 0) {
        prunedDobjs += dobjs[i];
        prunedFlags += flags[i];
      }
    }
    if (length(prunedDobjs) > 0) {
      dobjs = prunedDobjs;
      flags = prunedFlags;
      len = length(prunedDobjs);
      prunedDobjs = [];
      prunedFlags = [];
    }

    // Pass 3: Eliminate any objects where the noun phrase ended on an
    // adjective.
    for (i=1; i <= len; i++) {
      if ((flags[i] & PRSFLG_ENDADJ) == 0) {
        prunedDobjs += dobjs[i];
        prunedFlags += flags[i];
      }
    }
    if (length(prunedDobjs) > 0) {
      dobjs = prunedDobjs;
      flags = prunedFlags;
      len = length(prunedDobjs);
      prunedDobjs = [];
      prunedFlags = [];
    }

    /* We've pared it down all we can. Now lets look at what's left. */

    if (len == 1 && dobjs[1] == parserGetMe())
      "<<(a == parserGetMe()) ? 'yourself' : 'you'>>";
    else {
      for (i=1; i <= len; i++) {
        o = dobjs[i];
        if (o.isThem)
          numPlural++;
        else if (o.isHim && o.isHer)
          numAndro++;
        else if (o.isHim)
          numHim++;
        else if (o.isHer)
          numHer++;
      }
      if (numAndro == len)
        "it";
      else if (numHim == len)
        "him";
      else if (numHer == len)
        "her";
      else if (numHim + numHer == len)
        "them";
      else if (numHim + numHer + numPlural == len && v.doNPC)
        "them";
      else if (v.doNPC)
        "it";
      else
        "that";
    }
  }

  " <<p.sdesc>>?";
}

/* Makes a few of the standard parser messages more reasonable and/or more
 * friendly. In particular, message 16 is changed so that the player won't
 * see "You don't see that here" when they're answering a topic disambiguation
 * question.
 */
parseError: function (num, str) {
  if (num == 9)
    return 'I don\'t see that here.';
  else if (num == 16)
    return 'That wasn\'t one of the options.';
  else if (num == 100)
    return 'Please be more specific. ';
  return nil;
}

/* Our parseNounPhrase catches instances of valid noun phrases that don't
 * refer to any actual objects in the game (e.g. "red ball" where red exists
 * as an adjective and ball as a noun, but never together). It returns the
 * special object badNounPhraseObj, which will simply cause a reply of
 * "You don't see that here" for most verbs, and for topic-based verbs,
 * will cause a stock response to be printed out for the character.
 */
parseNounPhrase: function(wordlist, typelist, currentIndex, complainOnNoMatch,
                          isActorCheck) {
  local pnlResult, endIndex;
  local returnList = [];

  /* Since parseNounList calls parseNounPhrase, we must be careful here not
   * to allow an infinite loop to happen:
   */
  if (global.inParseNounPhrase)
    return PNP_USE_DEFAULT;

  global.inParseNounPhrase = true;
  pnlResult = parseNounList(wordlist, typelist, currentIndex, nil, nil,
               isActorCheck);
  global.inParseNounPhrase = nil;

  if (pnlResult == nil)
    return PNP_ERROR;
  else if (datatype(pnlResult) != DTY_LIST) {
    "\n[datatype is not list--this should never happen. Please tell the
    author.]\n";
    return PNP_ERROR;
  }

  endIndex = pnlResult[1];
  if (length(pnlResult) == 1 && endIndex > currentIndex) {
    returnList += endIndex;
    returnList += badNounPhraseObj;
    return returnList;
  }
  else
    return PNP_USE_DEFAULT;
}

/* You shouldn't have to modify the preparse function itself. Modify
 * global.preparseSequence instead.
 */
preparse: function(str) {
  local tokens, len;
  local i, seqLen, returnVal;
  local changed = nil;

  global.inConvModeTopic = nil;

  tokens = parserTokenize(str);
  if (tokens == nil)
    /* An invalid character was found in the string, but
     * parserTokenize didn't print an error message */
    return true;
  tokens = fixTokens(tokens, str);
  len = length(tokens);
  seqLen = length(global.preparseSequence);

  for (i=1; i <= seqLen; i++) {
    returnVal = (global.preparseSequence[i])(tokens, len);
    if (returnVal == nil)
      return nil;
    else if (datatype(returnVal) == DTY_LIST) {
      tokens = returnVal;
      len = length(tokens);
      changed = true;
    }
  }
 
  return (changed ? constructStringFromTokens(tokens) : true);
}

preparseValidateSay: function(tokens, len) {
  local i;

  for (i=1; i < len; i++) {
    if (tokens[i] == 'say' && substr(tokens[i+1], 1, 1) != '"') {
      "You should type what you want to say in double quotes, for example,
      SAY \"HELLO\" or SAY \"HELLO\" TO JOHN.";
      return nil;
    }
  }
  return true;
}

preparseRemoveComma: function(tokens, len) {
  local newTokens = [];
  local changed = nil;
  local verbLen, verbStr, verbList;
  local omitNextComma = nil;
  local i;

  for (i=1; i <= len; i++) {
    if (i < len-1 && tokens[i+1] == ',') {
      verbLen = 1;
      verbStr = tokens[i];
    }
    else if (i < len-2 && tokens[i+2] == ',') {
      verbLen = 2;
      verbStr = tokens[i] + ' ' + tokens[i+1];
    }
    else
      verbLen = 0;

    if (verbLen > 0) {
      verbList = parserDictLookup([verbStr], [PRSTYP_VERB]);
      if (length(verbList) > 0 &&
          find([yesVerb noVerb helloVerb goodbyeVerb thankVerb apologizeVerb],
                verbList[1]) != nil)
        omitNextComma = true;
    }
    if (omitNextComma && tokens[i] == ',') {
      omitNextComma = nil;
      changed = true;
    }
    else
      newTokens += tokens[i];
  }

  return (changed ? newTokens : true);
}

preparseConvertAutoTopic: function(tokens, len) {
  local newTokens;
  local pos, convVerb, convPrep, convTarget;
  local parseNounResult, targetVocab;
  local verbStr, verbList1 = [], verbList2 = [];
  local i, doublesLen;
  local tokenTypes, typesLen;

  /* If not in conversation or consult mode, if the command is empty, or if
   * the player entered multiple commands or multiple objects, stop now:
   */
  if ((global.talkingTo == nil && global.consulting == nil) ||
      len == 0 || find(tokens, ',') != nil || find(tokens, '.') != nil)
    return true;

  pos = 1;
  if (global.talkingTo != nil) {
    convVerb = 'ask';
    convPrep = 'about';
    convTarget = global.talkingTo;
    if (tokens[1] == 'tell') {
      if (len == 1) {
        "You'll have to specify what you want to tell about.";
        return nil;
      }
      else {
        pos++;
        convVerb = 'tell';
      }
    }
    else if (tokens[1] == 'ask') {
      if (len == 1) {
        "You'll have to specify what you want to ask about.";
        return nil;
      }
      else
        pos++;
    }
  }
  else {
    convVerb = 'consult';
    convPrep = 'on';
    convTarget = global.consulting;
  }

  if (pos == 1) {
    /* Check if the command can be interpreted as a no-object verb. This check
     * is not performed if the user typed "ask" or "tell" at the beginning of
     * the line.
     */
    verbStr = tokens[1];
    verbList1 = parserDictLookup([verbStr], [PRSTYP_VERB]);
    if (len >= 2) {
      verbStr += ' ' + tokens[2];
      verbList2 = parserDictLookup([verbStr], [PRSTYP_VERB]);
    }
    if ((len == 1 && length(verbList1) > 0) ||
        (len == 2 && length(verbList2) > 0))
      /* The command can be interpreted as a no-object verb. Stop now. */
      return true;
  }

  tokenTypes = parserGetTokTypes(tokens);
  typesLen = length(tokenTypes);

  /* Quit now if any unknown words are found */
  for (i=1; i <= typesLen; i++) {
    if ((tokenTypes[i] & PRSTYP_UNKNOWN) != 0)
      return true;
  }

  parseNounResult = parseNounList(tokens, tokenTypes, pos, nil, nil, nil);
  if (parseNounResult == nil)
    /* parseNounList found a syntax error and printed an error message */
    return nil;
  else if (parseNounResult[1] != len+1 &&
           (pos > 1 || length(verbList1) > 0 || length(verbList2) > 0))
    /* We stop here if the input doesn't appear to be a simple noun phrase,
     * -unless- no verb of any sort was found at the beginning of the command
     * (we don't want the player to get a "no verb" error message).
     */
    return true;

  // If the conversation target isn't present (isn't reachable), check its
  // "doubles" property, and switch targets if any object in the list is
  // currently present. Otherwise, error out.
  if (!convTarget.isReachable(parserGetMe())) {
    doublesLen = length(convTarget.doubles);
    for (i=1; i <= doublesLen; i++) {
      if (convTarget.doubles[i].isReachable(parserGetMe())) {
        convTarget = convTarget.doubles[i];
        break;
      }
    }
    if (i > doublesLen) {
      if (global.talkingTo != nil)
        "The person you were talking to doesn't seem to be present any
        longer.";
      else
        "The source you were consulting doesn't seem to be present any
        longer.";
      return nil;
    }
  }

  global.inConvModeTopic = true;

  newTokens = [convVerb];
  /* Now, find a piece of vocabulary with which to refer to the conversation
   * target. Study askVerb.disambigDobj to see why this is safe.
   */
  targetVocab = getwords(convTarget, &noun);
  if (length(targetVocab) == 0)
    targetVocab = getwords(convTarget, &plural);
  if (length(targetVocab) == 0)
    targetVocab = getwords(convTarget, &adjective);
  /* If targetVocab is still empty you'll get a run-time error now, but you
   * deserve it.
   */
  newTokens += targetVocab[1];
  newTokens += convPrep;
  for (i=pos; i <= len; i++)
    newTokens += tokens[i];
  return newTokens;
}

preparseConvertRequest: function(tokens, len) {
  local i, j, k;
  local newTokens = [];
  local changed = nil;

  for (i=1; i <= len; i++) {
    if (i > len-3 || (tokens[i] != 'ask' && tokens[i] != 'tell')) {
      newTokens += tokens[i];
      continue;
    }
    j = i;
    while (j < len && tokens[j] != '.' && tokens[j] != ',' &&
           tokens[j] != 'about' && tokens[j] != 'for' && tokens[j] != 'to')
      j++;
    if (tokens[j] != 'to' || j-i < 2) {
      newTokens += tokens[i];
      continue;
    }
    changed = true;
    for (k=i+1; k < j; k++)
      newTokens += tokens[k];
    newTokens += ',';
    i = j;
  }

  return (changed ? newTokens : true);
}

preparseConvertThem: function(tokens, len) {
  local i;
  local changed = nil;

  for (i=1; i <= len; i++) {
    if (i < len-1 &&
        (((tokens[i] == 'what' || tokens[i] == 'who') &&
          (tokens[i+1] == 'is' || tokens[i+1] == 'are')) ||
         (tokens[i] == 'look' && tokens[i+1] == 'up') ||
         (tokens[i] == 'read' && tokens[i+1] == 'about')) &&
        tokens[i+2] == 'T') {
      tokens[i+2] = 'I';
      changed = true;
    }
    else if (i < len-1 &&
            tokens[i] == 'look' && tokens[i+1] == 'T' && tokens[i+2] == 'up') {
      tokens[i+1] = 'I';
      changed = true;
    }
  }
  return (changed ? tokens : true);
}

preparseConvertHer: function(tokens, len) {
  local i, j;
  local toChange = nil;
  local changed = nil;
  local tokenTypes;
  local lookupList;

  /* If the adjective "hers" isn't used anywhere in the game, skip this
   * step:
   */
  lookupList = parserDictLookup(['hers'], [PRSTYP_ADJ]);
  if (length(lookupList) == 0)
    return true;

  tokenTypes = parserGetTokTypes(tokens);

  for (i=1; i < len; i++) {
    if (tokens[i] == 'R' && ((tokenTypes[i+1] & PRSTYP_NOUN) != 0 ||
                             (tokenTypes[i+1] & PRSTYP_ADJ) != 0 ||
                             (tokenTypes[i+1] & PRSTYP_PLURAL) != 0)) {
      toChange = true;
      if (i > 1 &&
          find(['feed' 'give' 'offer' 'show' 'throw'], tokens[i-1])
                != nil) {
        toChange = nil;
        j = i+2;
        while (j <= len && tokens[j] != '.') {
          if (find(['at' 'to' 'onto' 'into'], tokens[j]) != nil) {
            toChange = true;
            break;
          }
          j++;
        }
      }
      if (toChange) {
        tokens[i] = 'hers';
        changed = true;
      }
    }
  }
  return (changed ? tokens : true);
}

/************************** Utility Functions ****************************/

/* charAt: Returns the character at position pos in the given string. TADS
 * appears to be missing this function.
 */
charAt: function(str, pos) {
  return substr(str, pos, 1);
}

/* Takes the given token list (as returned by parserTokenize) and splices it
 * back into a string, converting special words as needed.
 */
constructStringFromTokens: function(tokens) {
  local len, i;
  local str = '';

  len = length(tokens);
  for (i=1; i <= len; i++) {
    if (i > 1)
      str += ' ';
    if (tokens[i] == 'A')
      str += 'all';
    else if (tokens[i] == 'X')
      str += 'but';
    else if (tokens[i] == 'I')
      str += 'it';
    else if (tokens[i] == 'T')
      str += 'them';
    else if (tokens[i] == 'M')
      str += 'him';
    else if (tokens[i] == 'R')
      str += 'her';
    else if (tokens[i] == 'Y')
      str += 'any';
    else if (tokens[i] == 'B')
      str += 'both';
    else
      str += tokens[i];
  }
  return str;
}

/* fixTokens: A kludge to repair the result of parserTokenize. TADS 2.5.1 and
 * under does not properly tokenize quoted strings when parserTokenize is
 * called. This function cross-references with the original input string to
 * create legitimate quoted-string tokens.
 */
fixTokens: function(tokens, str) {
  local tpos = 1, spos = 1;
  local tlen, slen;
  local quotedString;
  local quoteChar;

  tlen = length(tokens);
  slen = length(str);
  while (tpos <= tlen) {
    if (charAt(tokens[tpos], 1) == '"') {
      /* We've found a string token. Try to find it in the command string. */

      quotedString = '';

      /* Scan the string looking for the next start-quote. Skip '
       * (single-quote) unless it is at the beginning of the string, or
       * preceeded by a space, comma, or period.
       */
      while (spos <= slen && charAt(str, spos) != '"' &&
             (charAt(str, spos) != '\'' || (spos > 1 &&
                                            charAt(str, spos-1) != ' ' &&
                                            charAt(str, spos-1) != ',' &&
                                            charAt(str, spos-1) != '.')))
        spos++;
      if (spos <= slen)
        quoteChar = charAt(str, spos);
      spos++;

      /* Copy characters in the token string until we reach a matching
       * close-quote:
       */
      while (spos <= slen && charAt(str, spos) != quoteChar)
        quotedString += charAt(str, spos++);
      spos++;
      tokens[tpos] = '"' + quotedString + '"';
    }
    tpos++;
  }
  return tokens;
}

/* translateTopic: Performs various context-sensitive translations on the
 * given topic/object, including translation of non-topics to topics via the
 *  mytopic property.
 *
 * Params:
 *   + t: the topic
 *   + asked: the npc or information source which the player consulted
 *   + translateSelfRefs: true if you want to translate words like "himself"
 *     and "herself" to the npc/source, if gender and plurality match. For
 *     instance, "ask john about himself".
 *   + translateYou: true if you want to translate "you" or "yourself" to the
 *     npc/source. For instance, "john, who are you?"
 *
 * Returns: the translated topic
 */
translateTopic: function(t, asked, translateSelfRefs, translateYou) {
  if (!isclass(t, topic) && t.mytopic != nil)
    t = t.mytopic;
  if (translateSelfRefs && asked.mytopic != nil) {
    if (t == itselfTopic)
      return asked.mytopic;
    else if (t == himselfTopic && asked.isHim)
      return asked.mytopic;
    else if (t == herselfTopic && asked.isHer)
      return asked.mytopic;
    else if (t == themselvesTopic && asked.isThem)
      return asked.mytopic;
  }
  if (translateYou && t == youTopic && asked.mytopic != nil)
    return asked.mytopic;
  return t;
}

#pragma C-
