/* calleg.c: CSCR allegro interface
    This implements a set of text handling functions designed
    for compatability with curses.h, and adds a screen capture facility

    By L. Ross Raszewski <lraszewski@justice.loyola.edu>

    Uses Allegro by Sean Hargreaves
   

*/

#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <allegro.h>
#include "cscr.h"
#include "gtpref.h"
#define ERR 0

// Extended latin-1 output table - this is simple; we don't.
unsigned char OPT_AO_FF_OUTPUT[]= {
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
    0, 0, 0, 0, 0, 0, 0, 0,  
};

// Fake definitions to keep curses happy.
void *stdscr;
int COLS;
int LINES;

// Exportable definitions
int VHORIZ_REZ=0;
int VVERT_REZ=0;
BITMAP *Buffer;                         // The video buffer
struct mouse_info_s mouse_info;

// Internal variables
static int mouse_on;                    // A button was _down_ at last poll
                                        // We consider it a click when it
                                        // comes _up_ again
static int mouse_exists;                // Mouse install succeeded
static int mouse_alive=0;               // The Mouse is enabled
static int mouse_click=0;               // The mouse has a click pending

static int VERT_REZ=480;                // The "real" resolution
static int HORIZ_REZ=640;

static int horiz_vrez=0;                // The "virtual" resolution
static int vert_vrez=0;

static int V_window=0;                  // origin of the virtual viewport
static int H_window=0;

static int font_h;                      // Cached font metrics
static int font_w;

static chtype alleg_attr;               // Current typeface
static chtype alleg_tattr;
static int alleg_textbg;

static int just_cleared=-1;             // Prevent redundant screen clears


static int reso_tbl[]= {                // table of supported resolutions
                        320, 200,
                        640, 480,
                        800, 600,
                        1024, 768,
                        1280, 1024
                      };
#define NUM_RESOS  6

static int Key_remap[] = {              // allegro-> curses.h keymap
        47, 4, 295,                     // Alt-F1
        48, 4, 296,                     // Alt-F2
        76, 2, 477,                     // ^ins
        76, 4, 479,                     //A-ins
        76, 1, 479,                     // S-ins
        77, 2, 467,                     // ^Del
        76, 0, 0x14b,                   // Ins
        77, 0, 0x14a,                   // Del
        78, 0, 262,                     // Home
        79, 0, 358,                     // end
        80, 0, 339,                     // Pageup
        81, 0, 338,                     // Pagedown
        78, 2, 447,                     // ^Home
        79, 2, 448,                     // ^end
        80, 2, 445,                     // ^Pageup
        81, 2, 446,                     // ^Pagedown
        84, 0, 0x103,                   // Up
        82, 0, 0x104,                   // Left
        85, 0, 0x102,                   // Down
        83, 0, 0x105,                   // Right
        84, 2, 480,                     // ^Up
        82, 2, 443,                     // ^Left
        85, 2, 481,                     // ^Down
        83, 2, 444,                     // ^Right
        64, 127,482                     // ^Tab
        };
#define MAPPED_KEYS 25



static int alleg_color;                 // current drawing color
static int alleg_bg;

static int alleg_x;                     // Text cursor position
static int alleg_y;

static int allegcolor[]= {              // 4->16 bit color conversion
                     0x0000,      // 0000  0000000000000000
                     0x000F,      // 0001  0000000000001111
                     0x03E0,      // 0010  0000000011110000
                     0x07ff,      // 0011  0000000000001111
                     0x7800,
                     0x780F,
                     0x7BE0,
                     0xC618,
                     0x8210,
                     0x001F,      // 0001  0000000000001111
                     0x07E0,      // 0010  0000000011110000
                     0x07FF,      // 0011  0000000000001111
                     0xF800,
                     0xF8FF,
                     0xFFE0,
                     0xFFFF,
                     0x0000
                     };

// DEPRECIATED!!! REMOVE BY VERSION 25!!!
const static char dvorak_keyboard[][3]=
 {
   "-[",
   "=]",
   "q'",
   "w,",
   "e.",
   "rp",
   "ty",
   "yf",
   "ug",
   "ic",
   "or",
   "pl",
   "[/",
   "]=",
   "so",
   "de",
   "fu",
   "gi",
   "hd",
   "jh",
   "kt",
   "ln",
   ";s",
   "'-",
   "z;",
   "xq",
   "cj",
   "vk",
   "bx",
   "nb",
   ",w",
   ".v",
   "/z",
   "_{",
   "+}",
   "Q\"",
   "W<",
   "E>",
   "RP",
   "TY",
   "YF",
   "UG",
   "IC",
   "OR",
   "PL",
   "{?",
   "}+",
   "SO",
   "DE",
   "FU",
   "GI",
   "HD",
   "JH",
   "KT",
   "LN",
   ":S",
   "\"_",
   "Z:",
   "XQ",
   "CJ",
   "VK",
   "BX",
   "NB",
   "<W",
   ">V",
   "?Z",
   { 0,0,0 }
};
  


// Local helper functions

// cscr/curses color pair -> 4-bit foreground color 
static int alleg_colors_f(int c)
{
 return (c - 1 ) % 8;

}
// cscr/curses color pair -> 4-bit foreground color 
static int alleg_colors_b(int c)
{
 return (c - 1 ) / 8;

}

// Loader for font datafile
static DATAFILE *getdatfile()
{
 char *s; DATAFILE *f;
 s=getenv("GLKFONT");
 if (s==NULL) s="font.dat";
 f=load_datafile(s);
 if (!f) {
           char *w=getenv("GLKDIR");
           char z[1024];
           if (w!=NULL)
           {
            sprintf(z,"%s%s",w,s);
            f=load_datafile(z);

           }
         } 
 return f;
}

// Adjust a font character to the specified size
static void font_fix_char(BITMAP *s, BITMAP *d)
{
 clear(d);
 if (s->w > d->w)
  stretch_blit(s,d,0,0,s->w,s->h,0,0,d->w,d->h);
 else
  blit(s,d,0,0,(d->w-s->w)>>1,0,s->w,s->h);
}

// Determine of a font is fixed width
static int font_fixed(FONT *f)
{
  int i,j;
  j=((BITMAP *)(f->glyphs[45]))->w;

  for(i=1;i<=(f->end-f->start);i++)
  if (((BITMAP *)(f->glyphs[i]))->w!=j) return 0;

  return 1;
}

// Convert a font to fixed width. We only scale the character if
// it's too large; otherwise, we center it.
static FONT *font_fix_width(FONT *f)
{
 FONT *t;
 int h, w,i; 

 t=(FONT *)malloc(sizeof( FONT));
 t->mono=FALSE;
 t->start = f->start;
 t->end = f->end;
 t->next = NULL;
 t->renderhook = NULL;
 t->widthhook = NULL;
 t->heighthook = NULL;

 t->glyphs=(void *)malloc(sizeof(BITMAP *) * (t->end-t->start));
 h=( (BITMAP *)(f->glyphs[45]) )->h;
 w=( (BITMAP *)(f->glyphs[45]) )->w;

 for (i=0;i<=f->end-f->start;i++)
 {
 ((BITMAP *)(t->glyphs[i]))=create_bitmap_ex(8,w,h);
 clear((BITMAP *)(t->glyphs[i]));
 if (((BITMAP *)(f->glyphs[i]))->w > w)
 stretch_blit(((BITMAP *)(f->glyphs[i])),
              ((BITMAP *)(t->glyphs[i])),0,0,
              ((BITMAP *)(f->glyphs[i]))->w,
              ((BITMAP *)(f->glyphs[i]))->h,0,0,
              w,h);
 else
  blit(((BITMAP *)(f->glyphs[i])),
       ((BITMAP *)(t->glyphs[i])),0,0,
       (w-((BITMAP *)(f->glyphs[i]))->w)>>1,0,
       w,h);

 }
 return t;
}

// Scale a font by the given proportion
static FONT *font_scale(FONT *f, float prop)
{
 FONT *t;
 int h, w,i; 
 BITMAP *b;
 t=(FONT *)malloc(sizeof( FONT));
 t->mono=FALSE;
 t->start = f->start;
 t->end = f->end;
 t->next = NULL;
 t->renderhook = NULL;
 t->widthhook = NULL;
 t->heighthook = NULL;

 t->glyphs=(void * )malloc(sizeof(BITMAP *)*(t->end-t->start));
 h=((BITMAP *)(f->glyphs[45]))->h;
 w=((BITMAP *)(f->glyphs[45]))->w;
 b=create_bitmap_ex(8,w,h);
 h=((float)h) * prop;
 w=((float)w) * prop;
 for (i=0;i<=f->end-f->start;i++)
 {
  font_fix_char(((BITMAP *)(f->glyphs[i])),b);
  ((BITMAP *)(t->glyphs[i]))=create_bitmap_ex(8,w,h);
  stretch_blit(b,
              ((BITMAP *)(t->glyphs[i])),0,0,
              b->w,
              b->h,0,0,
              w,h);

 }
 destroy_bitmap(b);
 return t;
}

// Calculate the scaling factor to render a font at the given cpl
static float get_font_factor(FONT *f, int cpl, int wid)
{
 int t=text_length(f,"M");
 t=wid/t;
 return ((float) t)/cpl;
} 

// Draw a 1-character wide box by inverting the given region
static void not_box(BITMAP *b, int x, int y)
{
 int i,j;
 for(i=y;i<y+text_height(font);i++)
  for(j=x;j<x+text_length(font,"_");j++)
  {
   int k=getpixel(b,j,i);
   putpixel(b,j,i,~k);
  }
}

// Scroll the virtual screen according to keypress given
static void v_scroll(int a)
{
 if (a=='8') V_window-=10;
 else if (a=='2') V_window+=10;
 else if (a=='4') H_window-=10;
 else if (a=='6') H_window+=10;
 if (V_window<0 || a=='9') V_window=0;
 if (H_window<0 || a=='7') H_window=0;
 if (H_window>horiz_vrez || a=='1') H_window=horiz_vrez;
 if (V_window>vert_vrez || a=='3') V_window=vert_vrez;
 cscr_refresh();
}

// convert an allegro key to a curses.h chtype
static int tochtype(int a)
{
 int j; int f=0;
 int scan=a >> 8;
 int ascii=a & 0xFF;
 if (glk_preferences[pref_dvorak])
  for(j=0;dvorak_keyboard[j][0];j++)
   if (ascii==dvorak_keyboard[j][0])
    { ascii=dvorak_keyboard[j][1]; break;}

     
 if (scan>=38 && scan <=46 && !(key_shifts & KB_SCROLOCK_FLAG))
  {
  v_scroll(ascii);
  return -1;
 }
 for(j=0;j<MAPPED_KEYS*3;j+=3)
  if (scan==Key_remap[j] && ascii==Key_remap[j+1])
   f=Key_remap[j+2];
 if (f) return f;
 else return ascii;
}


#define MY_ABS(x) (((x) > 0) ? (x) : -(x))
// Update the mouse variables
static int check_mouse()
{
 int mic_x;
 int mic_y;
 if (!mouse_alive) return 0;
 if (mouse_click) return 1;
 if (mouse_b)
 {
  mouse_on=1;           // Indiacte that a click is in progress
  get_mouse_mickeys(&mic_x,&mic_y);     // Zero the mickeys
  return 0;
 }
 else if (mouse_on)
 {
  mouse_on=0;
  get_mouse_mickeys(&mic_x,&mic_y); // We'll assume the user changed his
                                    // Mind if he moves the mouse more
                                    // than 10 mickeys before releasing
  if (!(MY_ABS(mic_x)>10 || MY_ABS(mic_y)>10))
  {
   mouse_click=1;
   mouse_info.x=mouse_x;
   mouse_info.y=mouse_y;
   mouse_info.text_x=(mouse_info.x * COLS)/ HORIZ_REZ;
   mouse_info.text_y=(mouse_info.y * LINES)/VERT_REZ;
   return 1;
 }
}

 return 0;
}


// "really" set the attributes given
static void cscr_attrset_deep(chtype x)
{
 int c=x >> 24;
 int a=alleg_attr;
 alleg_attr=x;
 if (c==0)
  {
   if ((a & 0x00200000L) != (alleg_attr & 0x00200000L))
   {
   int temp=alleg_textbg;
   alleg_textbg=alleg_color;
   alleg_color=temp;
   }
   text_mode(alleg_textbg);
  }
 else {
  int f = alleg_colors_f(c);
  int b = alleg_colors_b(c);
  if (alleg_attr & 0x00800000L)
   f |= 0x08;
  if (alleg_attr & 0x00400000L)
   b |= 0x08;
  if (alleg_attr & 0x00200000L)
  {
    alleg_textbg=allegcolor[f];
    alleg_color=allegcolor[b];
  }
  else
  {
    alleg_textbg=allegcolor[b];
    alleg_color=allegcolor[f];
  }


  text_mode(alleg_textbg);
 }
}



// calleg -- Allegro Implementation
// Group 1
int cscr_version()
{
 return CSCR_ALLEGRO;
}
void cscr_start()
{
    int card=GFX_AUTODETECT;
    DATAFILE *d;
    
    install_allegro(SYSTEM_AUTODETECT,&errno,atexit);
    // Print the font list if so requested
    if (glk_preferences[pref_fontlist])
    {
     int i;
     d=getdatfile();
     if (!d)
     {
       printf("[glk]: Cannot load font file.\n");
       exit(1);
     }
     for(i=0;i<d[0].size;i++)
      putc(((char *)d[0].dat)[i], stdout);
     exit(1);
    }

    // Install allegro
    install_keyboard();
    mouse_exists= (install_mouse()>0);
    // Set up the real and virtual resolutions
    if (glk_preferences[pref_reso] < 1) glk_preferences[pref_reso]=2;
    glk_preferences[pref_reso]%=NUM_RESOS;
    HORIZ_REZ=reso_tbl[(glk_preferences[pref_reso]-1)*2];
    VERT_REZ=reso_tbl[(glk_preferences[pref_reso]-1)*2+1];
    if (!glk_preferences[pref_dbl_buf] && (VHORIZ_REZ || VVERT_REZ))
    {
     printf("[glk]: Virtual resolutions ignored since double-buffering is disabled\n");
     VHORIZ_REZ=0;
     VVERT_REZ=0;                                                     
    }
    if (!VHORIZ_REZ) VHORIZ_REZ=HORIZ_REZ;
    if (!VVERT_REZ) VVERT_REZ=VERT_REZ;
    horiz_vrez=VHORIZ_REZ-HORIZ_REZ;
    vert_vrez=VVERT_REZ-VERT_REZ;
    set_color_depth(16);
    if (set_gfx_mode(card,HORIZ_REZ,VERT_REZ,0,0) < 0)
     {
      printf("[glk]: Could not set the graphics mode as requested (Resolution set\n");
      printf("       %dx%d). Try a different resolution.\n",HORIZ_REZ,VERT_REZ);
      exit(1);
     }
    install_timer();

    // Load the font requested
    if (glk_preferences[pref_font])
    {
     int f;
     d=getdatfile();
     if (!d)
     {
       set_gfx_mode(GFX_TEXT,80,25,80,25);
       printf("[glk]: Cannot load font file.\n");
       exit(1);
     }

     ((char *)d[0].dat)[d[0].size-1]=0;
     f=atoi((char *)d[0].dat);
     font=d[glk_preferences[pref_font]%(f+1)].dat;
     if (glk_preferences[pref_cpl])
      font=font_scale(font,get_font_factor(font,glk_preferences[pref_cpl],VHORIZ_REZ));
     if (!font_fixed(font))
      font=font_fix_width(font);


    }
    font_w = text_length(font,"m");
    font_h = text_height(font);
    COLS=VHORIZ_REZ/font_w;
    LINES=VVERT_REZ/font_h;
    alleg_x=0;
    alleg_y=0;
    alleg_color=allegcolor[glk_preferences[pref_fg_color]];
    alleg_bg=allegcolor[glk_preferences[pref_bg_color]];
    // Set up the screen buffer
    if (glk_preferences[pref_dbl_buf])
    {
     Buffer=create_bitmap(VHORIZ_REZ,VVERT_REZ);
     if (!Buffer)
     {
      set_gfx_mode(GFX_TEXT,80,25,80,25);
      printf("[glk]: Could not create a video buffer in memory. (Resolution set");
      printf(" %dx%d).\nDisable double-buffering or try a smaller virtual resolution.\n",VHORIZ_REZ,VVERT_REZ);
      exit(1);
     }
 }
 else Buffer=screen;
 text_mode(alleg_bg);
 cscr_clear();
    
}
void cscr_exit()
{

}
void cscr_resume()
{
 install_allegro(SYSTEM_AUTODETECT,&errno,atexit);
 install_keyboard();
 set_color_depth(16);
 set_gfx_mode(GFX_AUTODETECT,HORIZ_REZ,VERT_REZ,0,0);
 install_timer();
 install_mouse();
} 
void cscr_suspend()
{
 allegro_exit();
}


// Group 2
void  cscr_charpos_to_pxl(int *x, int *y)
{
 *x*=font_w;
 *y*=font_h;
}
unsigned int cscr_getcols()
{
 return COLS;
}
unsigned int cscr_getlines()
{
 return LINES;
}

// Group 3:
void cscr_wake_mouse()
{
 if (mouse_alive) return;
 mouse_alive=1;
 show_mouse(screen);
}
void cscr_sleep_mouse()
{
 if (!mouse_alive) return;
 mouse_alive=0;
 show_mouse(NULL);
}
int cscr_query_mouse()
{
 return mouse_exists;
}

// Group 4a
void cscr_clear()
{
     if (just_cleared !=alleg_bg)
     {
      clear_to_color(Buffer,alleg_bg);
      just_cleared=alleg_bg;
     }
}
void cscr_clrtoeol()
{
  set_clip(Buffer,alleg_x, alleg_y,VHORIZ_REZ,alleg_y+font_h);
  clear_to_color(Buffer,alleg_bg);
  set_clip(Buffer,0, 0,VHORIZ_REZ,VVERT_REZ);
}
void cscr_refresh()
{
 // Note: screen updates happen automatically if buffering is off
 scare_mouse();
 if (glk_preferences[pref_dbl_buf])
  blit(Buffer,screen,H_window,V_window,0,0,HORIZ_REZ,VERT_REZ);
 unscare_mouse();
}
void cscr_moveyx(int y, int x)
{
   alleg_x=x*font_w;
   alleg_y=(y*font_h);
}
void cscr_screencap()
{
 char s[13];
 static int n=0;
 if (!glk_preferences[pref_dbl_buf]) return;
 // Generate a file name
 do {
  sprintf(s,"scr%05d.bmp",n);
  n++;
  }
 while(__file_exists(s) && n < 100000);
 if (n>99999) return;
 // Write out the video buffer
 save_bmp(s,Buffer,NULL);
}

//Group 4b
int cscr_addstr(char *s)
{
   int i; char *s1; chtype attr;
   for(i=0;i<strlen(s);i++)
    if (s[i]=='\n')
     {
      s[i]=0;
      s1=strdup(s);
      s[i]='\n';
      cscr_addstr(s1);
      free(s1);
      alleg_y+=font_h;
      cscr_addstr(s+i+1);
      return 1;
     }
    else if (s[i]=='\r')
    {
      s[i]=0;
      s1=strdup(s);
      s[i]='\r';
      cscr_addstr(s1);
      free(s1);
      alleg_x=0;
      cscr_addstr(s+i+1);
      return 1;
    }
   just_cleared=-1;
   attr=alleg_attr;
   cscr_attrset_deep(alleg_tattr);
   textout(Buffer,font,s,alleg_x,alleg_y, alleg_color);
   cscr_attrset_deep(attr);
   alleg_x+=text_length(font,s);
   if (alleg_x>VHORIZ_REZ) alleg_x=VHORIZ_REZ;
   return 0; 
}
void cscr_addch(chtype c)
{
  char f[2];
  f[1]=0;
  f[0]=c;
  cscr_addstr(f);

}
int cscr_printw(char *fmt, ...)
{
 va_list args;
 int retval=ERR;
 char c_printscanbuf[1024];
 va_start(args,fmt);
 vsprintf(c_printscanbuf, fmt, args);
 va_end(args);
 if (cscr_addstr(c_printscanbuf) != ERR)
  retval = (strlen(c_printscanbuf));
 return (retval);
}

// Group 4c

chtype cscr_color(int f, int b)
{
 int p,a;
 a=0;
 if (f >= 8) { a = 0x00800000L; f -= 8; }
 if (b >= 8) { a |= 0x00400000L; b -=8; }
 p=f + (8*b) + 1;
 return (p << 24) | a;
}
unsigned char cscr_color_to_8bit(chtype c)
{
 int cp = c >> 24;
 unsigned char fg = (cp - 1 ) % 8;
 unsigned char bg = (cp - 1 ) / 8;
 if (c & A_BOLD) fg |= 8;
 if (c & A_BGBRITE) bg |= 8;
 return fg | (bg << 4 );
}
void cscr_attron(chtype a)
{
 cscr_attrset(alleg_attr | a);
}
void cscr_attrset(chtype x)
{
  alleg_tattr=x;
}
chtype cscr_bkgd(chtype i)
{
   int j=i >> 24;
   if (i & 0x00400000L)
    alleg_bg=allegcolor[alleg_colors_b(j) & 0x08];   
   else
   alleg_bg=allegcolor[alleg_colors_b(j)];
  return i;
}

// Group 5
int cscr_kbhit()
{
 if (check_mouse()) return 1; else
 return keypressed();
}
void cscr_blinkey()
{
 not_box(Buffer,alleg_x,alleg_y);
 cscr_refresh();
}
chtype cscr_getch()
{
 int c=-1;
 if(check_mouse()) {  mouse_click=0; return MOUSE_CLICK_KEY; }
 while (c==-1) c=(tochtype(readkey()));
 just_cleared=-1;
 return c;
}
