/*
	FrostzApp.cpp: Defines the FrostzApp class.
	Copyright (C) 2005 Fabio Concas

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/



#include "StdAfx.h"
#include <Commdlg.h>

#include "resource.h"
#include "FrostzApp.h"


extern "C" {
	#include "Frotz/Frotz.h"
//	extern bool is_terminator (zchar key);
//	extern int completion (const zchar *buffer, zchar *result);

	// In blind mode, no text styles and no compass
	extern char cBlindMode;
}

#include "ZThread.h"


unsigned short* szClassName = L"Frostz";
unsigned short* szTitle = L"Frostz for PocketPC";

static BYTE DefaultColors;

bool GetAppDir (LPWSTR pathName, HINSTANCE appInstance);



/*** MRU Stuff ***/
//#define BASE_MRU_ID	41550 Now defined with the resources
#define MAX_MRU	4
struct tagMRU {
	unsigned short wFileName[256];
	unsigned short *wShortName;
} MRU[MAX_MRU];
/*****************/



//*** I'll use this to keep track of CBM palette while cycling between colors
static BYTE CBMPaletteCycle;



//****************************************************************************
//		FrostzApp
//	Constructor
//****************************************************************************
FrostzApp::FrostzApp () : hWnd (0), hInstance (0), bkBMP (0),
	isZmachineRunning (false), Initialized (false)
{
	memset (sStoryFile, 0, 256);
	//sStoryFile[0] = 0;

	bShowSip = true;	// SIP on by default

	InterpreterNumber = INTERP_MSDOS;
	SimulateCBM = false;
	bUnderline = false;

	/* - Experimental -
	numFonts = 0;
	memset (FontFace, 0, sizeof (unsigned short) * 256);
	*/

	pScreen = NULL;
	pAttributes = NULL;
	pFonts = NULL;

	imgCount = 0;
	imgVersion = 0;
	imgDir = NULL;
	imgFile = NULL;

	ReadKeyBuffer = KeyBuffer;
	WriteKeyBuffer = KeyBuffer;

	ExpectingInput = false;

	Redraw = true;

	InstantRefresh = false;

	State = STATE_NONE;

	CurrentFont = TEXT_FONT;//NULL;

	UpdateTimerMs = FROSTZ_UPDATE_MS;

	/* Set up palette	-	Now done in InitializeScreen
	Palette[PALETTE_BLACK]	= RGB(0,0,0);
	Palette[PALETTE_WHITE]	= RGB(0xff,0xff,0xff);
	Palette[PALETTE_RED]	= RGB(0xff,0,0);
	Palette[PALETTE_GREEN]	= RGB(0,0xff,0);
	Palette[PALETTE_YELLOW]	= RGB(0xff,0xff,0);
	Palette[PALETTE_BLUE]	= RGB(0,0,0xff);
	Palette[PALETTE_MAGENTA]= RGB(0xff,0,0xff);
	Palette[PALETTE_CYAN]	= RGB(0,0xff,0xff);
	Palette[PALETTE_GREY]	= RGB(0x80,0x80,0x80);
	*/
}



//****************************************************************************
//		~FrostzApp
//	Destructor
//****************************************************************************
FrostzApp::~FrostzApp ()
{
	// Free font object
	if (hCurrentFont) DeleteObject (hCurrentFont );

	// Free image directories
	if (imgDir) LocalFree (imgDir);

	// Free image file pointer
	if (imgFile) fclose (imgFile);

	// Free screen buffers
	if (pScreen) LocalFree (pScreen);
	if (pAttributes) LocalFree (pAttributes);
	if (pFonts) LocalFree (pFonts);

	if (hbmpScreen) DeleteObject (hbmpScreen);
}



//****************************************************************************
//		LoadRegistry
//	Loads settings from the registry
//****************************************************************************
void FrostzApp::LoadRegistry ()
{
	DWORD Type, Len;
	HKEY RegKey;

	if (RegOpenKeyEx (HKEY_CURRENT_USER, L"Software\\Frotz", 0, 0, &RegKey)
		!= ERROR_SUCCESS)
		return;

	Len = sizeof (FontHeight);
	if (RegQueryValueEx (RegKey, L"FontHeight", 0, &Type, (LPBYTE)&FontHeight, &Len))
		FontHeight = 11;
	Len = sizeof (FontWidth);
	if (RegQueryValueEx (RegKey, L"FontWidth", 0, &Type, (LPBYTE)&FontWidth, &Len))
		FontWidth = 4;

	// Be sure to have valid values
	if (FontWidth != 4 && FontWidth != 6 && FontWidth != 8)
		FontWidth = 4;
	if (FontHeight != 8 && FontHeight != 10 && FontHeight != 11 && FontHeight != 12)
		FontHeight = 11;

	Len = sizeof (DefaultStyle);
	if (RegQueryValueEx (RegKey, L"Style", 0, &Type, (LPBYTE)&DefaultStyle, &Len))
		DefaultStyle = NORMAL_STYLE;

	Len = sizeof (bUnderline);
	if (RegQueryValueEx (RegKey, L"Underline", 0, &Type, (LPBYTE)&bUnderline, &Len))
		bUnderline = false;

	Len = sizeof (UseStyles);
	if (RegQueryValueEx (RegKey, L"UseStyles", 0, &Type, (LPBYTE)&UseStyles, &Len))
		UseStyles = false;

	Len = sizeof (UseColours);
	if (RegQueryValueEx (RegKey, L"UseColors", 0, &Type, (LPBYTE)&UseColours, &Len))
		UseColours = true;

	Len = sizeof (UseSounds);
	if (RegQueryValueEx (RegKey, L"UseSounds", 0, &Type, (LPBYTE)&UseSounds, &Len))
		UseSounds = true;

	Len = sizeof (cBlindMode);
	if (RegQueryValueEx (RegKey, L"BlindMode", 0, &Type, (LPBYTE)&cBlindMode, &Len))
		cBlindMode = 0;

	Len = sizeof (ShowCompass);
	if (RegQueryValueEx (RegKey, L"ShowCompass", 0, &Type, (LPBYTE)&ShowCompass, &Len))
		ShowCompass = true;

	Len = sizeof (expand_abbreviations);
	if (RegQueryValueEx (RegKey, L"Expand", 0, &Type, (LPBYTE)&expand_abbreviations, &Len))
		expand_abbreviations = false;

	Len = sizeof (ignore_errors);
	if (RegQueryValueEx (RegKey, L"IgnoreRTE", 0, &Type, (LPBYTE)&ignore_errors, &Len))
		ignore_errors = true;

	Len = sizeof (TandyBit);
	if (RegQueryValueEx (RegKey, L"TandyBit", 0, &Type, (LPBYTE)&TandyBit, &Len))
		TandyBit = false;

	Len = sizeof (InterpreterNumber);
	if (RegQueryValueEx (RegKey, L"Interpreter", 0, &Type, (LPBYTE)&InterpreterNumber, &Len))
		InterpreterNumber = INTERP_MSDOS;

	Len = sizeof (SimulateCBM);
	if (RegQueryValueEx (RegKey, L"SimulateCBM", 0, &Type, (LPBYTE)&SimulateCBM, &Len))
		SimulateCBM = false;


	Len = sizeof (UpdateTimerMs);
	if (RegQueryValueEx (RegKey, L"TimerMs", 0, &Type, (LPBYTE)&UpdateTimerMs, &Len))
		UpdateTimerMs = FROSTZ_UPDATE_MS;

	unsigned int Cnt;

	/*** MRU Files ***/
	for (Cnt = 0; Cnt < MAX_MRU; ++Cnt)
	{
		unsigned short wValueName[8];

		wsprintf (wValueName, L"MRU%d", Cnt);
		Len = 256;
		if (RegQueryValueEx (RegKey, wValueName, 0, &Type, (LPBYTE)MRU[Cnt].wFileName, &Len))
		{
			MRU[Cnt].wFileName[0] = (unsigned short)0;
			MRU[Cnt].wShortName = 0;
		}
		else
			MRU[Cnt].wShortName = wcsrchr (MRU[Cnt].wFileName, (unsigned short)'\\') + 1;
	}
	/*****************/

	//*** Macros ***
	for (Cnt = 0; Cnt < MAX_MACROS; ++Cnt)
	{
		unsigned short wValueName[16];

		wsprintf (wValueName, L"Macro%dName", Cnt);
		Len = 256;
		if (RegQueryValueEx (RegKey, wValueName, 0, &Type, (LPBYTE)Macros[Cnt].wName, &Len))
		{
			Macros[Cnt].wName[0] = (unsigned short)0;
			Macros[Cnt].isDefined = false;
			Macros[Cnt].nlStyle = 0;
		}

		// If name is a null string, macro is undefined, else...
		if (Macros[Cnt].wName[0])
		{
			Macros[Cnt].isDefined = true;

			// Read action
			wsprintf (wValueName, L"Macro%dAction", Cnt);
			Len = 256;
			if (RegQueryValueEx (RegKey, wValueName, 0, &Type, (LPBYTE)Macros[Cnt].wAction, &Len))
				wcscpy (Macros[Cnt].wAction, L"Undefined");

			// Read newline style
			wsprintf (wValueName, L"Macro%dNewLine", Cnt);
			Len = sizeof (Macros[Cnt].nlStyle);
			if (RegQueryValueEx (RegKey, wValueName, 0, &Type, (LPBYTE)&Macros[Cnt].nlStyle, &Len))
				Macros[Cnt].nlStyle = 0; // No newline is the default
		}
	}



	UpdateMenuItems ();

	RegCloseKey (RegKey);
}



//****************************************************************************
//		SaveRegistry
//****************************************************************************
void FrostzApp::SaveRegistry ()
{
	HKEY RegKey;
	if (RegOpenKeyEx (HKEY_CURRENT_USER, L"Software\\Frotz", 0, 0, &RegKey)
		!= ERROR_SUCCESS)
	{
		DWORD Disp;
		// If key doesn't exist, then create it
		if (RegCreateKeyEx (HKEY_CURRENT_USER, L"Software\\Frotz",
			0, NULL, 0, NULL, NULL, &RegKey, &Disp) != ERROR_SUCCESS)
		{
			MessageBox (hWnd, L"Could not save settings to registry!", L"Frostz",
				MB_OK | MB_ICONEXCLAMATION);
			return;
		}
	}

	RegSetValueEx (RegKey, L"FontHeight", 0, REG_DWORD, (LPBYTE)&FontHeight, sizeof (FontHeight));

	RegSetValueEx (RegKey, L"FontWidth", 0, REG_DWORD, (LPBYTE)&FontWidth, sizeof (FontWidth));

	RegSetValueEx (RegKey, L"Style", 0, REG_DWORD, (LPBYTE)&DefaultStyle, sizeof (DefaultStyle));

	RegSetValueEx (RegKey, L"Underline", 0, REG_BINARY, (LPBYTE)&bUnderline, sizeof (bUnderline));

	RegSetValueEx (RegKey, L"UseStyles", 0, REG_BINARY, (LPBYTE)&UseStyles, sizeof (UseStyles));

	RegSetValueEx (RegKey, L"UseColors", 0, REG_BINARY, (LPBYTE)&UseColours, sizeof (UseColours));

	RegSetValueEx (RegKey, L"ShowCompass", 0, REG_BINARY, (LPBYTE)&ShowCompass, sizeof (ShowCompass));

	RegSetValueEx (RegKey, L"UseSounds", 0, REG_BINARY, (LPBYTE)&UseSounds, sizeof (UseSounds));

	RegSetValueEx (RegKey, L"BlindMode", 0, REG_BINARY, (LPBYTE)&cBlindMode, sizeof (cBlindMode));

	RegSetValueEx (RegKey, L"Expand", 0, REG_BINARY, (LPBYTE)&expand_abbreviations, sizeof (expand_abbreviations));

	RegSetValueEx (RegKey, L"IgnoreRTE", 0, REG_BINARY, (LPBYTE)&ignore_errors, sizeof (ignore_errors));

	RegSetValueEx (RegKey, L"TandyBit", 0, REG_BINARY, (LPBYTE)&TandyBit, sizeof (TandyBit));

	RegSetValueEx (RegKey, L"Interpreter", 0, REG_DWORD, (LPBYTE)&InterpreterNumber, sizeof (InterpreterNumber));

	RegSetValueEx (RegKey, L"SimulateCBM", 0, REG_BINARY, (LPBYTE)&SimulateCBM, sizeof (SimulateCBM));

	RegSetValueEx (RegKey, L"TimerMs", 0, REG_DWORD, (LPBYTE)&UpdateTimerMs, sizeof (UpdateTimerMs));

	unsigned int Cnt;

	/*** MRU Files ***/
	for (Cnt = 0; Cnt < MAX_MRU; ++Cnt)
	{
		unsigned short wValueName[8];

		wsprintf (wValueName, L"MRU%d", Cnt);
		if (MRU[Cnt].wFileName[0] != 0)
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)MRU[Cnt].wFileName, 256);
		else
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)L"\0", sizeof (unsigned short));
	}
	/*****************/

	//*** Macros ***
	for (Cnt = 0; Cnt < MAX_MACROS; ++Cnt)
	{
		unsigned short wValueName[16];

		if (Macros[Cnt].isDefined)
		{
			wsprintf (wValueName, L"Macro%dName", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)Macros[Cnt].wName, sizeof (Macros[Cnt].wName));
			wsprintf (wValueName, L"Macro%dAction", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)Macros[Cnt].wAction, sizeof (Macros[Cnt].wAction));
			wsprintf (wValueName, L"Macro%dNewLine", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_DWORD, (LPBYTE)&Macros[Cnt].nlStyle, sizeof (Macros[Cnt].nlStyle));
		}
		else
		{
			BYTE Tmp = 0;
			wsprintf (wValueName, L"Macro%dName", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)L"\0", sizeof (unsigned short));
			wsprintf (wValueName, L"Macro%dAction", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_SZ, (LPBYTE)L"\0", sizeof (unsigned short));
			wsprintf (wValueName, L"Macro%dNewLine", Cnt);
			RegSetValueEx (RegKey, wValueName, 0, REG_DWORD, (LPBYTE)&Tmp, sizeof (Tmp));
		}
	}


	RegCloseKey (RegKey);
}



//****************************************************************************
//		InitializeScreen
//****************************************************************************
void FrostzApp::InitializeScreen ()
{
	RECT cRect;
	HDC hdc;
	HFONT hOldFont;

	DefaultFontHeight = FontHeight;
	DefaultFontWidth = FontWidth;

	// Create keypress event object
	hKeyEvent = ::CreateEvent (NULL, FALSE, FALSE, NULL);

	// Initialise critical sections
	memset (&KbdCriticalSection, 0, sizeof (KbdCriticalSection));
	::InitializeCriticalSection (&KbdCriticalSection);
	memset (&ScrnCriticalSection, 0, sizeof (ScrnCriticalSection));
	::InitializeCriticalSection (&ScrnCriticalSection);

	// Set cursor state
	CursorOn = false;
	CursorSet = true;

	// Reset widths etc.
	Width = 0;
	CharWidth = 0;
	Height = 0;
	CharHeight = 0;

	// Get window rectangle
	GetClientRect (hWnd, &cRect);
	cRect.bottom += 4;

	// Get device context
	hdc = GetDC (hWnd);

	// Set up palette
	if (InterpreterNumber == INTERP_CBM_64 || InterpreterNumber == INTERP_CBM_128)
	{
		CBMPaletteCycle = (SimulateCBM)? 0 : 5;
		CycleCBMPalette ();
	}
	else
	{
		Palette[PALETTE_BLACK]	= RGB(0,0,0);
		Palette[PALETTE_WHITE]	= RGB(0xff,0xff,0xff);

		Palette[PALETTE_RED]	= RGB(0xff,0,0);
		Palette[PALETTE_GREEN]	= RGB(0,0xff,0);
		Palette[PALETTE_YELLOW]	= RGB(0xff,0xff,0);
		Palette[PALETTE_BLUE]	= RGB(0,0,0xff);
		Palette[PALETTE_MAGENTA]= RGB(0xff,0,0xff);
		Palette[PALETTE_CYAN]	= RGB(0,0xff,0xff);
		Palette[PALETTE_GREY]	= RGB(0x80,0x80,0x80);
	}

	Palette[PALETTE_DARKGREY]	= RGB(0x80, 0x80, 0x80);
	Palette[PALETTE_DARKRED]	= RGB(0x80, 0x00, 0x00);
	Palette[PALETTE_DARKYELLOW]	= RGB(0x80, 0x80, 0x00);
	Palette[PALETTE_DARKGREEN]	= RGB(0x00, 0x80, 0x00);
	Palette[PALETTE_DARKCYAN]	= RGB(0x00, 0x80, 0x80);
	Palette[PALETTE_DARKBLUE]	= RGB(0x00, 0x00, 0x80);
	Palette[PALETTE_VIOLET]		= RGB(0x80, 0x00, 0x80);

	Palette[PALETTE_USER_FG]	= RGB(0, 0, 0);
	Palette[PALETTE_USER_BG]	= RGB(0xff, 0xff, 0xff);

	Palette[PALETTE_USER_COLOR] = RGB(0xff, 0xff, 0xff);

	// Set colour
	if (InterpreterNumber == INTERP_CBM_64)
		DefaultColors = (PALETTE_CYAN << 2) | PALETTE_BLUE;
	else if (InterpreterNumber == INTERP_CBM_128)
		DefaultColors = (PALETTE_YELLOW << 2) | PALETTE_GREEN;
	else
		DefaultColors = DEF_TEXT_COLOUR;

	SetColour (hdc, DefaultColors);

	// Initialize logical font structures
	memset (&logFont[0], 0, sizeof (LOGFONT));
	wcscpy (logFont[0].lfFaceName,
		(InterpreterNumber == INTERP_CBM_64 || InterpreterNumber == INTERP_CBM_128)?
		L"CBM-P" : L"Frostz");
	logFont[0].lfCharSet = ANSI_CHARSET;
	logFont[0].lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
	logFont[0].lfHeight = DefaultFontHeight;
	logFont[0].lfWidth = DefaultFontWidth;

	memset (&logFont[1], 0, sizeof (LOGFONT));
	wcscpy (logFont[1].lfFaceName,
		(InterpreterNumber == INTERP_CBM_64 || InterpreterNumber == INTERP_CBM_128)?
		L"CBM Zork" : L"Beyond Zork");
	logFont[1].lfCharSet = ANSI_CHARSET;
	logFont[1].lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
	logFont[1].lfHeight = DefaultFontHeight;
	logFont[1].lfWidth = DefaultFontWidth;

	// Set default font
	CurrentFont = TEXT_FONT;

	// Using text styles?
	if (UseStyles)
	{
		// Select font into device context
		hOldFont = SetStyle (hdc, NORMAL_STYLE | BOLDFACE_STYLE | EMPHASIS_STYLE, TRUE);
	}
	else
	{
		// Select font into device context
		hOldFont = SetStyle (hdc, NORMAL_STYLE, TRUE);
	}


	// Get the width of a string
	SIZE sTextExtent;
	GetTextExtentPoint (hdc, L"WWWWWWWWWW", 10, &sTextExtent);

	// Save character dimensions
	CharWidth = (USHORT) (sTextExtent.cx / 10);
	CharHeight = (USHORT) sTextExtent.cy;

	// Set colour
	SetColour (hdc, DefaultColors);

	// Using text styles?
	if (UseStyles)
	{
		// Select previous font into device context
		SelectObject (hdc, hOldFont);
		// Select font into device context
		hOldFont = SetStyle (hdc, DefaultStyle);
	}

	// Select previous font into device context
	SelectObject (hdc, hOldFont);

	// Save screen dimensions
	Width = (cRect.right - cRect.left) / CharWidth;
	Height = (cRect.bottom - cRect.top) / CharHeight;

	// Allocate screen buffers
	pScreen = 
		(TCHAR *) ::LocalAlloc (LPTR, Width * Height * sizeof(TCHAR));
	pAttributes = 
		(BYTE *) ::LocalAlloc (LPTR, Width * Height);
	pFonts =
		(BYTE *) ::LocalAlloc (LPTR, Width * Height);

	if (hbmpScreen)
		DeleteObject (hbmpScreen);
	hbmpScreen = (h_version == V6 || story_id == BEYOND_ZORK)?
		CreateCompatibleBitmap (hdc, 240, 215)
		: NULL;

	// Release device context
	ReleaseDC (hWnd, hdc);

	// Clear the screen
	ClearScreen (FALSE);

	// Send a message to the view to create the caret
	SendMessage (hWnd, WM_CREATE_CURSOR, 0, 0);

	// Create timer to do update
	hUpdateTimer = SetTimer (hWnd, ID_UPDATE_TIMER, UpdateTimerMs, NULL);
}



//****************************************************************************
//		Initialize
//	Creates the main window and the menu bar
//****************************************************************************
bool FrostzApp::Initialize (HINSTANCE newInstance, int nCmdShow)
{
	WNDCLASS wc;
	SHMENUBARINFO mbInfo;

	if (hWnd != 0 || hInstance != 0)
		return false;

	hInstance = newInstance;

	// Register our class
    wc.style			= CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc		= (WNDPROC) WndProc;
    wc.cbClsExtra		= 0;
    wc.cbWndExtra		= 0;
    wc.hInstance		= hInstance;
    wc.hIcon			= LoadIcon (hInstance, MAKEINTRESOURCE (IDI_FROSTZ));
    wc.hCursor			= 0;
    wc.hbrBackground	= (HBRUSH) GetStockObject (WHITE_BRUSH);
    wc.lpszMenuName		= 0;
    wc.lpszClassName	= szClassName;

	if (RegisterClass (&wc) == 0)
		return false;

	// Create the main window
	hWnd = CreateWindowEx (
		WS_EX_TOPMOST,
		szClassName,
		szTitle,
		WS_VISIBLE | WS_POPUP,
		0, 0, CW_USEDEFAULT, CW_USEDEFAULT, NULL,
		NULL, hInstance, NULL);

	if (!hWnd)
		return false;

	ShowWindow (hWnd, nCmdShow);
	//UpdateWindow (hWnd);

	// Create the main menu bar
	memset (&mbInfo, 0, sizeof (SHMENUBARINFO));
	mbInfo.cbSize = sizeof (SHMENUBARINFO);
	mbInfo.hwndParent = hWnd;
	mbInfo.dwFlags = SHCMBF_COLORBK /*| SHCMBF_HIDESIPBUTTON | SHCMBF_HMENU*/;
	mbInfo.nToolBarId = IDR_APP_MENUBAR;
	mbInfo.hInstRes = hInstance;
	mbInfo.nBmpId = 0;//IDR_MAIN_TOOLBAR;
	mbInfo.cBmpImages = 0;//2;
	mbInfo.clrBk = RGB (0xFF, 0xFF, 0xFF);

	if (!SHCreateMenuBar (&mbInfo))
	{
		unsigned short wMsg[32];
		wsprintf (wMsg, L"Error #%d", GetLastError ());
		MessageBox (hWnd, L"Could not create menu bar", wMsg, MB_ICONSTOP);
		SendMessage (hWnd, WM_DESTROY, 0, 0);
	}
	else
	{
		hMenuBar = mbInfo.hwndMB;
		CommandBar_AddBitmap (hMenuBar, hInstance, IDR_MAIN_TOOLBAR, 3, 16, 16);
	}

	// Hide the taskbar
	//SetForegroundWindow (hWnd);
	//SHFullScreen (hWnd, SHFS_HIDETASKBAR);

	/* Get visible rectangle and set the window rectangle properly
	SIPINFO sSIP;
	sSIP.cbSize = sizeof (SIPINFO);
	SipGetInfo (&sSIP);
	/*** DEBUG ***
	{
		unsigned short wMsg[256];
		wsprintf (wMsg, L"%d, %d, %d, %d",
			sSIP.rcVisibleDesktop.left,
			sSIP.rcVisibleDesktop.top,
			sSIP.rcVisibleDesktop.right,
			sSIP.rcVisibleDesktop.bottom);
		MessageBox (hWnd, wMsg, L"DEBUG", MB_OK | MB_ICONINFORMATION);
	}
	/*************/

	/*
	ClientRect = sSIP.rcVisibleDesktop;
	ClientRect.top -= 26;
	ClientRect.bottom += 1;
	/*/
	ClientRect.top = -1;
	ClientRect.left = 0;
	ClientRect.bottom = (bShowSip && !cBlindMode)? 215 : 303;
	ClientRect.right = 240;


	/*
	MoveWindow (hWnd,
		ClientRect.left, // 0
		ClientRect.top, // -1
		ClientRect.right, // 240
		ClientRect.bottom, // 216
		FALSE);
	*/

	// Load registry settings
	LoadRegistry ();

	GoFullScreen ();

	return true;
}



//****************************************************************************
//		SetFont
//	Sets a new font face
//****************************************************************************
HFONT FrostzApp::SetFont (HDC hDC, BYTE nFont, BOOL bForce)
{
	// Font changed?
	if (nFont != CurrentFont || bForce)
	{
		//LOGFONT sLogFont;
		unsigned char nF = (nFont == GRAPHICS_FONT)? 1 : 0;
		HFONT hNewFont;

		CurrentFont = nFont;

		// Reverse style?
		if (!cBlindMode)
		if (CurrentStyle & REVERSE_STYLE)
		{
			// Set current text colour
			SetColour (hDC, CurrentColour);
		}

		// Set up logical font structure
		logFont[nF].lfItalic = FALSE;
		logFont[nF].lfWeight = FW_THIN;
		logFont[nF].lfHeight = DefaultFontHeight;
		logFont[nF].lfWidth = DefaultFontWidth;

		// Create font
		if ((hNewFont = ::CreateFontIndirect (&logFont[nF])) != NULL)
		{
			// Replace current font
			if (hCurrentFont) ::DeleteObject( (HGDIOBJ) hCurrentFont );
			hCurrentFont = hNewFont;

			// Select font into device context
			return (HFONT) SelectObject (hDC, hNewFont);
		}
		else return NULL;
	}
	else
	{
		// Select current font into device context
		return (HFONT) SelectObject(hDC, hCurrentFont);
	}

	return NULL;
}



//****************************************************************************
//		SetStyle
//	Sets a new font style
//****************************************************************************
HFONT FrostzApp::SetStyle (HDC hDC, BYTE nStyle, BOOL bForce)
{
	// Not using text styles?
	if (!UseStyles)
	{
		// Remove style flags
		nStyle &= ~BOLDFACE_STYLE;
		nStyle &= ~EMPHASIS_STYLE;
	}

	// Use default style
	nStyle |= DefaultStyle;

	// Style changed?
	if (nStyle != CurrentStyle || bForce)
	{
		//LOGFONT sLogFont;
		unsigned char nF = (CurrentFont == GRAPHICS_FONT)? 1 : 0;
		HFONT hNewFont;

		// Reverse style changing?
		//if (!cBlindMode)
		if ((CurrentStyle & REVERSE_STYLE) != (nStyle & REVERSE_STYLE))
		{
			// Set current text style
			CurrentStyle = nStyle;

			// Set current text colour
			SetColour (hDC, CurrentColour);
		}
		// Set current text style
		else CurrentStyle = nStyle;

		// Set up logical font structure
		logFont[nF].lfHeight = DefaultFontHeight;
		logFont[nF].lfWidth = DefaultFontWidth;
		logFont[nF].lfItalic = FALSE;
		logFont[nF].lfUnderline = FALSE;
		logFont[nF].lfWeight = FW_THIN;

		// Bold?
		if (CurrentStyle & BOLDFACE_STYLE)
		{
			if (bUnderline) //*** Use underlined text instead of bold?
				logFont[nF].lfUnderline = TRUE;
			else
			{
				logFont[nF].lfWeight = FW_SEMIBOLD;
				logFont[nF].lfWidth = DefaultFontWidth - 1;//*** This should correct spacing
			}
		}


		// Italic?
		if (CurrentStyle & EMPHASIS_STYLE)
		{
			//sLogFont.lfWeight = FW_BOLD;
			logFont[nF].lfUnderline = TRUE;
			logFont[nF].lfItalic = TRUE;
		}


		// Create font
		if ((hNewFont = ::CreateFontIndirect (&logFont[nF])) != NULL)
		{
			// Replace current font
			if (hCurrentFont) ::DeleteObject( (HGDIOBJ) hCurrentFont );
			hCurrentFont = hNewFont;

			// Select font into device context
			return (HFONT) SelectObject (hDC, hNewFont);
		}
		else return NULL;
	}
	else
	{
		// Select current font into device context
		return (HFONT) SelectObject(hDC, hCurrentFont);
	}

	return NULL;	
}



//****************************************************************************
//		SetColour
//	Sets a new text color
//****************************************************************************
void FrostzApp::SetColour (HDC hDC, BYTE nColour)
{
	// Set current text colour
	CurrentColour = nColour;

	// Reverse mode?
	if (!cBlindMode && (CurrentStyle & REVERSE_STYLE))
	{
		// Set text and background colours to reverse
		SetTextColor (hDC,
			GetNearestColor (hDC, Palette[CurrentColour >> 2]));
		SetBkColor (hDC,
			GetNearestColor (hDC, Palette[CurrentColour & 3]));
	}
	else
	{
		// Set text and background colours
		SetTextColor (hDC,
			GetNearestColor (hDC, Palette[CurrentColour & 3]));
		SetBkColor (hDC,
			GetNearestColor (hDC, Palette[CurrentColour >> 2]));
	}
}



//****************************************************************************
//		SetBgImage
//	Sets a new BG Image. Call with BmpId = 0 to remove the BG.
//****************************************************************************
void FrostzApp::SetBgImage (WORD BmpId)
{
//	RECT Rect;

//	Rect.left = Rect.top = 0;
//	Rect.right = 240;
//	Rect.bottom = 215;

	bkBMP = BmpId;

	InvalidateRect (hWnd, &ClientRect, TRUE);
	UpdateWindow (hWnd);
}



//****************************************************************************
//		ResetScreen
//****************************************************************************
void FrostzApp::ResetScreen ()
{
	//RECT Rect;

	// Free event object
	::CloseHandle (hKeyEvent);

	// Release critical sections
	::DeleteCriticalSection (&KbdCriticalSection );
	::DeleteCriticalSection (&ScrnCriticalSection );


	// Free font object
	if (hCurrentFont) 
	{
		::DeleteObject (hCurrentFont);
		hCurrentFont = NULL;
	}

	// Free screen buffers
	if (pScreen) 
	{
		::LocalFree (pScreen);
		pScreen = NULL;
	}
	if (pAttributes)
	{
		::LocalFree (pAttributes);
		pAttributes = NULL;
	}
	if (pFonts)
	{
		::LocalFree (pFonts);
		pFonts = NULL;
	}

	// Get frame window rectangle
	//GetClientRect (hWnd, &Rect);

	/*** Try to hide the TaskBar ***
	::SHFullScreen (MAINFRAME->m_hWnd, SHFS_HIDETASKBAR);

	cRect += CRect (0, 0, 0, +26);

	// Resize the view to cover the entire frame window client area
	MoveWindow( cRect, FALSE );
	*/

	// Free timer
	if (hUpdateTimer) KillTimer (hWnd, hUpdateTimer);

	// Clear screen
	InvalidateRect (hWnd, &ClientRect, TRUE);
}



//****************************************************************************
//		ClearScreen
//****************************************************************************
void FrostzApp::ClearScreen (BOOL Update)
{
	// Stop displaying background bitmap
	bkBMP = 0;

	InvalidateRect (hWnd, &ClientRect, TRUE);

	// Erase the entire screen
	EraseScreenArea (0, 0, Height-1, Width-1);

	// Set the cursor position to upper left-hand corner
	SetCursorPos (0, 0);
}



//****************************************************************************
//		SetTextColour
//****************************************************************************
void FrostzApp::SetTextColour (int Fg, int Bg)
{
	BYTE nColour = 0;

	/*** DEBUG ***
	{
		char Str[80];
		sprintf (Str, "FG%d-BG%d", Fg, Bg);
		DisplayString (Str);
	}
	/*************/

	// Use colours?
	if (UseColours)
	{
		// Map foreground colour
		switch (Fg)
		{
			case 0: // Default color
			case 1:
				nColour = PALETTE_BLACK;
				break;
			case WHITE_COLOUR:
				nColour = PALETTE_WHITE;
				break;
			case GREY_COLOUR:
			// same as LIGHTGREY_COLOUR
				nColour = PALETTE_GREY;
				break;
			case RED_COLOUR:
				nColour = PALETTE_RED;
				break;
			case GREEN_COLOUR:
				nColour = PALETTE_GREEN;
				break;
			case BLUE_COLOUR:
				nColour = PALETTE_BLUE;
				break;
			case YELLOW_COLOUR:
				nColour = PALETTE_YELLOW;
				break;
			case MAGENTA_COLOUR:
				nColour = PALETTE_MAGENTA;
				break;
			case CYAN_COLOUR:
				nColour = PALETTE_CYAN;
				break;
			case MEDIUMGREY_COLOUR:
			case DARKGREY_COLOUR:
				nColour = PALETTE_DARKGREY;
				break;
			case BLACK_COLOUR:
				nColour = PALETTE_BLACK;
				break;
			case DARKRED_COLOUR:
				nColour = PALETTE_DARKRED;
				break;
			case DARKGREEN_COLOUR:
				nColour = PALETTE_DARKGREEN;
				break;
			case DARKYELLOW_COLOUR:
				nColour = PALETTE_DARKYELLOW;
				break;
			case DARKBLUE_COLOUR:
				nColour = PALETTE_DARKBLUE;
				break;
			case VIOLET_COLOUR:
				nColour = PALETTE_VIOLET;
				break;
			case DARKCYAN_COLOUR:
				nColour = PALETTE_DARKCYAN;
				break;
			case USER_COLOUR:
				nColour = PALETTE_USER_COLOR;
				break;
			default:
				nColour = PALETTE_BLACK;
				break;
		}

		// Map background colour
		switch (Bg)
		{
			case 0: // Default color
			case 1:
				nColour |= (PALETTE_WHITE<<2);
				break;
			case WHITE_COLOUR:
				nColour |= (PALETTE_WHITE<<2);
				break;
			case GREY_COLOUR:
			// Same as LIGHTGREY_COLOUR
				nColour |= (PALETTE_GREY<<2);
				break;
			case RED_COLOUR:
				nColour |= (PALETTE_RED<<2);
				break;
			case GREEN_COLOUR:
				nColour |= (PALETTE_GREEN<<2);
				break;
			case BLUE_COLOUR:
				nColour |= (PALETTE_BLUE<<2);
				break;
			case YELLOW_COLOUR:
				nColour |= (PALETTE_YELLOW<<2);
				break;
			case MAGENTA_COLOUR:
				nColour |= (PALETTE_MAGENTA<<2);
				break;
			case CYAN_COLOUR:
				nColour |= (PALETTE_CYAN<<2);
				break;
			case BLACK_COLOUR:
				nColour |= (PALETTE_BLACK<<2);
				break;
			case MEDIUMGREY_COLOUR:
			case DARKGREY_COLOUR:
				nColour |= (PALETTE_DARKGREY<<2);
				break;
			case DARKRED_COLOUR:
				nColour |= (PALETTE_DARKRED<<2);
				break;
			case DARKGREEN_COLOUR:
				nColour |= (PALETTE_DARKGREEN<<2);
				break;
			case DARKYELLOW_COLOUR:
				nColour |= (PALETTE_DARKYELLOW<<2);
				break;
			case DARKBLUE_COLOUR:
				nColour |= (PALETTE_DARKBLUE<<2);
				break;
			case VIOLET_COLOUR:
				nColour |= (PALETTE_VIOLET<<2);
				break;
			case DARKCYAN_COLOUR:
				nColour |= (PALETTE_DARKCYAN<<2);
				break; 
			case USER_COLOUR:
				nColour |= (PALETTE_USER_COLOR<<2);
				break;
			default:
				nColour |= PALETTE_WHITE<<2;
				break;
		}
	}
	else nColour = DefaultColors;

	::EnterCriticalSection (&ScrnCriticalSection);

	// Colour change?
	if (nColour != CurrentColour)
	{
		HDC hdc = GetDC (hWnd);

		// Set current colour
		SetColour (hdc, nColour);

		// Release device context
		ReleaseDC (hWnd, hdc);
	}

	::LeaveCriticalSection (&ScrnCriticalSection);
}



//****************************************************************************
//		SetCursorState
//****************************************************************************
void FrostzApp::SetCursorState (bool newState)
{
		// Changing state?
	if (CursorOn != newState)
	{
		// Set cursor state
		CursorOn  = newState;
		// Flag that state has changed
		CursorSet = FALSE;

		// Update caret
		UpdateCaretPos ();
	}
}



//****************************************************************************
//		GetKey
//****************************************************************************
int FrostzApp::GetKey (int Timeout)
{
	DWORD dwTimeOut = INFINITE;
	unsigned char ch = 0;

	// Update cursor position
	UpdateCaretPos ();

	// Timeout specified?
	if (Timeout > 0) dwTimeOut = Timeout * 100;

	// Set flag to indicate we're expecting input
	ExpectingInput = true;

	// Got a key?
	if (ReadKeyBuffer != WriteKeyBuffer 
		|| ::WaitForSingleObject (hKeyEvent, dwTimeOut) != WAIT_TIMEOUT)
	{
		// Was a key pressed?
		if (ReadKeyBuffer != WriteKeyBuffer)
		{
			// Get key from buffer
			ch = *ReadKeyBuffer;
			ReadKeyBuffer++;

			// Gone past end of buffer?
			if (ReadKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE - 1])
			{
				// Go back to beginning of buffer
				ReadKeyBuffer = &KeyBuffer[0];
			}
		}
	}

	ExpectingInput = false;

	return (int) ch;
}



#define MAX_LINE 80
#define MAX_HISTORY 5

//****************************************************************************
//		GetLine
//	Gets a line from the keyboard
//****************************************************************************
extern "C" int FrostzApp::GetLine (char* Buf, int Max, int Timeout)
{
	DWORD dwTimeOut = INFINITE;
	int ch = 0;
	int nCurrChar = strlen (Buf);
	static nNextHistoryPos = 0, nCurrHistoryPos = 0;
	static char acHistory[MAX_HISTORY][MAX_LINE];

	// Timeout specified?
	if (Timeout > 0) dwTimeOut = Timeout * 100;

	// Update cursor position
	UpdateCaretPos ();

	// Set flag to indicate we're expecting input
	ExpectingInput = true;

	do
	{
		// Was a key pressed?
		if (ReadKeyBuffer != WriteKeyBuffer)
		{
			// Get key from buffer
			ch = *ReadKeyBuffer;
			ReadKeyBuffer++;

			// Gone past end of buffer?
			if (ReadKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
			{
				// Go back to beginning of buffer
				ReadKeyBuffer = &KeyBuffer[0];
			}
		}
		else
		{
			// Received a keypress notification?
			if (::WaitForSingleObject (hKeyEvent, dwTimeOut ) != WAIT_TIMEOUT)
			{
				continue;
			}
			else ch = 0;
		}


		// Add character to buffer
		if (ch && !is_terminator ((int)ch)) 
		{
			// Check key pressed
			switch (ch)
			{
			case BACKSPACE_KEY:
				// At beginning of line?
				if (nCurrChar > 0)
				{
					nCurrChar--;
					CursorCol--;
					DisplayChar (' ', true);
					CursorCol--;
					UpdateCaretPos ();
				}
				break;

			case TAB_KEY:
				char acExt[16];
				int nExtSize;

				// Terminate buffer
				Buf[nCurrChar] = '\0';

				// Completion successful?
				completion ((unsigned char *)Buf, (unsigned char *)acExt);
				if ((nExtSize = strlen (acExt)) > 0)
				{
					// Copy chars into buffer and display them
					for (int i=0; i < nExtSize && nCurrChar <= Max; i++)
					{
						Buf[nCurrChar++] = acExt[i];
						DisplayChar (acExt[i], true);
					}
				}
				break;

			case ESC_KEY:
				// Go until beginning of line
				while (nCurrChar > 0)
				{
					nCurrChar--;
					CursorCol--;
					DisplayChar (' ', true);
					CursorCol--;
				}
				UpdateCaretPos ();
				break;

			case UP_ARROW_KEY:
			case DOWN_ARROW_KEY:
				int nHistorySize;

				// Go until beginning of line
				while (nCurrChar > 0)
				{
					nCurrChar--;
					CursorCol--;
					DisplayChar (' ', true);
					CursorCol--;
				}
				UpdateCaretPos ();

				// Up arrow?
				if (ch == UP_ARROW_KEY) 
				{
					nCurrHistoryPos--;
					if (nCurrHistoryPos < 0) 
						nCurrHistoryPos = MAX_HISTORY - 1;
				}
				else 
				{
					nCurrHistoryPos++;
					if (nCurrHistoryPos >= MAX_HISTORY) 
						nCurrHistoryPos = 0;
				}

				// Got anything?
				if ((nHistorySize = strlen (acHistory[nCurrHistoryPos])) > 0)
				{
					// Copy chars into buffer and display them
					for (int i=0; i < nHistorySize && nCurrChar <= Max; i++)
					{
						Buf[nCurrChar++] = acHistory[nCurrHistoryPos][i];
						DisplayChar (acHistory[nCurrHistoryPos][i], true);
					}

					nNextHistoryPos = nCurrHistoryPos;
					UpdateCaretPos ();
				}
				break;

			case SPECIAL_KEY_HOME:
			case SPECIAL_KEY_END:
			case SPECIAL_KEY_DELETE:
			case SPECIAL_KEY_INSERT:
			case SPECIAL_KEY_PAGE_UP:
			case SPECIAL_KEY_PAGE_DOWN:
			case LEFT_ARROW_KEY:
			case RIGHT_ARROW_KEY:
				break;

			// Any other key...
			default:
				Buf[nCurrChar++] = (unsigned char) ch;
				DisplayChar (ch, true);
				UpdateCaretPos ();
				break;
			}
		}
		else break;
	
	}
	while (nCurrChar <= Max);

	// Null-terminate buffer
	Buf[nCurrChar] = '\0';

	// Anything entered?
	if (strlen (Buf) > 0 && Buf[0] != '\n')
	{
		// Copy string to history buffer
		strcpy (acHistory[nNextHistoryPos], Buf);
		nNextHistoryPos++;
		nCurrHistoryPos = nNextHistoryPos;
		if (nNextHistoryPos >= MAX_HISTORY) nNextHistoryPos = 0;
	}

	ExpectingInput = FALSE;
	State = STATE_NONE;
	return ch;
}



//****************************************************************************
//		EraseScreenArea
//****************************************************************************
void FrostzApp::EraseScreenArea (int Top, int Left, int Bottom, int Right)
{
	::EnterCriticalSection( &ScrnCriticalSection );

	// Make sure extents are in range
	if (Bottom > Height - 1) Bottom = Height - 1;
	if (Right > Width - 1) Right = Width - 1;

	// Fill bitmap rect with bg color
	if (hbmpScreen)
	{
		RECT rc;
		rc.left = (Left /*- 1*/) * FontWidth;
		rc.top = (Top /*- 1*/) * FontHeight;// - 1;
		rc.right = (Right + 1) * FontWidth;
		rc.bottom = (Bottom + 1) * FontHeight - 1;

		HDC hdcMem = CreateCompatibleDC (NULL);
		HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hbmpScreen);
		
		FillRect (hdcMem, &rc, CreateSolidBrush (RGB(255,255,255)));

		SelectObject (hdcMem, hOldBmp);
		DeleteObject (hOldBmp);
		DeleteDC (hdcMem);
	}

	// Fill screen buffer with spaces
	for (int Row = Top; Row <= Bottom; Row++)
	{
		for (int Col = Left; Col <= Right; Col++)
		{
			pScreen[(Row * Width) + Col] = _T(' ');
			pAttributes[(Row * Width) + Col] = 
				(CurrentColour << NUM_STYLE_BITS) | CurrentStyle;

			/*** Experimental ***/
			pFonts[(Row * Width) + Col] = CurrentFont;
			/********************/
		}
	}

	::LeaveCriticalSection( &ScrnCriticalSection );

	// Refresh instantly?
	if (InstantRefresh)
	{
		// Get device context
		HDC hdc = GetDC (hWnd);

		::EnterCriticalSection( &ScrnCriticalSection );

		// Update scrolled area
		UpdateScreenArea (hdc, Top, Left, Bottom, Right);

		::LeaveCriticalSection( &ScrnCriticalSection );

		// Release device context
		ReleaseDC (hWnd, hdc);
	}
	else
	{
		// Flag that screen must be redrawn
		Redraw = true;
	}
}



//****************************************************************************
//		UpdateScreenArea
//****************************************************************************
void FrostzApp::UpdateScreenArea (HDC hdc, int Top, int Left, int Bottom, int Right)
{
	HFONT hOldFont;

	// Hide cursor
	SendMessage (hWnd, WM_SHOW_CURSOR, FALSE, 0);

	// Set colour
	SetColour (hdc, CurrentColour);

	// Select font into device context
	hOldFont = SetStyle (hdc, CurrentStyle);

	// Go through each row in the screen buffer
	for (int Row = Top, YPos = Top * CharHeight; 
		Row <= Bottom; 
		Row++, YPos += CharHeight) 
	{
		for (int Col = Left, XPos = Left * CharWidth, Chars = 0;;)
		{
			// Count no. of chars with current colour and style
			for (Chars = 0; (Col + Chars) <= Right
				&& COLOUR(pAttributes[(Row * Width) + (Col + Chars)]) == CurrentColour
				&& STYLE(pAttributes[(Row * Width) + (Col + Chars)]) == CurrentStyle
				/*** Experimental ***/
				&& pFonts[(Row * Width) + (Col + Chars)] == CurrentFont
				/********************/
				;
				Chars++);

			// Any chars to output with this style?
			if (Chars)
			{
				RECT Rect;
				Rect.left = Col * CharWidth;
				Rect.top = Row * CharHeight;
				Rect.right = (Col + 1) * CharWidth;
				Rect.bottom = (Row + 1) * CharHeight;

				// Output the char to the device context
				ExtTextOut (hdc, Col * CharWidth, Row * CharHeight, ETO_OPAQUE,
					&Rect, (LPCTSTR)&pScreen[(Row * Width) + Col],
					Chars, NULL );
				//DrawText (hdc, (LPCTSTR)&pScreen[(Row * Width) + Col], Chars, &Rect, DT_NOCLIP);

				Col += Chars;
				XPos += (Chars * CharWidth);
			}

			// Not reached end of line?
			if (Col <= Right)
			{
				// Colour changed?
				if (COLOUR(pAttributes[(Row * Width) + Col]) != CurrentColour)
				{
					// Set current colour
					SetColour (hdc, COLOUR(pAttributes[(Row * Width) + Col]));
				}
				// Style changed?
				if (STYLE(pAttributes[(Row * Width) + Col]) != CurrentStyle)
				{
					// Select previous font into device context
					SelectObject (hdc, hOldFont);

					// Set current style
					hOldFont = SetStyle (hdc, (BYTE)STYLE(pAttributes[(Row * Width) + Col]));
				}
				/*** Font changed? ***/
				else
				if (pFonts[(Row * Width) + Col] != CurrentFont)
				{
					// Select previous font into device context
					SelectObject (hdc, hOldFont);
					hOldFont = SetFont (hdc, pFonts[(Row * Width) + Col]);
					//MessageBeep (MB_OK);
				}
				/*********************/
			}
			else break;
		}
	}

	// Select original font back into device context
	SelectObject (hdc, hOldFont);

	// Show cursor
	SendMessage (hWnd, WM_SHOW_CURSOR, TRUE, 0);
}



//****************************************************************************
//		ScrollScreenArea
//****************************************************************************
void FrostzApp::ScrollScreenArea (int Top, int Left, int Bottom, int Right,
								  int Units)
{
	int Row;

	::EnterCriticalSection (&ScrnCriticalSection);

	// Scroll each line upwards
	for (Row = Top + Units; Row <= Bottom; Row++)
	{
		memcpy (&pScreen[((Row - Units) * Width) + Left],
			&pScreen[(Row * Width) + Left], 
			((Right - Left) + 1) * sizeof (TCHAR));

		memcpy (&pAttributes[((Row - Units) * Width) + Left],
			&pAttributes[(Row * Width) + Left], 
			((Right - Left) + 1) );

		/*** Experimental ***/
		memcpy (&pFonts[((Row - Units) * Width) + Left],
			&pFonts[(Row * Width) + Left],
			((Right - Left) + 1));
		/********************/
	}


	/*** Scroll pictures too for Shogun ***/
	if ((story_id == SHOGUN || story_id == ZORK_ZERO) && hbmpScreen)
	{
		/* Convert character units (rows/cols) to pixels
		int pxUnits = Units * FontHeight;
		int pxTop = (Top - 1) * FontHeight;
		int pxLeft = (Left - 1) * FontWidth;
		int pxBottom = (Bottom - 1) * FontHeight;
		int pxRight = (Right - 1) * FontHeight;
		*/
		RECT scRect;
		scRect.left = Left * FontWidth;
		scRect.top = (Top * FontHeight) - (Units * FontHeight);
		scRect.right = (Right + 1) * FontWidth;
		scRect.bottom = (Bottom + 1) * FontHeight;
		int pxUnits = - (Units * FontHeight);
		if (story_id == SHOGUN)
			--pxUnits;

		HDC hdcMem = CreateCompatibleDC (NULL);
		HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hbmpScreen);

		ScrollDC (hdcMem,
			0, pxUnits,
			&scRect,
			&scRect,
			NULL, NULL);

		scRect.top = scRect.bottom - (Units * FontHeight) -  2;
		FillRect (hdcMem, &scRect, (HBRUSH)CreateSolidBrush (RGB(0xff,0xff,0xff)));

		SelectObject (hdcMem, hOldBmp);
		DeleteObject (hOldBmp);
		DeleteDC (hdcMem);

		//InvalidateRect (hWnd, &scRect, TRUE);
	}
	/***************************************/



	// Fill rows below scrolled area with spaces
	for (Row = Bottom; Row > Bottom - Units; Row--)
	{
		for (int Col = Left; Col <= Right; Col++)
		{
			pScreen[(Row * Width) + Col] = _T(' ');
			pAttributes[(Row * Width) + Col] = 
				(BYTE) ((CurrentColour << NUM_STYLE_BITS) | CurrentStyle);
			/*** Experimental ***/
			pFonts[(Row * Width) + Col] = CurrentFont;
			/********************/
		}
	}

	::LeaveCriticalSection (&ScrnCriticalSection);

	// Refresh instantly?
	if (InstantRefresh)
	{
		// Get device context
		HDC hdc = GetDC (hWnd);

		::EnterCriticalSection (&ScrnCriticalSection);

		// Update scrolled area
		UpdateScreenArea (hdc, Top, Left, Bottom, Right);

		::LeaveCriticalSection (&ScrnCriticalSection);

		// Release device context
		ReleaseDC (hWnd, hdc);
	}	
	else
	{
		// Flag that screen must be redrawn
		Redraw = true;
	}
}



//****************************************************************************
//		GetCursorPos
//****************************************************************************
void FrostzApp::SetCursorPos (int Row, int Col, bool Update)
{
		// Make sure row and column are in range
	if (Row < 0) Row = 0;
	else if (Row > Height-1) Row = Height - 1;
	if (Col < 0) Col = 0;
	else if (Col > Width-1) Col = Width - 1;

	// Save cursor position
	CursorRow = Row;
	CursorCol = Col;

	// Update caret position
	if (Update) UpdateCaretPos ();
}



//****************************************************************************
//		GetCursorPos
//****************************************************************************
void FrostzApp::GetCursorPos (int* Row, int* Col)
{
	// Save cursor position
	*Row = CursorRow;
	*Col = CursorCol;
}



//****************************************************************************
//		PeekColor
//	Return the colour of the screen unit below the cursor. (If the
//	interface uses a text mode, it may return the background colour
//	of the character at the cursor position instead.) This is used
//	when text is printed on top of pictures. Note that this coulor
//	need not be in the standard set of Z-machine colours. To handle
//	this situation, Frotz extends the colour scheme: Colours above
//	15 (and below 256) may be used by the interface to refer to non
//	standard colours. Of course, os_set_colour must be able to deal
//	with these colours.
//****************************************************************************
int FrostzApp::PeekColor ()
{
	HDC hdcMem = CreateCompatibleDC (NULL);
	HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hbmpScreen);

	Palette[PALETTE_USER_COLOR] = GetPixel (hdcMem, CursorCol, CursorRow);

	SelectObject (hdcMem, hOldBmp);
	DeleteObject (hOldBmp);
	DeleteDC (hdcMem);

	return USER_COLOUR;
}



//****************************************************************************
//		DisplayChar
//****************************************************************************
void FrostzApp::DisplayChar (char Ch, bool UpdateCaret)
{
	// Check for special characters
	switch (Ch)
	{
	// Paragraph indent?
	case 11:
		DisplayChar (' ');
		break;

	// Tab?
	case 9:
		DisplayChar(' ');
		Ch = ' ';
		break;

	// Everything else
	default:
		int nCurrPos;
		
		// Save cursor pos.
		nCurrPos = CursorCol;

		// Update cursor position
		CursorCol++;
		if (UpdateCaret) UpdateCaretPos ();

		// On screen?
		if (nCurrPos < Width && CursorRow < Height)
		{
			::EnterCriticalSection (&ScrnCriticalSection);
	

			// Put the char into the screen buffer
			pScreen[(CursorRow * Width) + nCurrPos] = (TCHAR) (unsigned char)Ch;
		
			pAttributes[(CursorRow * Width) + nCurrPos] = 
				(BYTE) ((CurrentColour << NUM_STYLE_BITS) | CurrentStyle);

			/*** Experimental ***/
			pFonts[(CursorRow * Width) + nCurrPos] = CurrentFont;
			/********************/


			// Refresh instantly?
			if (InstantRefresh || ExpectingInput)
			{
				HFONT hOldFont;
				RECT Rect = {
					nCurrPos * CharWidth,
					CursorRow * CharHeight,
					(nCurrPos + 1) * CharWidth,
					(CursorRow + 1) * CharHeight
				};

				// Get device context
				HDC hdc = GetDC (hWnd);

				// Set colour
				SetColour (hdc, CurrentColour);

				// Select font into device context
				hOldFont = SetStyle (hdc, CurrentStyle);

				// Output the char to the device context
				ExtTextOut (hdc, nCurrPos * CharWidth, CursorRow * CharHeight, ETO_OPAQUE, 
					&Rect, 
					(LPCTSTR) &pScreen[(CursorRow * Width) + nCurrPos], 
					1, NULL );
				//DrawText (hdc, (LPCTSTR)&pScreen[(CursorRow * Width) + nCurrPos], 1, &Rect, DT_NOCLIP);

				// Select previous font into device context
				SelectObject (hdc, hOldFont);

				// Release device context
				ReleaseDC (hWnd, hdc);
			}
			else
			{
				// Flag that screen must be redrawn
				Redraw = true;
			}

			::LeaveCriticalSection (&ScrnCriticalSection);

		}
	}
}



//****************************************************************************
//		DisplayString
//	Outputs a string to the current cursor coordinates
//****************************************************************************
#define OnScreen	(CursorCol < Width && CursorRow < Height)
#define PutChar(c)	pScreen[(CursorRow * Width) + CursorCol] = _T(c); \
					pAttributes[(CursorRow * Width) + CursorCol] =  \
						(BYTE) ((nCurrentColour << NUM_STYLE_BITS) | nCurrentStyle); \
					/*** Experimental ***/ \
					pFonts[(CursorRow * Width) + CursorCol] = nCurrentFont; \
					/********************/ \
					CursorCol++;

void FrostzApp::DisplayString (const char* Str)
{
	const unsigned char *pcCurr = (unsigned char *) Str;
	int nStartCol = CursorCol;
	int nCurrentStyle;
	int nCurrentColour;
	int nCurrentFont;

	::EnterCriticalSection (&ScrnCriticalSection);

	// Save current font, style and colours
	nCurrentFont = CurrentFont;
	nCurrentStyle = CurrentStyle;
	nCurrentColour = CurrentColour;

	::LeaveCriticalSection(&ScrnCriticalSection);

	// Process string
	for (; *pcCurr; pcCurr++)
	{
		// New style?
		if (*pcCurr == ZC_NEW_FONT || *pcCurr == ZC_NEW_STYLE) 
		{
			// Style change?
			if (*pcCurr == ZC_NEW_STYLE)
			{
				pcCurr++;

				// Set current style
				nCurrentStyle = *pcCurr;

				// Not using text styles?
				if (!UseStyles)
				{
					// Remove style flags
					nCurrentStyle &= ~BOLDFACE_STYLE;
					nCurrentStyle &= ~EMPHASIS_STYLE;
				}

				// Use default style
				nCurrentStyle |= DefaultStyle;
			}
			else if (*pcCurr == ZC_NEW_FONT)
			{
				pcCurr++;

				// Set current font
				nCurrentFont = *pcCurr;
			}
			else
			{
				pcCurr++;
			}
		}
		else 
		{
			/*** DEBUG ***
			if (*pcCurr > 25 && *pcCurr < 32)
			{
				unsigned short wMsg[256];
				wsprintf (wMsg, L"Char 0x%x (%d)", *pcCurr, *pcCurr);
				AfxMessageBox (wMsg);
			}

			else
			/*************/
			// Display char
			{
			unsigned char ch = (unsigned char) *pcCurr;

				// Check for special characters
				switch( *pcCurr )
				{
				// Paragraph indent?
				case 11:
					// On screen?
					if (OnScreen)
					{
						// Put the char into the screen buffer
						PutChar(' ');
					}
					break;
				// Tab?
				case 9:
					// On screen?
					if (OnScreen)
					{
						// Put the char into the screen buffer
						PutChar(' ');
					}
					ch = (unsigned char)' ';
					break;
				// Everything else
				default:
					/*** DEBUG ***
					if ((unsigned char)ch < 24 ||
						((unsigned char)ch > 127 && (unsigned char)ch < 179) ||
						(unsigned char)ch >223)
					{
						unsigned short wMsg[32];
						wsprintf (wMsg, L"'%c' (0x%x)", (TCHAR)(unsigned char)ch, (unsigned char)ch);
						MessageBox (hWnd, wMsg, L"DEBUG", MB_OK);
						GoFullScreen ();
					}
					/*************/

					// On screen?
					if (OnScreen)
					{
						// Put the char into the screen buffer
						pScreen[(CursorRow * Width) + CursorCol] = (TCHAR)ch;
						pAttributes[(CursorRow * Width) + CursorCol] =
							(BYTE) ((nCurrentColour << NUM_STYLE_BITS) | nCurrentStyle);
						/*** Experimental ***/
						pFonts[(CursorRow * Width) + CursorCol] = nCurrentFont;
						/********************/
						CursorCol++;
					}
					break;
				}
			}
		}
    }


	// Refresh instantly?
	if (InstantRefresh)
	{
		// Get device context
		HDC hdc = GetDC (hWnd);

		::EnterCriticalSection (&ScrnCriticalSection);

		// Anything to display?
		if (CursorCol > nStartCol)
		{
			// Update display
			UpdateScreenArea (hdc, CursorRow, nStartCol, CursorRow, CursorCol - 1);
		}

		// Set font?
		if (nCurrentFont != CurrentFont)
		{
			HFONT hOldFont;

			//MessageBeep (MB_OK); //*** DEBUG
			hOldFont = SetFont (hdc, nCurrentFont);

			// Select previous font into device context
			SelectObject (hdc, hOldFont);
		}

		// Set style?
		if (nCurrentStyle != CurrentStyle) 
		{
			HFONT hOldFont;

			hOldFont = SetStyle (hdc, nCurrentStyle);

			// Select previous font into device context
			SelectObject (hdc, hOldFont);
		}


		::LeaveCriticalSection (&ScrnCriticalSection);

		// Release device context
		ReleaseDC (hWnd, hdc);
	}
	else
	{
		::EnterCriticalSection (&ScrnCriticalSection);

		// Set font
		if (nCurrentFont != CurrentFont)
			CurrentFont = nCurrentFont;

		// Set style
		if (nCurrentStyle != CurrentStyle) 
			CurrentStyle = nCurrentStyle;

		::LeaveCriticalSection (&ScrnCriticalSection);

		// Flag that screen must be redrawn
		Redraw = true;
	}
}



//****************************************************************************
//		StuffKeyboard
//	Sends a string to the Z-Machine like if it comes from the keyboard
//****************************************************************************
void FrostzApp::StuffKeyboard (const char *pChars, bool bReturn, bool bEsc)
{
	::EnterCriticalSection (&KbdCriticalSection);

	// Clear input first?
	if (bEsc)
	{
		// Add ESC key to buffer
		*WriteKeyBuffer = ESC_KEY;
		WriteKeyBuffer++;

		// Gone past end of buffer?
		if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
		{
			// Go back to beginning of buffer
			WriteKeyBuffer = &KeyBuffer[0];
		}
	}

	// Stuff chars into buffer
	while (*pChars)
	{
		// Add char to buffer
		*WriteKeyBuffer = (int) *pChars;
		pChars++;
		WriteKeyBuffer++;

		// Gone past end of buffer?
		if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
		{
			// Go back to beginning of buffer
			WriteKeyBuffer = &KeyBuffer[0];
		}
	}

	// Want a return?
	if (bReturn)
	{
		// Add return key to buffer
		*WriteKeyBuffer = ENTER_KEY;
		WriteKeyBuffer++;

		// Gone past end of buffer?
		if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
		{
			// Go back to beginning of buffer
			WriteKeyBuffer = &KeyBuffer[0];
		}
	}

	::LeaveCriticalSection (&KbdCriticalSection);

	// Signal that a keypress event has occurred
	::SetEvent (hKeyEvent);
}



//****************************************************************************
//		CycleCBMPalette
//****************************************************************************
#define CBMBLACK		RGB(0,0,0)
#define CBMWHITE		RGB(0xff,0xff,0xff)
#define CBMRED			RGB(0xe0,0x40,0x40)
#define CBMTURQUOISE	RGB(0x60,0xff,0xff)
#define CBMPURPLE		RGB(0xe0,0x60,0xe0)
#define CBMGREEN		RGB(0x40,0xe0,0x40)
#define CBMBLUE			RGB(0x40,0x40,0xe0)
#define CBMYELLOW		RGB(0xff,0xff,0x40)
#define CBMORANGE		RGB(0xe0,0xa0,0x40)
#define CBMBROWN		RGB(0x9c,0x74,0x48)
#define CBMLIGHTRED		RGB(0xff,0xff,0xa0)
#define CBMLIGHTGREY	RGB(0x54,0x54,0x54)
#define CBMGREY			RGB(0x88,0x88,0x88)
#define CBMLIGHTGREEN	RGB(0xa0,0xff,0xa0)
#define CBMLIGHTBLUE	RGB(0xa0,0xa0,0xff)
#define CBMDARKGREY		RGB(0xc0,0xc0,0xc0)
void FrostzApp::CycleCBMPalette ()
{
	if (++CBMPaletteCycle > 5)
		CBMPaletteCycle = 0;

	switch (CBMPaletteCycle)
	{
		case 1:	// Light blue on Dark blue (C=64) / Green on Grey (C=128)
			// (Default while simulating CBM colors)
			Palette[PALETTE_BLACK]	= (InterpreterNumber == INTERP_CBM_64)? CBMLIGHTBLUE : CBMLIGHTGREEN;
			Palette[PALETTE_WHITE]	= (InterpreterNumber == INTERP_CBM_64)? CBMBLUE : CBMGREY;
			Palette[PALETTE_RED]	= CBMRED;
			Palette[PALETTE_GREEN]	= (InterpreterNumber == INTERP_CBM_64)? CBMGREEN : CBMBLACK;
			Palette[PALETTE_YELLOW]	= CBMYELLOW;
			Palette[PALETTE_BLUE]	= (InterpreterNumber == INTERP_CBM_64)? CBMWHITE : CBMBLUE;
			Palette[PALETTE_MAGENTA]= CBMPURPLE;
			Palette[PALETTE_CYAN]	= (InterpreterNumber == INTERP_CBM_64)? CBMBLACK : CBMLIGHTBLUE;
			Palette[PALETTE_GREY]	= (InterpreterNumber == INTERP_CBM_64)? CBMGREY : CBMWHITE;
			break;

		case 2:	// White on Black
			Palette[PALETTE_BLACK]	= CBMWHITE;
			Palette[PALETTE_WHITE]	= CBMBLACK;
			Palette[PALETTE_RED]	= CBMRED;
			Palette[PALETTE_GREEN]	= CBMGREEN;
			Palette[PALETTE_YELLOW]	= CBMYELLOW;
			Palette[PALETTE_BLUE]	= CBMBLUE;
			Palette[PALETTE_MAGENTA]= CBMPURPLE;
			Palette[PALETTE_CYAN]	= CBMLIGHTBLUE;
			Palette[PALETTE_GREY]	= CBMGREY;
			break;

		case 3:	// White on Blue (C=64) / Cyan on Black (C=128)
			Palette[PALETTE_BLACK]	= (InterpreterNumber == INTERP_CBM_64)? CBMWHITE : CBMTURQUOISE;
			Palette[PALETTE_WHITE]	= (InterpreterNumber == INTERP_CBM_64)? CBMBLUE : CBMBLACK;
			Palette[PALETTE_RED]	= CBMRED;
			Palette[PALETTE_GREEN]	= CBMGREEN;
			Palette[PALETTE_YELLOW]	= CBMYELLOW;
			Palette[PALETTE_BLUE]	= (InterpreterNumber == INTERP_CBM_64)? CBMLIGHTBLUE : CBMBLUE;
			Palette[PALETTE_MAGENTA]= CBMPURPLE;
			Palette[PALETTE_CYAN]	= (InterpreterNumber == INTERP_CBM_64)? CBMTURQUOISE : CBMLIGHTBLUE;
			Palette[PALETTE_GREY]	= CBMDARKGREY;
			break;

		case 4:	// White on Grey
			Palette[PALETTE_BLACK]	= CBMWHITE;
			Palette[PALETTE_WHITE]	= CBMGREY;
			Palette[PALETTE_RED]	= CBMRED;
			Palette[PALETTE_GREEN]	= CBMGREEN;
			Palette[PALETTE_YELLOW]	= CBMYELLOW;
			Palette[PALETTE_BLUE]	= CBMBLUE;
			Palette[PALETTE_MAGENTA]= CBMPURPLE;
			Palette[PALETTE_CYAN]	= CBMLIGHTBLUE;
			Palette[PALETTE_GREY]	= CBMDARKGREY;
			break;

		case 5: // Yellow on Black - light colors
			Palette[PALETTE_BLACK]	= CBMYELLOW;
			Palette[PALETTE_WHITE]	= CBMBLACK;
			Palette[PALETTE_RED]	= CBMLIGHTRED;
			Palette[PALETTE_GREEN]	= CBMLIGHTGREEN;
			Palette[PALETTE_YELLOW]	= CBMDARKGREY;
			Palette[PALETTE_BLUE]	= CBMLIGHTBLUE;
			Palette[PALETTE_MAGENTA]= CBMORANGE;
			Palette[PALETTE_CYAN]	= CBMTURQUOISE;
			Palette[PALETTE_GREY]	= CBMLIGHTGREY;
			break;

		case 0:	// Black on White (default while NOT simulating CBM colors)
		default:
			Palette[PALETTE_BLACK]	= CBMBLACK;
			Palette[PALETTE_WHITE]	= CBMWHITE;
			Palette[PALETTE_RED]	= CBMRED;
			Palette[PALETTE_GREEN]	= CBMGREEN;
			Palette[PALETTE_YELLOW]	= CBMYELLOW;
			Palette[PALETTE_BLUE]	= CBMBLUE;
			Palette[PALETTE_MAGENTA]= CBMPURPLE;
			Palette[PALETTE_CYAN]	= CBMLIGHTBLUE;
			Palette[PALETTE_GREY]	= CBMGREY;
			break;

	}

	Redraw = true;
}



//****************************************************************************
//		SetTextFont
//****************************************************************************
void FrostzApp::SetTextFont (int nFont)
{
	// Font change?
	if (nFont != CurrentFont)
	{
		HFONT hOldFont;

		::EnterCriticalSection (&ScrnCriticalSection);

		HDC hdc = GetDC (hWnd);

		// Set current style
		hOldFont = SetFont (hdc, nFont);

		//*** Tell the window we're setting a new font
		//SendMessage (hWnd, WM_SETFONT, (WPARAM)hOldFont, MAKELPARAM(TRUE, 0));

		// Select previous font into device context
		SelectObject (hdc, hOldFont);

		// Release device context
		ReleaseDC (hWnd, hdc);

		::LeaveCriticalSection (&ScrnCriticalSection);
	}
}



//****************************************************************************
//		SetTextStyle
//****************************************************************************
void FrostzApp::SetTextStyle (int Style)
{
	// Style change?
	if (Style != CurrentStyle)
	{
		HFONT hOldFont;

		::EnterCriticalSection (&ScrnCriticalSection);

		HDC hdc = GetDC (hWnd);

		// Set current style
		hOldFont = SetStyle (hdc, Style);

		//*** Tell the window we're setting a new font
		//SendMessage (hWnd, WM_SETFONT, (WPARAM)hOldFont, MAKELPARAM(TRUE, 0));

		// Select previous font into device context
		SelectObject (hdc, hOldFont);

		// Release device context
		ReleaseDC (hWnd, hdc);

		::LeaveCriticalSection (&ScrnCriticalSection);
	}
}



//****************************************************************************
//		ClearMRUList
//****************************************************************************
void FrostzApp::ClearMRUList ()
{
	for (BYTE Cnt = 0; Cnt < MAX_MRU; ++Cnt)
	{
		MRU[Cnt].wFileName[0] = (TCHAR)0;
		MRU[Cnt].wShortName = 0;
	}

	UpdateMenuItems ();
}



//****************************************************************************
//		UpdateMacros
//	Re-creates the macro menu items
//****************************************************************************
void FrostzApp::UpdateMacros ()
{
	unsigned int Cnt;
	HMENU hMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd),
					SHCMBM_GETSUBMENU, (WPARAM)0, (LPARAM)ID_GAMEMENU_MACROS);

	if (!hMenu)
		return;

	// First, remove all items from this menu
	for (Cnt = 0; Cnt < MAX_MACROS; ++Cnt)
		DeleteMenu (hMenu, BASE_MACRO_ID + Cnt, MF_BYCOMMAND);

	// Now, add an element for each macro
	for (Cnt = 0; Cnt < MAX_MACROS; ++Cnt)
		if (Macros[Cnt].isDefined)
			InsertMenu (hMenu, ID_GAMEMENU_EDITMACROS, MF_BYCOMMAND,
				BASE_MACRO_ID + Cnt, Macros[Cnt].wName);

	DrawMenuBar (hWnd);
}




//****************************************************************************
//		UpdateMenuItems
//	Checks/Unchecks the menu items
//****************************************************************************
void FrostzApp::UpdateMenuItems ()
{
	HMENU hMenu;
	DWORD Val;

	if (isZmachineRunning)
	{
		// GAME MENU BAR
		hMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd),
				SHCMBM_GETSUBMENU, (WPARAM)0, (LPARAM)ID_GAMEMENU_FILE);

		Val = (ShowCompass)? MF_CHECKED : MF_UNCHECKED;
			CheckMenuItem (hMenu, ID_GAMEMENU_COMPASS, MF_BYCOMMAND | Val);

		// Enable/Disable CBM Cycle Palette
		Val = (SimulateCBM &&
			(InterpreterNumber == INTERP_CBM_64 || InterpreterNumber == INTERP_CBM_128))?
			TRUE : FALSE;
		SendMessage (SHFindMenuBar (hWnd), TB_ENABLEBUTTON,
			(WPARAM)ID_GAMEMENU_CBMPAL, (LPARAM)MAKELONG(Val, 0));

		// Macros
		UpdateMacros ();
	}
	else
	{
		// MAIN MENU BAR

		hMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd),
				SHCMBM_GETSUBMENU, (WPARAM)0, (LPARAM)ID_MENU_OPTIONS);

		// Ignore Run-Time Errors
		Val = (ignore_errors != 0)? MF_CHECKED : MF_UNCHECKED;
			CheckMenuItem (hMenu, ID_IGNORERTERRORS, MF_BYCOMMAND | Val);

		// Expand Abbreviations
		Val = (expand_abbreviations != 0)? MF_CHECKED : MF_UNCHECKED;
			CheckMenuItem (hMenu, ID_MENU_EXPAND, MF_BYCOMMAND | Val);

		// Set Tandy Bit
		Val = TandyBit? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_TANDY, MF_BYCOMMAND | Val);

		// Use Text Styles
		Val = UseStyles? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_TEXTSTYLES, MF_BYCOMMAND | Val);

		// Underlined instead of bold
		if (bUnderline)
		{
			CheckMenuItem (hMenu, ID_EMPH_BOLD, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_EMPH_UNDERL, MF_BYCOMMAND | MF_CHECKED);
		}
		else
		{
			CheckMenuItem (hMenu, ID_EMPH_BOLD, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_EMPH_UNDERL, MF_BYCOMMAND | MF_UNCHECKED);
		}

		// Use Colors
		Val = UseColours? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_USECOLORS, MF_BYCOMMAND | Val);

		// Use Sounds
		Val = UseSounds? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_SOUND, MF_BYCOMMAND | Val);

		// Blind Mode
		Val = cBlindMode? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_BLINDMODE, MF_BYCOMMAND | Val);

		// Timer updates
		if (UpdateTimerMs == 125)
		{
			CheckMenuItem (hMenu, ID_MENU_TIMER125, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER250, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER500, MF_BYCOMMAND | MF_UNCHECKED);
		}
		else if (UpdateTimerMs == 500)
		{
			CheckMenuItem (hMenu, ID_MENU_TIMER125, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER250, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER500, MF_BYCOMMAND | MF_CHECKED);
		}
		else
		{
			UpdateTimerMs = FROSTZ_UPDATE_MS;
			CheckMenuItem (hMenu, ID_MENU_TIMER125, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER250, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_MENU_TIMER500, MF_BYCOMMAND | MF_UNCHECKED);
		}

		// Font dimensions
		if (FontHeight == 10)
		{
			CheckMenuItem (hMenu, ID_MENU_6x10, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_MENU_8x12, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x11, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x8, MF_BYCOMMAND | MF_UNCHECKED);
		}
		else if (FontHeight == 12)
		{
			CheckMenuItem (hMenu, ID_MENU_6x10, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_6x10, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_8x12, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x11, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x8, MF_BYCOMMAND | MF_UNCHECKED);
		}
		else if (FontHeight == 11)
		{
			CheckMenuItem (hMenu, ID_MENU_6x10, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_8x12, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x11, MF_BYCOMMAND | MF_CHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x8, MF_BYCOMMAND | MF_UNCHECKED);
		}
		else
		{
			CheckMenuItem (hMenu, ID_MENU_6x10, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_8x12, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x11, MF_BYCOMMAND | MF_UNCHECKED);
			CheckMenuItem (hMenu, ID_MENU_4x8, MF_BYCOMMAND | MF_CHECKED);
		}

		// Interpreter
		CheckMenuItem (hMenu, ID_MENU_INT_DEC20, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_APPLE2E, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_MAC, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_AMIGA, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_ATARIST, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_MSDOS, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_C128, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_C64, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_APPLE2C, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_APPLE2GS, MF_BYCOMMAND | MF_UNCHECKED);
		CheckMenuItem (hMenu, ID_MENU_INT_TANDY, MF_BYCOMMAND | MF_UNCHECKED);
		switch (InterpreterNumber)
		{
			case INTERP_DEC_20:
				CheckMenuItem (hMenu, ID_MENU_INT_DEC20, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_APPLE_IIE:
				CheckMenuItem (hMenu, ID_MENU_INT_APPLE2E, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_MACINTOSH:
				CheckMenuItem (hMenu, ID_MENU_INT_MAC, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_AMIGA:
				CheckMenuItem (hMenu, ID_MENU_INT_AMIGA, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_ATARI_ST:
				CheckMenuItem (hMenu, ID_MENU_INT_ATARIST, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_CBM_128:
				CheckMenuItem (hMenu, ID_MENU_INT_C128, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_CBM_64:
				CheckMenuItem (hMenu, ID_MENU_INT_C64, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_APPLE_IIC:
				CheckMenuItem (hMenu, ID_MENU_INT_APPLE2C, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_APPLE_IIGS:
				CheckMenuItem (hMenu, ID_MENU_INT_APPLE2GS, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_TANDY:
				CheckMenuItem (hMenu, ID_MENU_INT_TANDY, MF_BYCOMMAND | MF_CHECKED);
				break;

			case INTERP_MSDOS:
			default:
				InterpreterNumber = INTERP_MSDOS;
				CheckMenuItem (hMenu, ID_MENU_INT_MSDOS, MF_BYCOMMAND | MF_CHECKED);
				break;
		}

		// Simulate CBM Colors
		Val = SimulateCBM? MF_CHECKED : MF_UNCHECKED;
		CheckMenuItem (hMenu, ID_MENU_SIMCBM, MF_BYCOMMAND | Val);


		/*** MRU Files ***/
		{
			unsigned short MenuItemText[64];
			MENUITEMINFO mii;

			memset (MenuItemText, 0, sizeof (unsigned short) * 64);
			memset (&mii, 0, sizeof (MENUITEMINFO));

			mii.cbSize = sizeof (MENUITEMINFO);
			mii.fMask = MIIM_TYPE;
			mii.fType = MFT_STRING;
			mii.dwTypeData = MenuItemText;
			mii.cch = 64;

			// Add an item for each valid MRU file in the structure
			for (int Cnt = 0; Cnt < MAX_MRU; ++Cnt)
			{
				// First, see if an item already existed
				if (GetMenuItemInfo (hMenu, BASE_MRU_ID + Cnt, FALSE, &mii) != 0)
				{
					// If so, replace text
					mii.fMask = MIIM_TYPE;
					wcscpy (mii.dwTypeData, MRU[Cnt].wShortName);
					SetMenuItemInfo (hMenu, BASE_MRU_ID + Cnt, FALSE, &mii);
				}
				else
				{
					// Else, check if this MRU is a valid entry
					HMENU hSubMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd), SHCMBM_GETSUBMENU,
									(WPARAM)0, (LPARAM)ID_MENU_FILE);
					if (MRU[Cnt].wFileName[0])
					{
						// If so, add a voice to the menu
						DeleteMenu (hSubMenu, BASE_MRU_ID + Cnt, MF_BYCOMMAND);
						InsertMenu (hSubMenu, ID_MENU_CLRMRU, MF_BYCOMMAND, BASE_MRU_ID + Cnt,
							MRU[Cnt].wShortName);
					}
				}
			}
		}
		/*****************/


		/* FONT SUBMENU - experimental
		unsigned short MenuItemText[256];
		MENUITEMINFO mii;

		memset (MenuItemText, 0, sizeof (unsigned short) * 256);
		memset (&mii, 0, sizeof (MENUITEMINFO));

		mii.cbSize = sizeof (MENUITEMINFO);
		mii.fMask = MIIM_TYPE;
		mii.fType = MFT_STRING;
		mii.dwTypeData = MenuItemText;
		mii.cch = 256;

		for (unsigned int Cnt = 0; Cnt < numFonts; ++Cnt)
		{
			// Get menu item text
			GetMenuItemInfo (hMenu, 41000 + Cnt, FALSE, &mii);

			if (wcsncmp (FontFace, MenuItemText, wcslen (MenuItemText)) == 0)
				CheckMenuItem (hMenu, 41000 + Cnt, MF_BYCOMMAND | MF_CHECKED);
			else
				CheckMenuItem (hMenu, 41000 + Cnt, MF_BYCOMMAND | MF_UNCHECKED);
		}
		*/
	}
}



//****************************************************************************
//		GoFullScreen
//	Takes the focus and hides the taskbar
//****************************************************************************
bool FrostzApp::GoFullScreen ()
{
	if (SetForegroundWindow (hWnd) == 0)
		return false;

	/*if (SetWindowPos (hWnd, HWND_TOPMOST, 0, -1, 240,
		bShowSip? 215 : 303,
		SWP_SHOWWINDOW) == 0)
	/*/
	if (MoveWindow (hWnd,
		ClientRect.left, ClientRect.top,
		ClientRect.right, ClientRect.bottom,
		FALSE) == 0)
		return false;

	if (SHFullScreen (hWnd, SHFS_HIDETASKBAR) == FALSE)
		return false;

	return true;
}



//****************************************************************************
//		OnTimer
//	Handles the WM_TIMER message
//****************************************************************************
void FrostzApp::OnTimer ()
{
	// Need to do a refresh?
	if (Redraw)
	{
		//RECT Rect = {0, 0, 240, 215};

		InvalidateRect (hWnd, &ClientRect, (UseStyles)? TRUE : FALSE);
		/* Using text styles?
		if (UseStyles)
		{
			// Invalidate and clear client area
 			InvalidateRect (hWnd, &ClientRect, TRUE);
		}
		else
		{
			// Invalidate client area
 			InvalidateRect (hWnd, &ClientRect, FALSE);
		}
		*/

		// Flag that refresh has been done
		Redraw = false;
	}
}



//****************************************************************************
//		OnSetFocus
//	Handles the WM_SETFOCUS message
//****************************************************************************
void FrostzApp::OnSetFocus ()
{
	if (isZmachineRunning)
	{
		POINT sCaretPos;

		// Calculate new caret position
		sCaretPos.x = CursorCol * CharWidth;
		sCaretPos.y = CursorRow * CharHeight;

		CreateCaret (hWnd, NULL, CharWidth, CharHeight );
		SetCaretPos (sCaretPos.x, sCaretPos.y);
		ShowCaret (hWnd);
	}
}



//****************************************************************************
//		OnPaint
//	Handles the WM_PAINT message
//****************************************************************************
void FrostzApp::OnPaint ()
{
	PAINTSTRUCT ps;

	HDC hdc = BeginPaint (hWnd, &ps);

	// Draw background picture if any
	if (bkBMP)
	{
		HBITMAP hBMP = LoadBitmap (hInstance, MAKEINTRESOURCE(bkBMP));

		HDC hdcMem = CreateCompatibleDC (NULL);
		HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hBMP);

		BitBlt (hdc,
			ClientRect.left,
			ClientRect.top + 2,
			ClientRect.right,
			ClientRect.bottom + 2, hdcMem, 0, 0, SRCCOPY);

		SelectObject (hdcMem, hOldBmp);
		DeleteObject (hBMP);
		DeleteDC (hdcMem);
	}

	// Screen for the game
	if (pScreen)
	{
		::EnterCriticalSection (&ScrnCriticalSection);

		BYTE nOldColour = CurrentColour;
		int nOldStyle = CurrentStyle;
		BYTE nOldFont = CurrentFont;

		// Set default colour
		CurrentColour = DefaultColors;//DEF_TEXT_COLOUR;

		// Set default font
		CurrentFont = TEXT_FONT;

		// Set default style
		CurrentStyle = NORMAL_STYLE;

		// Update the entire screen
		UpdateScreenArea (hdc, 0, 0, Height-1, Width-1 );

		// Restore previous style and colours
		if (nOldColour != CurrentColour) SetColour (hdc, nOldColour);
		if (nOldFont != CurrentFont)
		{
			HFONT hOldFont;

			hOldFont = SetFont (hdc, nOldFont);
			SelectObject (hdc, hOldFont);
		}
		if (nOldStyle != CurrentStyle) 
		{
			HFONT hOldFont;

			hOldFont = SetStyle (hdc, nOldStyle);
			SelectObject (hdc, hOldFont);
		}

		// Bitmap memory
		if (hbmpScreen)
		{
			HDC hdcMem = CreateCompatibleDC (NULL);
			HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hbmpScreen);
			BitBlt (hdc,
				ClientRect.left,
				ClientRect.top + 2,
				ClientRect.right,
				ClientRect.bottom + 2,
				hdcMem, 0, 0,
				SRCAND);
			SelectObject (hdcMem, hOldBmp);
			DeleteObject (hOldBmp);
			DeleteDC (hdcMem);
		}

		// Draw compass if needed
		if (ShowCompass && !cBlindMode)
		{
			unsigned short wMsg[256];
			HBITMAP hBMP = LoadBitmap (hInstance, MAKEINTRESOURCE(IDB_COMPASS));

			if (!hBMP)
			{
				wsprintf (wMsg, L"Error #%d", GetLastError ());
				MessageBox (hWnd, L"Could not load compass bmp", wMsg, MB_OK);
				ShowCompass = false;
			}

			HDC hdcMem = CreateCompatibleDC (NULL);
			HBITMAP hOldBmp = (HBITMAP)SelectObject (hdcMem, hBMP);

			if (!BitBlt (hdc, 180, 20, 49, 49, hdcMem, 0, 0, SRCAND))
			{
				wsprintf (wMsg, L"Error #%d", GetLastError ());
				MessageBox (hWnd, L"Could not blit compass", wMsg, MB_OK);
				ShowCompass = false;
			}


			SelectObject (hdcMem, hOldBmp);
			DeleteObject (hBMP);
			DeleteDC (hdcMem);
		}

		::LeaveCriticalSection (&ScrnCriticalSection);
	}

	EndPaint (hWnd, &ps);
}



//****************************************************************************
//		OnExit
//	Cleaning up
//****************************************************************************
void FrostzApp::OnExit ()
{
	SaveRegistry ();

	// Hide the SIP
	SipShowIM (SIPF_OFF);
}



//****************************************************************************
//		OnUpdateMenu
//	Destroy old menu bar and create the new one
//****************************************************************************
void FrostzApp::OnUpdateMenu ()
{
	SHMENUBARINFO mbInfo;
	memset (&mbInfo, 0, sizeof (SHMENUBARINFO));
	mbInfo.cbSize = sizeof (SHMENUBARINFO);

	// Destroy current menu bar
	DestroyWindow (hMenuBar);

	if (isZmachineRunning)
	{
		// Create game menu bar

		mbInfo.hwndParent = hWnd;
		mbInfo.dwFlags = SHCMBF_COLORBK;
		mbInfo.nToolBarId = IDR_GAME_MENUBAR;
		mbInfo.hInstRes = hInstance;
		mbInfo.nBmpId = 0;
		mbInfo.cBmpImages = 0;
		mbInfo.clrBk = RGB (0xFF, 0xFF, 0xFF);

		if (!SHCreateMenuBar (&mbInfo))
		{
			unsigned short wMsg[128];
			wsprintf (wMsg, L"Error #%d", GetLastError ());
			MessageBox (hWnd, L"Could not create menu bar", wMsg, MB_ICONSTOP);
			SendMessage (hWnd, WM_DESTROY, 0, 0);
		}
		else
		{
			hMenuBar = mbInfo.hwndMB;
			CommandBar_AddBitmap (hMenuBar, hInstance, IDR_MAIN_TOOLBAR, 10, 16, 16);
		}
	}
	else
	{
		// Create main application bar
		mbInfo.hwndParent = hWnd;
		mbInfo.dwFlags = SHCMBF_COLORBK;
		mbInfo.nToolBarId = IDR_APP_MENUBAR;
		mbInfo.hInstRes = hInstance;
		mbInfo.nBmpId = 0;//IDR_MAIN_TOOLBAR;
		mbInfo.cBmpImages = 0;//2;
		mbInfo.clrBk = RGB (0xFF, 0xFF, 0xFF);

		if (!SHCreateMenuBar (&mbInfo))
		{
			unsigned short wMsg[128];
			wsprintf (wMsg, L"Error #%d", GetLastError ());
			MessageBox (hWnd, L"Could not create menu bar", wMsg, MB_ICONEXCLAMATION);
			SendMessage (hWnd, WM_DESTROY, 0, 0);
		}
		else
		{
			hMenuBar = mbInfo.hwndMB;
			CommandBar_AddBitmap (hMenuBar, hInstance, IDR_MAIN_TOOLBAR, 3, 16, 16);

			/* Enumerate fonts and add corresponding items to the menu
				- Experimental -
			FoundFonts = 0;
			ID_FIRSTFONT = 41000;
			HMENU hMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd), SHCMBM_GETSUBMENU,
								(WPARAM)0, (LPARAM)ID_MENU_OPTIONS);
			EnumFontFamilies (GetDC (hWnd), NULL, (FONTENUMPROC)EnumFontProc, (LPARAM)hMenu);
			numFonts = FoundFonts;
			*/
		}
	}

	UpdateMenuItems ();
}



//****************************************************************************
//		OnUpdateCursor
//	Handles the WM_UPDATE_CURSOR (user) message
//****************************************************************************
void FrostzApp::OnUpdateCursor ()
{
	POINT sCaretPos;

	// Calculate new caret position
	sCaretPos.x = CursorCol * CharWidth;
	sCaretPos.y = CursorRow * CharHeight;

	SetCaretPos (sCaretPos.x, sCaretPos.y);

	// Cursor state needs setting
	if (!CursorSet)
	{
		CursorSet = TRUE;
		
		// Show or hide cursor
		if (CursorOn)
			ShowCaret (hWnd);
		else
			HideCaret (hWnd);
	}
}



void FrostzApp::OnUndo ()
{
	::EnterCriticalSection( &KbdCriticalSection );

	// Add char to buffer
	*WriteKeyBuffer = ZC_HKEY_UNDO;
	WriteKeyBuffer++;

	// Gone past end of buffer?
	if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
	{
		// Go back to beginning of buffer
		WriteKeyBuffer = &KeyBuffer[0];
	}

	::LeaveCriticalSection( &KbdCriticalSection );

		// Signal that a keypress event has occurred
	::SetEvent( hKeyEvent );
}



//****************************************************************************
//		OnChar
//	Handles the WM_CHAR message
//****************************************************************************
void FrostzApp::OnChar (UINT Char, UINT RepCnt, UINT Flags)
{
		// Expecting input?
	if (ExpectingInput)
	{
		::EnterCriticalSection (&KbdCriticalSection);

		// Add key to buffer
		*WriteKeyBuffer = Char;
		WriteKeyBuffer++;

		// Gone past end of buffer?
		if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
		{
			// Go back to beginning of buffer
			WriteKeyBuffer = &KeyBuffer[0];
		}

		::LeaveCriticalSection (&KbdCriticalSection);

		// Signal that a keypress event has occurred
		::SetEvent (hKeyEvent);
	}
}



//****************************************************************************
//		OnLButtonDown
//	Handles the WM_LBUTTONDOWN message
//****************************************************************************
void FrostzApp::OnLButtonDown (UINT Flags, int X, int Y)
{
	// Expecting input?
	if (ExpectingInput) 
	{
		int nCharRow, nCharCol;
		char *pszBuffer = new char[Width+1];
		int i = 0;
		bool bReturn = true;

		int gridx = 52/3;
		int gridy = 46/3;
		int origx = 178;
		int origy = 18;

		if (story_id == JOURNEY || story_id == ZORK_ZERO ||
			(story_id == BEYOND_ZORK && X > 41*FontWidth && Y < 14*FontHeight))
		if (h_flags & MOUSE_FLAG)
		{
			pszBuffer[0] = (unsigned char)ZC_SINGLE_CLICK;
			pszBuffer[1] = (unsigned char)'\0';

			mouse_x = X / FontWidth;
			mouse_y = Y / FontHeight;

			StuffKeyboard (pszBuffer, false);

			return;
		}

		// Use compass if visible
		if(!cBlindMode && ShowCompass && (X > origx )  && (X <= 3 * gridx + origx ) && (Y > origy )  && (Y <= 3 * gridy + origy ) )
		{
			MessageBeep (MB_OK);

			// WEST QUADRANT
			if(X > origx && X <= (origx + gridx) )
			{
				// NW
				if(Y > origy && Y <= (origy + gridy) )
					StuffKeyboard( "NorthWest", bReturn );
				else if(Y > (origy + gridy) && Y <= (origy + 2 * gridy) )
					StuffKeyboard( "West", bReturn );
				else if(Y > (origy + 2 * gridy) && Y <= (origy + 3 * gridy) )
					StuffKeyboard( "SouthWest", bReturn );
			}
			// MID QUADRANT
			else if(X > (origx + gridx) && X <= (origx + 2 * gridx) )
			{
				// NW
				if(Y > origy && Y <= (origy + gridy) )
					StuffKeyboard( "North", bReturn );
//				else if(point.y > (origy + gridy) && point.y <= (origy + 2 * gridy) )
//					StuffKeyboard( "West", bReturn );
				else if(Y > (origy + 2 * gridy) && Y <= (origy + 3 * gridy) )
					StuffKeyboard( "South", bReturn );
			}
			// EAST QUADRANT
			else if(X > (origx + 2 * gridx) && X <= (origx + 3 * gridx) )
			{
				// NW
				if(Y > origy && Y <= (origy + gridy) )
					StuffKeyboard( "NorthEast", bReturn );
				else if(Y > (origy + gridy) && Y <= (origy + 2 * gridy) )
					StuffKeyboard( "East", bReturn );
				else if(Y > (origy + 2 * gridy) && Y <= (origy + 3 * gridy) )
					StuffKeyboard( "SouthEast", bReturn );
			}
			else
					StuffKeyboard( "", bReturn );

			
		}
		else  // Look at words
		{

			// Calculate char position
			nCharRow = Y / CharHeight;
			nCharCol = X / CharWidth;

			// Go backwards until beginning of word
			while (nCharCol)
			{
				if (pScreen[(nCharRow * Width) + nCharCol-1] != _T(' ')) nCharCol--;
				else break;
			}

			// Valid row?
			if (nCharRow < Height)
			{
				// Copy word to buffer
				for (; nCharCol < Width; i++)
				{
					pszBuffer[i] = (char) pScreen[(nCharRow * Width) + nCharCol + i];

					if (!((pszBuffer[i] >= 'A' && pszBuffer[i] <= 'Z') || 
						(pszBuffer[i] >= 'a' && pszBuffer[i] <= 'z'))) break;
				}
			}

			pszBuffer[i] = '\0';

			// Check current state
			switch (State)
			{
				case STATE_NONE:
					if (strlen( pszBuffer )) bReturn = false;
					break;
				case STATE_LOOKING:
				//	if (strlen( pszBuffer )) StuffKeyboard( " at ", false );
					break;
				case STATE_TAKING:
					if (!strlen( pszBuffer )) StuffKeyboard( "all", false );
					break;
				case STATE_OPENING:
					if (!strlen( pszBuffer )) StuffKeyboard( "Door", false );
					break;
				case STATE_MOVING:
					pszBuffer[0] = '\0';
					break;
				case STATE_MACRONL:
					if (strlen( pszBuffer )) bReturn = true;
					break;
			}

			// Copy buffer into keyboard buffer
			StuffKeyboard( pszBuffer, bReturn );

			// Add a space if we're not adding a return
			if (!bReturn) StuffKeyboard( " ", false );

			delete [] pszBuffer;
		}
	}
}



//****************************************************************************
//		OnKeyDown
//	Handles the WM_KEYDOWN message
//****************************************************************************
void FrostzApp::OnKeyDown (UINT Char, UINT RepCnt, UINT Flags)
{
	// Expecting input?
	if (ExpectingInput)
	{
		int ch;

		// Check for recognised keys
		switch (Char)
		{
		// Left arrow
		case 37:
			ch = LEFT_ARROW_KEY;
			break;
		// Right arrow
		case 39:
			ch = RIGHT_ARROW_KEY;
			break;
		// Up arrow
		case 38:
			ch = UP_ARROW_KEY;
			break;
		// Down arrow
		case 40:
			ch = DOWN_ARROW_KEY;
			break;
		// Page up
		case 33:
			ch = SPECIAL_KEY_PAGE_UP;
			break;
		// Page down
		case 34:
			ch = SPECIAL_KEY_PAGE_DOWN;
			break;
		// Home
		case 36:
			ch = SPECIAL_KEY_HOME;
			break;
		// End
		case 35:
			ch = SPECIAL_KEY_END;
			break;
		// Del
		case 46:
			ch = SPECIAL_KEY_DELETE;
			break;
		// Insert
		case 45:
			ch = SPECIAL_KEY_INSERT;
			break;
		default:
			ch = 0;
			break;
		}

		// Got anything?
		if (ch)
		{
			::EnterCriticalSection (&KbdCriticalSection);

			// Add hotkey to buffer
			*WriteKeyBuffer = ch;
			WriteKeyBuffer++;

			// Gone past end of buffer?
			if (WriteKeyBuffer > &KeyBuffer[KEYBUFFER_SIZE-1])
			{
				// Go back to beginning of buffer
				WriteKeyBuffer = &KeyBuffer[0];
			}

			::LeaveCriticalSection (&KbdCriticalSection);

			// Signal that a keypress event has occurred
			::SetEvent (hKeyEvent);
		}
	}
}



//****************************************************************************
//		OnLoadRecent
//****************************************************************************
void FrostzApp::OnLoadRecent (BYTE MRU_Idx)
{
	DWORD tmp = 0;

	//*** Set Story File Name
	//	- This string will be used by the Z-Machine Thread!!!
	memset (sStoryFile, 0, strlen (sStoryFile));
	wcstombs (sStoryFile, MRU[MRU_Idx].wFileName, 256);

	// Start the Z-Machine
	hZThread = CreateThread (NULL, 0,
		(LPTHREAD_START_ROUTINE )ZMachineThreadFunction,
		(LPVOID)NULL, 0, &tmp);

	if (hZThread == 0)
	{
		unsigned short wMsg[32];
		wsprintf (wMsg, L"Error #%d", GetLastError ());
		MessageBox (hWnd, L"Could not start the Z-Machine", wMsg, MB_OK | MB_ICONSTOP);
		SendMessage (hWnd, WM_DESTROY, 0, 0);
	}
	else
		ResumeThread (hZThread);
}



//****************************************************************************
//		OnStoryOpen
//****************************************************************************
void FrostzApp::OnStoryOpen ()
{
	OPENFILENAME ofn;
	BOOL IsOK;
	unsigned short szDirName[256];
	unsigned short szFile[256];
	UINT  i, cbString;
	unsigned short  chReplace;
	unsigned short  szFilter[256], szDefaultDir[256];

	szDirName[0] = (unsigned short)0;
	memset (szFile, 0, sizeof(unsigned short) * 256);
	wcscpy (szFilter, L"Story files (*.DAT;*.ZIP)|*.DAT;*.ZIP|Other Story Files (*.Z?)|*.Z5;*.Z6;*.Z7;*.Z8)|All files (*.*)|*.*|\0");

	cbString = wcslen(szFilter);

	chReplace = szFilter[cbString - 1];

	for (i = 0; szFilter[i] != 0x0000; ++i)
	{
		if (szFilter[i] == chReplace)
			szFilter[i] = 0x0000;
	}

	// Get application dir
	GetAppDir (szDefaultDir, hInstance);

	memset (&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hWnd;
	ofn.lpstrFilter = szFilter;
	ofn.nFilterIndex = 1;
	ofn.lpstrFile= szFile;
	ofn.nMaxFile = 256;
	//ofn.lpstrFileTitle = szFileTitle;
	//ofn.nMaxFileTitle = sizeof(szFileTitle);
	ofn.lpstrInitialDir = szDefaultDir;
	ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;

	IsOK = GetOpenFileName (&ofn);

	unsigned short wMsg[256];

	if (IsOK)
	{
		DWORD tmp;
		int Cnt;

		//*** Save complete story file
		//	- This value will be used by the Z-Machine thread!!!
		memset (sStoryFile, 0, 256);
		for (Cnt = 0; Cnt < 256 && ofn.lpstrFile[Cnt] != 0; ++Cnt)
			sStoryFile[Cnt] = (char)ofn.lpstrFile[Cnt];
		
		/*** Store this new MRU entry ***/
		// First, check if there was already such an entry
		tmp = FALSE; // We'll use tmp as a flag: it will become TRUE if entry was found
		for (Cnt = 0; Cnt < MAX_MRU; ++Cnt)
		{
			if (!wcscmp (ofn.lpstrFile, MRU[Cnt].wFileName))
			{
				tmp = TRUE;
				break;
			}
		}
		// Now, if there wasn't such an entry, add it - shifting down the others
		if (tmp == FALSE)
		{
			wcscpy (MRU[3].wFileName, MRU[2].wFileName);
			if (MRU[3].wFileName[0])
				MRU[3].wShortName = wcsrchr (MRU[3].wFileName, (unsigned short)'\\') + 1;
			else
				MRU[3].wShortName = 0;
			wcscpy (MRU[2].wFileName, MRU[1].wFileName);
			if (MRU[2].wFileName[0])
				MRU[2].wShortName = wcsrchr (MRU[2].wFileName, (unsigned short)'\\') + 1;
			else
				MRU[2].wShortName = 0;
			wcscpy (MRU[1].wFileName, MRU[0].wFileName);
			if (MRU[1].wFileName[0])
				MRU[1].wShortName = wcsrchr (MRU[1].wFileName, (unsigned short)'\\') + 1;
			else
				MRU[1].wShortName = 0;
			wcscpy (MRU[0].wFileName, ofn.lpstrFile);
			MRU[0].wShortName = wcsrchr (MRU[0].wFileName, (unsigned short)'\\') + 1;
		}
		/********************************/

		// Start the Z-Machine
		tmp = 0;
		hZThread = CreateThread (NULL, 0,
			(LPTHREAD_START_ROUTINE )ZMachineThreadFunction,
			(LPVOID)NULL, 0, &tmp);

		if (hZThread == 0)
		{
			wsprintf (wMsg, L"Error #%d", GetLastError ());
			MessageBox (hWnd, L"Could not start the Z-Machine", wMsg, MB_OK | MB_ICONSTOP);
			SendMessage (hWnd, WM_DESTROY, 0, 0);
		}
		else
			ResumeThread (hZThread);

	}

	GoFullScreen ();
}


/* - Experimental -
//****************************************************************************
//		OnSelectFont
//	A new font face is selected from the menu
//****************************************************************************
void FrostzApp::OnSelectFont (int SelId)
{
	unsigned short MenuItemText[256];
	MENUITEMINFO mii;
	HMENU hMenu = (HMENU)SendMessage (SHFindMenuBar (hWnd),
		SHCMBM_GETSUBMENU, (WPARAM)0, (LPARAM)ID_MENU_OPTIONS);

	memset (MenuItemText, 0, sizeof (unsigned short) * 256);
	memset (&mii, 0, sizeof (MENUITEMINFO));

	mii.cbSize = sizeof (MENUITEMINFO);
	mii.fMask = MIIM_TYPE;
	mii.fType = MFT_STRING;
	mii.dwTypeData = MenuItemText;
	mii.cch = 256;

	// Get face name from menu item
	GetMenuItemInfo (hMenu, SelId, FALSE, &mii);
	memset (FontFace, 0, sizeof (unsigned short) * 256);
	wcsncpy (FontFace, MenuItemText, wcslen (MenuItemText));

	// Check selected item, uncheck others
	for (unsigned int Cnt = 0; Cnt < numFonts; ++Cnt)
	{
		// Get menu item text
		GetMenuItemInfo (hMenu, 41000 + Cnt, FALSE, &mii);

		if (wcsncmp (FontFace, MenuItemText, wcslen (MenuItemText)) == 0)
			CheckMenuItem (hMenu, 41000 + Cnt, MF_BYCOMMAND | MF_CHECKED);
		else
			CheckMenuItem (hMenu, 41000 + Cnt, MF_BYCOMMAND | MF_UNCHECKED);
	}
}
*/
