/*=============================================================
 * standard.c -- Some standard routines.
 * Copyright(c) by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 07 Aug 93 - modified
 *=============================================================
 */
#include <stdio.h>
#include "standard.h"

static BOOLEAN logopen = FALSE;
static FILE *log = NULL;
extern BOOLEAN alloclog;
static char scratch[80];

/*==========================================================
 * __allocate - Allocates memory - called by sdtalloc macro.
 *==========================================================*/
char *__allocate (len, file, line)
int len;    /* number of bytes to allocate */
char *file; /* name of C file requesting memory */
int line;   /* line number in file */
{
	char *p;
	ASSERT(p = malloc(len));
	if (alloclog) {
		sprintf(scratch, "A  %s\t%d\t%d\t%d", file, line,
		    len, p);
		alloc_out(scratch);
	}
	return p;
}
/*=========================================================
 * __deallocate - Returns memory - called by sdtfree macro.
 *========================================================*/
__deallocate (ptr, file, line)
char *ptr;  /* address of memory being returned */
char *file; /* name of C file returning memory */
int line;   /* line number in file */
{
	if (alloclog) {
		sprintf(scratch, "F  %s\t%d\t\t%d", file,
		    line, ptr);
		alloc_out(scratch);
	}
	free(ptr);
}
/*=======================================================
 * alloc_out -- Output alloc message if log file is open.
 *=====================================================*/
alloc_out (str)
STRING str;
{
	if (!alloclog) return;
	if (!logopen) {
		log = fopen("alloc.log", "w");
		logopen = TRUE;
	}
	fprintf(log, "%s\n", str);
}
/*=================================
 * strsave -- Saves copy of string.
 *===============================*/
STRING strsave (str)
STRING str;
{
	STRING strcpy();
	return strcpy(stdalloc(strlen(str) + 1), str);
}
/*====================================
 * strconcat -- Catenates two strings.
 *==================================*/
STRING strconcat (s1, s2)
STRING s1, s2;
{
	INT c, len;
	STRING s3, p;

	if (!s1) return strsave(s2);
	if (!s2) return strsave(s1);
	len = strlen(s1) + strlen(s2);
	p = s3 = (STRING) stdalloc(len+1);
	while (c = *s1++)
		*p++ = c;
	while (c = *s2++)
		*p++ = c;
	*p = 0;
	return s3;
}
/*=======================================
 * chartype -- Returns type of character.
 *=====================================*/
INT chartype (c)
INT c;
{
	if (c == ' ' || c == '\t' || c == '\n' || c == '\r')
		return WHITE;
	if (c >= 'a' && c <= 'z') return LETTER;
	if (c >= 'A' && c <= 'Z') return LETTER;
	if (c >= '0' && c <= '9') return DIGIT;
	return c;
}
/*=============================
 * fatal - Fatal error routine.
 *===========================*/
__fatal (file, line)
STRING file;
int line;
{
	wprintf("FATAL ERROR: %s: line %d\n", file, line);
	gedclose();
	user_fatal_calls();
	endwin();
	abort();
	exit(0);
}
/*=================================
 * __assert -- Checks an assertion.
 *===============================*/
__assert (exp, file, line)
BOOLEAN exp;
STRING file;
int line;
{
	if (exp) return;
	__fatal(file, line);
}
/*=============================================
 * iswhite -- Checks for white space character.
 *===========================================*/
BOOLEAN iswhite (c)
INT c;
{
	return c == ' ' || c == '\t' || c == '\n' || c == '\r';
}
/*=========================================
 * isletter -- Checks for letter character.
 *=======================================*/
BOOLEAN isletter (c)
INT c;
{
	if (c >= 'a' && c <= 'z') return TRUE;
	return c >= 'A' && c <= 'Z';
}
/*========================================
 * lower -- Converts string to lower case.
 *======================================*/
STRING lower (str)
STRING str;
{
	static char scratch[50];
	STRING p = scratch;
	INT c;
	while (c = *str++)
		*p++ = tolower(c);
	*p = 0;
	return scratch;
}
/*========================================
 * upper -- Converts string to upper case.
 *======================================*/
STRING upper (str)
STRING str;
{
	static char scratch[50];
	STRING p = scratch;
	INT c;
	while (c = *str++)
		*p++ = toupper(c);
	*p = 0;
	return scratch;
}
/*==================================
 * capitalize -- Capitalizes string.
 *================================*/
STRING capitalize (str)
STRING str;
{
	STRING p = lower(str);
	*p = toupper(*p);
	return p;
}
/*=========================================
 * toupper -- Converts letter to uppercase.
 *=======================================*/
INT toupper (c)
INT c;
{
	if (c < 'a' || c > 'z') return c;
	return c + 'A' - 'a';
}
/*=========================================
 * tolower -- Converts letter to lowercase.
 *=======================================*/
INT tolower (c)
INT c;
{
	if (c < 'A' || c > 'Z') return c;
	return c + 'a' - 'A';
}
/*==================================
 * trim -- Trims string if too long.
 *================================*/
STRING trim (str, len)
STRING str;
INT len;
{
	static char scratch[MAXLINELEN+1];
	if (!str || strlen(str) > MAXLINELEN) return NULL;
	if (len < 0) len = 0;
	if (len > MAXLINELEN) len = MAXLINELEN;
	strcpy(scratch, str);
	scratch[len] = 0;
	return scratch;
}
