/*================================================================
 * miscutils.c -- Supports the miscellaneous commands.
 * Copyright(c) 1992 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 15 Aug 93 - modified
 *   Version 2.3.6 - 12 Oct 93 - modified
 *================================================================
 */
#include "standard.h"
#include "btree.h"
#include "table.h"
#include "gedcom.h"
#include <sys/types.h>
#include <time.h>

extern STRING idgedf, gdcker, gdnadd, gdnerr;
extern STRING btreepath, llarchives;
extern BTREE BTR;
extern BOOLEAN resnames;

static TABLE indikeys = NULL;
static TABLE famkeys = NULL;

STRING map_indi_key();
STRING map_fam_key();

/*==============================================
 * restore -- Reads a GEDCOM file into database.
 *============================================*/
BOOLEAN restore_from_file ()
{
	FILE *fp, *ask_for_file();
	NODE node, conv;
	STRING msg, fname;
	BOOLEAN emp;
	INT nindi = 0, nfam = 0;
	int fboth();

/* Open GEDCOM file and validate it */
	fp = ask_for_file("r", idgedf, &fname, NULL);
	if (!fp) return FALSE;
	wprintf(gdcker, fname);
	if (!validate_gedcom(fp)) {
		wprintf(gdnadd);
		return FALSE;
	}
	wprintf(gdnerr);
	rewind(fp);
	indikeys = create_table();
	famkeys = create_table();

/* Add records in GEDCOM file to database */
	node = first_fp_to_node(fp, FALSE, &msg, &emp);
	while (node) {
		STRING tag = ntag(node);
		if (!(conv = node_to_node(node)))
			free_nodes(node);
		else if (strcmp("INDI", tag) == 0) {
			nindi++;
			restore_indi(conv);
			free_nodes(node);
		} else if (strcmp("FAM", tag) == 0) {
			nfam++;
			restore_fam(conv);
			free_nodes(conv);
		} else
			FATAL();
		node = next_fp_to_node(fp, FALSE, &msg, &emp);
	}
	wprintf("\n");
	mprintf("Added %d persons and %d families from file `%s'.",
	    nindi, nfam, fname);
	remove_table(indikeys, fboth);
	remove_table(famkeys, fboth);
	return TRUE;
}
/*==============================================
 * restore_indi -- Restore a person to database.
 *============================================*/
BOOLEAN static restore_indi (indi)
NODE indi;
{
	STRING old, new;
	NODE fnode;
	if (!indi) return FALSE;

 /* Translate record key to internal format */
	old = nxref(indi);
	ASSERT(old && NAME(indi));
	new = map_indi_key(old);
	if (resnames)
		wprintf("%s %s %s\n", rmvat(old), rmvat(new),
		    indi_to_name(indi, 78));
	else
		wprintf("i");
	if (strcmp(old, new)) {
		stdfree(old);
		nxref(indi) = strsave(new);
	}

/* Fix all links in record */
	if (fnode = FAMC(indi)) fix_fam_code(fnode);
	for (fnode = FAMS(indi); fnode; fnode = nsibling(fnode))
		fix_fam_code(fnode);
	add_linked_indi(indi);
	return TRUE;
}
/*=============================================
 * restore_fam -- Restore a family to database.
 *===========================================*/
static restore_fam (node)
NODE node;
{
	STRING old, new, str;
	NODE indi;

	if (!node) return;
	old = nxref(node);
	ASSERT(old);
	new = map_fam_key(old);
	if (resnames)
		wprintf("%s %s New Family\n", rmvat(old), rmvat(new));
	else
		wprintf("f");
	if (strcmp(old, new)) {
		stdfree(old);
		nxref(node) = strsave(new);
	}
	if (indi = HUSB(node)) fix_indi_code(indi);
	if (indi = WIFE(node)) fix_indi_code(indi);
	indi = CHIL(node);
	for ( ; indi; indi = nsibling(indi))
		fix_indi_code(indi);
	ASSERT(str = node_to_string(node));
	ASSERT(store_record(rmvat(nxref(node)), str, strlen(str)));
	stdfree(str);
}
/*========================================================
 * fix_fam_code -- Translates FAM node to internal format.
 *======================================================*/
static fix_fam_code (node)
NODE node;
{
	STRING new = map_fam_key(nval(node));
	stdfree(nval(node));
	nval(node) = strsave(new);
}
/*==========================================================
 * fix_indi_code -- Translates INDI node to internal format.
 *========================================================*/
static fix_indi_code (node)
NODE node;
{
	STRING new = map_indi_key(nval(node));
	stdfree(nval(node));
	nval(node) = strsave(new);
}
/*==================================================================
 * map_indi_key -- Translate person code from extern to intern form.
 *================================================================*/
STRING map_indi_key (old)
STRING old;
{
	STRING new;
	if (new = (STRING) valueof(indikeys, old)) return new;
	new = getixref();
	insert_table(indikeys, strsave(old), strsave(new));
	return new;
}
/*=================================================================
 * map_fam_key -- Translate family code from extern to intern form.
 *===============================================================*/
STRING map_fam_key (old)
STRING old;
{
	STRING new;
	if (new = (STRING) valueof(famkeys, old)) return new;
	new = getfxref();
	insert_table(famkeys, strsave(old), strsave(new));
	return new;
}
/*========================================================
 * archive_in_file -- Archive a database as a GEDCOM file.
 *======================================================*/
STRING mabbv[] = {
	"JAN", "FEB", "MAR", "APR", "MAY", "JUN",
	"JUL", "AUG", "SEP", "OCT", "NOV", "DEC",
};
static FILE *fn = NULL;
BOOLEAN archive_in_file ()
{
	FILE *ask_for_file();
	char dat[30], tim[20];
	struct tm *pt;
	time_t curtime;
	STRING fname;
	extern STRING version;
	BOOLEAN archive();

	fn = ask_for_file("w", "Enter name of output archive file.",
	    &fname, llarchives);
	if (!fn) {
		mprintf("The database was not saved.");
		return FALSE; 
	}
	curtime = time(NULL);
	pt = localtime(&curtime);
	sprintf(dat, "%d %s %d", pt->tm_mday, mabbv[pt->tm_mon],
	    1900 + pt->tm_year);
	sprintf(tim, "%d:%.2d", pt->tm_hour, pt->tm_min);
	fprintf(fn, "0 HEAD\n1 SOUR LIFELINES %s\n1 DEST ANY\n", version);
	fprintf(fn, "1 DATE %s\n1 TIME %s\n", dat, tim);
	traverse(BTR, bmaster(BTR), NULL, archive);
	fprintf(fn, "0 TRLR\n");
	fclose(fn);
	wprintf("\n");
	mprintf("Database `%s' has been saved in `%s'.", btreepath, fname);
	return TRUE;
}
/*==============================================================
 * archive -- Traverse function called on each BTREE data block.
 *============================================================*/
BOOLEAN archive (btree, block)
BTREE btree;
BLOCK block;
{
	INT i, n, l;
	char scratch[100];
	STRING key;
	FILE *fo;

	sprintf(scratch, "%s/%s", bbasedir(btree), fkey2path(iself(block)));
	ASSERT(fo = fopen(scratch, "r"));
	n = nkeys(block);
	wprintf(".");
	for (i = 0; i < n; i++) {
		key = rkey2str(rkeys(block, i));
		if (*key != 'I' && *key != 'F') continue;
		if (fseek(fo, (long)(offs(block, i) + BUFLEN), 0))
			FATAL();
		if ((l = lens(block, i)) > 6)	/* filter deleted records */
			filecopy(fo, l, fn);
	}
	fclose(fo);
	return TRUE;
}
/*=========================================
 * key_util -- Returns key value of person.
 *=======================================*/
key_util ()
{
	NODE indi = ask_for_indi("Whose key value do you want?", FALSE, FALSE);
	if (!indi) return;
	mprintf("%s - %s", rmvat(nxref(indi)), indi_to_name(indi, 70));
}
/*======================================================
 * who_is_he_she -- Find who a person is from key value.
 *====================================================*/
who_is_he_she ()
{
	STRING key, str, rec, ask_for_string();
	NODE indi;
	INT len;
	char nkey[100];

	key = ask_for_string("Please enter person's internal key value.",
	    "enter key:");
	if (!key || *key == 0) return;
	nkey[0] = 'I';
	if (*key == 'I')
		strcpy(nkey, key);
	else
		strcpy(&nkey[1], key);
	if (!(rec = retrieve_record(nkey, &len))) {
		mprintf("No one in database has key value %s.", key);
		return;
	}
	if (!(indi = string_to_node(rec))) {
		mprintf("No one in database has key value %s.", key);
		stdfree(rec);
		return;
	}
	if (!(str = indi_to_name(indi, 60)) || *str == 0) {
		mprintf("No one in database has key value %s.", key);
		stdfree(rec);
		return;
	}
	mprintf("%s - %s", key, str);
}
/*========================================================
 * fboth -- Remove tables used for restoring GEDCOM files.
 *======================================================*/
static fboth (ent)
ENTRY ent;
{
	stdfree(ent->ekey);
	stdfree(ent->evalue);
}
/*===============================================
 * show_database_stats -- Show stats of database.
 *=============================================*/
show_database_stats ()
{
	INT n = num_indis();
	INT m = num_fams();
	mprintf("Database `%s' contains %d persons and %d families.",
	    btreepath, n, m);
}
