/*================================================================
 * valgdcom.c -- Validates GEDCOM file for closure.
 * Copyright(c) 1993 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 02 Sep 93 - modified
 *   Version 2.3.6 - 31 Oct 93 - modified
 *================================================================
 */
#include <stdio.h>
#include "standard.h"
#include "table.h"
#include "sequence.h"
#include "gedcom.h"

typedef struct {
	STRING key;
	INT line;
	INT sex;
	INT famc;
	SEQUENCE fams;
}  *PERSON;

typedef struct {
	STRING key;
	INT line;
	INT husb;
	INT wife;
	SEQUENCE chil;
}  *FAMILY;

static BOOLEAN in_indi, in_fam;
static SEQUENCE indiseq;
static SEQUENCE famseq;
static TABLE inditable, famtable;
static BOOLEAN sexed = FALSE;
static BOOLEAN named = FALSE;
static PERSON person = NULL;
static FAMILY family = NULL;
static INT num_indis, num_fams;
static INT numindis, numfams;
static INT num_errors;

STRING misixr = "Line %d: The person defined here has no key.\n";
STRING misfxr = "Line %d: The family defined here has no key.\n";
STRING mulper = "Lines %d and %d: Person %s is multiply defined.\n";
STRING mulfam = "Lines %d and %d: Family %s is multiply defined.\n";
STRING misval = "Line %d: This %s line is missing a value field.\n";
STRING misfms = "Person %s is missing a FAMS link to Family %s.\n";
STRING misfmc = "Person %s is missing a FAMC link to Family %s.\n";
STRING mishsb = "Family %s is missing a HUSB link to Person %s.\n";
STRING miswif = "Family %s is missing a WIFE link to Person %s.\n";
STRING mischl = "Family %s is missing a CHIL link to Person %s.\n";
STRING oppsex = "Person %s has wrong sex for Family %s.\n";
STRING unsex  = "Person %s is a parent but has unknown or no recorded sex.\n";
STRING undper = "Person %s is referred to but not defined.\n";
STRING undfam = "Family %s is referred to but not defined.\n";
STRING badlev = "Line %d: This line has a level number that is too large.\n";
STRING noname = "Person %s has no name line.\n";

/*=============================================================
 * validate_gedcom -- Validate set of GEDCOM records from file.
 *===========================================================*/
BOOLEAN validate_gedcom (fp)
FILE *fp;
{
	INT lev, noline = 0, rc, idx, curlev = 0;
	STRING xref, tag, val, msg;
	char scratch[20];

	inditable = create_table();
	famtable = create_table();
	indiseq = crtseq(100);
	famseq = crtseq(100);
	num_indis = num_fams = num_errors = numindis = numfams = 0;
	curlev = 0;
	rc = file_to_line(fp, &lev, &xref, &tag, &val, &msg);
	noline++;
	while (rc != DONE)  {
		if (lev > curlev + 1) {
			wprintf(badlev, noline);
			num_errors++;
			curlev = lev;
			rc = file_to_line(fp, &lev, &xref, &tag, &val, &msg);
			noline++;
			continue;
		}
		if (rc == ERROR) {
			wprintf("%s\n", msg);
			num_errors++;
			curlev = lev;
			rc = file_to_line(fp, &lev, &xref, &tag, &val, &msg);
			noline++;
			continue;
		}
		if (lev > 1) {
			curlev = lev;
			rc = file_to_line(fp, &lev, &xref, &tag, &val, &msg);
			noline++;
			continue;
		}
		if (lev == 0) {
			if (in_indi && !named) {
				wprintf(noname, scratch);
				num_errors++;
			}
			strcpy(scratch, xref);
			if (eqstr("INDI", tag)) {
				num_indis++;
if (num_indis % 100 == 0) wprintf("%d Persons Processed\n", num_indis);/**/
				in_indi = TRUE;
				in_fam = FALSE;
				named = FALSE;
				sexed = FALSE;
				idx = add_indi_defn(xref, noline);
				person = (PERSON) elseq(indiseq, idx);
			} else if (eqstr("FAM", tag)) {
				num_fams++;
if (num_fams % 100 == 0) wprintf("%d Families Processed\n", num_fams);/**/
				in_indi = FALSE;
				in_fam = TRUE;
				idx = add_fam_defn(xref, noline);
				family = (FAMILY) elseq(famseq, idx);
			} else {
				in_indi = FALSE;
				in_fam = FALSE;
			}
		} else {
			if (in_indi)
				add_indi_ref(tag, val, noline);
			else if (in_fam)
				add_fam_ref(tag, val, noline);
		}
		curlev = lev;
		rc = file_to_line(fp, &lev, &xref, &tag, &val, &msg);
		noline++;
	}
	check_references();
	remove_sequences();
	remove_table(inditable, NULL);
	remove_table(famtable, NULL);
	return num_errors == 0;
}
/*========================================
 * add_indi_defn -- Add person definition.
 *======================================*/
INT add_indi_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	INT idx;
	PERSON per;
	BOOLEAN there;
	if (!xref || *xref == 0) {
		wprintf(misixr, line);
		num_errors++;
		return -1;
	}
	idx = (INT) valueofbool(inditable, xref, &there);
	if (there) {
		per = (PERSON) elseq(indiseq, idx);
		if (per->line && line) {
			wprintf(mulper, per->line, line, xref);
			num_errors++;
		} else if (line)
			per->line = line;
	} else {
		idx = numindis++;
		per = (PERSON) malloc(sizeof(*per));
		per->key = strsave(xref);
		per->line = line;
		per->sex = SEX_UNKNOWN;
		per->famc = -1;
		per->fams = crtseq(2);
		insert_table(inditable, per->key, idx);
		tinsseq(indiseq, per);
	}
	return idx;
}
/*=======================================
 * add_fam_defn -- Add family definition.
 *=====================================*/
INT add_fam_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	INT idx;
	FAMILY fam;
	BOOLEAN there;
	if (!xref || *xref == 0) {
		wprintf(misfxr, line);
		num_errors++;
		return -1;
	}
	idx = (INT) valueofbool(famtable, xref, &there);
	if (there) {
		fam = (FAMILY) elseq(famseq, idx);
		if (fam->line && line) {
			wprintf(mulfam, fam->line, line, xref);
			num_errors++;
		} else if (line)
			fam->line = line;
	} else {
		idx = numfams++;
		fam = (FAMILY) malloc(sizeof(*fam));
		fam->key = strsave(xref);
		fam->line = line;
		fam->husb = -1;
		fam->wife = -1;
		fam->chil = crtseq(2);
		insert_table(famtable, fam->key, idx);
		tinsseq(famseq, fam);
	}
	return idx;
}
/*======================================
 * add_indi_ref -- Add person reference.
 *====================================*/
add_indi_ref (tag, val, line)
STRING tag, val;
INT line;
{
	INT idx;
	if (eqstr(tag, "FAMC")) {
		if (!val || *val == 0) {
			wprintf(misval, line, "FAMC");
			num_errors++;
			return;
		}
		if (person && person->famc != -1) {
			wprintf("Line %d: Multiple FAMC line.\n", line);
			num_errors++;
			return;
		}
		if (person) {
			idx = add_fam_defn(val, 0);
			person->famc = idx;
		}
	} else if (eqstr(tag, "FAMS")) {
		if (!val || *val == 0) {
			wprintf(misval, line, "FAMS");
			num_errors++;
			return;
		}
		idx = add_fam_defn(val, 0);
		if (person && inseq(person->fams, idx)) {
			wprintf("Line %d: Multiple FAMS line.\n", line);
			num_errors++;
			return;
		}
		if (person) tinsseq(person->fams, idx);
	} else if (eqstr(tag, "SEX")) {
		if (sexed) {
			wprintf("Line %d: Multiple SEX lines.\n", line);
			num_errors++;
			return;
		}
		if (!person) return;
		if (!val || *val == 0)
			person->sex = SEX_UNKNOWN;
		else if (*val == 'M')
			person->sex = SEX_MALE;
		else if (*val == 'F')
			person->sex = SEX_FEMALE;
		else
			person->sex = SEX_UNKNOWN;
	} else if (eqstr(tag, "NAME")) {
		named = TRUE;
		if (!val || *val == 0 || !valid_name(val)) {
			wprintf("Line %d: Bad NAME syntax.\n", line);
			num_errors++;
			return;
		}
	}
}
/*=====================================
 * add_fam_ref -- Add family reference.
 *===================================*/
add_fam_ref (tag, val, line)
STRING tag, val;
INT line;
{
	INT idx;
	if (eqstr(tag, "HUSB")) {
		if (!val || *val == 0) {
			wprintf(misval, line, "HUSB");
			num_errors++;
			return;
		}
		if (family && family->husb != -1) {
			wprintf("Line %d: Multiple HUSB line.\n", line);
			num_errors++;
			return;
		}
		if (family) {
			idx = add_indi_defn(val, 0);
			family->husb = idx;
		}
	} else if (eqstr(tag, "WIFE")) {
		if (!val || *val == 0) {
			wprintf(misval, line, "WIFE");
			num_errors++;
			return;
		}
		if (family && family->wife != -1) {
			wprintf("Line %d: Multiple WIFE line.\n", line);
			num_errors++;
			return;
		}
		if (family) {
			idx = add_indi_defn(val, 0);
			family->wife = idx;
		}
	} else if (eqstr(tag, "CHIL")) {
		if (!val || *val == 0) {
			wprintf(misval, line, "CHIL");
			num_errors++;
			return;
		}
		idx = add_indi_defn(val, 0);
		if (family && inseq(family->chil, idx)) {
			wprintf("Line %d: Multiple CHIL line.\n", line);
			num_errors++;
			return;
		}
		if (family) tinsseq(family->chil, idx);
	}
}
/*=============================================================
 * check_references -- Check for undefined persons or families.
 *============================================================*/
check_references ()
{
	INT i;
	for (i = 0; i < numindis; i++)
		check_indi_links(i);
	for (i = 0; i < numfams; i++)
		check_fam_links(i);
}
/*========================================
 * check_indi_links -- Check person links.
 *======================================*/
check_indi_links (idx)
INT idx;
{
	PERSON per;
	FAMILY fam;
	INT fdx, sdx, odx;
	STRING msg;
	per = (PERSON) elseq(indiseq, idx);
	if (per->line == 0) {
		wprintf(undper, per->key);
		num_errors++;
		return;
	}
	if ((fdx = per->famc) != -1) {
		fam = (FAMILY) elseq(famseq, fdx);
		if (fam->line != 0) {
			if (!inseq(fam->chil, idx)) {
				wprintf(mischl, fam->key, per->key);
				num_errors++;
			}
		}
	}
	if (per->sex == SEX_UNKNOWN && Size(per->fams)) {
		wprintf(unsex, per->key);
		num_errors++;
	} else {
		FORALL(i,el,per->fams)
			fam = (FAMILY) elseq(famseq, el);
			if (fam->line != 0) {
				if (per->sex == SEX_MALE) {
					sdx = fam->husb;
					odx = fam->wife;
					msg = mishsb;
				} else {
					sdx = fam->wife;
					odx = fam->husb;
					msg = miswif;
				}
				if (odx == idx) {
					wprintf(oppsex, per->key, fam->key);
					num_errors++;
				} else if (sdx != idx) {
					wprintf(msg, fam->key, per->key);
					num_errors++;
				}
			}
		ENDLOOP
	}
}
/*=======================================
 * check_fam_links -- Check family links.
 *=====================================*/
check_fam_links (fdx)
INT fdx;
{
	FAMILY fam = (FAMILY) elseq(famseq, fdx);
	PERSON per;
	INT sdx;
	if (fam->line == 0) {
		wprintf(undfam, fam->key);
		return;
	}
	if ((sdx = fam->husb) != -1) {
		per = (PERSON) elseq(indiseq, sdx);
		if (per->line != 0) {
			if (!inseq(per->fams, fdx)) {
				wprintf(misfms, per->key, fam->key);
				num_errors++;
			}
		}
	}
	if ((sdx = fam->wife) != -1) {
		per = (PERSON) elseq(indiseq, sdx);
		if (per->line != 0) {
			if (!inseq(per->fams, fdx)) {
				wprintf(misfms, per->key, fam->key);
				num_errors++;
			}
		}
	}
	FORALL(i,el,fam->chil)
		per = (PERSON) elseq(indiseq, el);
		if (per->line != 0) {
			if (per->famc != fdx) {
				wprintf(misfmc, per->key, fam->key);
				num_errors++;
			}
		}
	ENDLOOP
}
/*======================================
 * inseq -- See if value is in sequence.
 *====================================*/
BOOLEAN inseq (seq, val)
SEQUENCE seq;
INT val;
{
	FORALL(i,el,seq)
		if ((INT) el == val)
			return TRUE;
	ENDLOOP
	return FALSE;
}
/*===============================================
 * remove_sequences -- Remove two main sequences.
 *=============================================*/
remove_sequences ()
{
	PERSON per;
	FAMILY fam;
	FORALL(i,el,indiseq)
		per = (PERSON) el;
		stdfree(per->key);
		rmvseq(per->fams);
		stdfree(per);
	ENDLOOP
	rmvseq(indiseq);
	FORALL(i,el,famseq)
		fam = (FAMILY) el;
		stdfree(fam->key);
		rmvseq(fam->chil);
		stdfree(fam);
	ENDLOOP
	rmvseq(famseq);
}
