/*===================================================================
 * record.c -- This file contains the routines that handle BTREE
 *   records.
 * Copyright(c) 1991 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 01 Jul 93 - modified
 *===================================================================
 */
#include "standard.h"
#include "btree.h"

/*===================================================================
 * addrecord -- Add a record to a btree database.
 *===================================================================
 * This routine adds a record to a btree.  The first parameter is the
 * tree, the second is the rkey of the record being added, the third
 * is the record itself, and the fourth is the length of the record.
 * This is one of the main user interfaces to the btree software.
 *===================================================================*/
BOOLEAN addrecord (btree, rkey, record, len)
BTREE btree;	/* btree to add record to */
RKEY rkey;	/* key of the record to add */
RECORD record;	/* record to add to btree */
INT len;	/* length of record */
{
	INDEX index;
	BLOCK old, new, xtra;
	FKEY nfkey;
	FKEY last = 0;
	FKEY parent;
	SHORT i, j, k, l, n;
	SHORT lo, hi;
	BOOLEAN found = FALSE;
	INT off = 0;
	FILE *fo, *fn1, *fn2;
	char scratch1[200];
	char scratch2[200];
	char *p = record;

/*wprintf("ADDRECORD: rkey = %s\n",rkey2str(rkey));dumpbtree(btree);/*DEBUG*/
/*---------------------------------------------------------------------
 * Starting at the master index, search for the data block that does
 * or would hold the record.  Read index blocks, compare keys, and read
 * new index blocks, until a data block is reached.
 *--------------------------------------------------------------------*/
	if ((index = bmaster(btree)) == NULL)  FATAL();
	while (itype(index) == BTINDEXTYPE) {
/*--------------------------------------------------------------------
 * The following maintains "lazy" parent chaining in the btree.  The
 * need for parent links occurs when new keys must be added to
 * indices.  The links must be correct just before a new record is
 * added.  This code rewrites the indexes that lead to a new
 * record, but only if the parent link in the index is incorrect.
 *--------------------------------------------------------------------*/
		if (iparent(index) != last) {
			if (index == bmaster(btree))  FATAL();
			iparent(index) = last;
			writeindex(bbasedir(btree), index);
		}
		last = iself(index);
		n = nkeys(index);
		nfkey = fkeys(index, 0);
		for (i = 1; i <= n; i++) {
			if (strncmp(rkey.r_rkey, rkeys(index, i).r_rkey, 8) < 0)
				break;
			nfkey = fkeys(index, i);
		}
		if ((index = getindex(btree, nfkey)) == NULL)  FATAL();
	}
/*--------------------------------------------------------------------
 * We have the block (index holds its header) that may contain an
 * older version of the record.  There will be room in the header to
 * store information about the new record, because of an invariant
 * that splits a data block if it ever fills up (code later in this
 * routine maintains the invariant).
 *--------------------------------------------------------------------*/
	iparent(index) = last;
	old = (BLOCK) index;
	if (nkeys(old) >= NORECS)  FATAL();
/*--------------------------------------------------------------------
 * We now search the block to see if it contains an earlier version of
 * the record.  Records are sorted by key so a binary search is used.
 * Variable found is set to TRUE if the record is already there.
 *--------------------------------------------------------------------*/
	lo = 0;
	hi = nkeys(old) - 1;
	found = FALSE;
	while (lo <= hi) {
		SHORT md = (lo + hi)/2;
		INT rel = strncmp(rkey.r_rkey, rkeys(old, md).r_rkey, 8);
		if (rel < 0)
			hi = --md;
		else if (rel > 0)
			lo = ++md;
		else {
			found = TRUE;
			lo = md;
			break;
		}
	}
/*--------------------------------------------------------------------
 * We now know where in the data block the record will go, indexed by
 * the value of lo.  We must now construct the header for the updated
 * data block.  Here we allocate and initialize the block.
 *--------------------------------------------------------------------*/
	new = allocblock();
	itype(new) = itype(old);
	iparent(new) = iparent(old);
	iself(new) = iself(old);
	n = nkeys(new) = nkeys(old);
/*--------------------------------------------------------------------
 * We now put information about all records with keys less than the
 * one being added into the new header.
 *--------------------------------------------------------------------*/
	for (i = 0; i < lo; i++) {
		rkeys(new, i) = rkeys(old, i);
		lens(new, i) = lens(old, i);
		offs(new, i) = off;
		off += lens(old, i);
	}
/*--------------------------------------------------------------------
 * We then put in the information about the record being added.
 * This record may be an entirely new record, with a new key, or it
 * may be a replacement for an old record, with the same key.
 *--------------------------------------------------------------------*/
	rkeys(new, lo) = rkey;
	lens(new, lo) = len;
	offs(new, lo) = off;
	off += len;
/*--------------------------------------------------------------------
 * We then put in the information about the records with keys greater
 * than the one being added.  The variable found is used to skip over
 * the old copy of the record, if it were in the old data block.
 *--------------------------------------------------------------------*/
	if (found)
		j = 0, i++;
	else
		j = 1;
	for (; i < n; i++) {
		rkeys(new, i + j) = rkeys(old, i);
		lens(new, i + j) = lens(old, i);
		offs(new, i + j) = off;
		off += lens(old, i);
	}
	if (!found) nkeys(new) = n + 1;
/*--------------------------------------------------------------------
 * The new header has been built.  We must now rewrite the data block
 * file with the new record inserted.  We do this by reading from the
 * original file and writing a temporary file.  The following code
 * opens these two files.
 *--------------------------------------------------------------------*/
	sprintf(scratch1, "%s/%s", bbasedir(btree), fkey2path(iself(old)));
	if ((fo = fopen(scratch1, "r")) == NULL) FATAL();
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	if ((fn1 = fopen(scratch1, "w")) == NULL) FATAL();
/*--------------------------------------------------------------------
 * Before continuing we must check if adding the new record will cause
 * the current data block to split.  Of course, splitting will never
 * occur if the new record replaces an old one.  If splitting is
 * needed, we branch to special code immediately.
 *--------------------------------------------------------------------*/
	if (!found && n == NORECS - 1) goto splitting;
/*--------------------------------------------------------------------
 * Okay, we're not splitting.  We must therefore write the new header
 * to the temporary file, and then write all the records that precede
 * the new record to the temporary file.  These records are read from
 * the old data block.
 *--------------------------------------------------------------------*/
	if (fwrite(new, BUFLEN, 1, fn1) != 1)  FATAL();
	putheader(btree, new);
	for (i = 0; i < lo; i++) {
		if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
			FATAL();
		filecopy(fo, lens(old, i), fn1);
	}
/*--------------------------------------------------------------------
 * Next, the new record itself is written to the temp file.
 *--------------------------------------------------------------------*/
	p = record;
	while (len >= BUFLEN) {
		if (fwrite(p, BUFLEN, 1, fn1) != 1)  FATAL();
		len -= BUFLEN;
		p += BUFLEN;
	}
	if (len && fwrite(p, len, 1, fn1) != 1)  FATAL();
/*--------------------------------------------------------------------
 * And finally, the records that follow the new record must be written
 * to the new data block file.
 *--------------------------------------------------------------------*/
	if (found) i++;
	for ( ; i < n; i++) {
		if (fseek(fo, (long)(offs(old, i)+BUFLEN), 0))  FATAL();
		filecopy(fo, lens(old, i), fn1);
	}
/*--------------------------------------------------------------------
 * At this point a non-splitting, normal add record has been done.
 * The two files must be closed, and the temp file must replace the
 * original data block file.  The only thing slightly tricky here
 * involves the header blocks.  The memory where the old header is
 * stored must be freed, while the new header must be cached (the
 * caching was done earlier with the putheader call).
 *--------------------------------------------------------------------*/
	fclose(fn1);
	fclose(fo);
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(iself(old)));
	stdfree(old);
	movefiles(scratch1, scratch2);
/*--------------------------------------------------------------------
 * This is the normal return point for the non-splitting case.
 *--------------------------------------------------------------------*/
	return TRUE;
/*--------------------------------------------------------------------
 * Here begins the code that splits the current data block (becuase
 * the addition of the new record would fill it up).  This code knows
 * we are adding a new record, not just replacing an old one.  At this
 * point the full header has been created, the old data block file is
 * open for reading, and one of the now two required temp output files
 * is open for writing.  We first open the second temp file.
 *--------------------------------------------------------------------*/
splitting:
	sprintf(scratch1, "%s/tmp2", bbasedir(btree));
	if ((fn2 = fopen(scratch1,"w")) == NULL)  FATAL();
/*--------------------------------------------------------------------
 * We now write the first temp file, the "first half" of the old data
 * block file.  We first write the header block, but after modifying
 * the nkeys field.  We then write out the first half of the records
 * to the first temp file.  We don't worry whether the new record goes
 * into the first or second half, simply checking for this case in
 * both sections (see the "if (j == lo)" checks).
 *--------------------------------------------------------------------*/
	nkeys(new) = n/2;	/*temporarily*/
	if (fwrite(new, BUFLEN, 1, fn1) != 1)  FATAL();
	putheader(btree, new);
	for (i = j = 0; j < n/2; j++) {
		if (j == lo) {
			p = record;
			while (len >= BUFLEN) {
				if (fwrite(p, BUFLEN, 1, fn1) != 1)
					FATAL();
				len -= BUFLEN;
				p += BUFLEN;
			}
			if (len && fwrite(p, len, 1, fn1) != 1)
				FATAL();
		} else {
			if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
				FATAL();
			filecopy(fo, lens(old, i), fn1);
			i++;
		}
	}
/*--------------------------------------------------------------------
 * The first temp file has been written and it's time to write the
 * second.  We must first create a new block header using crtblock,
 * which initializes the header with a new file key.  We have to put
 * information into this header from the last half of the other new
 * header.  We therefore move the second half of the header in new
 * into the first half of this newer header.
 *--------------------------------------------------------------------*/
	nfkey = iself(new);
	parent = iparent(new);
	xtra = crtblock(btree);
	iparent(xtra) = parent;
	off = 0;
	for (k = 0, l = n/2; k < n - n/2 + 1; k++, l++) {
		rkeys(xtra, k) = rkeys(new, l);
		lens(xtra, k) = lens(new, l);
		offs(xtra, k) = off;
		off += lens(new, l);
	}
	nkeys(xtra) = n - n/2 + 1;
	if (fwrite(xtra, BUFLEN, 1, fn2) != 1)  FATAL();
	putheader(btree, xtra);
/*--------------------------------------------------------------------
 * And now we write the second half of the records to the second temp
 * file.
 *--------------------------------------------------------------------*/
	for (j = n/2; j <= n; j++) {
		if (j == lo) {
			p = record;
			while (len >= BUFLEN) {
				if (fwrite(p, BUFLEN, 1, fn2) != 1)
					FATAL();
				len -= BUFLEN;
				p += BUFLEN;
			}
			if (len && fwrite(p, len, 1, fn2) != 1)
				FATAL();
		} else {
			if (fseek(fo, (long)(offs(old, i) + BUFLEN), 0))
				FATAL();
			filecopy(fo, lens(old, i), fn2);
			i++;
		}
	}
/*--------------------------------------------------------------------
 * Both temp files are now written, and the record adding is done.  We
 * must now close the files, take care of memory management, and make
 * the temp files permanent.
 *--------------------------------------------------------------------*/
	fclose(fo);
	fclose(fn1);
	fclose(fn2);
	stdfree(old);
	sprintf(scratch1, "%s/tmp1", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(nfkey));
	movefiles(scratch1, scratch2);
	sprintf(scratch1, "%s/tmp2", bbasedir(btree));
	sprintf(scratch2, "%s/%s", bbasedir(btree), fkey2path(iself(xtra)));
	movefiles(scratch1, scratch2);
/*--------------------------------------------------------------------
 * And one last thing remains.  Because we split a data block we must
 * add an index entry for the newly created data block the the parent
 * index.  This may lead to further splitting and other new keys.
 *--------------------------------------------------------------------*/
	addkey(btree, parent, rkeys(xtra, 0), iself(xtra));
	return TRUE;
}
/*================================================================
 * filecopy - Copy a record from one data file to another.
 *================================================================*/
filecopy (fo, len, fn)
FILE *fo, *fn;
INT len;
{
	char buffer[BUFLEN];
	while (len >= BUFLEN) {
		if (fread(buffer, BUFLEN, 1, fo) != 1)  FATAL();
		if (fwrite(buffer, BUFLEN, 1, fn) != 1)  FATAL();
		len -= BUFLEN;
	}
	if (len) {
		if (fread((char *)buffer, len, 1, fo) != 1)  FATAL();
		if (fwrite(buffer, len, 1, fn) != 1)  FATAL();
	}
}
/*=================================================================
 * getrecord - Retrieves a record from a BTREE.
 *=================================================================*/
RECORD getrecord (btree, rkey, plen)
BTREE btree;
RKEY rkey;
INT *plen;
{
	INDEX index;
	SHORT i, n, lo, hi;
	FKEY nfkey;
	BLOCK block;
	BOOLEAN found = FALSE;
	char scratch[200];
	FILE *fr;
	RECORD record;
	INT len;

	*plen = 0;
/*wprintf("GETRECORD: rkey: %s\n", rkey2str(rkey));/*DEBUG*/
	if ((index = bmaster(btree)) == NULL)  FATAL();

   /* SEARCH FOR THE DATA BLOCK THAT DOES OR "SHOULD" CONTAIN RECORD */

	while (itype(index) == BTINDEXTYPE) {
		n = nkeys(index);
		nfkey = fkeys(index, 0);
		for (i = 1; i <= n; i++) {
			if (strncmp(rkey.r_rkey, rkeys(index, i).r_rkey, 8) < 0)
				break;
			nfkey = fkeys(index, i);
		}
		if ((index = getindex(btree, nfkey)) == NULL)
			FATAL();
	}

   /* HAVE FOUND THE BLOCK THAT MAY CONTAIN RECORD */

	block = (BLOCK) index;

   /* SEARCH FOR RECORD KEY */

	lo = 0;
	hi = nkeys(block) - 1;
	while (lo <= hi) {
		SHORT md = (lo + hi)/2;
		INT rel = strncmp(rkey.r_rkey, rkeys(block, md).r_rkey, 8);
		if (rel < 0)
			hi = --md;
		else if (rel > 0)
			lo = ++md;
		else {
			found = TRUE;
			lo = md;
			break;
		}
	}
	if (!found) return NULL;

	sprintf(scratch, "%s/%s", bbasedir(btree), fkey2path(iself(block)));
	if ((fr = fopen(scratch, "r")) == NULL) FATAL();
	if (fseek(fr, (long)(offs(block, lo) + BUFLEN), 0)) FATAL();
	len = lens(block, lo) + 1;
	record = (RECORD) stdalloc(len);
	if (fread(record, len - 1, 1, fr) != 1) FATAL();
	fclose(fr);
	record[len-1] = 0;
	*plen = lens(block, lo);
	return record;
}
/*===============================================
 * movefiles -- Move the first file to the second.
 *=============================================*/
movefiles (from, to)
STRING from, to;
{
	unlink(to);
	link(from, to);
	unlink(from);
}
