/*================================================================
 * openbtree.c -- Code for opening and creating BTREE databases.
 * Copyright(c) 1991 by Thomas T. Wetmore IV; all rights reserved.
 *   Version 2.3.4 - 24 Jun 93 - controlled
 *   Version 2.3.5 - 05 Jul 93 - modified
 *   Version 2.3.6 - 17 Oct 93 - modified
 *   Version 3.0.0 - 17 Oct 93 - modified
 *================================================================
 */
#include <sys/types.h>
#include <sys/stat.h>
#include "standard.h"
#include "btree.h"

/*================================================
 * openbtree -- Allocate and init BTREE structure.
 *==============================================*/
BTREE openbtree (name, dir, crtflag)
STRING name;		/*name of btree*/
STRING dir;		/*base directory of btree*/
BOOLEAN crtflag;	/*create btree if it doesn't exist?*/
{
	BTREE btree;
	char scratch[200];
	FILE *fp;
	struct stat sbuf;
	KEYFILE kfile;
	INDEX master;

	bterrno = 0;
	if (strlen(dir) > 150) {         /*validate directory*/
		bterrno = BTERRLNGDIR;
		return NULL;
	}

/* See if base directory exists */
	if (stat(dir, &sbuf)) {
		sprintf(scratch, "%s/", dir);
		if (!crtflag || !mkalldirs(scratch)) {
			bterrno = BTERRNOBTRE;
			return NULL;
		}
	}
	if (stat(dir, &sbuf) || !sbuf.st_mode&S_IFDIR) {
		bterrno = BTERRNOBTRE;
		return NULL;
	}

/* See if key file exists */
	sprintf(scratch, "%s/key", dir);
	if (stat(scratch, &sbuf)) {
		if (!crtflag || !initbtree(dir)) {
			bterrno = BTERRNOKFL;
			return NULL;
		}
	}
	if (stat(scratch, &sbuf) || !sbuf.st_mode&S_IFREG) {
		bterrno = BTERRNOKFL;
		return NULL;
	}

/* Open and read key file */
	if (!(fp = fopen(scratch, "r+")) ||
	    fread(&kfile, sizeof(KEYFILE), 1, fp) != 1) {
		bterrno = BTERRRDKFL;
		return NULL;
	}

/* Get master index */
	if (!(master = readindex(dir, kfile.k_mkey)))
		return NULL;	/* bterrno set by getindex */

/* Create new BTREE */
	btree = (BTREE) stdalloc(sizeof *btree);
	bname(btree) = name;
	bbasedir(btree) = dir;
	bmaster(btree) = master;
	bkfp(btree) = fp;
	btree->b_kfile.k_mkey = kfile.k_mkey;
	btree->b_kfile.k_fkey = kfile.k_fkey;
	initcache(btree, 20);
	return btree;
}

/*===================================
 * initbtree -- Initialize new BTREE.
 *=================================*/
BOOLEAN initbtree (basedir)
STRING basedir;
{
	KEYFILE kfile;
	INDEX master;
	BLOCK block;
	FILE *fk, *fi, *fd;
	char scratch[200];

/* Open file for writing keyfile */
	sprintf(scratch, "%s/key", basedir);
	if ((fk = fopen(scratch, "w")) == NULL) {
		bterrno = BTERROPNKFL;
		return FALSE;
	}

/* Open file for writing master index */
	sprintf(scratch, "%s/aa/aa", basedir);
	if (!mkalldirs(scratch) || (fi = fopen(scratch, "w")) == NULL) {
		bterrno = BTERROPNIDX;
		fclose(fk);
		return FALSE;
	}

/* Open file for writing first data block */
	sprintf(scratch, "%s/ab/aa", basedir);
	if (!mkalldirs(scratch) || (fd = fopen(scratch, "w")) == NULL) {
		bterrno = BTERROPNDAT;
		fclose(fk);
		fclose(fi);
		return FALSE;
	}

/* Write key file */
	kfile.k_mkey = path2fkey("aa/aa");
	kfile.k_fkey = path2fkey("ab/ab");
	if (fwrite(&kfile, sizeof(KEYFILE), 1, fk) != 1) {
		bterrno = BTERRWRTKFL;
		fclose(fk);
		fclose(fi);
		fclose(fd);
		return FALSE;
	}
	fclose(fk);

/* Write master index */
	master = (INDEX) stdalloc(BUFLEN);
	itype(master) = BTINDEXTYPE;
	iself(master) = path2fkey("aa/aa");
	iparent(master) = NULL;
	master->i_nkeys = 0;
	master->i_fkeys[0] = path2fkey("ab/aa");
	if (fwrite(master, BUFLEN, 1, fi) != 1) {
		bterrno = BTERRWRTMST;
		fclose(fi);
		fclose(fd);
		return FALSE;
	}
	fclose(fi);

/* Write first data block */
	block = (BLOCK) stdalloc(BUFLEN);
	itype(block) = BTBLOCKTYPE;
	iself(block) = path2fkey("ab/aa");
	iparent(block) = NULL;
	block->i_nkeys = 0;
	if (fwrite(block, BUFLEN, 1, fd) != 1) {
		bterrno = BTERRWRTMST;
		fclose(fd);
		return FALSE;
	}
	fclose(fd);
	return TRUE;
}

/*=============================
 * llmkdir -- Create directory.
 *===========================*/
int llmkdir (dir)
STRING dir;	/* directory to create */
{
	static status;
	register pid;

	if (pid = fork())
		while (wait(&status) != pid);
	else  {
		close(2);
		execl("/bin/mkdir", "mkdir", dir, 0);
		exit(2);
	}
	return status>>8 == 0;
}
/*===========================================
 * mkalldirs -- Make all directories in path.
 *=========================================*/
#define ENDSTR  0
#define exists(p)  (!(*p) || access((p),00) == 0)

BOOLEAN mkalldirs (path)
char  *path;		/* path with dir's to be made */
{
	register  int   i, n;
	register  char  *p = path;

	for (i = 0, n = strlen(path);  i < n;  i++, p++)  {
		if (*p != '/')
			continue;
		*p = ENDSTR;
		if (exists (path) || llmkdir (path))  {
			*p = '/';
			continue;
		}
		wprintf ("Can't create directory %s", path);
		return FALSE;
	}
	return TRUE;
}
