#
# This is a small debugger for Wafe. It is very experimental and
# has probably many bugs. it is quite dangerous to try to debug
# the debugger with itself. However, i found already a bug with it
# in a 6000 line Wafe application (tcl part), so other might like
# to share it.
#
# Essentialy, this debugger supports 
#   - call level tracing on 
#       + Tcl procedures,
#       + Wafe built-ins, and
#       + global Tcl variables (on read, write, unset)
#   - inspection of the current state of 
#       + Tcl procedures and
#       + global Tcl variables
#
# The debugger can be called by the Tcl Command "wafeDebug"
# and should be loaded via autoload. The current version uses
# the Athena widget set, it should be quite easy to provide
# OSF/Motif support as well
#
# Gustaf Neumann                         Mohegan Lake, May 5 1994

set wafeDebugDepth 0

set _wafeDebugIgnore(procs,wafeDebug) {}
set _wafeDebugIgnore(procs,wafeDebugLog) {}
set _wafeDebugIgnore(procs,wafeDebug:ShowStep) {}
set _wafeDebugIgnore(procs,wafeDebug:TraceLine) {}
set _wafeDebugIgnore(procs,wafeDebug:ListElement) {}
set _wafeDebugIgnore(procs,wafeDebug:ProcContents) {}
set _wafeDebugIgnore(procs,wafeDebug:Create) {}
set _wafeDebugIgnore(procs,wafeDebug:Redraw) {}
set _wafeDebugIgnore(procs,wafeDebug:SetInfo) {}
set _wafeDebugIgnore(procs,wafeDebugLog:Create) {}
set _wafeDebugIgnore(procs,wafeDebugLog:DisplayText) {}
set _wafeDebugIgnore(procs,wafeDebugLog:ClearIfInvisible) {}
set _wafeDebugIgnore(procs,pvars) {}
set _wafeDebugIgnore(procs,pset) {}
set _wafeDebugIgnore(procs,$) {}

# the "object" wafeDebug has
#   method create
#   method setSelection
#   method getSelection
#   method setMode
#   method getMode
#   method dismiss
#   method info
#   method reDraw
#   method trace
#   static instance variables: selection selectionList mode sh
proc wafeDebug { args } {
  # p is the name of the object class (equals procedure name)
  set self [lindex [info level 0] 0]

  static selection selectionList mode sh

  if [string match "" $args] {
    set method create
    set arguments {}
  } elseif [llength $args]==1 {
    set method $args
    set arguments {}
  } else {
    pset {method arguments} $args
  }

  # pvars method arguments
  switch $method {
    create {
      # if wafeDebug is called without arguments, 
      # its widget tree is created (if needed) and is popped up
      if ![isWidget $self] {
	set sh [wafeDebug:Create $self]
	set selection {}
	set mode undefined
      }
      popup $sh none
      wafeDebugLog
    }

    reDraw {
      wafeDebug:Redraw $sh $selectionList $selection
    }

    isTraced {
      global _wafeDebugTrace
      pset {type name} $arguments
      return [info exists _wafeDebugTrace($type,$name)]
    }

    info {
      wafeDebug:SetInfo $sh [lindex $arguments 0]
    }

    getSelection {
      return $selection
    }

    setSelection {
      sV $sh*$self-show sensitive true
      sV $sh*$self-trace sensitive true
      set selection [string trimleft $arguments *\ ]
    }

    getMode {
      return $mode
    }

    setMode {
      global _wafeDebugIgnore
      set mode $arguments

      switch $mode {
	commands {
	  set selectionList {}
	  foreach element [lsort [info commands]] {
	    if [catch {info args $element}] { 
	      if ![info exists _wafeDebugIgnore($mode,$element)] {
		lappend selectionList [wafeDebug:ListElement $mode $element]
	      } elseif [regexp {(.*)-orig} $element x oname] {
		lappend selectionList [wafeDebug:ListElement $mode $oname]
	      }
	    }
	  }
	}
	vars -
	procs {
	  set selectionList {}
	  foreach element [lsort [uplevel #0 "info $mode"]] {
	    if ![info exists _wafeDebugIgnore($mode,$element)] {
	      lappend selectionList [wafeDebug:ListElement $mode $element]
	    }
	  }
	}
      }
      $self reDraw
    }
 
    trace {
      global _wafeDebugTrace _wafeDebugIgnore
      pset {type name} $arguments

      if [info exists _wafeDebugTrace($type,$name)] {
	unset _wafeDebugTrace($type,$name)
	switch $type {
	  vars {
	    uplevel #0 "trace vdelete $name w {wafeDebugLog traceVar}"
	    uplevel #0 "trace vdelete $name u {wafeDebugLog traceVar}"
	    uplevel #0 "trace vdelete $name r {wafeDebugLog traceVar}"
	  }
	  commands -
	  procs {
	    unset _wafeDebugIgnore($type,$name-orig)
	    rename $name ""
	    rename $name-orig $name
	  }
	}
	$self info "Removing Tracepoint for $name"
	$self setMode $type 
	return
      }
      switch $type {
	vars {
	  uplevel #0 "trace variable $name w {wafeDebugLog traceVar}"
	  uplevel #0 "trace variable $name u {wafeDebugLog traceVar}"
	  uplevel #0 "trace variable $name r {wafeDebugLog traceVar}"
	}
	commands -
	procs {
	  append new "proc $name args \{\n"
	  append new "  uplevel 1 wafeDebug:ShowStep $name $name-orig \[list \$args\]\n\}"
          catch {rename $name-orig ""}
          rename $name $name-orig
	  eval $new
	  set _wafeDebugIgnore($type,$name-orig) {}
	}
      }
      set _wafeDebugTrace($type,$name) {}
      $self info "Setting Trace Point for $name"
      $self setMode $type 
    }

    dismiss {
      popdown $sh
    }

  }
}


################### Athena specific commands ##########################

proc wafeDebug:Create-athena {self} {

  set sh [TopLevelShell $self topLevel title "Wafe Debugger"]
  #  if [window topLevel] {callback $sh popupCallback positionCursor 0}
  set smallFixedFont -b&h-*-medium-r-*-*-12-*-*-*-*-*-*-*
  set clearEnterTranslations {#override
    <Enter>: exec()
    <Leave>: exec()
  }
  mergeResources topLevel \
      *wafeDebug.geometry +0+0 \
      *wafeDebugLog.geometry -0+0 \
      *wafeDebug*left chainLeft \
      *wafeDebug*Command.right chainLeft \
      *wafeDebug*top chainTop \
      *wafeDebug*bottom chainTop \
      *wafeDebug*wafeDebug-List*font $smallFixedFont \
      *wafeDebug*translations $clearEnterTranslations \
      *wafeDebugLog*left chainLeft \
      *wafeDebugLog*Command.right chainLeft \
      *wafeDebugLog*top chainTop \
      *wafeDebugLog*bottom chainTop \
      *wafeDebugLog*Text*font $smallFixedFont \
      *wafeDebugLog*translations $clearEnterTranslations 

  set width 350
  set F [Form     wafeDebug-form $sh ]
  set I [Label    wafeDebug-info $F width $width label {} ]
  set V [Command  wafeDebug-vars $F {
    callback "$self setMode vars" label "Global Variables"
    fromVert $I 
  }]
  set P [Command  wafeDebug-procs $F {
    callback "$self setMode procs" label Procedures
    fromVert $I fromHoriz $V 
  }]
  set P [Command  wafeDebug-commands $F {
    callback "$self setMode commands" label Built-Ins
    fromVert $I fromHoriz $P
  }]
  set VP [Viewport wafeDebug-view $F {
    width $width height 250
    allowHoriz true allowVert true
    fromVert $V
    top chainTop bottom chainBottom right chainRight
  }]
  set L [List    wafeDebug-list $VP {
    callback "$self setSelection %s" list {}
    width $width height 250
  }]
  set Q [Command wafeDebug-quit $F {
    callback "$self dismiss" label "Dismiss"
    fromVert $VP
    top chainBottom bottom chainBottom
  }]
  set S [Command wafeDebug-show $F {
    callback {wafeDebugLog show [wafeDebug getMode] [wafeDebug getSelection]}
    label "Show" sensitive false
    fromVert $VP fromHoriz $Q
    top chainBottom bottom chainBottom
  }]
  set T [Command wafeDebug-trace $F {
    callback {wafeDebug trace [wafeDebug getMode] [wafeDebug getSelection]}
    label "Trace" sensitive false
    fromVert $VP fromHoriz $S
    top chainBottom bottom chainBottom
  }]
  set GL [Label wafeDebug-grepLabel $F {
    label "Grep:" borderWidth 0
    fromVert $VP fromHoriz $T
    top chainBottom bottom chainBottom
  }]
  set G [Text wafeDebug-grep $F {
    editType Edit callback "$self reDraw"
    fromVert $VP fromHoriz $GL
    top chainBottom bottom chainBottom
  }]
  return $sh
}

proc wafeDebug:Redraw-athena {sh selectionList selection} {
  set grepString [gV $sh*wafeDebug-grep string]
  if [string compare "" $grepString] {
    set list ""
    foreach e $selectionList {
      if [string match *$grepString* $e] { lappend list $e }
    }
    set theList list
  } else {
    set theList selectionList
  }
  XawListChange $sh*wafeDebug-list 0 0 1 List [set $theList]
  set currentItem [lsearch -glob [set $theList] ?$selection]
  # pvars currentItem selection 
  if [string compare -1 $currentItem] {
    XawListHighlight $sh*wafeDebug-list $currentItem
  }
}

proc wafeDebug:SetInfo-athena {sh string} {
      sV $sh*wafeDebug-info label $string
}


proc wafeDebugLog:Create-athena {self} {
  set sh [TransientShell $self wafeDebug title "Wafe Debug Log"]
  # if [window topLevel] {callback $sh popupCallback positionCursor 0}
  set width 530
  set F [Form  $self-form $sh ]
  set I [Label $self-info $F width $width label {}]
  set T [Text  $self-text $F {
    editType edit type string string "" width $width height 300
    scrollVertical whenNeeded scrollHorizontal whenNeeded
    fromVert $I right chainRight bottom chainBottom
  }]
  Command $self-quit $F {
    callback "$self dismiss" label "Dismiss"
    fromVert $T
    top chainBottom bottom chainBottom
  }
  return $sh
}

proc wafeDebugLog:DisplayText-athena {self sh string} {
  set self wafeDebugLog
  set w [widgetId $sh*$self-text]
  callActionProc $w {} end-of-file
  set cursorPosition [gV $w insertPosition]
  set text(firstPos) 0
  set text(length)   [set length [string length $string]]
  set text(ptr)      $string
  XawTextReplace $w $cursorPosition $cursorPosition text
  callActionProc $w {} end-of-file
}

proc wafeDebugLog:ClearIfInvisible-athena {self sh} {
  if [string compare IsViewable [mapState $sh]] {
    sV $sh*$self-text string ""
  }
}

################### Motif specific commands ##########################

proc wafeDebug:Create-motif {self} {
  # Motif Version
  # if [window topLevel] {callback $sh popupCallback positionCursor 0}
  set smallFixedFont -b&h-*-medium-r-*-*-12-*-*-*-*-*-*-*
  set clearEnterTranslations {#override
    <Enter>: exec()
    <Leave>: exec()
  }
  mergeResources topLevel \
      *wafeDebug.geometry +0+0 \
      *wafeDebugLog.geometry -0+0 \
      *wafeDebug-actions.labelString "Actions" \
      *wafeDebug-actions.mnemonic "A" \
      *wafeDebug-views*labelString "Views" \
      *wafeDebug-views*mnemonic "V" \
      *wafeDebug-vars.labelString "Global Variables" \
      *wafeDebug-procs.labelString "Procedures" \
      *wafeDebug-commands.labelString "Built-Ins" \
      *wafeDebug-quit.labelString "Dismiss" \
      *wafeDebug-show.labelString "Show" \
      *wafeDebug-trace.labelString "Trace" \
      *wafeDebug-grep.labelString "Grep:" \
      *wafeDebug-info.scrolledWindowChildType  COMMAND_WINDOW \
      *wafeDebug-btns2.scrolledWindowChildType MESSAGE_WINDOW \
      *wafeDebugLog*Text*font $smallFixedFont \
      *wafeDebugLog*translations $clearEnterTranslations 

  set sh [TopLevelShell $self topLevel title "Wafe Debugger"] 
  set width 350
  set F [XmMainWindow $self-form $sh width $width]
  set M [XmMenuBar $self-menubar $F]

  set P [XmPulldownMenu $self-m1 $M unmanaged]
  XmPushButtonGadget $self-quit $P \
      accelerator Ctrl<Key>C \
      acceleratorText Ctrl+C \
      activateCallback "$self dismiss"
  XmCascadeButtonGadget $self-actions $M subMenuId $self-m1
  
  set P [XmPulldownMenu $self-m2 $M unmanaged]
  foreach b {vars procs commands} {
    XmPushButtonGadget $self-$b $P activateCallback "$self setMode $b"
  }
  XmCascadeButtonGadget $self-views $M subMenuId $self-m2
  
  XmLabel $self-info $F width $width labelString { }
  XmList $self-list $F height 250 \
    browseSelectionCallback "$self setSelection %s" 
  set Btns2 [XmRowColumn wafeDebug-btns2 $F orientation horizontal]
  XmPushButton $self-show $Btns2 sensitive false \
      activateCallback {
    wafeDebugLog show [wafeDebug getMode] [wafeDebug getSelection]
  }
  XmPushButton $self-trace $Btns2 sensitive false \
      activateCallback {
    wafeDebug trace [wafeDebug getMode] [wafeDebug getSelection]
  } 
  XmLabel $self-grep $Btns2 borderWidth 0
  XmTextField $self-grepText $Btns2 valueChangedCallback "$self reDraw"
  
  sV $F workWindow $self-list messageWindow $Btns2 commandWindow $self-info

  return $sh
}

proc wafeDebug:Redraw-motif {sh selectionList selection} {
  set grepString [gV $sh*wafeDebug-grepText value]
  if [string compare "" $grepString] {
    set list ""
    foreach e $selectionList {
      if [string match *$grepString* $e] { lappend list $e }
    }
    set theList $list
  } else {
    set theList $selectionList
  }
  sV $sh*wafeDebug-list \
      items $theList itemCount [llength $theList]

  #  set currentItem [lsearch -glob $theList ?$selection]
  # pvars currentItem selection 
  #  if [string compare -1 $currentItem] {
  #    XmListSelectPos $sh*wafedebug-list $currentItem false
  #  }
}

proc wafeDebug:SetInfo-motif {sh string} {
  sV $sh*wafeDebug-info labelString $string
}

proc wafeDebugLog:Create-motif {self} {
  mergeResources topLevel *Attachment ATTACH_FORM
  set sh [TransientShell $self wafeDebug  title "Wafe Debug Log"]
  # if [window topLevel] {callback $sh popupCallback positionCursor 0}
  set width 530
  set F [XmForm  $self-form $sh unmanaged ]
  set I [XmLabel $self-info $F width $width labelString { } \
      bottomAttachment ATTACH_NONE]
  set T [XmScrolledText  $self-text $F \
      rows 20 columns 80 height 300 editMode MULTI_LINE_EDIT \
      topAttachment ATTACH_WIDGET topWidget $I bottomAttachment ATTACH_NONE ]
  XmPushButton $self-quit $F \
      activateCallback "$self dismiss" labelString "Dismiss" \
      topAttachment ATTACH_WIDGET topWidget [parent $T] \
      rightAttachment ATTACH_NONE
  manageChild $F
  return $sh
}

proc wafeDebugLog:DisplayText-motif {self sh string} {
  # puts stderr "wafeDebugLog:DisplayText-motif $sh $string"
  set w [widgetId $sh*$self-text]
  set cursorPosition [gV $w cursorPosition]
  XmTextInsert $w $cursorPosition $string
  sV $w cursorPosition [incr cursorPosition [string length $string]]
  XmTextShowPosition $w $cursorPosition
}

proc wafeDebugLog:ClearIfInvisible-motif {self sh} {
  if [string compare IsViewable [mapState $sh]] {
    sV $sh*$self-text value ""
  }
}

################### Generic wafeDebugLog commands ##########################

proc wafeDebugLog { args } {
  # the common prefix is the procedure name
  set self [lindex [info level 0] 0]

  static sh

  if [string match "" $args] {
    set method create
    set arguments {}
  } elseif [llength $args]==1 {
    set method $args
    set arguments {}
  } else {
    pset {method arguments} $args
  }
  # pvars method arguments

  switch $method {
    create {
      # if wafeDebugLog is called without arguments, 
      # it's widget structure is created if needed
      # and its shell is popped up

      if ![isWidget $self] {
	set sh [$self:Create $self]
      }
      $self:ClearIfInvisible $self $sh
      popup $sh none
    }

    displayText {
      if ![info exists sh] $self
      $self:DisplayText $self $sh [lindex $arguments 0]
    }

    traceVar {
      pset {name element op} $arguments
      # pvars name element op
      global $name
      if [string match "" $element] {
	set var $name
	if [string compare $op u] {set n [set $var]}
      } else {
	set var "${name}($element)"
	upvar #0 $var n
      }
      switch $op {
	w {$self:DisplayText $self $sh "\n### $var = <$n>\n"}
	r {$self:DisplayText $self $sh "\n### $var == <$n> was read\n"}
	u {
	  $self:DisplayText $self $sh "\n### $var was unset\n"
	  wafeDebug trace vars $var
	}
	default {
	  puts stderr "unknown operation for traceVar : <$op>"
	}
      }
    }

    show {
      $self
      #if ![info exists sh] $self
      pset {type name} $arguments
      switch $type {
	vars {
	  if [wafeDebug isTraced $type $name] {
	    # reading the variable is enough
	    global $name; set $name
	    return
	  }
	  upvar #0 $name v
	  if [catch {array size v} error] {
	    $self displayText "\n### $name = <$v>\n"
	  } else {
	    $self displayText "\n### $name is an array\n"

	    #  parray v
	    # the following lines are from parray.tcl
	    set maxl 0
	    foreach n [lsort [array names v]] {
	      if {[string length $n] > $maxl} {
		set maxl [string length $n]
	      }
	    }
	    set maxl [expr {$maxl + [string length $name] + 2}]
	    foreach n [lsort [array names v]] {
	      set nameString [format %s(%s) $name $n]
	      $self displayText [format "%-*s = %s\n" $maxl $nameString $v($n)]
	    }
	  }
	}
	procs {
	  $self displayText "\n### The Definition of Procedure $name is:\n"
	  $self displayText [wafeDebug:ProcContents $name]
	}
	commands {
	  $self displayText [wafeHelp $name returnText]\n-----------\n
	}
      }
    }

    dismiss {
      popdown $sh
      $self:DisplayText $self $sh {}
    }

  }
}


################### Generic tracing command ##########################

proc $ args {
  global wafeDebugDepth 
  set func [lindex $args 0]
  set backupDebugDepth $wafeDebugDepth 
  wafeDebug:TraceLine call [incr wafeDebugDepth] "$args"
  if [catch {set result [uplevel 1 "eval $args"]} error] {
    wafeDebug:TraceLine exit $wafeDebugDepth "$func aborted ($error)"
    set wafeDebugDepth $backupDebugDepth
    return -code error -errorinfo $error
  } else {
    wafeDebug:TraceLine exit $wafeDebugDepth "$func returns <$result>"
    incr wafeDebugDepth -1
    return $result
  }
}

proc wafeDebug:ShowStep {func newFunc args} {
  global wafeDebugDepth 
  set backupDebugDepth $wafeDebugDepth 
  wafeDebug:TraceLine call [incr wafeDebugDepth] "$func $args"
  if [catch {set result [uplevel 1 "eval $newFunc $args"]} error] {
    wafeDebug:TraceLine exit $wafeDebugDepth "$func aborted ($error)"
    set wafeDebugDepth $backupDebugDepth
    return -code error -errorinfo $error
  } else {
    wafeDebug:TraceLine exit $wafeDebugDepth "$func returns <$result>"
    incr wafeDebugDepth -1
    return $result
  }
}

proc wafeDebug:TraceLine {port depth text} {
  wafeDebugLog displayText "#$port ($depth): $text\n"
}


proc wafeDebug:ListElement {mode element} {
  global _wafeDebugTrace
#  puts stderr "looking for ($mode,$element)"
  if [info exists _wafeDebugTrace($mode,$element)] {
    return *$element
  } else {
    return \ $element
  }
}

proc wafeDebug:ProcContents {procName} {
  if [uplevel #0 info exists _wafeDebugTrace(procs,$procName)] {
    set name $procName-orig
  } else {
    set name $procName
  }
  append proc {proc } $procName \ \{ [info args $name] \}\ \{ \
      [info body $name] \} \n
  return $proc
}


# general routines, which should go into a different
# library once sufficiently debugged

proc pset {lvals rvals} {
  if {[llength $lvals] > [llength $rvals]} {
    error "pset argument list does not match value list ({$lvals}, {$rvals})"
  }
  set i 0
  foreach left $lvals {
    set val [lindex $rvals $i]
    uplevel 1 [list set $left $val]
    incr i
  }
  incr i -1
  uplevel 1 [list set $left [lrange $rvals $i end]]
}

proc static {args} {
  set procName [lindex [info level -1] 0]
  foreach varName $args {
    uplevel 1 "upvar #0 _staticvars($procName:$varName) $varName"
  }
}

proc pvars {args} {
  foreach varName $args {
    puts stderr [concat \
	[uplevel 1 {lindex [info level 0] 0}]:\ \
	[uplevel 1 "concat $varName = <$$varName>"]]
  }
}

#proc backtrace {} {
#  for {set l [info level]} { $l>0 } { incr l -1 } {
#    puts stderr "$l: [info level $l]"
#  }
#}


###### test procedures for tracing
#
#proc a {} {
#  set x 1
#  $ b $x 2
#  puts stderr "c returns [c $x]"
#  puts stderr "y=$y"
#}
#
#proc b {x y} {
#  return [expr $x+$y]
#}
#
#proc c {x} {
#  uplevel 1 set y 2
#  return [expr $x+$x]
#}
#

proc wafeDebug:Init {} {
  if [string compare "" [info command Form]] {
    set widgetSet athena
  } else {
    set widgetSet motif
  }
  foreach cmd {
    wafeDebug:Create
    wafeDebug:Redraw 
    wafeDebug:SetInfo
    wafeDebugLog:Create
    wafeDebugLog:DisplayText
    wafeDebugLog:ClearIfInvisible
  } { alias $cmd $cmd-$widgetSet }
  # wafeDebug
}

wafeDebug:Init


