#
# PanedWindow
# ----------------------------------------------------------------------
# Implements a paned widnow widget using primitive widgets as the
# building blocks.  The PanedWindow creates the top and bottom frames,
# and the pane itself.  The user may use the top and bottom methods to
# retrieve these frames and add children to them as needed.
#
# The base logic of this class is from a post to comp.lang.tcl from
# Jay Schmidgall.  I added additional features and made it a class.
#
#   PUBLIC ATTRIBUTES:
#
#     -fraction ...... initial percentage of sash
#     -low ........... minimum percentage of sash
#     -high .......... maximum percentage of sash
#     -width ......... width of displayed list in characters
#     -height ........ height of displayed list in lines
#
#   METHODS:
#
#     config ....... used to change public attributes
#     top .......... return the top pane
#     bottom ....... return the bottom pane
#
#   USAGE:
#
#     PanedWindow .pw -fraction .6 -width 600 -height 300 -high .8 -low .2
#     set top [.pw top]
#     set bot [.pw bottom]
#
#     button $top.b1 -text "Hello"
#     pack $top.b1 -fill both -expand yes -padx 10 -pady 10
#     button $bot.b2 -text "World"
#     pack $bot.b2 -fill both -expand yes -padx 10 -pady 10
#     pack .pw -fill both -expand yes 
#     wm minsize . 0 0
#  
#   X11 OPTION DATABASE ATTRIBUTES
#
#     ...and the rest of the usual widget attributes
#
#   ACKNOWLEDGEMENTS:
#     James Noble - 1993
#     Jay Schmidgall - 1994
#
# ----------------------------------------------------------------------
#   AUTHOR:  Mark L. Ulferts          Phone: (214) 519-3947
#            DSC Communications Corp  E-mail: mulferts@spd.dsccc.com
# ----------------------------------------------------------------------


itcl_class PanedWindow {
    # ------------------------------------------------------------------
    #  CONSTRUCTOR - create new paned window
    # ------------------------------------------------------------------
    constructor {config} {
	set class [$this info class]
	::rename $this $this-tmp-
	::frame $this -class $class
	::rename $this $this-win-
	::rename $this-tmp- $this

	frame $this.pw
	#
	# Make the sash button
	#
	frame $this.f -geometry 10x10 -borderwidth 2 -relief raised  \
		-cursor crosshair
	bind $this.f <Button-1> "$this start-grip %y"
	bind $this.f <B1-Motion> "$this handle-grip %y"
	bind $this.f <B1-ButtonRelease-1> "$this end-grip %y"

	#
	# Make the separator
	#
        frame $this.sep -height 2 -width 2 -borderwidth 1 -relief sunken

	#
	# Make the windows
	#
	frame $this.top -borderwidth 2 -relief raised
	frame $this.bot -borderwidth 2 -relief raised

	pack $this.pw -fill both -expand yes -anchor w

        replace

	#
	#  Explicitly handle configs that may have been ignored earlier
	#
	foreach attr $config {
	    config -$attr [set $attr]
	}

    }

    # ------------------------------------------------------------------
    #  METHOD:  config - used to change public attributes
    # ------------------------------------------------------------------
    method config {config} {}

    # ------------------------------------------------------------------
    #  DESTRUCTOR - destroy window containing widget
    # ------------------------------------------------------------------
    destructor {
	::rename $this-win- {}
	destroy $this
    }

    # ------------------------------------------------------------------
    #  METHOD calc-fraction - determines the fraction for the sash
    # ------------------------------------------------------------------
    method calc-fraction {where} {
	set fraction \
		[expr "(($where.0 - $drag_start.0) / [winfo height $this]) + \
		$fraction"]

	if {$fraction < $min} {
	    set fraction $min
	} elseif {$fraction > $max} {
	    set fraction $max
	}
    }

    # ------------------------------------------------------------------
    #  METHOD start-grip - Starts the sash drag and drop operation
    # ------------------------------------------------------------------
    method start-grip {where} {
	grab $this.f
	raise $this.sep
	raise $this.f
	$this.f configure -relief sunken
	set drag_start $where
    }

    # ------------------------------------------------------------------
    #  METHOD end-grip - Ends the sash drag and drop operation
    # ------------------------------------------------------------------
    method end-grip {where} {
	calc-fraction $where
	$this.f configure -relief raised
	grab release $this.f
	replace
    }

    # ------------------------------------------------------------------
    #  METHOD handler-grip - Motion action for sash
    # ------------------------------------------------------------------
    method handle-grip {where} {
	calc-fraction $where
	place $this.sep -in $this.pw -relx 0 -relwidth 1 \
		-rely $fraction -anchor w
	place $this.f -in $this.pw -relx .95 \
		-rely $fraction -anchor center
    }

    # ------------------------------------------------------------------
    #  METHOD replace - Resets the panes of the window following 
    #                   movement of the sash
    # ------------------------------------------------------------------
    method replace {} {
	place $this.sep -in $this.pw -relx 0 -relwidth 1 \
		-rely $fraction -anchor w
	place $this.f -in $this.pw -relx .95 -rely $fraction -anchor center
	place $this.top -in $this.pw -x 0 -y 0 -relwidth 1 -relheight $fraction
	place $this.bot -in $this.pw -x 0 -rely $fraction \
		-relwidth 1 -relheight [expr "1.0 - $fraction"]
	lower $this.sep
	raise $this.f
    }

    # ------------------------------------------------------------------
    #  METHOD top - Return the top pane
    # ------------------------------------------------------------------
    method top {} {
	return $this.top
    }

    # ------------------------------------------------------------------
    #  METHOD bottom - Return the bottom pane
    # ------------------------------------------------------------------
    method bottom {} {
	return $this.bot
    }

    # ------------------------------------------------------------------
    #  ATTRIBUTE low - Minimum percentage sash
    # ------------------------------------------------------------------
    public low 0.1 {
	if {[winfo exists $this]} {
	    if {[expr $low > 0.0 && $low < 1.0]} {
		set max [expr 1.0 - $low]
		set drag_start [winfo y $this.f]
		calc-fraction [winfo y $this.f]
		replace
	    }
	}
    }

    # ------------------------------------------------------------------
    #  ATTRIBUTE high - Maximum percentage of sash
    # ------------------------------------------------------------------
    public high 0.9 {
	if {[winfo exists $this]} {
	    if {[expr $high > 0.0 && $high < 1.0]} {
		set min [expr 1.0 - $high]
		set drag_start [winfo y $this.f]
		calc-fraction [winfo y $this.f]
		replace
	    }
	}
    }

    # ------------------------------------------------------------------
    #  ATTRIBUTE fraction - Initial percentage of visible area
    # ------------------------------------------------------------------
    public fraction .5 {
	if {[winfo exists $this]} {
	    if {[expr $fraction > $min && $fraction < $max]} {
		set drag_start [winfo y $this.f]
		calc-fraction [winfo y $this.f]
		replace
	    }
	}
    }

    # ------------------------------------------------------------------
    #  ATTRIBUTE width - Set the frame width
    # ------------------------------------------------------------------
    public width 50 {
	if {[winfo exists $this]} {
	    $this.pw config -width $width
	}
    }

    # ------------------------------------------------------------------
    #  ATTRIBUTE height - Set the frame height
    # ------------------------------------------------------------------
    public height 50 {
	if {[winfo exists $this]} {
	    $this.pw config -height $height
	}
    }

    protected max 0.9
    protected min 0.1
    protected drag_start 0
}
