;;;
;;; Routines for handling Lisp-to-Wish communication
;;;

(in-package 'user)

;; Variable to hold stream used to communicate with Wish
(defvar *lisp-to-wish-stream*)

;;;;;;;;;;;;;;;;;;;;;;;;;
;; Start the Wish process

(defun init-interface ()
  (setq *lisp-to-wish-stream*  (#+:lucid   run-program
				#+:allegro run-shell-command
				   "test.tcl"
				   :input :stream
				   :output :stream
				   :wait nil)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Handle interaction between program and Wish
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun interface-interact ()
  (let (fn)

    ;; Anything you want to tell Wish before it starts waiting for
    ;; user input should go here

    ;; Tell Wish we're done initializing windows, so start up
    (finish-lisp-broadcast)

    ;; Main processing loop.  Get a code from Wish, look it up in our hash
    ;; table and dispatch the function associated with that code.  When the
    ;; function is done, return control to Wish.

    (do ((wish-read (read-line *lisp-to-wish-stream*)
		    (read-line *lisp-to-wish-stream*)))
	;; Let user exit without closing connection
	((equal wish-read "exit") (close-wish-stream))

	(setq fn (gethash wish-read *code-to-func*))
	(if fn 
	    (funcall fn)
	  (format t "No procedure defined for ~S~%" wish-read))
	(finish-lisp-broadcast))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Function to read a line from Wish.  Wish must send first a code
;; indicating what it's returning.  The caller of read-from-wish gives
;; an "accept-code" to filter anything Wish sends that might not be
;; relevant to the information the caller desires from the user.

;; I use this routine to exchange information between Lisp and Wish without
;; having to wait for the user to do something.

(defun read-from-wish (accept-code)

  (let ((ans (read-line *lisp-to-wish-stream*)))

    (if accept-code
	(progn
	  ;; First get acknowledgement code
	  (loop until (equalp ans accept-code)
		do
		(finish-lisp-broadcast)
		(setf ans (read-line *lisp-to-wish-stream*)))

	  ;; Now get the info
	  (read-line *lisp-to-wish-stream*))
      ans)))
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Example routine to call a procedure defined in test.tcl
;;
;; Give the user some message

(defun display-message (&rest args)
  (tellwish (concatenate 'string "{telluser .message {"
			 (apply #'format (cons nil args))
			 "}}")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Protocol for communication with the Wish process 
;;
;;
;; Lisp waits for the Wish process to send a code.  All codes have been
;; registered in *code-to-func*, identifying which action the user
;; requested.  When Lisp sees that code, it goes through a 2 step
;; process:
;; 
;; 1. Invoke the function in the hash table corresponding to the code
;; received
;; 
;;  2. Call "finish-lisp-broadcast" sending Wish the end-of-transmission
;; symbol.  Wish knows then to take control again and wait for the user's
;; next action.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Hash table mapping codes received from Wish to functions.  Codes are
;; case-sensitive, so we use "equal" instead of "eq".
(defparameter *code-to-func*  (make-hash-table :test #'equal :size 100))


;; Given a code that Wish will send and a Lisp function, this function
;; registers the code with the function so when Wish sends the
;; code, the dispatcher calls the function.

(defun wish-code-register (code function)
  (setf (gethash code *code-to-func*) function))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; UTILITY FUNCTIONS

;; Send "format"-type string to Wish

(defmacro tellwish (&rest args)
  `(progn
     (format *lisp-to-wish-stream* ,(car args) ,@(cdr args))
     (terpri *lisp-to-wish-stream*)
     (force-output *lisp-to-wish-stream*)))

;; Send end-of-transmission symbol, giving control back to Wish
(defmacro finish-lisp-broadcast ()
  '(tellwish "@"))

;; Close the stream to Wish
(defun close-wish-stream ()
  (finish-lisp-broadcast)  ;; When we're done shutting down
  (close *lisp-to-wish-stream*))
