
// Simple test case for the objectify program.  Two objects are defined -- Foo,
// which has a few slots and methods, and Bar, which is a widget that doesn't
// do anything.  objectify will read this file and produce a file called
// test_objects.cc

#include "tcl.h"
#include "tk.h"

// This file is required to be able to interpret the TCL_ macros correctly for
// C++.

#include "objectify.h"

class TCL_OBJECT("foo", Foo, object,
		  "This is a foo object.  It has no particular value in life.") {
  public:
    Foo(Tcl_Interp* interp) { }
    ~Foo() { }
    
    TCL_SLOT(int, i, "17", "A random integer value");
    TCL_SLOT(double, f, "13.31", "A random floating-point value");
    
    TCL_METHOD("printall", PrintAllCmd(Tcl_Interp* interp),
	       "Print some useful things") {
	char buf[64];
	sprintf(buf, "-i %d -f %f diff %f", i, f, d);
	Tcl_AppendResult(interp, buf, 0);
	return (TCL_OK);
    }
    TCL_METHOD("echo", EchoCmd(Tcl_Interp* interp, int ac, char** av),
	       "Print arguments");
 
    int AfterConfigure(Tcl_Interp* interp);
    int BeforeDelete(Tcl_Interp* interp) {
	printf("before delete foo\n");
	return (TCL_OK);
    }
    
    float d;
};

// These are two random types that are used below to illustrate how to
// use user-defined types with objectify.

struct Fruitcake { int f; };
struct Frogs { int f; };

class TCL_OBJECT("baz", Baz, object(Foo), "This is an object type that inherits from Foo") : public Foo {
  public:
    Baz(Tcl_Interp* interp) : Foo(interp) { }
    ~Baz() { }
    
    TCL_SLOT_FULL(INT, int, c, "-c", "c", "C", "13", 0, NULL,
		  "", "This is an example of the full slot declaration.");
        
    // This method takes 3 args: a fruitcake pointer, an int, and a frog.
    // Note that the interp can be in any position in the arg list.
    TCL_METHOD("method1", Method1Cmd(Fruitcake* ff, Tcl_Interp* interp,
				     int a, Frogs fr), "") {
	char buf[256];
	sprintf(buf, "fruitcake pointer = 0x%lx, a = %d, fr.f = %d",
		(long) ff, a, fr.f);
	Tcl_AppendResult(interp, buf, 0);
	return (TCL_OK);
    }
    
    int AfterConfigure(Tcl_Interp* interp) {
	printf("baz configure\n");
	return (TCL_OK);
    }
    int BeforeDelete(Tcl_Interp* interp) {
	printf("baz delete\n");
	return (TCL_OK);
    }
};

struct TCL_OBJECT("bar", Bar, widget, "This is a bar widget") {
    Bar(Tcl_Interp* interp, Tk_Window win) {
	tkwin = win;
	bgBorder = NULL;
	bgColor = NULL;
	cursor = None;
	redraw_pending = 0;
    }
    ~Bar() { }
    
    TCL_SLOT_FULL(BORDER, Tk_3DBorder, bgBorder, "-bg", "bg", "Bg", "black", 0, NULL,
		  "color", "Color of the border");
    TCL_SLOT_FULL(COLOR,  XColor*, bgColor, NULL, NULL, NULL, NULL, 0, NULL, "", "");
    TCL_SLOT_FULL(RELIEF, int, relief, "-relief", "relief", "Relief", "flat", 0,
		  NULL, "relief-value", "The relief to use for the object");
    TCL_SLOT_FULL(PIXELS, int, borderwidth, "-bw", "bw", "Bw", "0", 0, NULL, "pixels",
		  "Width of the border");
    TCL_SLOT_FULL(PIXELS, int, height, "-height", "height", "Height",
		  "100", 0, NULL, "pixels", "Height of the window");
    TCL_SLOT_FULL(PIXELS, int, width, "-width", "width", "Width",
		  "100", 0, NULL, "pixels", "Width of the window");
    TCL_SLOT_FULL(ACTIVE_CURSOR, Cursor, cursor, "-cursor", "cursor", "Cursor",
		  NULL, TK_CONFIG_NULL_OK, NULL, "cursor spec",
		  "Cursor to use in the window");
    
    int AfterConfigure(Tcl_Interp* interp);
    int BeforeDelete(Tcl_Interp* interp) {
	printf("before delete bar\n");
	return (TCL_OK);
    }
        
    Tk_Window tkwin;
    int redraw_pending;

    void Draw();
};

