;;; -*- Syntax: Common-lisp; Package: USER -*-

;;; Hairy index constructor for Scheme report.

;;; (index-entry "SORTKEY" "HEADCS" TYPE PAGENUMBER)
;;;   TYPE = main or aux

(defmacro iterate (tag specs &body body)
  (let ((vars (mapcar #'car specs))
	(id (gensym)))
    `(block ,id
       (let ,specs
	 (tagbody
	   ,id (macrolet ((,tag ,vars
			   `(progn (psetq ,@(list ,@(mapcan #'(lisp:lambda (var)
								`(',var ,var))
							    vars)))
				   (go ,',id))))
		 (return-from ,id (progn ,@body))))))))

(defun make-index ()
  (write-index (read-index "/zu/arthur/this/manual.idx")
	       "/zu/arthur/this/theindex.tex"))

(defvar foo-readtable
	(let ((*readtable* (copy-readtable nil)))
	  (set-syntax-from-char #\\ #\/)
	  *readtable*))

(defstruct pageref
  key						;sort key
  headcs 					;heading control seq
  type						;main or aux
  pagenum)					;page number

(defvar foo-package *package*)

(defun read-index (infile)
  (format t "~%Reading...")
  (with-open-file (in infile :direction :in)
    (let ((*readtable* foo-readtable)
	  (*package* foo-package))		;G.M.O.
      (do ((form (read in nil nil) (read in nil nil))
	   (l '() (cons (let* ((form (cdr form))
			       (sortkey (car form))
			       (headcs (cadr form))
			       (type (caddr form))
			       (pagenum (cadddr form)))
			  (make-pageref :key (massage-sort-key sortkey)
					:headcs headcs
					:type type	;main or aux
					:pagenum pagenum))
			l)))
	  ((null form)
	   (format t "~%~a nominal entries" (length l))
	   l)))))

(defun massage-sort-key (sortkey)
  (let* ((sortkey (string-downcase sortkey))
	 (magic "\\discretionary {->}{}{->}{}")
	 (probe (search magic sortkey)))
    (cond (probe (concatenate 'string
			      (subseq sortkey 0 probe)
			      "->"
			      (subseq sortkey (+ probe (length magic)))))
	  (t sortkey))))

(defun write-index (entries outfile)
  (let ((entries (delete-duplicates (sort entries #'entry-lessp)
				    :test #'entry-equalp)))
    (format t "~%~a entries after duplicates removed" (length entries))
    (with-open-file (out outfile :direction :out)
      (iterate loop ((z entries)
		     (prev nil)
		     (count 0))
	(cond ((null z)
	       (format t "~%~a items written" count))
	      (t
	       (let ((p (1+ (or (position (car z) (cdr z)
					  :test-not #'(lambda (e1 e2)
							(and (string= (pageref-key e1)
								      (pageref-key e2))
							     (string= (pageref-headcs e1)
								      (pageref-headcs e2)))))
				(length (cdr z))))))
		 (when (and prev
			    (alpha-char-p (aref (pageref-key (car z)) 0))
			    (not (char-equal (aref (pageref-key (car z)) 0)
					     (aref (pageref-key prev) 0))))
		   (format out "\\indexspace~%"))
		 (write-index-item (subseq z 0 p) out)
		 (loop (nthcdr p z)
		       (car z)
		       (1+ count)))))))))

(defun write-index-item (these out)
  (let* ((e (car these))
	 (q (or (position 'main these :key #'pageref-type)
		(length these)))
	 (aux-pages (mapcar #'pageref-pagenum (subseq these 0 q)))
	 (main-pages (mapcar #'pageref-pagenum (nthcdr q these))))
    (if (> (length main-pages) 1)
	(format t "~% *** ~a has multiple main entries" (pageref-key e)))
    (format out "\\item{\\~a{~a}}{\\hskip .75em}" (pageref-headcs e) (pageref-key e))
    (write-comma-separated-list main-pages out)
    (if (and (not (null main-pages)) (not (null aux-pages)))
	(princ "; " out))
    (write-comma-separated-list aux-pages out)
    (terpri out)))

(defun write-comma-separated-list (l out)
  (if (not (null l))
      (do ((l l (cdr l)))
	  ((null (cdr l))
	   (prin1 (car l) out))
	(prin1 (car l) out)
	(princ ", " out))))

;;; Make sure main entries come last

(defun entry-lessp (e1 e2)
  (or (string-lessp (pageref-key e1) (pageref-key e2))
      (and (string-equal (pageref-key e1) (pageref-key e2))
	   (or (string-lessp (pageref-headcs e1) (pageref-headcs e2))
	       (and (string-equal (pageref-headcs e1) (pageref-headcs e2))
		    (or (and (eq (pageref-type e1) 'aux)
			     (eq (pageref-type e2) 'main))
			(and (eq (pageref-type e1) (pageref-type e2))
			     (< (pageref-pagenum e1) (pageref-pagenum e2)))))))))

(defun entry-equalp (e1 e2)
  (and (string-equal (pageref-key e1) (pageref-key e2))
       (string-equal (pageref-headcs e1) (pageref-headcs e2))
       (= (pageref-pagenum e1) (pageref-pagenum e2))))
