#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2015 -- leonerd@leonerd.org.uk

package Device::Chip;

use strict;
use warnings;

our $VERSION = '0.01';

use Carp;

use Future 0.26; # ->done / ->fail as constructor

=head1 NAME

C<Device::Chip> - an abstraction of a hardware chip IO driver

=head1 DESCRIPTION

=over 2

B<Note>: this document is currently under heavy development. Details will be
added, changed, and evolved as it progresses. Be warned that currently
anything may be changed from one version to the next.

=back

This package describes an interface that classes can use to implement a driver
to talk to a specific hardware chip or module. An instance implementing this
interface would communicate with the actual hardware device via some instance
of the related interface, L<Device::Chip::Adapter>.

It is suggested that a driver for a particular hardware chip or module
provides a concrete class named within the C<Device::Chip> heirarchy, adding
the basic name of the chip or module as a suffix; for example the driver for
a I<Maxim> I<MAX7219> LED driver would be called:

   package Device::Chip::MAX7219;

This package provides a base class that such a specific implementation class
could use as a superclass, but it is not required to. The important detail is
that it provides the interface described by this documentation.

=head2 USING A CHIP DRIVER

To actually use a chip driver to talk to a specific piece of hardware that is
connected to the computer, an adapter must be supplied. This will be an
instance of some class that satisfies the L<Device::Chip::Adapter> interface.
The chip driver will use this adapter instance to access the underlying
hardware port used to electrically connect to the chip and communicate with
it. This is supplied by invoking the L</connect> method. For example:

   my $chip = Device::Chip::MAX7219->new;

   my $adapter = Device::Chip::Adapter::FTDI->new;

   $chip->connect( $adapter )->get;

=cut

=head1 CONSTRUCTOR

=cut

=head2 new

   $chip = Device::Chip->new

Returns a new instance of a chip driver object.

=cut

sub new
{
   my $class = shift;
   bless {}, $class;
}

=head1 METHODS

The following methods documented with a trailing call to C<< ->get >> return
L<Future> instances.

This allows them to easily be used as a simple synchronous method by using the
trailing L<Future/get> call. Alternatively, if the underlying adapter allows a
fully asynchronous mode of operation, they can be combined in the usual ways
for futures to provide more asynchronous use of the device.

=cut

=head2 adapter

   $adapter = $chip->adapter

Returns the current adapter that the chip is connected to. This will be some
instance implementing the companion interface, L<Device::Chip::Adapter>.

This method is primarily used by the concrete driver subclass; it is unlikely
to be useful to the containing application using the chip.

=cut

sub adapter
{
   my $self = shift;
   return $self->{adapter} //
      croak "This chip has not yet been connected to an adapter";
}

=head2 protocol

   $protocol = $chip->protocol

Returns the adapter protocol module that the chip driver code will use to
actually communicate with the chip.

This method is primarily used by the concrete driver subclass; it is unlikely
to be useful to the containing application using the chip.

=cut

sub protocol
{
   my $self = shift;
   return $self->{protocol} //
      croak "This chip has not yet been connected to a protocol";
}

=head2 connect

   $chip = $chip->connect( $adapter )->get

Supplies the chip driver with the means to actually communicate with the
connected device, via some electrical interface connected to the computer.

=cut

sub connect
{
   my $self = shift;
   ( $self->{adapter} ) = @_;

   $self->{adapter}->make_protocol( $self->PROTOCOL )
      ->on_done( sub { ( $self->{protocol} ) = @_ } )
      ->then_done( $self );
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
