package Term::Table2;

use v5.14;
use warnings FATAL => qw(all);

use List::Util       qw(max min);
use Module::Load     qw(load);
use Params::Validate qw(ARRAYREF CODEREF SCALAR validate);
use POSIX            qw(floor ceil);
use Term::ReadKey    qw(GetTerminalSize);

use Class::XSAccessor
  getters => {
    map { $_ => $_; }
        qw(
          broadColumn
          broadHeader
          broadRow
          collapse
          columnWidth
          currentRow
          endOfTable
          header
          pad
          pageHeight
          rows
          separateRows
          tableWidth
        )
  };

sub _overrideLength {                                       # Consider wide Unicode characters if possible i.e.
  no warnings qw(redefine once);                            # if Unicode::GCString can be used.
  use subs qw(length);
  eval { load('Unicode::GCString') };                       # Otherwise table content can be twisted
  *length = $@ ? sub { return CORE::length($_[0]) } : sub { return Unicode::GCString->new($_[0])->columns() };
  return;
}

BEGIN { _overrideLength() }

use constant {                                              # Boolean values
  FALSE => 0,
  TRUE  => 1,
};
use constant {                                              # Table flags
  ADJUST  => 0,
  CUT     => 0,
  SPLIT   => 1,
  WRAP    => 2,
};
use constant {                                              # Integer that hopefully can never be exceeded
  BIG_INT => ~0,
};
use constant {                                              # Valid option combinations in form required by
  ALL_OPTIONS => {                                          # Params::Validate::validate
    header       => {
      default     => [],
      optional    => 1,
    },
    rows         => {
      default     => [],
      optional    => 1,
    },
    broadColumn  => {
      default     => [WRAP],
      optional    => 1,
    },
    broadHeader  => {
      default     => [WRAP],
      optional    => 1,
    },
    broadRow     => {
      default     => WRAP,
      optional    => 1,
    },
    collapse     => {
      default     => [FALSE],
      optional    => 1,
    },
    columnWidth  => {
      default     => [ADJUST],
      optional    => 1,
    },
    pad          => {
      default     => 1,
      optional    => 1,
    },
    pageHeight   => {
      default     => \&_screenHeight,
      optional    => 1,
    },
    separateRows => {
      default     => FALSE,
      optional    => 1,
    },
    tableWidth   => {
      default     => \&_screenWidth,
      optional    => 1,
    },
  },
};
use constant {
  OPTIONS_ARRAY => {
    %{ALL_OPTIONS()},
    rows        => {
      type        => ARRAYREF,
      callbacks   => {
        q('rows' element is an array reference)                                   => \&_isEachRowArray,
        q(all 'rows' elements have same length)                                   => \&_areAllRowsOfEqualLength,
        q('rows' elements contain defined scalars only)                           => \&_isEachCellScalar,
      },
    },
    broadRow    => {
      type        => SCALAR,
      optional    => 1,
      callbacks   => { q('broadRow' is either 'CUT', or 'SPLIT', or 'WRAP')       => \&_isCutOrSplitOrWrap },
    },
    collapse    => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
    },
    columnWidth => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('columnWidth' is undefined or a positive integer)        => \&_isEachColumnWidthUndefOrInt },
    },
  },
  OPTIONS_CALLBACK => {
    %{ALL_OPTIONS()},
    broadRow    => {
      type        => SCALAR,
      optional    => 1,
      callbacks   => { q('broadRow' is either 'CUT' or 'WRAP')                    => \&_isCutOrWrap },
    },
    columnWidth => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('columnWidth' is a positive integer)                     => \&_isEachColumnWidthInt },
    },
  },
  OPTIONS_GENERAL => {
    %{ALL_OPTIONS()},
    header       => {
      type        => ARRAYREF,
      optional    => 1,
      callbacks   => { q('header' element is a scalar)                            => \&_isScalar },
    },
    broadColumn  => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('broadColumn' is / contains either 'CUT' or 'WRAP' only) => \&_isEachColumnFlagCutOrWrap },
    },
    broadHeader  => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('broadHeader' is / contains either 'CUT' or 'WRAP' only) => \&_isEachColumnFlagCutOrWrap },
    },
    columnWidth  => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('columnWidth' is a positive integer)                     => \&_isEachColumnWidthInt },
    },
    pad          => {
      type        => ARRAYREF | SCALAR,
      optional    => 1,
      callbacks   => { q('pad' is undefined or a positive integer)                => \&_isUndefOrInt },
    },
    pageHeight   => {
      type        => SCALAR,
      optional    => 1,
      callbacks   => { q('pageHeight' is undefined or a positive integer)         => \&_isUndefOrInt },
    },
    separateRows => {
      type        => SCALAR,
      optional    => 1,
    },
    tableWidth   => {
      type        => SCALAR,
      optional    => 1,
      callbacks   => { q('tableWidth' is undefined or a positive integer)         => \&_isUndefOrInt },
    },
  },
};

use Exporter qw(import);
our @EXPORT_OK = qw(ADJUST CUT SPLIT WRAP);

our $VERSION = '1.0.0';

sub fetch {                                                 # Provides current line
  my ($self) = @_;

  return if $self->{'endOfTable'} && !@{$self->{':rowLines'}};

  $self->_getNextLines() unless @{$self->{':rowLines'}} && $self->{':lineOnPage'};
  $self->{':lineOnPage'}++;
  $self->{':lineOnPage'} = 0 if $self->{':lineOnPage'} == $self->{':linesPerPage'};

  my $row = shift(@{$self->{':rowLines'}});
  $self->{':rowBuffer'} = [] unless @{$self->{':rowLines'}};

  return $_ = $row;
}

sub fetch_all {                                             # Provides all table lines at once
  my ($self) = @_;

  my @lines;
  push(@lines, $_) while $self->fetch();

  return \@lines;
}

sub new {                                                   # Instantiate table object
  my ($class, @params) = @_;

  my $self = bless(
    {
      ':endOfChunk'      => FALSE,                          # End of vertical chunk in case of horizontal splitting
      ':headerLines'     => [],                             # One array element per header line
      ':lineFormat'      => '|',                            # Row line format
      ':lineOnPage'      => 0,                              # Line number relative to the current page
      ':linesPerPage'    => 1,                              # Considers multiple lines per page depending on 'broadRow'
      ':linesPerRow'     => 1,                              # Considers multiple lines per row depending on 'broadRow'
      ':numberOfColumns' => undef,                          # Number of columns as supplied via 'rows' array
      ':rowBuffer'       => [],                             # Current row
      ':rowLines'        => [],                             # One array element per row line
      ':separatingAdded' => FALSE,                          # Separating line is among ':rowLines'
      ':separatingLine'  => '+',                            # Line separating table / header content
      ':splitOffset'     => 0,                              # Horizontal offset from the table left side
      ':totalWidth'      => 0,                              # Table width independently of possible horizontal splitting
      'currentRow'       => 0,
      'endOfTable'       => FALSE,                          # End of table in general (end of last chunk)
    },
    $class,
  )->_validate(\@params);

  return ref($self->{'rows'}) eq 'ARRAY' ? $self->_setDefaults()->_init()
                                         : $self->_copyOptions(OPTIONS_CALLBACK, \@params);
}

sub _areAllRowsOfEqualLength {
  my ($rows) = @_;

  return !grep { @$_ != @{$rows->[0]} } @$rows;
}

sub _copyOptions {                                          # Takes over values of required options into object
  my ($self, $options, $params) = @_;

  my %params = @$params;

  $self //= {};
  foreach my $option (keys(%$options)) {
    $self->{$option} = $params{$option} if exists($params{$option});
  }

  return $self;
}

sub _cutOrWrapLine {
  my ($self, $line) = @_;

  $line = substr($line, $self->{':splitOffset'}) if $self->{':splitOffset'};

  my $lineTooLong = length($line) > $self->{'tableWidth'};;

  return unpack('(A' . $self->{'tableWidth'} . ')*', $line) # Wrap is required and line is long enough to be wrapped
    if $self->{'broadRow'} == WRAP && $lineTooLong;

  return substr($line, 0, $self->{'tableWidth'})            # Wrap is not required and line is too long
    if $lineTooLong;

  return $line;                                             # Line is too short for any change
}

sub _extractLine {                                          # Extract 1st remaining line from current table row
  my ($self, $row, $broadFlags) = @_;

  my @line;

  foreach my $columnNo (0 .. $self->{':numberOfColumns'} - 1) {
    my $columnWidth   = $self->{'columnWidth'}[$columnNo];
    my $field         = $row->[$columnNo];

    $row->[$columnNo] = do {
      if (length($field) > $columnWidth) {
        push(@line, substr($field, 0, $columnWidth));
        $broadFlags->[$columnNo] == CUT ? '' : substr($field, $columnWidth);
      }
      else {
        push(@line, $field);
        '';
      }
    };
  }

  return \@line;
}

sub _extractLines {                                         # Converts table row to array of output cell arrays
  my ($self, $row, $broadFlags) = @_;

  my @row = @$row;

  my @lines;
  if (@row) {
    do {push(@lines, $self->_extractLine(\@row, $broadFlags))} while grep { $_ ne '' } @row;
  }

  return \@lines;
}

sub _getNextLines {                                         # Provides next lines from the current row
  my ($self) = @_;

  if (!@{$self->{':rowLines'}} && @{$self->_getNextRow()}) {
    push(@{$self->{':rowLines'}},
         map { $self->_cutOrWrapLine($_) } @{$self->_prepareRow($self->{':rowBuffer'}, $self->{'broadColumn'})});
    $self->{':separatingAdded'} = FALSE;
  }

  my $headerAdded;
  if ((ref($self->{'rows'}) eq 'ARRAY' && $self->{'currentRow'} == 1 || !$self->{':lineOnPage'})
  &&  !$self->{'endOfTable'}) {
    unshift(@{$self->{':rowLines'}}, map { $self->_cutOrWrapLine($_) } @{$self->{':headerLines'}});
    $headerAdded = TRUE;
  }

  if (($self->{':endOfChunk'}                               # Ends up the table or separates two rows if required
  ||   $self->{'separateRows'} && $self->{':lineOnPage'} + @{$self->{':rowLines'}} < $self->{'pageHeight'} - 1)
  && !$self->{':separatingAdded'}) {
    push(@{$self->{':rowLines'}}, $self->_cutOrWrapLine($self->{':separatingLine'}));
    $self->{':separatingAdded'} = TRUE;
  }

  return;
}

sub _getNextRow {                                           # Takes over next row
  my ($self) = @_;

  if ($self->{'endOfTable'}                                 # End of table already reached or being reached just now
  ||  ref($self->{'rows'}) eq 'ARRAY' && !$self->_getNextRowFromArray()
  ||  ref($self->{'rows'}) eq 'CODE'  && !$self->_getNextRowFromCallback()) {
    $self->{':rowBuffer'} = [];
  }
  else {
    $self->{'currentRow'}++;
  }

  return $self->{':rowBuffer'};
}

sub _getNextRowFromArray {                                  # Takes over next row from array
  my ($self) = @_;

  my $currentRow = $self->{'currentRow'};

  $self->{':endOfChunk'} = $currentRow == $#{$self->{'rows'}};

  if ($currentRow > $#{$self->{'rows'}}) {
    if ($self->{'broadRow'} != SPLIT || $self->{':splitOffset'} + $self->{'tableWidth'} >= $self->{':totalWidth'}) {
      $self->{'endOfTable'} = TRUE;
      return FALSE;
    }

    $self->{'currentRow'}    = $currentRow = 0;
    $self->{':splitOffset'} += $self->{'tableWidth'};
  }

  $self->{':rowBuffer'} = $self->{'rows'}[$currentRow];
  $self->{'endOfTable'} = FALSE;

  return TRUE;
}

sub _getNextRowFromCallback {                               # Takes over next row delivered by callback function
  my ($self) = @_;

  my $row = &{$self->{'rows'}};

  unless (defined($self->{':numberOfColumns'})) {
    $self->{':numberOfColumns'} = ref($row) eq 'ARRAY' ? @$row : 0;
    die("Row $self->{'currentRow'}: not an array reference") if ref($row) ne 'ARRAY';
    $self->_validateForCallback()->_setDefaults()->_init();
  }

  unless (defined($row)) {
    $self->{':endOfChunk'} = $self->{'endOfTable'} = TRUE;
    return FALSE;
  }

  my $numberOfColumns = $self->{':numberOfColumns'};

  die("Row $self->{'currentRow'}: not an array reference") if ref($row) ne 'ARRAY';
  die("Row $self->{'currentRow'}: wrong number of elements (", scalar(@$row), " instead of $numberOfColumns)")
    if scalar(@$row) != $numberOfColumns;
  foreach (0 .. $numberOfColumns - 1) {
    die("Row $self->{'currentRow'}: element No. $_ is ", ref($row->[$_]), ' not a scalar') if ref($row->[$_]);
  }

  $self->{':rowBuffer'} = _stripTrailingBlanks($row);

  return TRUE;
}

sub _init {                                                 # Set remaining attributes during object instantiating
  my ($self) = @_;

  $self->{'pageHeight'} ||= BIG_INT;
  $self->{'tableWidth'} ||= BIG_INT;

  $self->{'pad'} = [($self->{'pad'}) x max(scalar(@{$self->{'columnWidth'}}), 1)] unless ref($self->{'pad'});

  die(
    "Table width ($self->{'tableWidth'}) is lower than the width of the narrowest possible column i.e. 1 character, ",
    'left-side column separator, and the lowest left-side padding (', min(@{$self->{'pad'}}), ')'
  ) if $self->{'tableWidth'} < 1 + 1 + min(@{$self->{'pad'}});

  $self->{'columnWidth'} = [map { $self->{'columnWidth'}[$_] == ADJUST ? $self->_maxColumnWidth($_)
                                                                       : $self->{'columnWidth'}[$_] }
                            0 .. $#{$self->{'columnWidth'}}];

  $self->_setLineFormat();
  $self->{':headerLines'} = $self->_prepareRow($self->{'header'}, $self->{'broadHeader'});
  if (@{$self->{':headerLines'}}) {
    unshift(@{$self->{':headerLines'}}, $self->{':separatingLine'});
    push   (@{$self->{':headerLines'}}, $self->{':separatingLine'});
  }

  $self->{':linesPerRow'}  = $self->{'broadRow'} == WRAP
                           ? ceil(length($self->{':separatingLine'}) / $self->{'tableWidth'})
                           : 1;
  my $headerHeight         = @{$self->{':headerLines'}} * $self->{':linesPerRow'};
  $self->{':linesPerPage'} = $headerHeight
                           + floor(($self->{'pageHeight'} - $headerHeight) / $self->{':linesPerRow'})
                           * $self->{':linesPerRow'};

  if (@{$self->{':headerLines'}}) {                         # At least one row or one separating line under the header
    my $pageHeight = $headerHeight + $self->{':linesPerRow'};
    die("Page height ($self->{'pageHeight'}) is lower than the minimum possible page height ($pageHeight)")
      if $self->{'pageHeight'} < $pageHeight;
  }

  $self->{'currentRow'} = 0;

  return $self;
}

sub _isCutOrSplitOrWrap {                                   # Check if each column flag is CUT, or SPLIT, or WRAP
  my ($flag) = @_;

  return FALSE unless _isInt($flag);                        # This split-up in 2 "returns" is only necessary due to
  return $flag == CUT || $flag == SPLIT || $flag == WRAP;   # a weakness of Devel::Cover
}

sub _isCutOrWrap {                                          # Check if flag is CUT or WRAP
  my ($flag) = @_;

  return FALSE unless _isInt($flag);                        # This split-up in 2 "returns" is only necessary due to
  return $flag == CUT || $flag == WRAP;                     # a weakness of Devel::Cover
}

sub _isEachCellScalar {                                     # Check if each cell in each row is a defined scalar
  my ($rows) = @_;

  return !grep { !_isScalar($_) } @$rows;
}

sub _isEachColumnFlagCutOrWrap {                            # Check if each column flag is CUT or WRAP
  my ($flag) = @_;

  return ref($flag) ? !grep { !_isCutOrWrap($_) } @$flag : _isCutOrWrap($flag);
}

sub _isEachColumnWidthInt {                                 # Check if each column width is positive integer
  my ($width) = @_;

  return ref($width) ? !grep { !$_ || !_isInt($_) } @$width : $width && _isInt($width);
}

sub _isEachColumnWidthUndefOrInt {                          # Check if each column width is udefined or positive integer
  my ($width) = @_;

  return ref($width) ? !grep { !_isUndefOrInt($_) } @$width : _isUndefOrInt($width);
}

sub _isEachRowArray {                                       # Check if each row is an array
  my ($rows) = @_;

  return !grep { ref ne 'ARRAY' } @$rows;
}

sub _isInt {                                                # Check if defined value is an integer
  my ($value) = @_;

  return !ref($value) && $value =~ /^\d+$/;
}

sub _isScalar {                                             # Check if each cell in a row is a defined scalar
  my ($value) = @_;

  return !grep { !defined || ref } @$value;
}

sub _isUndefOrInt {                                         # Check if value is defined or an integer
  my ($value) = @_;

  return !defined($value) || _isInt($value);
}

sub _maxColumnWidth {                                       # Estimates maximum length of text in column
  my ($self, $columnNo) = @_;

  my $width = @{$self->{'header'}} ? length($self->{'header'}[$columnNo]) : 0;

  return max($width, map { length($_->[$columnNo]) } @{$self->{'rows'}});
}

sub _prepareRow {                                           # Converts table row to array of output strings
  my ($self, $row, $broadFlags) = @_;
                                                            # Ignore possible redundant columns in header
  @$row = @$row[0 .. $self->{':numberOfColumns'} - 1] if $#$row >= $self->{':numberOfColumns'};

  return [map { sprintf($self->{':lineFormat'}, @$_) } @{$self->_extractLines($row, $broadFlags)}];
}

sub _screenHeight { return (GetTerminalSize())[1] }

sub _screenWidth  { return (GetTerminalSize())[0] }

sub _setDefaults {                                          # Set default attributes if they are omitted
  my ($self) = @_;

  for my $option (keys(%{ALL_OPTIONS()})) {
    if (ref(ALL_OPTIONS->{$option}{'default'}) eq 'ARRAY') {
      my $default = defined($self->{$option}) && !ref($self->{$option}) ? $self->{$option}
                                                                        : ALL_OPTIONS->{$option}{'default'}[0];
      $self->{$option}       = [] unless ref($self->{$option});
      next if $option eq 'header' || $option eq 'rows';
      $self->{$option}[$_] //= $default foreach 0 .. $self->{':numberOfColumns'} - 1;
    }
    else {
      my $default = ALL_OPTIONS->{$option}{'default'};
      $self->{$option} = ref($default) eq 'CODE' ? &$default() : $default unless defined($self->{$option});
    }
  }

  return $self;
}

sub _setLineFormat {
  my ($self) = @_;

  my $tableWidth = 1;

  foreach my $columnNo (0 .. $self->{':numberOfColumns'} - 1) {
    my $columnWidth = $self->{'columnWidth'}[$columnNo];
    if ($self->{'collapse'}[$columnNo] && !$columnWidth) {
      $self->{':lineFormat'} .= '%s';
    }
    else {
      my $pad = $self->{'pad'}[$columnNo];
      $self->{':lineFormat'}     .= ' ' x  $pad . '%-' . $columnWidth . 's' . ' ' x $pad  . '|';
      $self->{':separatingLine'} .= '-' x ($pad +        $columnWidth +             $pad) . '+';
      $tableWidth                +=        $pad +        $columnWidth +             $pad  + 1;
    }

    last if $self->{'broadRow'} == CUT && $tableWidth >= $self->{'tableWidth'};
  }

  $self->{':totalWidth'} = $tableWidth;
  $self->{'currentRow'} = 0 if $tableWidth == 1;            # This table has no content

  return $self;
}

sub _stripTrailingBlanks {                                  # Strips down trailing blanks from all cell values in row
  my ($row) = @_;

  return [map { s/\s+$//r } @$row];
}

sub _validate {
  my ($self, $params) = @_;

  validate(@$params, {%{ALL_OPTIONS()}, rows => {'optional' => TRUE, 'type' => ARRAYREF | CODEREF}});

  my %params      = @$params;
  $self->{'rows'} = $params{'rows'} // [];
  return ref($params{'rows'}) eq 'ARRAY' ? $self->_validateForArray([%{_copyOptions(undef, OPTIONS_ARRAY, $params)}])
                                         : $self;
}

sub _validateForArray {
  my ($self, $params) = @_;

  $self->{'rows'}             = [map { _stripTrailingBlanks($_) } @{$self->{'rows'}}];
  $self->{':numberOfColumns'} = @{$self->{'rows'}} ? @{$self->{'rows'}[0]} : 0;

  validate(@$params, OPTIONS_ARRAY);

  return $self->_copyOptions(OPTIONS_ARRAY, $params)->_validateGeneral($params);
}

sub _validateForCallback {
  my ($self, $params) = @_;

  validate(@$params, OPTIONS_CALLBACK);

  return $self->_copyOptions(OPTIONS_CALLBACK, $params)->_validateGeneral($params);
}

sub _validateGeneral {
  my ($self, $params) = @_;

  validate(@$params, OPTIONS_GENERAL);

  my %params = @$params;
  die("The 'header' parameter contains less elements than an element of the 'rows' parameter")
    if exists($params{'header'}) && exists($params{'rows'}) && @{$params{'header'}} < $self->{':numberOfColumns'};

  return $self->_copyOptions(OPTIONS_GENERAL, $params);
}

1;