/* common.c:
 *
 * routines common to both xbgsun and xviewsun
 *
 * jim frost 01.06.88
 *
 * this is in the public domain
 *
 * 01.06.88 ability to handle run-length encoding added.
 * 12.30.88 original version derived from xbgsun.
 */

#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <X11/Xlib.h>
#include "rasterfile.h"

/* global variables.  we could just pass these around but this is simpler
 * since there are so many.
 */

Display		*disp;                /* X display */
Screen		*screen;		/* Screen of above display */
Visual		*visual;		/* Visual of above screen/display */
int		 width, height, depth; /* dimensions of image */
int		 xorigin, yorigin;     /* where to put image */
int		 pwidth, pheight;      /* dimensions of pixmap */
int		 cx, cy;               /* clip origin */
int		 cwidth, cheight;      /* clip dimensions */
int		 verbose;              /* talkative mode flag */
int		 raw;			/* raw (non-Sun) image */
long		 offset;		/* seek offset into file */
char            *fgcolor, *bgcolor;   /* foreground/background colors */
Pixmap           pic;                  /* destination pixmap */
XWindowAttributes wa;                   /* root window attributes */
unsigned char	*data;                  /* image data area */
int		bytesperline;           /* bytes per image line */

/* local globals
 */

static unsigned long *pixdata;             /* pixel data area */
static unsigned char *red, *blue, *green;  /* colormap data areas */
static int	 mapsize;              /* number of colors in colormap */
static int	 imagebyte, linebyte;

/* return true if option matches argument for a minimum number of chars
 */

int isoption(op, arg, min)
char *op, *arg;
int min;
{
  if ((strlen(arg) < min) && !strncmp(op, arg, strlen(arg))) {
    printf("not enough characters specified on option '%s'\n", arg);
    exit(1);
  }
  return(!strncmp(op, arg, strlen(arg)));
}

/* memToVal and valToMem convert 68000-ordered numbers into whatever the
 * local ordering is.  if you have bit or byte ordering problems, fix
 * them here.  this was tested on machines with differing bit and byte
 * orders so it should work fine.  your mileage may vary.
 */

static unsigned int memToVal(d, l)
unsigned char *d; /* char array of number */
int      l;       /* length of array */
{ int a;
  unsigned int i;

  i= 0;
  for (a= 0; a < l; a++)
    i= (i << 8) + *(d++);
  return(i);
}

static void valToMem(d, l, v)
unsigned char *d;
int l;
unsigned long v;
{ int a;

  for (a= l - 1; a >= 0; a--) {
    *(d + a)= v & 0xff;
    v >>= 8;
  }
}

/* macro to do 68000 to local integer conversions
 */

#define localint(d) memToVal(d, 4)

/* this attempts to find the image.   order:
 *   name
 *   name.Z
 *   nameSUFFIX
 *   nameSUFFIX.Z
 *   PATH/name
 *   PATH/name.Z
 *   PATH/nameSUFFIX
 *   PATH/nameSUFFIX.Z
 */

static char *findImage(name)
char *name;
{ static char fname[BUFSIZ];
  struct stat sbuf; /* dummy */

  if (!stat(name, &sbuf))
    return(name);
  sprintf(fname, "%s.Z", name);
  if (!stat(fname, &sbuf))
    return(fname);
#ifdef SUFFIX
  sprintf(fname, "%s%s", name, SUFFIX);
  if (!stat(fname, &sbuf))
    return(fname);
  sprintf(fname, "%s%s.Z", name, SUFFIX);
  if (!stat(fname, &sbuf))
    return(fname);
#endif
#ifdef PATH
  if ((*name == '.') || (*name == '/'))
    return(name);
  sprintf(fname, "%s/%s", PATH, name);
  if (!stat(fname, &sbuf))
    return(fname);
  sprintf(fname, "%s/%s.Z", PATH, name);
  if (!stat(fname, &sbuf))
    return(fname);
#ifdef SUFFIX
  sprintf(fname, "%s/%s%s", PATH, name, SUFFIX);
  if (!stat(fname, &sbuf))
    return(fname);
  sprintf(fname, "%s/%s%s.Z", PATH, name, SUFFIX);
  if (!stat(fname, &sbuf))
    return(fname);
#endif
#endif
  return(name);
}

/* this is called if there is a read error getting the image
 */

static int badRead(fname, f)
char *fname;
FILE *f;
{
  printf("%s: read problem at offset %ld, attempting to use what I have\n",
	fname, ftell(f));
  return(0);
}

/* read in run length encoded data and decode it
 */

static int readEncoded(f, buf, size)
     FILE          *f;
     unsigned char *buf;
     int           size;
{ static int           remaining= 0;
  static unsigned char repeating;

  while (size--)
    if (remaining) {
      remaining--;
      *(buf++)= repeating;
    }
    else {
      if (fread(&repeating, 1, 1, f) != 1)
	return(-1);
      if (repeating == RESC) {
	if (fread(&repeating, 1, 1, f) != 1)
	  return(-1);
	if (repeating == 0)
	  *(buf++)= RESC;
	else {
	  remaining= repeating;
	  if (fread(&repeating, 1, 1, f) != 1)
	    return(-1);
	  *(buf++)= repeating;
	}
      }
      else
	*(buf++)= repeating;
    }
}

/* this loads the rasterfile into memory
 */

int loadImage(name)
     char *name;
{ FILE           *f;
  char           *fname, cmd[BUFSIZ];
  struct rheader header;
  unsigned char  *colormap, byte;
  int            a, b, len, ilen, maplen, rlencoded;

  /* get rasterfile; we get it from uncompress -c if it ends in .Z, or
   * look for a .Z if we don't find an uncompressed one.
   */

  if(strcmp(name, "-") == 0) {
    f = stdin;
  } else {
    fname= findImage(name);
    if ((strlen(fname) > 2) && (!strcmp(fname + strlen(fname) - 2, ".Z"))) {
      sprintf(cmd, "uncompress -c %s", fname);
      f= popen(cmd, "r");
    }
    else
      f= fopen(fname, "r");
  }
  if (f == NULL) {
    perror(fname);
    return(-1);
  }

  if(offset > 0) {
    /* Is file seekable? */
    if(lseek(fileno(f), 0L, 1) < 0) {
      int n;
      char buf[8192];

      for(n = offset; n > 0; n -= sizeof(buf)) {
	if(fread(buf, n<sizeof(buf) ? n : sizeof(buf), 1, f) <= 0) {
	    printf("%s: error seeking to offset %d\n", fname, offset);
	    return(-1);
	}
      }
    } else {
	(void) fseek(f, offset, 0);
    }
  }

		
  if(!raw) {
    /* Not raw, this is a Sun rasterfile */
    if (fread(&header, sizeof(struct rheader), 1, f) != 1) {
      printf("%s: error loading rasterfile header.\n", fname);
      return(-1);
    }

    /* check magic number
     */

    if (!raw && localint(header.magic) != RMAGICNUMBER) {
      printf("I don't know what '%s' is, but it's not a Sun raster image.\n",
	     fname);
      return(-1);
    }

    /* filter out unsupported rasterfiles
     */

    switch(localint(header.type)) {
    case RSTANDARD :
      rlencoded= 0;
      break;
    case RRLENCODED :
      rlencoded= 1;
      break;
    default :
      printf("%s: unsupported rasterfile type\n", fname);
      return(-1);
    }

    if ((localint(header.maptype) != RNOMAP) &&  /* no map, no problem */
	(localint(header.maptype) != RRGBMAP)) {
      printf("%s: unsupported colormap type\n", fname);
      return(-1);
    }

    width= localint(header.width);
    height= localint(header.height);
    depth= localint(header.depth);

    /* read the colormap
     */

    if ((maplen= localint(header.maplen)) > 0) {
      if ((colormap= (unsigned char *)malloc(maplen)) == NULL) {
	printf("%s: malloc error (cannot load colormap)\n", fname);
	exit(1);
      }
      if (fread(colormap, maplen, 1, f) != 1) {
	printf("%s: error reading colormap\n", fname);
	return(-1);
      }
    }
  }

  if (verbose) {
    printf("Hmm, %s is a %dx%d %s image", name, width, height,
	   (depth == 1 ? "monochrome" : "color"));
    if (depth > 1)
      printf(" with %d planes", depth);
    printf(".\n");
  }

  bytesperline= ((width * (depth>8 ? 8 : depth)) / 8);
#ifdef notdef
  if ((width) % 16 > 8)                /* images are rounded out to 16 bits */
    bytesperline += 2;
  else if (width % 16)
    bytesperline += 1;
#endif
  bytesperline = (bytesperline+1) & ~1;	/* Round up to multiple of 16 bits */
  if (depth) {
    mapsize= localint(header.maplen) / 3;
    red= colormap;
    green= colormap + mapsize;
    blue= colormap + (mapsize * 2);
  }
  else {
    mapsize= 0;
    red= green= blue= NULL;
  }

  /* load image data
   */

  if ((data= (unsigned char *)malloc((height+1) * bytesperline)) == NULL) {
    printf("%s: malloc error (cannot load image)\n", fname);
    exit(1);
  }

  for (a= 0; a < height; a++) {
    if (rlencoded) {
      if (readEncoded(f, data + (a * bytesperline), bytesperline) < 0)
	return(badRead(fname, f));
    }
    else {
      if (fread(data + (a * bytesperline),
		depth>8 ? bytesperline*2 : bytesperline,
		1, f) != 1) {
	fprintf(stderr, "bytesperline %d depth %d a %d height %d\n", bytesperline, depth, a, height);
	return(badRead(fname, f));
      }
      if(depth > 8) {
	  register unsigned char *cp;
	  register unsigned short *wp;
	  register int n;
	  register int sc = depth - 8;

	  n = bytesperline;
	  cp = data + (a * bytesperline);
	  wp = (unsigned short *)cp;
	  do { *cp++ = *wp++ >> sc; } while(--n > 0);
      }
    }
  }
  if(depth > 8)
    depth = 8;
  return(0);
}

/* convert a color name to a pixel value
 */

unsigned long nameToPixel(name, pixel)
    char          *name;
    unsigned long *pixel;
{ XColor color;

  if (!XParseColor(disp, wa.colormap, name, &color)) {
    printf("Unknown color '%s'", name);
    return(-1);
  }
  if (!XAllocColor(disp, wa.colormap, &color)) {
    printf("Cannot allocate color '%s'", name);
    return(-1);
  }
  *pixel= color.pixel;
  return(0);
}

/* find the best color in our colormap
 */

static void findBestColor(xcolor)
XColor *xcolor;
{ XColor qcolor;
  int    a;
  int    bcolor; /* best color */
  long   dist;   /* our distance from the color */
  long   bdist;  /* distance for best color yet */
  long   qdist;

  bdist= 256 * 256 * 3;
  xcolor->red >>= 8;   /* shifted so the distance value will fit into */
  xcolor->green >>= 8; /* a long comfortably.  why use floats? */
  xcolor->blue >>= 8;
  
  dist= (xcolor->red * xcolor->red) + (xcolor->green * xcolor->green) +
    (xcolor->blue * xcolor->blue);
  for (a= 0; a < (1 << wa.depth); a++) {
    XQueryColor(disp, wa.colormap, &qcolor);
    qcolor.red >>= 8;
    qcolor.green >>= 8;
    qcolor.blue >>= 8;
    qdist= (qcolor.red * qcolor.red) + (qcolor.green * qcolor.green) +
      (qcolor.blue * qcolor.blue) - dist;
    if (qdist < 0)
      qdist= -qdist;
    if (qdist < bdist) {
      bdist= qdist;
      bcolor= a;
    }
  }
  xcolor->pixel= bcolor;
}

/* translate the image into local colors
 */

void translateColors()
{ int           x, y;
  int           pixlen;     /* length of image pixel in bytes */
  char          *haspixval; /* 1 if we've allocated this pixel's color */
  unsigned long *pixval;    /* local pixel for image's pixel value */
  unsigned char *pixrow;    /* start of image row */
  unsigned char *pixloc;    /* current pixel we're playing with */
  unsigned long pixel;      /* actual pixel value from image */
  Colormap      cmap;       /* color map we're using */
  XColor        xcolor;
  int           rval, gval, bval; /* rgb values of image pixel */
  int           colors;     /* number of colors actually allocated */
  unsigned long *dptr;      /* pointer into pixdata */
  int           greyscale;  /* 1 if colormap is greyscale */

  /* in the interest of keeping our normal colormap, we only allocate those
   * pixel values which are actually used in the image and translate the
   * pixel values in the image to those we've allocated.  this can be quite
   * time consuming, but that's the cost of color!
   */

  pixlen= depth >> 3;
  cmap= wa.colormap;
  xcolor.flags= DoRed | DoGreen | DoBlue;
  if (((haspixval= malloc(1 << depth)) == NULL) ||
      ((pixval= (unsigned long *)malloc((1 << depth) * sizeof(long)))
       == NULL)) {
    printf("Malloc failure.\n");
    XCloseDisplay(disp);
    exit(1);
  }

  /* if we're sending to a different depth of display, we must translate
   * the image into pixel values and put it somewhere else.  this grabs
   * the new area.
   */

  if (wa.depth != depth)
    if ((pixdata= dptr= (unsigned long *)malloc(sizeof(long) * width *
						height)) == NULL) {
      printf("Malloc failure (can't allocate pixel data area).\n");
      exit(1);
    }

  greyscale= 1;
  colors= 0;
  for (x= 0; x < (1 << depth); x++) /* none yet */
    *(haspixval + x)= 0;

  for (y= 0; y < height; y++) {
    pixrow= data + (y * bytesperline);
    for (x= 0; x < width; x++) {
      pixloc= pixrow + (x * pixlen);
      pixel= memToVal(pixloc, pixlen);

      if (pixel >= mapsize) {
	printf("Something's bogus -- found a pixel with no colormap entry.\n");
	exit(1);
      }

      if (! *(haspixval + pixel)) {
	rval= memToVal(red + pixel, 1);
	gval= memToVal(green + pixel, 1);
	bval= memToVal(blue + pixel, 1);

	if (greyscale && ((rval != gval) || (rval != bval)))
	  greyscale= 0;

	/* if we're color, grab a new colormap entry
         */

	if (wa.depth > 1) {
	  xcolor.red= rval << 8;     /* X colors are 16 bits */
	  xcolor.green= gval << 8;
	  xcolor.blue= bval << 8;
	  if (!XAllocColor(disp, cmap, &xcolor)) {
	    cmap= XCopyColormapAndFree(disp, cmap);
	    if (!XAllocColor(disp, cmap, &xcolor))
	      findBestColor(&xcolor);
	  }
	}

	/* if we're mono, figure out if the color is whiter or blacker.  most
	 * servers do this automagically in XAllocColor() but often they
	 * don't do as good a job.
	 */

	else {
	  xcolor.pixel= ((rval * rval) + (gval * gval) + (bval * bval));
	  if (xcolor.pixel < ((255 * 255 * 3) - xcolor.pixel))
	    xcolor.pixel= BlackPixelOfScreen(screen);
	  else
	    xcolor.pixel= WhitePixelOfScreen(screen);
	}

        *(haspixval + pixel)= 1;
	*(pixval + pixel)= xcolor.pixel;
	colors++;
      }
      if (wa.depth == depth)
	valToMem(pixloc, pixlen, *(pixval + pixel));
      else
	*(dptr++)= *(pixval + pixel);
    }
  }
  if (verbose) {
    if (greyscale)
      printf("This is a greyscale image.\n");

    if (colors < (1 << depth))
      printf("Image only used %d of %d colors in its colormap.\n", colors,
	     1 << depth);
    else
      printf("Image used all of the colors in its colormap.\n");

    if (wa.depth == 1)
      printf("Monochrome destination -- this is going to be ugly.\n");
    else if (wa.depth < depth)
      printf("Fewer destination than source planes -- could be \
interesting.\n");
  }
  wa.colormap= cmap;
}

/* send across a monochrome image
 */

void sendMonoImage()
{ XImage    *image;   /* XImage for our raster image */
  Pixmap    picplane; /* monochrome plane used in transfer */
  XGCValues gcv;
  GC        gc;
  GC        planegc;  /* gc for picplane */

  /* set up an XImage structure that points to our bitmap data
   */

  image= XCreateImage(disp, visual, 1,
		      XYPixmap, 0, data, width, height, 16, bytesperline);

  /* since our data will be in MC68000 format, we force the image structure
   * to agree with it.  neat results if you don't do this.
   */

  image->byte_order= MSBFirst;
  image->bitmap_bit_order= MSBFirst;

  /* set up gc that tells server what colors to use
   */

  gcv.function= GXcopy;
  gcv.foreground= BlackPixelOfScreen(screen);
  gcv.background= WhitePixelOfScreen(screen);
  if (fgcolor && nameToPixel(fgcolor, &gcv.foreground))
    printf(" requested for foreground, using black.\n");
  if (bgcolor && nameToPixel(bgcolor, &gcv.background))
    printf(" requested for background, using white.\n");
  gc= XCreateGC(disp, pic, GCFunction | GCForeground | GCBackground, &gcv);

  /* send image to pixmap.  if we're on a mono screen, we just send it to
   * the destination pixmap.  otherwise we have to send to a mono pixmap
   * and copy that pixmap to the destination pixmap
   */

  if (wa.depth > 1) {
    picplane= XCreatePixmap(disp, RootWindowOfScreen(screen),
			    pwidth, pheight, 1);
    gcv.function= GXcopy;
    gcv.foreground= 1;
    gcv.background= 0;
    planegc= XCreateGC(disp, picplane, GCFunction | GCForeground |
		       GCBackground, &gcv);

    XPutImage(disp, picplane, planegc, image, cx, cy, xorigin, yorigin,
	      cwidth, cheight);
    XCopyPlane(disp, picplane, pic, gc, xorigin, yorigin,
	       cwidth, cheight, xorigin, yorigin, 1);
    XFreePixmap(disp, picplane);
    XFreeGC(disp, planegc);
  }
  else
    XPutImage(disp, pic, gc, image, cx, cy, xorigin, yorigin,
	      cwidth, cheight);
  XFreeGC(disp, gc);
}

/* send across a color image to a display of the same depth.
 */

void sendColorImage()
{ XImage    *image;
  XGCValues gcv;
  GC        gc;

  image= XCreateImage(disp, visual,  depth>8 ? 8 : depth,
		      ZPixmap, 0, data, width, height, 16, bytesperline);
  image->byte_order= MSBFirst;
  image->bitmap_bit_order= MSBFirst;

  gcv.function= GXcopy;
  gc= XCreateGC(disp, pic, GCFunction, &gcv);
  XPutImage(disp, pic, gc, image, cx, cy, xorigin, yorigin, cwidth, cheight);
  XFreeGC(disp, gc);
}

/* send across a color image to whatever
 */

void sendColorPixels()
{ int           x, y;
  unsigned long *dptr;
  XGCValues     gcv;
  GC            gc;

  if (verbose)
    printf("This may take a minute, a pixel at a time is tedious.\n");

  gcv.function= GXcopy;
  gc= XCreateGC(disp, pic, GCFunction, &gcv);

  dptr= pixdata;
  for (y= cy; y < cheight; y++)
    for (x= cx; x < cwidth; x++) {
      XSetForeground(disp, gc, *(dptr++));
      XDrawPoint(disp, pic, gc, x, y);
    }
  XFreeGC(disp, gc);
}
