/*
 * Copyright (C) Rutherford Appleton Laboratory 1987
 * 
 * This source may be copied, distributed, altered or used, but not sold for profit
 * or incorporated into a product except under licence from the author.
 * It is not in the public domain.
 * This notice should remain in the source unaltered, and any changes to the source
 * made by persons other than the author should be marked as such.
 * 
 *	Crispin Goswell @ Rutherford Appleton Laboratory caag@uk.ac.rl.vd
 */
#include <sgtty.h>

#include "main.h"

#define MAXFILENAME	1000

/*
 * This file implements the input/output calls in PostScript.
 *
 * A PostScript file corresponds reasonably closely to a Unix file, so this is not
 * too hard.
 *
 * The file operator itself can generate three major file types:
 *
 *	Regular filing system files
 *	Standard I/O channels (called %stdin, %stdout and %stderr)
 *	Peculiar temporary files called %lineedit and %statementedit
 *
 * %lineedit and %statementedit are files generated by the file operator, rather
 * than merely opened for reading or writing.
 * 
 * %lineedit generates a temporary file containing the text that the user types
 * in on the %stdin stream for one line only.
 * The %stdin is required to be edited while this is going on, which means that it
 * must be in cooked mode (-cbreak). %stdin is normally in cbreak mode.
 *
 * %statementedit is similar to %lineedit (in that it generates a temporary file),
 * but the input process continues until PostScript sees a complete expression, with
 * "(" matching ")" and "{" matching "}". "[" is not required to match "]", as these
 * are not actually parsed differently from operators (except for their delimiting
 * properties).
 *
 * There are essentially two ways of implementing these temporary files:
 *
 *	o	We could use real files, and copy user input into them, allowing the
 *		PostScript interpreter to just read from them.
 *	o	We could provide some kind of cache using a buffer in main memory.
 *
 * The second alternative is likely to reduce certain overheads and, since the code
 * for parsing from strings is needed anyway (token is defined to parse from strings),
 * there is little extra effort in doing things this way.
 *
 * For this reason, files under this implementation are typed. We have:
 *
 *	StreamFiles	which are conventional stdio FILE *s.
 *	StringFiles	which are pointers in strings.
 *
 * StringFiles are used internally by token to parse from strings, to avoid duplicating
 * the entire parser. These file types are defined with an enumerated type.
 *
 * It is defined that closing the %stdin file have no effect other than resetting an
 * end-of-file flag. It is not defined whether the output files are required to be
 * similarly permanent. We assume they are, and introduce a feature of files which is
 * that they may be PERMANENT. This is a flag. It is also not clear if Readable and
 * Writeable access to files is an Access Restriction, as such. We assume so.
 *
 */

extern int errno;

static Object StdIn, Terminal, StdOut, StdErr, LineEdit;
Object StatementEdit, Fstdin, Fterminal, Fstdout, Fstderr;

FILE *popen ();

static int Token (), Eq ();

static int PFile (), PCloseFile (), PRead (), PReadHexString (), PReadLine (), PReadString ();
static int PBytesAvailable (), PWrite (), PWriteHexString (), PWriteString (), PFlush ();
static int PFlushFile (), PStatus (), PCurFile (), PPrint (), PEcho ();

int ExecFile ();

InitFile ()
 {
	TypeInstallOp (File, "token",	Token, 	1, 2, 0, 0, File);
 	TypeInstallOp (File, "exec", 	ExecFile, 	1, 0, 0, 0, File);
  	TypeInstallOp (File, "eq", 	Eq, 	2, 1, 0, 0, File, File);
	
	Fstdin  = ReadOnly (FileFrom (stdin));   Fstdin.flags |= PERMANENT;
	Fterminal  = FileFrom (fopen ("/dev/tty", "r"));   Fterminal.flags |= PERMANENT;
	Fstdout = WriteOnly (FileFrom (stdout)); Fstdout.flags |= PERMANENT;
	Fstderr = WriteOnly (FileFrom (stderr)); Fstderr.flags |= PERMANENT;
	
	StdIn 		= StringFrom ("%stdin");
	StdOut 		= StringFrom ("%stdout");
	StdErr 		= StringFrom ("%stderr");
	Terminal	= StringFrom ("%terminal");
	StatementEdit 	= StringFrom ("%statementedit");
	LineEdit 	= StringFrom ("%lineedit");
	
	InstallOp ("file",		PFile,		2, 1, 0, 0, String, String);
	InstallOp ("closefile",		PCloseFile,	1, 0, 0, 0, File);
	InstallOp ("read",		PRead,		1, 1, 0, 0, File);
	InstallOp ("readhexstring",	PReadHexString,	2, 1, 0, 0, File, String);
	InstallOp ("readline",		PReadLine,	2, 1, 0, 0, File, String);
	InstallOp ("readstring",	PReadString,	2, 1, 0, 0, File, String);
	InstallOp ("bytesavailable",	PBytesAvailable,1, 1, 0, 0, File);
	InstallOp ("write",		PWrite,		2, 0, 0, 0, File, Integer);
	InstallOp ("writehexstring",	PWriteHexString,2, 0, 0, 0, File, String);
	InstallOp ("writestring",	PWriteString,	2, 0, 0, 0, File, String);
	InstallOp ("flush",		PFlush,		0, 0, 0, 0);
	InstallOp ("flushfile",		PFlushFile,	1, 0, 0, 0, File);
	InstallOp ("status",		PStatus,	1, 1, 0, 0, File);
	InstallOp ("currentfile",	PCurFile,	0, 1, 0, 0);
	InstallOp ("print",		PPrint,		1, 0, 0, 0, String);
	InstallOp ("echo",		PEcho,		1, 0, 0, 0, Boolean);
	
	Cbreak (TRUE);
 }

static int Token (f) Object f;
 {
 	Object res;
 	res = Parse (f);
	if (TypeOf (res) == Condition)
	 	return Error (PSyntaxError);
	else if (TypeOf (res) == Boolean)
	 	VOID Push (OpStack, False);
	else if (TypeOf (res) != Null)
	 {
	 	VOID Push (OpStack, res);
		VOID Push (OpStack, True);
	 }
	else
		return Error (PUnResult);
	return TRUE;
 }

static struct file_struct *Body (item) Object item;
 {
	return item.u.File;
 }

int getchbuf;

#ifdef notdef
struct file_struct *BodyFile (item) Object item;
 {
	return item.u.File;
 }

static int Getc (fp) FILE *fp;
 {
 	int c;
 	
 	for (;;)
 	 {
 	 	errno = 0;
 	 	if ((c = getc (fp)) != EOF)
 	 		return c;
/* 	 	if (errno == 0 || Interrupted ())*/
 	 		return EOF;
 	 }
 }
#endif

#define Getc(a) getc(a)

static int Getchar ()
 {
 	return Getc (stdin);
 }

#define Getchar getchar

static int Status (file) Object file;
 {
 	return Body (file)->available;
 }

#define Status(f) (Body(f)->available)

static int Eq (a, b) Object a, b;
 {
	return Push (OpStack, MakeBoolean (Body (a) == Body (b)));
 }

/*
 *	FileFrom generates a file object from an existing file pointer
 */
Object FileFrom (fp) FILE *fp;
 {
 	Object res;
 	struct file_struct *f;
 	
 	res = MakeObject (File);
 		f = res.u.File = (struct file_struct *) Malloc (sizeof (struct file_struct));
 		f->file_type = StreamFile;
 		f->f.f_ptr = fp;
 		f->available = 1;
 	
 	return res;
 }

/*
 *	FileString is used to generate a file-like object which actually
 *	reads from a string. This is used by "file" and "token" when parsing
 *	strings.
 */
Object FileString (o) Object o;
 {
 	Object res;
 	struct file_struct *f;
 	
 	res = MakeObject (File);
 		f = res.u.File = (struct file_struct *) Malloc (sizeof (struct file_struct));
 		f->file_type	= StringFile;
 		f->f.c_ptr	= BodyString (o);
 		f->available	= lengthString (o);
 	
 	return res;
 }

int GeneralGetch (file) Object file;
 {
 	int c;
 	
 	if (!Status (file))
 		return EOF;
	switch (Body (file)->file_type)
	 {
	 	default:
	 		PanicIf (TRUE, "unknown file type in Getch\n");
	 	    return 0; /* shuts lint up */
	 	
		case StreamFile:
			c = Getc (Body (file)->f.f_ptr);
			if (c == EOF)
			 {
				Body (file)->available = 0;
				Close (file);
			 }
		    return c;
		
 		case StringFile:
 			if (Body (file)->available == 0)
 				return EOF;
 			--Body (file)->available;
  			c = *Body (file)->f.c_ptr++;
                   			return c;
	 }
 }

Ungetch (o, c) Object o; char c;
 {
 	struct file_struct *f = Body (o);
 	
 	++f->available;
 	switch (f->file_type)
 	 {
 	 	case StringFile: f->f.c_ptr--; break;
 	 	case StreamFile: VOID ungetc (c, f->f.f_ptr); break;
 	 	
 	 	default:
 	 		Panic ("Don't know how to Ungetch to this file\n");
 	 }
 }

int ExecFile (item) Object item;
 {
 	Object res;
 	res = Parse (item);
	if (TypeOf (res) == Boolean)
	 {
		if (BodyBoolean (res))
			return Error (PSyntaxError);
	 }
	else if (TypeOf (res) == Condition)
	 	return Error (PSyntaxError);
	else if (TypeOf (res) == Null)
		return Error (PUnResult);
	else if (TypeOf (res) == Name && xCheck (res))
	 {
		/*Object binding;
		
		binding = Load (res);
		
		if (TypeOf (binding) == Operator)
			res = binding;*/
		VOID Push (ExecStack, item);
		VOID Push (ExecStack, res);
	 }
	else
	 {
		VOID Push (ExecStack, item);
		VOID Push (/*TypeOf (res) == Name ? ExecStack : */OpStack, res);
	 }
	return TRUE;
 }

Cleanup ()
 {
 	Echo (TRUE);
 	Cbreak (FALSE);
 }

static cbreak = FALSE, echoing = TRUE;

/*ARGSUSED*/
Cbreak (cond) int cond;
 {
/*	struct sgttyb s;
	
 	if (cond == cbreak)
 		return;
 	ioctl (0, TIOCGETP, &s);
 	cbreak = cond;
 	if (cbreak)
 		s.sg_flags |= CBREAK;
 	else
 		s.sg_flags &= ~CBREAK;
 	ioctl (0, TIOCSETP, &s);
*/ }

static Echo (cond) int cond;
 {
	struct sgttyb s;
	
 	if (cond == echoing)
 		return;
 	ioctl (0, TIOCGETP, &s);
 	echoing = cond;
 	if (echoing)
 		s.sg_flags |= ECHO;
 	else
 		s.sg_flags &= ~ECHO;
 	ioctl (0, TIOCSETP, &s);
 }

static FILE *open_pipes [32];/* ICK! */
static int n_open_pipes = 0;

FILE *Fopen (name, mode) char *name, *mode;
 {
	if (name[0] == '|')
	 {
		FILE *fp = popen (name + 1, mode);
		int i;
		
		for (i = 0; i < n_open_pipes && open_pipes [i] != NULL; i++)
			;
		open_pipes [i] = fp;
		if (i == n_open_pipes)
			++n_open_pipes;
		return fp;
	 }
	else
		return fopen (name, mode);
 }

void Fclose (fp) FILE *fp;
 {
 	int i;
 	
 	for (i = 0; i < n_open_pipes; i++)
 		if (open_pipes [i] == fp)
 		 {
 		 	pclose (fp);
 		 	open_pipes [i] = NULL;
 		 }
 	fclose (fp);
 }

static int PFile (string1, string2) Object string1, string2;
 {
 	int mode;
 	
	if (lengthString (string1) == 0
 			|| lengthString (string2) != 1
 			|| (mode = getString (string2, 0)) != 'r' && mode != 'w')
 		return Error (PInvFileAccess);
 	else if (getString (string1, 0) == '%')
		return PseudoFile (string1, mode);
 	else if (lengthString (string1) > MAXFILENAME)
 		return Error (PUnFilename);
 	else
 	 {
 	 	char name[MAXFILENAME+1], modes[2];
 		FILE *fp;
 		
 		modes [0] = mode; modes [1] = '\0';
 	 	VOID Bcopy (name, BodyString (string1), lengthString (string1));
 	 	name [lengthString (string1)] = '\0';
 	 	if (fp = Fopen (name, modes))
 	 	 	return Push (OpStack, mode == 'r' ? ReadOnly (FileFrom (fp)) : WriteOnly (FileFrom (fp)));
	 	return Error (PUnFilename);
 	 }
 }

static int PseudoFile (string1, mode) Object string1; int mode;
 {
 	     if (EqString (string1, StdIn))	    return mode == 'r' ? Push (OpStack, Fstdin)  : Error (PInvFileAccess);
 	else if (EqString (string1, Terminal))	    return Push (OpStack, Fterminal);
	else if (EqString (string1, StdOut))	    return mode == 'w' ? Push (OpStack, Fstdout) : Error (PInvFileAccess);
	else if (EqString (string1, StdErr))	    return mode == 'w' ? Push (OpStack, Fstderr) : Error (PInvFileAccess);
	else if (EqString (string1, StatementEdit)) return mode == 'r' ? EditStatement () 	 : Error (PInvFileAccess);
	else if (EqString (string1, LineEdit))	    return mode == 'r' ? EditLine () 		 : Error (PInvFileAccess);
	else
 		return Error (PUnFilename);
 }

static int EditLine ()
 {
 	unsigned char buf [BUFSIZE];
 	int c, length = 0;
 	
 	Cbreak (FALSE);
	do {
		if (length >= BUFSIZE)
		 {
		 	Cbreak (TRUE);
		 	return Error (PLimitCheck);
		 }
		if ((c = Getchar ()) == EOF)
		 	break;
		buf [length++] = c;
		if (Interrupted ())
		 {
		 	Cbreak (TRUE);
			return FALSE;
		 }
	} while (buf [length - 1] != '\n');
	Cbreak (TRUE);
	return Push (OpStack,
			ReadOnly (FileString (MakeString (Bcopy (Malloc ((unsigned) length), buf, length),
							  length))));
 }

static int EditStatement ()
 {
 	char buf[BUFSIZE];
 	int c, length = 0, slevel = 0, blevel = 0, escaped = FALSE;
 	
 	Cbreak (FALSE);
 	do {
		if (length >= BUFSIZE)
		 {
		 	Cbreak (TRUE);
		 	return Error (PLimitCheck);
		 }
		if ((c = Getchar ()) == EOF)
		 	break;
		buf [length++] = c;
		if (Interrupted ())
		 {
		 	Cbreak (TRUE);
 			return FALSE;
		 }
 		if (slevel != 0)
		 	if (escaped)
		 		escaped = FALSE;
		 	else if (c == '(')
		 		++slevel;
 		 	else if (c == ')')
 		 		--slevel;
 		 	else if (c == '\\')
 		 		escaped = TRUE;
 		 	else
 		 		;
 		else
 		 {
 		 	if (blevel == 0 && c == '}' || c == ')')
 			 {
				Cbreak (TRUE);
				return Error (PSyntaxError);
			 }
			else if (c == '{')
				++blevel;
			else if (c == '}')
				--blevel;
			else if (c == '(')
				++slevel;
		 }
	} while (!(slevel == 0 && blevel == 0 && buf [length - 1] == '\n'));
 	Cbreak (TRUE);
 	return Push (OpStack,
 			ReadOnly (FileString (MakeString (Bcopy (Malloc ((unsigned) length), buf, length),
 							  length))));
 }

static int PCloseFile (file) Object file;
 {
	if (Status (file))
		Close (file);
	return TRUE;
 }

int Close (file) Object file;
 {
 	if (file.flags & PERMANENT)
		return TRUE;
 	switch (Body (file)->file_type)
	 {
	 	default:
	 		Panic ("Unknown file type in closefile\n");
	 		
	 	case StringFile:
	 		Body (file)->available = 0;
	 		Body (file)->f.c_ptr = NULL;
	 	    return TRUE;
	 	
	 	case StreamFile:
	 		Body (file)->available = 0;
	 		if (Body (file)->f.f_ptr)
	 			Fclose (Body (file)->f.f_ptr),
	 			Body (file)->f.f_ptr = NULL;
	 	    return TRUE;
	 }
 }

static int PRead (file) Object file;
 {
 	int c = Getch (file);
 	
 	if (c != EOF)
		return Push (OpStack, MakeInteger (c)), Push (OpStack, True);
 	else
 	 	return Push (OpStack, False);
 }

static int PReadHexString (file, string) Object file, string;
 {
	int cc = 0, i, odd = FALSE, l = lengthString (string);
	
	for (i = 0; i < l; )
	 {
	 	int c = Getch (file);
	 	
	 	if (c == EOF)
	 		break;
	 	else if (c >= '0' && c <= '9')
	 		c -= '0';
	 	else if (c >= 'a' && c <= 'f')
	 		c -= 'a' - 10;
	 	else if (c >= 'A' && c <= 'F')
	 		c -= 'A' - 10;
	 	else
	 		continue;
	 	if (odd)
	 		putString (string, i++, (cc << 4) | c);
	 	else
	 		cc =  c;
	 	odd = !odd;
	 }
	VOID Push (OpStack, getIString (string, 0, i));
	VOID Push (OpStack, MakeBoolean (Status (file)));
	
	return TRUE;
 }

static int PReadLine (file, string) Object file, string;
 {
 	int i, c = 0, l = lengthString (string);
  	 	
	for (i = 0; (c = Getch (file)) != EOF && i < l && c != '\n'; i++)
		putString (string, i, c);
	if (c != '\n' && c != EOF)
	 	return Error (PRangeCheck);
	VOID Push (OpStack, getIString (string, 0, i));
	VOID Push (OpStack, MakeBoolean (c != EOF));
	
	return TRUE;
 }

static int PReadString (file, string) Object file, string;
 {
 	int i, c, l = lengthString (string);
 	
	for (i = 0; i < l && (c = Getch (file)) != EOF; i++)
 		putString (string, i, c);
 	VOID Push (OpStack, getIString (string, 0, i));
 	VOID Push (OpStack, MakeBoolean (c != EOF));
 	
 	return TRUE;
 }

/*ARGSUSED*/
static int PBytesAvailable (file) Object file;
 {
 	return Push (OpStack, MakeInteger (-1));
 }

static int PWrite (file, integer) Object file, integer;
 {
 	if (!Status (file))
 		return Error (PIOError);
 	switch (Body (file)->file_type)
 	 {
 	 	default:
 	 		Panic ("unknown file type in write\n");
 	 	
 	 	case StringFile:
 	 		return Error (PInvFileAccess);
 	 	
	 	case StreamFile:
 	 		if (Body (file)->f.f_ptr)
	 		 {
	 			VOID fputc (BodyInteger (integer), Body (file)->f.f_ptr);
	 			return TRUE;
	 		 }
	 		return Error (PIOError);
	  }
 }

static int PWriteHexString (file, string) Object file, string;
 {
 	FILE *fp;
 	unsigned char *p = BodyString (string);
 	int l = lengthString (string);
 	 	
  	if (!Status (file))
 		return Error (PIOError);
 	switch (Body (file)->file_type)
 	 {
 	 	default:
 	 		Panic ("unknown file type in writestring\n");
 	 	
	 	case StringFile:
	 		return Error (PInvFileAccess);
	 	
	 	case StreamFile:
	 		fp = Body (file)->f.f_ptr;
			while (l--)
			 {
			 	int c = *p++;
			 	static char hex[] = "0123456789abcdef";
			 	
		 		fprintf (fp, "%c%c", hex[c >> 4], hex[c & 0xF]);
	 		 }
		    return TRUE;
	 }
 }

static int PWriteString (file, string)	Object file, string;
 {
 	if (!Status (file) || !wCheck (file))
 		return Error (PIOError);
	switch (Body (file)->file_type)
	 {
	 	default:
	 		Panic ("unknown file type in writestring\n");
	 	
	 	case StringFile:
	 		return Error (PInvFileAccess);
	 	
	 	case StreamFile:
	 		fprintf (Body (file)->f.f_ptr,
	 			 "%.*s",
	 			 lengthString (string),
	 			 BodyString (string));
		    return TRUE;
	 }
 }

static int PFlush ()		/* --- */
 {
 	VOID fflush (stdout);
 	HardUpdate ();	/* yes, well ... you didn't see this... */
 	return TRUE;
 }

static int PFlushFile (file) Object file;
 {
 	if (!Status (file))
 		return Error (PIOError);
 	else if (wCheck (file))
 		switch (Body (file)->file_type)
 		 {
 		 	default:
 		 		Panic ("unknown file type in flushfile\n");
 		 		return FALSE; /* shuts lint up */
 		 	
 		 	case StringFile:	break;
 		 		
 		 	case StreamFile:
 		 		if (Body (file)->f.f_ptr)
 		 			VOID fflush (Body (file)->f.f_ptr);
 		 		break;
 		 }
 	else if (rCheck (file))
 		while (Getch (file) != EOF)
 			;
 	return TRUE;
 }

static int PStatus (file) Object file;
 {
 	return Push (OpStack, MakeBoolean (Status (file)));
 }

static int PCurFile ()		/* --- file */
 {
 	int c = CountTo (File, ExecStack);
 	Object f;
 	
 	PanicIf (c < 0, "currentfile finds no run context");
 	f = ExecStack->stack_body[Height (ExecStack) - c - 1];
 	return Push (OpStack, Cvlit (f));
 }

static int PPrint (string) Object string;
 {
 	int i, l = lengthString (string);
 	unsigned char *s = BodyString (string);
 	
 	for (i = 0; i < l; i++)
 		putchar (*s++);
 	/* PRINTF BUG: printf ("%.*s", lengthString (string), BodyString (string));*/
 	return TRUE;
 }

static int PEcho (res)	Object res;
 {
	Echo (BodyBoolean (res));
 	return TRUE;
 }
