;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: rule-requests.lisp,v 1.3 1993/06/04 06:27:36 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; process-requests.rule
; ---------------------
;
;       nonlocal-vars : various registers of the current *NODE*
;                         *NODE*, *REQUESTS*
;
;       description   : processes all requests received by the current node
;
;       side-effects  : registers affected:  *REQUESTS*, OUTGOING-CHANNELS:,
;                         *RULE-USE-CHANNELS*
;
;       implementation: 
;
;                                        written :  rgh  2/02/86
;                                        modified:  rgh  2/09/86
;                                                   rgh  2/22/86
;                                                   scs  4/22/88
;                                                   ssc  5/10/89
;
;
(defun process-requests.rule ()
  (let ((remark-sent nil) pathfrom-result)
    (declare (special pathfrom-result remark-sent))
    (do.set (ch *REQUESTS*)
      (if (is-cq-to-rule.req ch)
	  (process-rule-use-request.rule ch)
	  (process-one-request.non-rule ch)))
    (setq *REQUESTS* (new.chset))))
;
;
; =============================================================================
;
; antecedents
; -----------
;
;       arguments     : rule - <node>
;                       cq - <node>
;
;       returns       : <node set>
;
;       description   : returns a <node set> of those nodes which are in
;                       rule antecedent position relative to "cq"
;
;                                        written :  rgh  3/31/86
;                                        modified:  scs  4/22/88
;                                        modified:  scs  6/21/88
;
;
(defun antecedents (rule cq)
  "Returns a <node set> of nodes in antecedent position relative to cq in the rule."
  (cond ((is-and.n rule) nil)
	((is-nor.n rule) nil)
	(t (or (nodeset.n rule 'sneps::ant)
	       (nodeset.n rule 'sneps::&ant)
	       (remove.ns cq (nodeset.n rule 'sneps::arg))))))
;
;
; =============================================================================
;
; process-rule-use-request.rule
; ------------------------------
;
;       arguments     : request - <request>
;
;       nonlocal-vars : *RULE-USE-CHANNELS*, *NODE*, *USABILITY-TEST* registers
;
;       description   : handles a request from a consequent of the current
;                        rule node
;
;       side-effects  : registers affected:  *RULE-USE-CHANNELS* of the
;                        current node, and INCOMING-CHANNELS: of the
;                         consequent node
;
;       implementation: 
;
;                                        written :  rgh  2/02/86
;                                        modified:  rgh  2/09/86
;                                                   rgh  3/31/86
;                                                   rgh  4/03/86
;                                                   njm 10/24/88
;                                                   njm 11/11/88
;                                                   choi 9/19/90
;
(defun process-rule-use-request.rule (request)
  (let* ((ants (antecedents *NODE* (destination.ch request)))
	 (cqch (install-rule-use-channel
		 request (union.ns ants (if (eq *TYPE* 'NUM-QUANTIFIER)
					    (nodeset.n *NODE* 'cq)
					    nil))))
	 (crntct (context.ch request)))
    (declare (special crntct))
    (cond ((and (isassert.n *NODE*) (funcall *USABILITY-TEST* 'POS))
	   (if (member *TYPE* '(AND NOR))
	       (funcall *RULE-HANDLER*
			(make.rep (filter.ch request)
				  (sneps:node-asupport *NODE*)
				  'POS
				  (destination.ch request)
				  nil
				  crntct)
			cqch)
	       (send-ant-requests.rule cqch)))
	  (t (let ((rule-sbst (restrict-binding-to-pat (filter.ch request) *NODE*))
		   (any-instances-applied nil)
		   (restr nil))
	       (declare (special any-instances-applied))
	       (apply-known-instances.rule cqch)
	       (when (or (not any-instances-applied)
			 (is-wh-question rule-sbst))
		 (setq restr (make.restr rule-sbst))
		 (when (not-working-on restr (context.ch request) request)
		   (send-rule-use-requests restr
					   (context.ch request)
					   (destination.ch request))
		   (when (enough-resources)
		     (decrease-resources)
		     (remark '"~%I wonder if" (makeone.ns *NODE*) restr (context.ch request))
		     (send-node-to-node-requests restr (context.ch request))))))))))
;
;
; =============================================================================
;
; install-rule-use-channel
; ------------------------
;
;       arguments     : request - <channel>
;                       ants - <node set>
;
;       returns       : <cq-channel>
;
;       nonlocal-vars : *RULE-USE-CHANNELS*, *NODE*, INCOMING-CHANNELS:
;
;       description   : If there is not already an existing rule use channel
;                       whose channel is "request", one is inserted in
;                       *RULE-USE-CHANNELS*.  If there is one already, but it
;                       is closed, it is reopened.  A similar modification is
;                       made to the INCOMING-CHANNELS: register of the node
;                       which is the destination of the channel.  The newly
;                       inserted or reopened cq-channel is returned.
;
;       side-effects  : updates the registers mentioned above
;
;                                        written :  rgh  2/02/86
;                                        modified:  rgh  2/09/86
;                                                   rgh  3/31/86
;                                                   rgh  4/03/86
;                                                   rgh  4/13/86
;                                                   scs  6/20/88
;                                                   njm 10/24/88
;
;
;
(defun install-rule-use-channel (request ants)
  (let* ((dest (destination.ch request))
	 (destsub (restrict-binding-to-pat (filter.ch request) dest))
	 (cqch (make.cqch (make.ch destsub
				   (switch.ch request)
				   (context.ch request)
				   dest
				   (valve.ch request))
			  ants
			  (makeone.ruis
			    (make.rui destsub 0 0 (nodeset-to-fnodeset ants) nil)))))
    (setq *RULE-USE-CHANNELS* (insert.cqchset cqch *RULE-USE-CHANNELS*))
    (activate.n dest)
    cqch))
;
;
; =============================================================================
;
; apply-known-instances.rule
; --------------------------
;
;       arguments     : cqch - <cq-channel>
;
;       returns       : <boolean>
;
;       nonlocal-vars : *KNOWN-INSTANCES* register
;                       any-instances-applied
;
;       description   : applies known rule instances to the given "cqch",
;                       returning "true" if any are successfully applied,
;                       and "false" otherwise.
;
;                                        written :  rgh  3/31/86
;                                        modified:  rgh  4/03/86
;
;
(defun apply-known-instances.rule (cqch)
  (declare (special any-instances-applied))
  (do ((instances *KNOWN-INSTANCES* (others.iset instances)))
      ((isnew.iset instances))
    (when (try-to-apply-instance.rule (choose.iset instances) cqch)
      (setq any-instances-applied t))))
;
;
; =============================================================================
