;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: rule-finfers.lisp,v 1.3 1993/06/04 06:27:32 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; process-forward-inferences.rule
; -------------------------------
;
;       nonlocal-vars : *PENDING-FORWARD-INFERENCES* register
;
;       description   : handles all the reports in the PENDING-FORWARD-
;                         INFERENCES: register of the current rule node
;
;       side-effects  : updates various registers
;
;                                        written :  rgh  4/20/86
;                                        modified:
;
;
(defun process-forward-inferences.rule ()
  (let ((repset *PENDING-FORWARD-INFERENCES*))
    (setq *PENDING-FORWARD-INFERENCES* (new.repset))
    (do ()
	((isnew.repset repset))
	(process-one-forward-inference.rule (choose.repset repset))
	(setq repset (others.repset repset)))))
;
;
; =============================================================================
;
; process-one-forward-inference.rule
; ----------------------------------
;
;       arguments     : report - <report>
;
;       description   : handles a single forward inference report for a rule
;                       node
;
;       side-effects  : updates several process registers
;
;       implementation: sets up rule use channels to all possible consequents
;                       of the rule.  If the forward inference report came
;                       from an antecedent, it broadcasts it to all the
;                       rule-use channels.  If the report is of a new rule
;                       instance, requests are broadcast to all antecedents
;                       so that the rule may be used -- and the new rule
;                       instance is also handled as a forward inference would
;                       be for a non-rule.
;
;                                        written :  rgh  4/20/86
;                                        modified:  rgh  4/24/86
;                                                   choi 9/19/90
;
(defun process-one-forward-inference.rule (report)
  (set-up-rule-use-channels report)
  (cond ((or (is-ant-to-rule.rep report)
	     (and (eq *TYPE* 'NUM-QUANTIFIER)
		  (is-cq-to-rule.rep report)))
	 (setq *INCOMING-CHANNELS*
	       (insert.feedset
                 (make.feeder (make.restr (subst.rep report))
			      (context.rep report)
			      (signature.rep report)
			      'OPEN)
                 *INCOMING-CHANNELS*))
	 (broadcast-ant-report.rule report))
        (t (process-one-forward-inference.non-rule report)
           (try-applying-one-report report))))
;
;
; =============================================================================
;
; consequents
; -----------
;
;       arguments     : rule - <node>
;                       ant - <node>
;
;       returns       : <node set>
;
;       description   : returns the set of all nodes which are in consequent
;                       position relative to "ant" in "rule"
;
;                                        written :  rgh  4/20/86
;                                        modified:  njm 10/25/88
;                                        modified:  
;
;
(defmacro consequents (rule ant)
  `(let ((cqs (union.ns (nodeset.n ,rule 'sneps::cq) (nodeset.n ,rule 'sneps::dcq)))
         (args (remove.ns ,ant (nodeset.n ,rule 'sneps::arg))))
     (if (not (isnew.ns cqs))
	 cqs
	 args)))
;
;
; =============================================================================
;
; all-consequents
; ---------------
;
;       arguments     : rule - <node>
;
;       returns       : <node set>
;
;       description   : returns the set of all nodes which are in consequent
;                       position in the "rule"
;
;                                        written :  rgh  4/20/86
;                                        modified:  njm 10/25/88
;                                        modified:
;
;
(defmacro all-consequents (rule)
  `(let ((cqs (union.ns (nodeset.n ,rule 'sneps::cq) (nodeset.n ,rule 'sneps::dcq)))
         (args (nodeset.n ,rule 'sneps::arg)))
      (if (not (isnew.ns cqs))
	  cqs
          args)))
;
;
; =============================================================================
;
; set-up-rule-use-channels
; ------------------------
;
;       arguments     : report - <report>
;
;       nonlocal-vars : *RULE-USE-CHANNELS* register
;
;       description   : installs new rule use channels on all appropriate
;                       consequent arcs (based on where the report came from)
;                       whose channels contain the reported substitution as
;                       their filters
;
;       side-effects  : updates *RULE-USE-CHANNELS*
;
;       implementation: if the report came from a node at the end of an arg
;                       arc emanating from the current node, then all other
;                       args are possible consequents.  If the report is of
;                       a new rule instance, and the rule has arg arcs, then
;                       all args are possible consequents.
;
;                                        written :  rgh  4/20/86
;                                        modified:  rgh  4/24/86
;                                        modified:  scs  6/17/88
;                                                   choi 9/19/90
;
(defun set-up-rule-use-channels (report)
  (let ((subst (subst.rep report))
	(ct (context.rep report)))
    (do.ns (cq (if (is-ant-to-rule.rep report)
		   (consequents *NODE* (signature.rep report))
		   (all-consequents *NODE*)))
      (let* ((ants (antecedents *NODE* cq))
	     (rui-ants (union.ns ants (if (eq *TYPE* 'NUM-QUANTIFIER) 
					  (nodeset.n *NODE* 'sneps::cq) nil)))
	     (chsub (restrict-binding-to-pat subst cq)))
	(setq *RULE-USE-CHANNELS*
	      (insert.cqchset
		(make.cqch (make.ch chsub (new.sbst) ct cq 'open) rui-ants
			   (makeone.ruis (make.rui chsub 0 0 (nodeset-to-fnodeset rui-ants) nil)))
		*RULE-USE-CHANNELS*))
	))))
;
;
; =============================================================================
