;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: nrn-finfers.lisp,v 1.3 1993/06/04 06:27:10 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; process-forward-inferences.non-rule
; -----------------------------------
;
;       returns       : <boolean>
;
;       nonlocal-vars : the PENDING-FORWARD-INFERENCES: register of the
;                       current node
;
;       description   : goes through the PENDING-FORWARD-INFERENCES: register
;                       and processes each report in it.
;
;       side-effects  : the PENDING-FORWARD-INFERENCES: register is cleared
;                       and the INCOMING-CHANNELS: and the OUTGOING-CHANNELS:
;                       registers may be updated.
;
;                                        written :  rgh 10/06/85
;                                        modified:  rgh  3/22/86
;                                                   njm 10/23/88
;
;
;
(defun process-forward-inferences.non-rule ()
  (let ((repset *PENDING-FORWARD-INFERENCES*))
    (setq *PENDING-FORWARD-INFERENCES* (new.repset))
    (do ()
	((isnew.repset repset))
      (process-one-forward-inference.non-rule (choose.repset repset))
      (setq repset (others.repset repset)))))
;
;
; =============================================================================
;
; process-one-forward-inference.non-rule
; --------------------------------------
;
;       arguments     : rep - <report>
;
;       returns       : <boolean>
;
;       nonlocal-vars : the *NODE* and *INCOMING-CHANNELS* registers of the
;                       current process
;
;       description   : handles the processing of a single forward-inference
;                       report, sending it to possibly interested rules and
;                       calling forward-match to find possibly interested
;                       antecedent nodes.
;
;       side-effects  : possibly adds channels to *OUTGOING-CHANNELS* 
;
;                                        written :  rgh 10/05/85
;                                        modified:  rgh  2/02/86
;                                                   rgh  3/09/86
;                                                   rgh  4/24/86
;                                                   njm 10/23/88
;                                                   njm 06/30/89
;
;
;
(defun process-one-forward-inference.non-rule (rep)
  (let* ((crntct (context.rep rep))
	 (sub (subst.rep rep))
	 (updated-rep (make.rep sub
				(support.rep rep)
				(sign.rep rep)
				*NODE*
				nil
				crntct)))
    (setq *INCOMING-CHANNELS*
          (insert.feedset
	    (make.feeder (make.restr (subst.rep rep))
			 crntct
			 (signature.rep rep)
			 'OPEN)
	    *INCOMING-CHANNELS*))
    (push-forward updated-rep (sneps:in-context.ns (nodeset.n *NODE* 'sneps:ant-) t))
    (push-forward updated-rep (sneps:in-context.ns (nodeset.n *NODE* 'sneps:&ant-) t))
    (push-forward updated-rep
		  (sneps:in-context.ns (sneps:remove-if-not.ns
					 #'(lambda (n) (not (or (is-and.n n) (is-nor.n n))))
					 (nodeset.n *NODE* 'sneps:arg-)) t))
    (cond ((and (not (is-node-to-node.rep rep)) (enough-resources))
	   (decrease-resources)
	   (do.supmatchingset (m (forward-match-in-context *NODE* crntct sub))
	     (unless (eq (tnode.supmatching m) *NODE*)
	       (let (ch)
		 (setq ch (make.ch (target-sub.supmatching m)
				   (source-sub.supmatching m)
				   crntct
				   (tnode.supmatching m)
				   'OPEN))
		 (install-channel ch) 
		 (send-reports (makeone.repset updated-rep) ch))))))))
;
;
; =============================================================================
;
; push-forward
; ------------
;
;       arguments     : report  - <report>
;                       nodeset - <nodefun set>
;                  
;
;       returns       : <boolean>
;
;       description   : opens channels to all the nodes in "nodeset" and
;                       sends "report" to each of them
;
;       side-effects  : *OUTGOING-CHANNELS* register is updated by call to
;                       "install-channel"
;
;                                        written :  rgh 10/05/85
;                                        modified:  rgh  2/02/86
;                                                   njm 10/23/88
;
;
(defun push-forward (report nodeset)
  (do ((ns nodeset (others.ns ns))
       (ch))
      ((isnew.ns ns))
      (setq ch (make.ch (new.sbst)
			(new.sbst)
			(context.rep report)
			(choose.ns ns)
			'OPEN))
      (install-channel ch)
      (send-reports (makeone.repset report) ch)))
;
;
; =============================================================================
