;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: outnet.lisp,v 1.5 1993/07/20 06:25:06 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; ==========================================================================
;
; outnet
; ------
;
;       arguments     : file - <filename> 
;
;       returns       : <nothing>
;
;       description   : "User function" to be called from SNePS environment.
;                       It dumps the current network to the FILE.
;
;       side-effects  : It prints a message.
;
;                                          written : jgn 10/10/83
;                                          modified: ejm 06/01/84
;                                                    ssc 02/13/87
;                                                    njm 09/26/88
;                                                    hc  29/06/93
;                                                    hc  07/18/93
;
(defsnepscom outnet ((file) (top) t)
  (with-open-file (outunit file :direction :output
		   :if-does-not-exist :create
		   :if-exists :new-version)
    (let ((*package* (find-package 'snepsul))
	  (*print-pretty* nil)
	  (*print-level* nil)
	  (*print-length* nil))
      (format outunit "|SNePS network 2.0|~%")
      (outindices outunit)
      (outrelations outunit)
      (outpaths outunit)
      (outcontexts outunit)
      (outnodes outunit)
      (outsysvars outunit)))
  (format t "~%~%Network dumped to file: ~A~%" file)
  (values))


(defun outindices (outunit)
  (format outunit "~%;; Node/Context-ID indices:~%")
  (dolist (node-name-prefix '(b m v p tm tv tp))
    (format outunit "~s " (get 'gennewnode node-name-prefix)))
  (format outunit "~s~%" (get 'gennewcontext 'c)))

;
;
;
; ==========================================================================
;
; outrelations
; ------------
;
;       arguments     : outunit - <output port> 
;
;       returns       : Sequence of relations 
;
;       description   : Prints a list of the defined relations to
;                       the output port.
;
;       nonlocal-vars : SNePS variable "relations"
;
;       side-effects  : Additions to output file. 
;
;                                          written : jgn 10/10/83
;                                          modified: hc  06/29/93
;
(defun outrelations (outunit)
  (let ((relations (value.sv 'relations)))
    (format outunit "~%~d ;; Relation definitions:"
	    (length relations))
    (dolist (relation relations)
      (print.r relation outunit))
    (terpri outunit)))

(defun outpaths (outunit)
  (let ((paths (remove-if-not #'(lambda (r) (get r :pathdef))
			      (value.sv 'relations))))
    (format outunit "~%~d ;; Path definitions:"
	    (length paths))
    (dolist (path paths)
      (print-path.r path outunit))
    (terpri outunit)))
 
;
;
;
; ==========================================================================
;
; outcontexts
; -----------
;
;       arguments     : outunit - <output port> 
;
;       returns       : Sequence of the current list of contexts 
;
;       nonlocal-vars : System variable "contexts"
;
;       description   : Prints a list of the network contexts to output port.
;
;       side-effects  : Additions to output file. 
;
;                                          written : njm 09/26/88
;                                          modified: hc  06/29/93
;
(defun outcontexts (outunit)
  (let ((contexts (allct)))
    (format outunit "~%~d ;; Context definitions:" (length contexts))
    (dolist (context contexts)
      (print.ct context outunit))
    (terpri outunit)))

;
;
;
; ==========================================================================
;
; outnodes
; --------
;
;       arguments     : outunit - <output port> 
;
;       returns       : Sequence of the current list of nodes 
;
;       nonlocal-vars : System variable "nodes"
;
;       description   : Prints a list of the network nodes to output port. 
;
;       side-effects  : Additions to output file. 
;
;                                          written : jgn 10/10/83
;                                          modified: hc  06/29/93
;
(defun outnodes (outunit)
  (let ((nodes (value.sv 'nodes)))
    (format outunit "~%~d ;; Node definitions:" (length nodes))
    (dolist (node nodes)
      (print.n node outunit))
    (terpri outunit)))

;
;
; ==========================================================================
;
; outsysvars
; ----------
;
;       arguments     : outunit - <output port> 
;
;       returns       : Irrelevant, but returns a list of system 
;                       variables.
;
;       nonlocal-vars : The system variables
;
;       description   : Prints the values of the system variables 
;                       to the output port.
;
;       side-effects  : Additions made to output file 
;
;                                          written : jgn 10/10/83
;                                          modified: hc  06/29/93
;
(defun outsysvars (outunit)
  (let ((variables (value.sv 'variables))
	(excluded-variables
	 '(;; context hashtable is unprintable and gets
	   ;; automatically reconstructed:
	   contexts
	   ;; these are redundant because they get automatically
	   ;; constructed by `innodes' and `inrelations':
	   nodes relations
	   ;; are these any interesting?
	   ;;command lastcommand errorcommand
	   )))
    (format outunit "~%~d ;; SNePSUL variable definitions:"
	    (lisp:- (length variables) (length excluded-variables)))
    (dolist (variable variables)
      (unless (member variable excluded-variables)
	(print.sv variable outunit)))
    (terpri outunit)))
