;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: rel2.lisp,v 1.4 1993/07/02 00:30:36 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; =============================================================================
; Data Type:  <relation>
; =============================================================================
;
; 
; =============================================================================
;
; new.r        
; -----
;
;       arguments     : id - <identifier>
;
;       returns       : <relation>
;
;       description   : creates a new <relation> having the name "id". In
;                       addition, it automatically creates the converse
;                       <relation> having the name "id-".
;                       It also adds the <relation> to the list of
;                       <relation>s "relations".
;
;       side-effects  : it side-effects the <atom>s "id", "id-", and
;                       "relations".
;
;                                        written:  mja 07/28/83
;                                        modified: ejm 02/11/84, 06/05/84
;                                        modified: ssc 11/04/87
;                                        modified: scs 06/23/89
;
;

(defmacro new.r (id) 
  "Creates a new arc label."
  `(let ((id- (intern (concatenate 'string (symbol-name ,id) "-")))) 
     (setf (get ,id '=conv) id-)
     (setf (get ,id '=dnrel) t )
     (setf (get id- '=conv) ,id)
     (setf (get id- '=uprel) t )
     (set.sv 'relations
	     (insert.rs ,id
			(value.sv 'relations)))
     ,id))

;
;
;
; =============================================================================
;
; converse.r
; ----------
;
;       arguments     : r - <relation>
;
;       returns       : <relation>
;
;       description   : returns the converse-relation to "r".
;
;                                        written:  mja 07/28/83
;                                        modified: ejm 06/05/84
;
;
(defmacro converse.r (r)
   `(get ,r '=conv) )
;
; =============================================================================
;
; undefine.r
; ----------
;
;       arguments     : r - <relation>
;
;       returns       : <nothing>
;
;       description   : it undefines the <relation> "r" and its converse
;                       <relation>
;
;       side-effects  : it side-effects the property list of the <atom>s 
;                       representing  the <relation> and its converse,
;                       and removes "r" from "relations".
;
;                                        written : ejm 06/05/84
;                                        modified: 
;
;
(defmacro undefine.r (r)
   `(prog (conv)
	  (declare (special conv))
          (set.sv 'relations 
                  (remove.rs ,r 
                             (value.sv 'relations)))
          (setq conv (converse.r ,r))
          (remprop ,r '=conv)
          (remprop ,r '=dnrel)
          (remprop conv '=conv)
          (remprop conv '=uprel)))
;
;
;
; =============================================================================
;
; isup.r       
; ------
;
;       arguments     : r - <relation>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "r" is an <=uprelation>, otherwise
;                       "false".
;
;                                        written:  mja 07/28/83
;                                        modified: ejm 06/05/84
;
;
(defmacro isup.r (r)
   `(get ,r '=uprel) )
;
;
; =============================================================================
;
; isdn.r       
; ------
;
;       arguments     : r - <relation>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "r" is a <=dnrelation>, otherwise
;                       "false".
;
;                                        written:  mja  07/28/83 
;                                        modified: ejm 06/05/84
;
;
(defmacro isdn.r (r)
   `(get ,r '=dnrel) )
;
;
; =============================================================================
;
; isquant.r
; ---------
;
;       arguments     : r - <relation>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "r" is a quantifier <relation>,
;                       otherwise "false".
;
;                                        written:  mja 08/08/83
;                                        modified:
;
;
(defmacro isquant.r (r)
  `(member ,r '(forall exists pevb) :test #'eq))
;
;
; =============================================================================
;
; read.r
; ------
;
;       arguments     : inunit - <unit>
;
;       returns       : <relation>
;
;       description   : It reads a <relation> from "inunit" and defines it
;                       together with any associated path definition.
;
;       side-effects  : It sides effect the <plist> of the <relation>.
;
;                                        written :  ejm 10/04/83
;                                        modified:  hc  06/29/93
;

(defun read.r (inunit)
  (let ((r (read inunit)))
    (new.r r)))

(defun read-path.r (inunit)
  ;; Reads the path information of a relation
  (let* ((r (read inunit)))
    ;; Read/add path information if there was any:
    (read-plist r '(:pathdef :fwd-paths) inunit)
    (read-plist (converse.r r) '(:pathdef :fwd-paths) inunit)))

;
; =============================================================================
;
; Print.r
; -------
;
;       arguments     : r - <relation>
;                       outunit - <unit>
;
;       returns       : nil
;
;       description   : Prints the <relation> "r" to "outunit".
;
;       side-effects  : It prints the <relation>
;
;                                        written :  ejm 10/04/83
;                                        modified:  hc  06/29/93
;
;

(defun print.r (r outunit)
  (format outunit "~%~s" r))

(defun print-path.r (r outunit)
  ;; Prints the path definition associated with R.
  (format outunit "~%~s" r)
  (print-plist r '(:pathdef :fwd-paths) outunit)
  (print-plist (converse.r r) '(:pathdef :fwd-paths) outunit))

;
;
; =============================================================================

(defun isless.r (rel1 rel2)
  "defines an ordering among relations"
  (cond ((eq rel1 rel2) nil) ; a relation is not less than itself
	((and (isup.r rel1) (isup.r rel2)) ; up relations have the same ordering as down
	 (isless.r (converse.r rel1) (converse.r rel2)))
	((isup.r rel1) t) ; up relations sort before down relations
	((isup.r rel2) nil)
	;; both are down relations
	(t (let* ((priority-list '(FORALL EXISTS PEVB
				       MIN MAX THRESH THRESHMAX EMIN EMAX ETOT
				       ANT &ANT CQ DCQ ARG
				       FNAME DEFAULT)) ; an explicit ordered list
		  (less-list (member rel1 priority-list)))
	     (cond (less-list ; rel1 is a priority relation
		    (cond ((member rel2 less-list) t) ; rel2 is a lower priority relation
			  ((member rel2 priority-list) nil) ;rel2 is a higher priority relation
			  (t) ; rel2 is not a priority relation
			  ))
		   ((member rel2 priority-list) nil) ; rel2 is a priority relation
		   (t ; neither is a priority relation
		    (string< rel1 rel2)) ; so alphabetize
		   )))))