;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: ENGLEX; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: lenglex.lisp,v 1.10 1993/07/20 06:27:43 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :englex)


;;morphological functions

;;issues

;;ISnaturalword
;;allows spaces, hypens,aposts etc to be valid chars, but parser uses them as
;;single character words

;;lookup
;;make changes to tracer information

;;lexin
;;expects that the file is in the new-format.
;;that is it contains quoted forms of symbols corresponding to actual words 
;;such as the entry word and any features whos values are words versus 
;;predefined value tokens.
;;now must be called with the file-name as a string -- else everything gets
;;converted to upper case.

;;puth:
;;substitute for the original call (put entry '=dict val)
;;for (puth entry val)
;;associates with a word(the entry) a list of feature-lists

;;geth:
;;substitute for the original call (get entry '=dict)
;;for (geth entry)
;;gets the feature lists associated with entry

;;lookup-lexical-feature:
;;lexentry needs to be declared a special in calling parser functions
;;needs definition to be shadowed in parser

;;evalt:
;;use parser definition
;;*register-* is a global

(defconstant *Abort-String-No-Constant* 350
 "*Abort-String-No-Constant* controls the number of strings that are tested 
  on a call to lookup. This is used to prevent an infinite loop. When 350 
  strings are generated, this package will abort and give up.")

(defvar *lexicon* nil "hashtable for lexicon.")
  
(defvar *Current_Word* "is the word that is originally passed to routine root")

(defvar *Change_Root_Feature* t 
 "*Change_Root_Feature* - 
  When it has value t:
  Routine lexic will change 'root' value to be *Current_Word* symbol in 
  feature list else 'root' value is assigned the root of the *Current_Word*
  unless *C_R_F* is 'Inner_Suffix.")

(defvar *Negative_Feature* nil 
 "*Negative_Feature* -
  When has value t add '(neg . t) to feature lists.")

;;***************************************************************************;;
;; morphological functions : lookup, lexic, root, Prefix, suffix             ;;
;;***************************************************************************;;

(defvar *Count-New-Strings* 0
 "*Count-New-Strings* - This variable prevents infinite loops. It counts the
  number of new strings that are generated by the package. If more than 350 
  strings (based on lookup ['ungentlemanliness']) are generated then the 
  package will abort and return nil.")

(defun lexic (word)
  "lexic - args: word returns: all the feature lists of word."
  (declare (special *Change_Root_Feature* *Tracing-On* *Count-New-Strings*
                    *trace-ending*))
  (if (eq *Count-New-Strings* *Abort-String-No-Constant*)
        (throw 'abort 'nil)
        (setq *Count-New-Strings* (1+ *Count-New-Strings*)))
  (when *Tracing-On*
 	(write-string "[Ending ")
	(when (stringp *trace-ending*)
	      (write-string *trace-ending*))
        (write-string "]")
        (write-string " Tracing --> ")
	(write-string word) (terpri))
  (if word
      (mapcar #'(lambda (feature-list)
		  (cond ((and (not *Change_Root_Feature*)
			      (assoc 'root feature-list)) feature-list)
			((and *Change_Root_Feature* (assoc 'root feature-list))
			 (subst (cons 'root *Current_Word*)
				(assoc 'root feature-list) feature-list
				:test 'equal))
			(t 
			 (if *Change_Root_Feature*
			     (cons (cons 'root *Current_Word*) feature-list)
			     (cons (cons 'root word) feature-list)
			 )
			)
		  )
                )
	      (geth word)
      )
  )
)

(defun root (word)
 "root - arg - word is actual string representing word. 
  Used to assign initial values to global symbols *Current_Word* and 
  *Negative_Feature*, and to enter *Current_Word*'s definition in lexicon at 
  end of the calls it makes to other routines for efficiency. Also adds feature
  neg if *Negative_Feature* is set etc."
 (declare (special *Change_Root_Feature* *Current_Word* *Negative_Feature* 
		   *Negatives* *Trace* *Tracing-On* *Count-New-Strings*
                   *trace-ending* *C_R_F*))
  (let (Def_Returned temp
	 (*package* (find-package 'snepsul)))
   (catch 'abort
    (setq temp
     (or 
      (geth word)
      (progn 
       (if (or *Trace* *Tracing-On*)
	  (setq *Tracing-On* 't)
	  (setq *Tracing-On* 'nil))
       (setq *Current_Word* word *Change_Root_Feature* t 
             *Negative_Feature* nil *Count-New-Strings* 0 
             *C_R_F* nil Def_Returned (root2 word))
       (definewords (cons *Current_Word* 
			 (setq temp 
			       (if (or
				    (and *Negative_Feature* Def_Returned)
				    (member word *Negatives* :test 'string=))
				 (Add_Negative Def_Returned)
				 Def_Returned)))
       )
       temp)
    )
   ))
  (setq *Tracing-On* nil *trace-ending* "")
  temp
  )
)

(defun root2 (word)
  "root2 - args: word returns: all the feature lists of word"
    (and (ISnaturalword word)
         (add-default-feats
           (or (lexic word)
               (suffix word)
               (Prefix word)))))

(defun Add_Negative (def_returned)
 "Add_Negative - arg def_returned is list of feature lists
  Adds neg feature to each feature list if it's not already there"
  (let (new_def)
    (dolist (feat_list def_returned new_def)
	    (cond ((assoc 'neg feat_list)
		   (setq new_def (cons feat_list new_def)))
		  (t (setq new_def (cons (cons '(neg . t) feat_list) new_def)))
	    )
    )
  )
)

(defvar *Negatives* '("a" "barely" "but" "cannot" "counter" "de" "di" "dis" 
  "doubt" "either" "fail" "few" "hardly" "il" "ill" "im" "in" "infrequently" 
  "ir" "less" "little" "negligible" "neither" "never" "no" "nobody" "non" 
  "none" "nor" "not" "nothing" "occasionally" "off" "only" "rarely" "scarely" 
  "seldom" "stop" "un")
 "Currently is what is accepted as getting a feature (neg . t) by program, 
  unless it already has that 'feature' in lexicon.")

(defun ISnaturalword (word)
  "ISnaturalword - args: word returns: t if word is composed of legal chars"
              (not(find-if-not #'standard-char-p word)))

(defun Has8CharStart (word)
 "Has8CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 8 letters"
   (member (subseq word 0 8)
			 '("spermato""straight""thorough") :test 'string=)
)

(defun Has7CharStart (word)
 "Has7CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 7 letters"
   (member (subseq word 0 7)   '("chamber""checker""chicken""clothes""concert"
      "council""counter""country""electro""feather""knuckle""leather""mercuro"
      "oscillo""passion""postero""quarter""scatter""spectro""through""thunder"
      "weather")   :test 'string=)
)
  
(defun Has6CharStart (word)
 "Has6CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 6 letters"
   (member (subseq word 0 6)  '("antero""bitter""breast""breech""bridge"
	"butter""calori""camera""candle""cardio""centro""change""cheese"
	"church""circum""coffee""collar""common""copper""corner""cotton"
	"dodeca""dollar""double""dragon""engine""figure""finger""fronto"
	"gastro""gentle""golden""ground""letter""master""megalo""methyl"
	"middle""needle""neutro""paddle""pepper""pigeon""pocket""psycho"
	"quadri""quanti""rudder""saddle""school""sclero""second""seismo"
	"shadow""shovel""silver""single""sports""spring""square""stereo"
	"stetho""street""strike""string""strong""switch""thermo""turtle"
	"window""yellow") :test 'string=)
)

(defun Has5CharStart (word)
 "Has5CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 5 letters"
   (member (subseq word 0 5)      '("acuti""after""alter""amnio""amphi""astro"
	"beach""belly""birth""black""block""blood""brain""bread""break""broad"
	"brown""brush""camel""cameo""carbo""carto""catch""centi""chain""chalk"
	"cheap""check""chemo""child""chuck""cirro""class""clavi""clean""cloak"
	"clock""close""cloth""cloud""coach""coast""color""coral""court""cover"
	"crack""crank""crazy""cream""cross""crown""cruci""curvi""cyclo""dairy"
	"death""devil""dorso""drain""dream""dress""drift""drill""drive""earth"
	"eight""elbow""ethno""every""extra""false""fiber""field""flash""flood"
	"floor""forth""frost""fungi""ghost""glass""globe""goose""grand""grape"
	"grass""grave""green""guard""guide""heart""heavy""horse""house""hydro"
	"hyper""hypno""icono""inter""intra""intro""ivory""juris""knock""light"
	"liver""macro""match""matri""metro""micro""mille""milli""money""mouse"
	"mouth""multi""never""nitro""north""ortho""other""paleo""palin""paper"
	"patch""patri""peace""pearl""penny""penta""pepto""petro""pheno""philo"
	"phono""photo""phylo""piece""piezo""pilot""pinch""place""plain""plano"
	"plate""porno""power""press""proto""punch""quasi""quick""quint""radio"
	"razor""ready""retro""right""river""rough""round""sacro""sales""sauce"
	"scape""scare""screw""seven""shake""sharp""sheep""shell""shirt""short"
	"sight""sleep""small""smoke""snake""solid""south""space""spark""spear"
	"speed""spino""stage""stair""stand""steam""steel""steno""stick""stink"
	"stock""stone""store""story""straw""sugar""super""supra""sweat""sweep"
	"sweet""sword""table""tetra""there""thick""third""thumb""tibio""tight"
	"tooth""touch""trade""train""trans""turbo""under""waist""waste""watch"
	"water""wheel""where""whirl""whole""world""wrist") :test 'string=)
)

(defun Has4CharStart (word)
 "Has4CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 4 letters"
   (member (subseq word 0 4)  '("alti""ante""anti""aqua""arch""arti""atmo"
        "auto""back""bare""barn""baro""base""basi""bean""bear""beef""bell"
	"bene""bili""bird""blow""blue""boat""body""bomb""bone""book""boot"
	"boro""bull""call""card""care""case""cash""cast""cata""cess""city"
	"clap""claw""clod""club""coal""coat""cock""cole""coli""come""cook"
	"coon""copy""core""cork""corn""crab""crib""crow""curb""dark""dash"
	"date""dead""deck""deer""dino""dish""dock""door""dove""down""drag"
	"draw""drop""drum""duck""dumb""dust""ecto""equi""etho""ever""fall"
	"farm""fast""feed""fire""fish""fist""flag""flat""flea""food""fool"
        "foot""fore""foul""free""frog""gain""game""gang""gate""gear""genu"
	"giga""gold""good""goof""gyro""hair""hand""hang""hare""head""hell"
	"hemi""hemo""here""hexa""hide""high""hill""hind""home""homo""hook"
	"horo""hour""hypo""inch""iono""iron""jack""jail""kick""kilo""king"
	"knot""lace""lady""lamp""land""life""line""lock""long""look""love"
	"lung""mail""main""make""manu""meat""medi""mega""melo""meno""meta"
	"milk""mine""mini""mito""mono""moon""muck""nano""neck""news""nine"
	"noon""note""noto""octo""omni""over""pack""para""pari""pass""peep"
	"peri""pick""pico""pilo""pipe""play""plow""plug""pneo""poly""poor"
	"pork""poro""post""pull""pump""push""race""rail""rain""ribo""ring"
        "road""rock""rool""root""rope""safe""sail""salt""sand""self""semi"
	"shin""ship""shoe""shop""shot""show""shut""sick""side""sign""silk"
	"skin""slap""slip""slow""snap""snow""soap""soft""some""soup""sour"
	"spin""spit""star""step""stop""tail""take""taxi""tear""tele""tera"
	"tide""time""toad""toll""tool""trap""tree""true""turn""type""vine"
	"vivi""walk""wall""wash""wave""what""when""whip""wide""wild""wind"
	"with""wood""wool""word""work""worm""yard""year") :test 'string=)
)

(defun Has3CharStart (word)
 "Has3CharStart - args word
  function that returns true if have a stored prefix or word beginning, of 3 
  letters"
  (member (subseq word 0 3)'("abs""acu""air""ali""all""ant""any""arm""avi""bad"
    "bag""bar""bed""big""bin""bio""bou""bow""box""bug""bur""car""cat""cod""cog"
    "col""com""con""cor""cow""cup""cut""day""dew""die""dif""dis""dog""dry""dys"
    "ear""eco""eel""egg""elk""end""epi""eye""fan""far""fat""fin""fly""fog""for"
    "fox""gar""gas""gem""geo""get""god""gum""hat""hay""hen""hip""hog""hot""ice"
    "ill""ink""iso""jaw""jet""key""lay""law""lee""leg""log""mad""mal""man""may"
    "mer""mid""mis""mon""mud""neo""new""non""nor""nut""odd""off""oxy""pan""par"
    "pay""pea""peg""pen""per""pig""pin""pit""pop""pot""pre""pro""put""rag""rat"
    "raw""rib""rip""run""sap""saw""sea""set""sit""sub""sun""sur""sym""syn""tax"
    "tea""ten""tie""tin""tip""tom""top""tow""tri""twi""uni""war""wet""who")
	  :test 'string=)
)

(defun Has2CharStart (word)
 "Has2CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 2 letters"
   (member (subseq word 0 2)   '("ab""ac""ad""af""ag""al""am""an""ap""ar""as"
     "at""be""by""co""de""di""ec""ef""em""en""ex""go""il""im""in""ir""no""oc"
     "on""op""re""se""so""un""up")
	   :test 'string=)
)

(defun Has1CharStart (word)
 "Has1CharStart - args word
  function that returns true if have a stored prefix or word beginning,
  of 1 letter"
   (member (subseq word 0 1) '("a""e""-")
	 :test 'string=)
)

(defvar *Actual_Prefixes* '("ab" "abs" "ac" "acu" "acuti" "ad" "af" "ag" "al" 
  "ali" "alti" "am" "amnio" "amphi" "an" "antero" "anti" "ap" "ar" "arti" 
  "astro" "atmo" "avi" "baro" "basi" "bene" "bili" "bio" "boro" "bou" "bur" 
  "calori" "carbo" "cardio" "carto" "cata" "centi" "centro" "chemo" "circum" 
  "cirro" "clavi" "co" "coli" "com" "con" "cor" "cruci" "curvi" "cyclo" "de" 
  "di" "dif" "dino" "dis" "dodeca" "dorso" "dys" "e" "ec" "eco" "ecto" "ef" 
  "electro" "em" "en" "epi" "equi" "ethno" "etho" "ex" "fronto" "gastro" "genu"
  "geo" "giga" "hemi" "hemo" "hexa" "horo" "hydro" "hyper" "hypno" "icono" "il"
  "ill" "im" "inter" "intra" "intro" "iono" "ir" "iso" "juris" "macro" "mal" 
  "manu" "matri" "medi" "mega" "megalo" "melo" "meno" "mer" "mercuro" "meta" 
  "methyl" "metro" "micro" "mille" "milli" "mini" "mis" "mito" "mon" "mono" 
  "multi" "nano" "neo" "neutro" "nitro" "non" "nor" "noto" "oc" "octo" "omni" 
  "op" "ortho" "oscillo" "oxy" "paleo" "palin" "para" "pari" "patri" "penta" 
  "pepto" "per" "peri" "petro" "pheno" "philo" "phono" "phylo" "pico" "piezo" 
  "pilo" "plano" "pneo" "poly" "porno" "poro" "postero" "pre" "proto" "quadri" 
  "quanti" "re" "retro" "ribo" "sacro" "sclero" "se" "seismo" "semi" "spectro"
  "spermato" "spino" "stetho" "sub" "supra" "sur" "sym" "syn" "tele" "tera" 
  "tetra" "thermo" "tibio" "trans" "tri" "turbo" "twi" "un" "uni" "vivi" "-")
 "*Actual_Prefixes* is a list of current prefixes implemented in this program.")

(defvar *Simple_Prefixes* '("anti" "dis" "il" "im" "in" "ir" "mis" "pre" "re" 
			    "semi" "un")
 "*Simple_Prefixes* -
  These prefixes were in the package originally. These prefixes are the only
  ones which can be used without suffixes etc. and get an answer from package.")

(defun StartWord_NotIn_Lex (removed_prefix whole_word word)
 "StartWord_NotIn_Lex - args - word is string representing the found word
                               removed_prefix is string without prefix or word
                               whole_word is the unmodified actual word string.
 Routine is based on the idea of having a default lexicon of words which if
 encountered during run of program they are added to the lexicon definition.
 If a word is a prefix or if a word is already in the lexicon then call routine
 root2 with prefix or word beginning removed, else construct the words 
 definition, and add it to the lexicon hash table *lexicon*, and also output 
 the new entry to the terminal screen, and call routine root2 with the 
 unmodified word string."
 (declare (special *Actual_Prefixes* *AtoK_Nouns_Start_Word* *Verbs_Start_Word*
    *LtoZ_Nouns_Start_Word* *Adjectives_Start_Word* *Adverbs_Start_Word* 
    *Prep_Start_Word* *Negatives* *Negative_Feature* *Simple_Prefixes*
    *Tracing-On*))
 (when (member word *Negatives* :test 'string=)
       (setq *Negative_Feature* t))
 (if (and (not (member word *Simple_Prefixes* :test 'string=))
	  (geth removed_prefix))
   nil
   (if (or (member word *Actual_Prefixes* :test 'string=)
	   (geth word))
     (root2 removed_prefix)
     (let (def)
       (when *Tracing-On*
	  (write-string "(")
	  (setq def (remove nil (list
		(when (or  (member word *AtoK_Nouns_Start_Word* :test 'string=)
			   (member word *LtoZ_Nouns_Start_Word* :test 'string=))
			(write `((ctgy . n) (num . sing) (root . ,word))))
		(when (member word *Verbs_Start_Word* :test 'string=)
			(print `((ctgy . v) (tense . pres) (root . ,word))))
		(when (member word *Adjectives_Start_Word* :test 'string=)
			(print `((ctgy . adj) (root . ,word))))
		(when (member word *Adverbs_Start_Word* :test 'string=)
			(print `((ctgy . adv) (root . ,word))))
		(when (member word *Prep_Start_Word* :test 'string=)
			(print `((ctgy . prep) (root . ,word)))))
	  ))
	  (write-string ")")
	  (terpri))
       (when (not *Tracing-On*)
	  (setq def (remove nil (list
		(when (or  (member word *AtoK_Nouns_Start_Word* :test 'string=)
			   (member word *LtoZ_Nouns_Start_Word* :test 'string=))
			 `((ctgy . n) (num . sing) (root . ,word)))
		(when (member word *Verbs_Start_Word* :test 'string=)
			 `((ctgy . v) (tense . pres) (root . ,word)))
		(when (member word *Adjectives_Start_Word* :test 'string=)
			 `((ctgy . adj) (root . ,word)))
		(when (member word *Adverbs_Start_Word* :test 'string=)
			 `((ctgy . adv) (root . ,word)))
		(when (member word *Prep_Start_Word* :test 'string=)
			 `((ctgy . prep) (root . ,word)))))
       ))
       (let ((*package* (find-package 'snepsul)))
	    (definewords (cons word def)))
       (root2 whole_word)
       )
    )
  )
)

(defvar *AtoK_Nouns_Start_Word* 
 '("air" "all" "ant" "ante" "aqua" "arch" "arm" "as" "at" "auto" "back" "bad" 
   "bag" "bar" "barn" "base" "beach" "bean" "bear" "bed" "beef" "bell" "belly"
   "bin" "bird" "birth" "black" "block" "blood" "blow" "blue" "boat" "body" 
   "bomb" "bone" "book" "boot" "bow" "box" "brain" "bread" "break" "breast" 
   "breech" "bridge" "broad" "brown" "brush" "bug" "bull" "butter" "call" 
   "camel" "cameo" "camera" "candle" "car" "card" "care" "case" "cash" "cast"
   "cat" "catch" "cess" "chain" "chalk" "chamber" "change" "check" "checker"
   "cheese" "chicken" "child" "church" "city" "clap" "class" "claw" "cloak" 
   "clock" "clod" "cloth" "clothes" "cloud" "club" "coach" "coal" "coast"
   "coat" "cock" "cod" "coffee" "cog" "col" "cole" "collar" "color" "common" 
   "concert" "cook" "coon" "copper" "copy" "coral" "core" "cork" "corn" 
   "corner" "cotton" "council" "counter" "country" "court" "cover" "cow" "crab"
   "crack" "crank" "crazy" "cream" "crib" "cross" "crow" "crown" "cup" "curb"
   "cut" "dairy" "dark" "dash" "date" "day" "death" "deck" "deer" "devil" "dew"
   "die" "dish" "dock" "dog" "dollar" "door" "double" "dove" "down" "drag"
   "dragon" "drain" "draw" "dream" "dress" "drift" "drill" "drive" "drop" 
   "drum" "duck" "dust" "ear" "earth" "eel" "egg" "eight" "elbow" "elk" "end" 
   "engine" "extra" "eye" "fall" "fan" "far" "farm" "fat" "feather" "feed" 
   "fiber" "field" "figure" "fin" "finger" "fire" "fish" "fist" "flag" "flash"
   "flat" "flea" "flood" "floor" "fly" "fog" "food" "fool" "foot" "fox" "frog"
   "frost" "fungi" "gain" "game" "gang" "gar" "gas" "gate" "gear" "gem" "get"
   "ghost" "glass" "globe" "god" "gold" "good" "goof" "goose" "grand" "grape" 
   "grass" "grave" "green" "ground" "guard" "guide" "gum" "gyro" "hair" "hand"
   "hare" "hat" "hay" "head" "heart" "heavy" "hell" "hen" "high" "hill" "hind"
   "hip" "hog" "home" "homo" "hook" "horse" "hour" "house" "hypo" "ice" "ill" 
   "in" "inch" "ink" "iron" "ivory" "jack" "jail" "jaw" "jet" "key" "kick" 
   "kilo" "king" "knock" "knot" "knuckle")
 "Default nouns (form a to k in alphabet) which might be added to the hash 
  table *lexicon* dynamically by the program, if they are encountered and 
  aren't already in the lexicon. 2 noun symbols are used for readability.")

(defvar *LtoZ_Nouns_Start_Word*
 '("lace" "lady" "lamp" "land" "law" "leather" "lee" "leg" "letter" "life" 
  "light" "line" "liver" "lock" "log" "long" "look" "love" "lung" "mad" "mail"
  "main""make" "man" "master" "match" "meat" "methyl" "mid" "middle" "milk" 
  "mine" "money" "moon" "mouse" "mouth" "muck" "mud" "neck" "needle" "news"
  "nine" "noon" "north" "note" "nut" "odd" "other" "pack" "paddle" "pan" 
  "paper" "par" "pass" "passion" "patch" "pay" "pea" "peace" "pearl" "peep"
  "peg" "pen" "penny" "pepper" "photo" "pick" "piece" "pig" "pigeon" "pilot"
  "pin" "pinch" "pipe" "pit" "place" "plain" "plate" "play" "plow" "plug" 
  "pocket" "pop" "pork" "post" "pot" "power" "press" "pro" "psycho" "pull" 
  "pump" "punch" "put" "quarter" "quint" "race" "rag" "radio" "rail" "rain" 
  "rat" "razor" "ready" "rib" "right" "ring" "rip" "river" "road" "rock" "rool"
  "root" "rope" "rough" "round" "rudder" "run" "saddle" "safe" "sail" "sales" 
  "salt" "sand" "sap" "sauce" "saw" "scape" "scare" "scatter" "school" "screw"
  "sea" "second" "self" "set" "seven" "shadow" "shake" "sharp" "sheep" "shell"
  "shin" "ship" "shirt" "shoe" "shop" "shot" "shovel" "show" "shut" "side" 
  "sight" "sign" "silk" "silver" "single" "skin" "slap" "sleep" "slip" "small"
  "smoke" "snake" "snap" "snow" "soap" "soft" "solid" "some" "soup" "sour" 
  "south" "space" "spark" "spear" "speed" "spin" "spit" "sports" "spring" 
  "square" "stage" "stair" "stand" "star" "steam" "steel" "steno" "step"
  "stereo" "stick" "stink" "stock" "stone" "stop" "store" "story" "straight"
  "straw" "street" "strike" "string" "sub" "sugar" "sun" "super" "sweat"
  "sweep" "sweet" "switch" "sword" "table" "tail" "take" "tax" "taxi" "tea"
  "tear" "ten" "there" "thick" "third" "thumb" "thunder" "tide" "tie" "time"
  "tin" "tip" "toad" "toll" "tom" "tool" "tooth" "touch" "tow" "trade" "train"
  "trap" "tree" "true" "turn" "turtle" "type" "up" "vine" "waist" "walk" "wall"
  "war" "wash" "waste" "watch" "water" "wave" "weather" "wet" "what" "wheel" 
  "where" "whip" "whirl" "who" "whole" "wind" "window" "wood" "wool" "word" 
  "work" "world" "worm" "wrist" "yard" "year" "yellow")
 "Default nouns (from l to z in alphabet) which might be added to the hash 
  table *lexicon* dynamically by the program, if they are encountered and 
  aren't already in the lexicon. 2 noun symbols are used for readability.")

(defvar *Verbs_Start_Word*
 '("air" "alter" "arm" "back" "bag" "be" "bear" "block" "blow" "bomb" "boot" 
   "bow" "box" "brain" "break" "breast" "breech" "brush" "butter" "call" "care"
   "cash" "cast" "catch" "change" "check" "chuck" "clap" "claw" "clean" "cloak"
   "clock" "close" "cloud" "club" "coach" "coast" "coat" "cock" "cog" "collar"
   "color" "come" "cook" "copy" "cork" "court" "cover" "crack" "crank" "cream"
   "cross" "crow" "crown" "cup" "curb" "cut" "dash" "date" "die" "dish" "dock"
   "dove" "drag" "drain" "draw" "dream" "dress" "drift" "drill" "drive" "drop"
   "drum" "dry" "duck" "dust" "eight" "elbow" "end" "eye" "fall" "fan" "farm" 
   "feather" "feed" "field" "figure" "finger" "fire" "fish" "flag" "flash" 
   "flood" "floor" "fly" "fog" "food" "fool" "foot" "foul" "fox" "free" "frost"
   "gang" "gear" "get" "go" "goof" "grass" "ground" "guard" "guide" "gum"
   "hand" "hang" "head" "hide" "hill" "hog" "hook" "iron" "jack" "jet" "kick" 
   "knock" "knot" "lace" "land" "lay" "letter" "light" "line" "lock" "log" 
   "look" "love" "mail" "make" "match" "may" "milk" "mine" "mouth" "muck" 
   "neck" "needle" "note" "over" "pack" "paddle" "pan" "pass" "patch" "pay"
   "peep" "pepper" "pick" "pilot" "pin" "pinch" "pipe" "place" "plate" "play"
   "plow" "plug" "pocket" "pop" "post" "pot" "press" "pull" "pump" "punch"
   "push" "quarter" "race" "radio" "rain" "right" "ring" "rip" "rock" "rool" 
   "rope" "rough" "run" "saddle" "sail" "salt" "sand" "saw" "scare" "scatter"
   "screw" "second" "set" "shadow" "shake" "shell" "ship" "shop" "shot" 
   "shovel" "show" "shut" "side" "sight" "sign" "sit" "skin" "slap" "sleep" 
   "slip" "slow" "smoke" "snap" "snow" "soap" "space" "spark" "spear" "speed"
   "spin" "spit" "sports" "spring" "square" "stage" "stand" "star" "steam"
   "step" "stick" "stink" "stock" "stone" "stop" "store" "strike" "string"
   "sugar" "sun" "sweat" "sweep" "switch" "table" "tail" "take" "tax" "taxi"
   "tear" "thumb" "thunder" "tide" "tie" "time" "tip" "toll" "top" "touch"
   "tow" "trade" "train" "trap" "turn" "type" "up" "walk" "wall" "war" "wash"
   "waste" "watch" "water" "wave" "weather" "wet" "wheel" "whip" "whirl" "wind"
   "work")
 "Default verbs which might be added to the hash table *lexicon* dynamically
  by the program, if they are encountered and aren't already in the lexicon.")

(defvar *Adjectives_Start_Word*
 '("all" "any" "back" "bad" "bare" "base" "big" "bitter" "black" "blood" 
   "blue" "breech" "broad" "brown" "bull" "cheap" "chicken" "clean" "close" 
   "common" "con" "concert" "copper" "corner" "council" "counter" "crazy" 
   "dairy" "dark" "dead" "double" "down" "dress" "dry" "dumb" "eight" "end"
   "every" "extra" "false" "fast" "fat" "flat" "fore" "foul" "free" "game"
   "gentle" "go" "gold" "golden" "good" "grand" "grave" "green" "ground"
   "heavy" "here" "high" "hind" "home" "hot" "ill" "in" "ivory" "jet" "key"
   "lay" "leather" "lee" "light" "log" "long" "mad" "main" "master" "mid"
   "middle" "mud" "new" "nine" "no" "noon" "north" "odd" "off" "on" "other" 
   "over" "paper" "par" "passion" "pay" "pearl" "pilot" "plain" "pocket" "poor"
   "pop" "power" "pro" "quarter" "quick" "radio" "rag" "raw" "ready" "right"
   "rool" "rough" "round" "safe" "second" "set" "seven" "sharp" "short" "sick"
   "side" "silk" "silver" "single" "slow" "small" "snap" "soft" "solid" "some"
   "sour" "south" "sports" "square" "steel" "stock" "straight" "straw" "strong"
   "super" "sweet" "table" "tail" "ten" "there" "thick" "third" "thorough" 
   "through" "tight" "time" "top" "true" "under" "up" "wet" "what" "whole"
   "wide" "wild" "yellow")
 "Default adjectives which might be added to the hash table *lexicon* 
  dynamically by the program, if they are encountered and aren't already in the
  lexicon.")

(defvar *Adverbs_Start_Word*
 '("after" "all" "any" "as" "back" "big" "by" "clean" "con" "counter" "dead" 
   "double" "down" "ever" "false" "far" "fast" "flat" "fore" "forth" "free"
   "here" "home" "ill" "light" "long" "mouse" "never" "no" "off" "on" "over"
   "post" "pro" "quasi" "right" "rool" "second" "sharp" "short" "slow" "small"
   "so" "soft" "some" "south" "straight" "strong" "there" "tight" "true" 
   "under" "up" "what" "when" "where" "whole" "wide" "wild")
 "Default adverbs which might be added to the hash table *lexicon* dynamically
  by the program, if they are encountered and aren't already in the lexicon.")

(defvar *Prep_Start_Word*
 '("a" "after" "as" "at" "bar" "by" "down" "for" "in" "mid" "off" "on" "over" 
   "rool" "round" "through" "under" "up" "with")
 "Default prepositions which might be added to the hash table *lexicon* 
  dynamically by the program, if they are encountered and aren't already in the
  lexicon.")

(defun Prefix (word)
  "Prefix - args: word
   Checks to see if word begins with a sublist corresponding to valid prefix or
   word beginning. involves some 800 word beginnings and prefixes."
    (declare (special *Change_Root_Feature* *trace-ending*))
    (setq *Change_Root_Feature* t *trace-ending* "")
    (cond ((string= word "") nil)
	  ((and (> (length word) 7) (Has8CharStart word)
	   (StartWord_NotIn_Lex (subseq word 8) word (subseq word 0 8))))
	  ((and (> (length word) 6) (Has7CharStart word)
	   (StartWord_NotIn_Lex (subseq word 7) word (subseq word 0 7))))
	  ((and (> (length word) 5) (Has6CharStart word)
	   (StartWord_NotIn_Lex (subseq word 6) word (subseq word 0 6))))
	  ((and (> (length word) 4) (Has5CharStart word)
	   (StartWord_NotIn_Lex (subseq word 5) word (subseq word 0 5))))
	  ((and (> (length word) 3) (Has4CharStart word)
	   (StartWord_NotIn_Lex (subseq word 4) word (subseq word 0 4))))
	  ((and (> (length word) 2) (Has3CharStart word)
	   (StartWord_NotIn_Lex (subseq word 3) word (subseq word 0 3))))
	  ((and (> (length word) 1) (Has2CharStart word)
	   (StartWord_NotIn_Lex (subseq word 2) word (subseq word 0 2))))
	  ((and (> (length word) 0) (Has1CharStart word) 
		(not (geth (subseq word 1)))
		(root2 (subseq word 1))))
          (t nil))
)

(defvar *Trace* nil
 "*Tracing* - if this variable is non-nil then all the strings that are
 looked up in the lexicon will be output, until it becomes 'nil. This
 variable can be set by the user for continuous trace output.")

(defvar *Tracing-On* nil
 "*Tracing-On* - if this variable is non-nil then all the strings that are
 looked up in the lexicon will be output.")

(defun lookup (word &optional set-trace)
  "lookup - args:  word,  set-trace is  non-nil  for tracing"
 (declare (special *Tracing-On*))
  (cond ((not (stringp word)) nil)
        (set-trace 
	 (setq *Tracing-On* t) (root word))
	((null (symbol-function 'root))
	 (lexic word))
	(t (root word))
	;;the following is what is in the original code:
	;;(or (null (valuep 'tlvlxx)) (greaterp tlvlxx 0))
	;;i changed it to this temporarily:
        ;;((or (null *trace-level*)(> *trace-level* 0))
        ;;(format t  "% ~s not in dictionary" word)
        ;;nil)
	)
)

(defun s (without_suffix)
  "s - args: without_suffix"
  (declare (special *Change_Root_Feature* *C_R_F*))
  (when (not *C_R_F*)
	(setq *Change_Root_Feature* nil))
  (if (or (null without_suffix) (string= "" without_suffix)) nil
      (let (temp)
	(append 
	 (up* '((num . plur)) '(n) (setq temp (Spelling_Rules without_suffix)))
	 (up* '((ctgy . v)) '(n v) temp)
	)
      )
  )
)

(defun en (without_suffix) 
 "en - args: without_suffix" 
  (declare (special *Change_Root_Feature* *C_R_F*))
  (when (not *C_R_F*)
	(setq *Change_Root_Feature* nil))
  (if (or (null without_suffix) (string= "" without_suffix)) nil
    (let ((temp (Spelling_Rules without_suffix)) temp2)
      (setq temp2 
	    (when 
	     (char-equal (char without_suffix (1- (length without_suffix)))
			 '#\r)
	     (up* '((num . plur)) '(n)
		  (Spelling_Rules 
		   (subseq without_suffix 0 (1- (length without_suffix)))))))
      (setq temp2 
	    (append temp2
		    (up* '((num . plur)) '(n)
			 temp)
		    (up* '((presprt . t)) '(v)
			 temp)))
      (when (not *C_R_F*)
	    (setq *Change_Root_Feature* t))
      (setq temp (Spelling_Rules without_suffix))
      (append temp2
	      (up* '((ctgy . adj)) '(n adj) temp)
	      (up* '((ctgy . v)) '(n adj adv) temp))
    )
  )
)

(defun y (without_suffix)
  "y - args: without_suffix"
  (if (or (null without_suffix) (string= "" without_suffix)) nil
    (let (temp) 
      (append 
       (up* 
	'((ctgy . adj)) '(v n adj) 
	(setq temp (Spelling_Rules without_suffix)))
       (up* '((ctgy . n)) '(v n) temp))
    )
  )
)

(defun lim (limit def)	
  "lim - args: limit def"
  ;; returns only one of the feature-pair sublists of the feature-pair list of
  ;; a given word (which is gotton by call to (lexic word))  
  ;; -- that has in it the feature-pair element : (ctgy . limit)
  ;; example: limit: (z) def : (((ctgy . z)..)((ctgy . z)..)((ctgy . other)..))
  ;; --> (((ctgy . z)..))
  (declare (special *Change_Root_Feature*))
    (cond ((null def) nil)
          ((member (get-lexical-feature 'ctgy (car def)) limit)
	       (list (car def)))
          (t (lim limit (cdr def)))))

(defun helper-remove-feature (lst)
 "helper-remove-feature - arg lst is feature list.
  Removes non-ctgy, or non-root features if they're present."
  (cond ((null lst) nil)
	((not (or (eql (caar lst) 'root)
		  (eql (caar lst) 'ctgy)))
	 (helper-remove-feature (rest lst)))
	(t (cons (first lst) (helper-remove-feature (rest lst))))
  )
)

(defun remove-features (def)
 "remove-features arg - def is definition from lexicon.
  If a tense is in a lexicon definition then remove it. This is because
  the package adds tenses but doesn't use them."
  (let (adjusted-def)
    (dolist (feature-list def adjusted-def)
	    (setq adjusted-def 
		  (cons (helper-remove-feature feature-list) adjusted-def))
    )
  )
)

(defun up* (cng limit def)
  "up* - cdg limit def"
   (update cng (lim limit (remove-features def)))
)

(defun update (cng def)
  "update - args: cng def"
    (cond ((null def) nil)
          ((null cng) def)
          (t
           (update (cdr cng) (list (enterp (car cng) (caar cng) (car def)))))))

(defun enterp (entry type dlist)
  "enterp - args: entry type dlist"
  ;; given the feature pair sublist, dlist, (gotton by call to lexic word)
  ;; the feature pair element element entry (a pair) -- as well the the name 
  ;; of the feature type -- enterp searches for the feature type in dlist and 
  ;; 1- adds it if  its not there
  ;; 2- changes the pair to look like entry if theres a feature value for type
  ;; 3- leaves alone if it contains it
  ;; entry= (num . plural) type=num dlist=((ctgy.n)(num.sing))
  ;; --> ((ctgy.n)(num.plural))
    (cond ((null dlist) (list entry))
          ((eq (caar dlist) type) (cons entry (cdr dlist)))
          (t (cons (car dlist) (enterp entry type (cdr dlist))))))

(defun get-lexical-feature (feature feature-list)
  "get-lexical-feature  - args: feature feature-list"
   ;;returns the value (if any) for given feature in the feature-list
    (cdr (assoc (intern (symbol-name feature) (find-package 'snepsul))
		feature-list)))	


(defun lookup-lexical-feature (feature lexeme)
  "looks up lexeme in the lexicon,
   and returns the first value of the feature it finds among the lexcical entries."
  (do* ((ifeature (intern (symbol-name feature) (find-package 'snepsul)))
	(lexentries (geth lexeme) (rest lexentries))
	(value (cdr (assoc ifeature (first lexentries)))
	       (cdr (assoc ifeature (first lexentries)))))
       ((or value (null (rest lexentries))) value)))

(defun add-feats (pairs1 pairs2)
 "add-feats args pairs1 is a feature list unchanged
                pairs2 is possible modified feature list or nil."
  (if (null pairs2)
      pairs1
      pairs2
  )
)

(defun add-default-feats (lx)
  "add-default-feats - args: lx"
    (mapcar #'(lambda (pairs)
		(add-feats pairs
			   (case (get-lexical-feature 'ctgy pairs)
				 (n (noun_add-num pairs))
				 (v (verb_add-tense pairs))
			   )
	        )
	      )
    lx)
)

(defun noun_add-num (pairs)
 "noun_add-num - args  pairs is a single feature list of a definition.
  If a num feature isn't already present, then routine adds a default."
    (if (assoc 'num pairs) 
	pairs
        (cons '(num . sing) pairs)))

(defun verb_add-tense (pairs)
 "verb_add-tense - args  pairs is a single feature list of a definition.
  If one of 6 tenses isn't already present, then routine adds a default tense."
    (cond ((assoc 'pastp pairs) pairs)
	  ((assoc 'pprt pairs) pairs)
	  ((assoc 'presprt pairs) pairs)
	  ((assoc 'presp pairs) pairs)
          ((assoc 'tense pairs) pairs)
          (t (cons '(tense . pres) pairs))))

;;to substitute for the original call (put entry '=dict val)
;;associates with a word(the entry) a list of feature-lists
(defun puth (entry val)
  "putf - args: word feature-lists"
  (when (null *lexicon*)
    (setq *lexicon* (make-hash-table :test 'equal)))
  (setf (gethash entry *lexicon*)
    (add-default-root entry (add-default-feats val))))

(defun add-default-root (lexeme feats)
  "Returns the list of feature entries, FEATS, with a root entry of LEXEME
added to every feature list that doesn't already have one."
  (mapcar #'(lambda (featlist)
	      (if (assoc 'root featlist)
		  featlist
		(cons (cons 'root lexeme) featlist)))
	  feats))

;; to substitute for the original call (get entry '=dict)
;; gets the feature lists associated with entry
(defun geth (entry)
  "geth - args: entry"
  (if (null *lexicon*)
      nil
      (gethash entry *lexicon*))
)

;; expects that lexicon file contains lexical entries as defined in the
;; sneps manual. 
;; if you want to preserve case in filename -- call function with
;; a string: (lexin "lexicon")...else all gets converted to upper case.
(defsnepscom lexin ((file))
  "lexin (macro) - args: list of file names"
   (init)
   (let (lwords undefs input *feature-values*
	 (*package* (find-package 'snepsul)))
	 (declare (special undefs *feature-values*))
	 ;; don't waste hash tables
	 (cond ((and (boundp '*lexicon*)
		      (hash-table-p *lexicon*))
		(clrhash *lexicon*))
	       (t (setq *lexicon* (make-hash-table :test 'equal))))
	 (with-open-file (inunit file :direction :input)
	   (loop
	     (setq input (read inunit nil :eof))
	     (cond ((eq input :eof)
		    (return t))
		   (t
		    (setq lwords
			  (nconc lwords (definewords input t))))))
	   )
	 (mapc #'(lambda (word)
		   (setq undefs (find-and-del word undefs)))
	       lwords)
	 (if undefs
	     (format t "undefined- ~s~%" undefs))
	 lwords))

(defmacro lex-in (x)
  "lex-in - args: file-names"
  `(apply 'lexin ,x))

(defun find-and-del (target undefs)
  "near as i can figure, this steps through an a-list removing references to 
   the target."
  (do
    ((sofar nil (append sofar (list (car undefs)))))
    ((null undefs) sofar)
    (if (equal (cadar undefs) target)
	(return (append sofar (cdr undefs))))
    (setq undefs (cdr undefs))
  )
)

(defun mem* (item list)
  "``reverse member function'' -- like rassoc, but returns t if item is
   second element (not just cdr) of any item in list, and nil otherwise"
  (if list
      (or (null item)
	  (not (null (member item list :test #'equal :key #'second)))
       )))    ; changed 07/01/88 ssa, force use of member rather than zlc:member.

(defun definewords (words &optional lexin-flag &aux (word (car words)) (lpairs (cdr words)) rt)
  "defines words in *lexicon*, identifies words with roots, and exports/imports
   values of features to the parser package. - arg: words"
  (declare (special undefs *feature-values*))
  (puth word lpairs)
  (dolist (lpair lpairs (list word))
    (setq rt (get-lexical-feature (intern "root") lpair))
    (when lexin-flag
	  (unless (mem* rt undefs)
		  (push (list word rt) undefs)))
    ;; export/import feature values into the parser package.
    (dolist (pair lpair)			; one feature at a time.
      (let ((feature (cdr pair)))
	(unless (or (stringp feature)
		    (and lexin-flag (member feature *feature-values*)))
	  (cond ((listp feature)
		   (setq feature (parser::flatten feature))          ; change 07/01/88 ssa: allow 
		   (if (atom feature) (setq feature (list feature))) ; 1-element feature lists.
	           (dolist (elt feature)
		     (unless (or (characterp elt)
				 (stringp elt))
		       (export elt)
		       (shadowing-import elt (find-package 'parser)))))
		((not (characterp feature))
		   (export feature)
		   (shadowing-import feature (find-package 'parser))))
	  (when lexin-flag
		(push feature *feature-values*)))
	))))

(defvar *Seven_Char_End* nil
 "*Seven_Char_End* contains all 7 character word endings and suffixes")

(defvar *Six_Char_End* nil
 "*Six_Char_End* contains all 6 character word endings and suffixes")

(defvar *Five_Char_End* '("stand")
 "*Five_Char_End* contains all 5 character word endings and suffixes")

(defvar *Four_Char_End* '("fold" "meal" "ough" "side" "some")
 "*Four_Char_End* contains all 4 character word endings and suffixes")

(defvar *Three_Char_End* '("ant""ate""ent""ept""est""ful""ile""ine""ing""ite"
                           "oid""ory")
 "*Three_Char_End* contains all 3 character suffixes and word endings.")

(defvar *Two_Char_End* '("al""ed""en""er""es" "id" "re")
 "*Two_Char_End* contains all 2 character suffixes.")

(defvar *One_Char_End* '("y")
 "*One_Char_End* contains all 1 character suffixes.")

(defun Has_Size_Ending (word size)
 "Has_Size_Ending - args - word string representing actual word
                          size is size of ending routine is testing for.
  Routine checks if have a suffix or word ending reconized by this program,
  and sets global variable (*trace-ending*) to that ending."
 (declare (special *Seven_Char_End* *Six_Char_End* *Five_Char_End* 
		   *Four_Char_End* *Three_Char_End* *Two_Char_End* 
		   *One_Char_End* *trace-ending*))
 (and (> (length word) (1- size))
      (setq *trace-ending* 
       (first 
        (member (subseq word (- (length word) size) (length word))
	      (case size
		    (4 *Four_Char_End*)
		    (3 *Three_Char_End*)
		    (5 *Five_Char_End*)
		    (6 *Six_Char_End*)
		    (2 *Two_Char_End*)
		    (7 *Seven_Char_End*)
		    (1 *One_Char_End*)
	      )
	      :test 'string=)
       )
      )
 )
)

(defun remove_suffix (l length_suffix)
 "remove_suffix - args: l is a word with a suffix, length_suffix is length of 
  suffix. Routine is used to reduce duplicated code."
   (subseq l 0 (- (length l) length_suffix))
)

(defun get_current_suff (l length_suffix)
 "get_current_suff -args l is a word with a suffix, length_suffix is length of
  suffix. Routine is used to reduce duplicated code."
   (subseq l (- (length l) length_suffix) (length l))
)

(defvar *Actual_Suffixes* '("al""ate""ed""en""er""es""ept""est""ful""ile""ine"
			    "ing""ite""oid""ory""y")
 "*Actual_Suffixes* is a list of actual suffixes this program handles. 
  Initialized by routine Var_Suffixes mostly.")

(defun EndWord_NotIn_Lex (word)
 "EndWord_NotIn_Lex - args - word is string
  routine is based on having a default lexicon of words, so that if one of
  these words is encountered during the run of the program it is added to the
  hash table *lexicon* which represents the lexicon.
  if word is a suffix or if word is already in the lexicon return nil, else
  construct the definition of the word and enter it in lexicon, as well as
  output the new definition to the terminal screen."
 (declare (special *Actual_Suffixes* *Nouns_End_Word* *Verbs_End_Word* 
	*Adjectives_End_Word* *Adverbs_End_Word* *Preps_End_Word* *Negatives* 
	*Negative_Feature* *Tracing-On*))
 (when (and (member word *Negatives* :test 'string=) 
            (not (string= "im" word)))
       (setq *Negative_Feature* t))
 (if (or (member word *Actual_Suffixes* :test 'string=)
	 (geth word))
	nil
	(let (def)
	  (when *Tracing-On*
		(write-string "(")
		(setq def (remove nil (list
			 (when (member word *Nouns_End_Word* :test 'string=)
			       (write `((ctgy . n) (num . sing) (root . ,word))))
			 (when (member word *Verbs_End_Word* :test 'string=)
			       (print `((ctgy . v) (tense . pres) (root . ,word))))
			 (when (member word *Adjectives_End_Word* :test 'string=)
			       (print `((ctgy . adj) (root . ,word))))
			 (when (member word *Adverbs_End_Word* :test 'string=)
			       (print `((ctgy . adv) (root . ,word))))
			 (when (member word *Preps_End_Word* :test 'string=)
			       (print `((ctgy . prep) (root . ,word)))))
	         ))
		(write-string ")")
		(terpri))
	  (when (not *Tracing-On*)
		(setq def (remove nil (list
			 (when (member word *Nouns_End_Word* :test 'string=)
			       `((ctgy . n) (num . sing) (root . ,word)))
			 (when (member word *Verbs_End_Word* :test 'string=)
			       `((ctgy . v) (tense . pres) (root . ,word)))
			 (when (member word *Adjectives_End_Word* :test 'string=)
			       `((ctgy . adj) (root . ,word)))
			 (when (member word *Adverbs_End_Word* :test 'string=)
			       `((ctgy . adv) (root . ,word)))
			 (when (member word *Preps_End_Word* :test 'string=)
			       `((ctgy . prep) (root . ,word))))
	         ))
	  )
	  (let ((*package* (find-package 'snepsul)))
	    (definewords (cons word def)))
	)
  )
)

(defvar *Nouns_End_Word*
 '("band" "bell" "bid" "bird" "blood" "board" "bone" "bottle" "bow" "boy" 
   "breed" "bridge" "bud" "burger" "bush" "buster" "cade" "cake" "card"
   "castle" "charge" "cloth" "cord" "craft" "cycle" "dame" "dew" "edge" "eye" 
   "face" "fall" "fame" "fare" "feed" "fest" "field" "figure" "fish" "fold" 
   "foot" "fore" "frame" "geld" "girl" "god" "ground" "guard" "hand" "handle"
   "head" "herd" "hill" "hold" "hole" "hood" "horse" "hound" "house" "jack" 
   "kind" "knowledge" "lace" "land" "leg" "less" "line" "maid" "man" "mate" 
   "meal" "mold" "monger" "mouth" "nose" "pan" "pate" "path" "people" "person"
   "piece" "pipe" "place" "pole" "poll" "port" "pot" "proof" "quake" "race"
   "red" "right" "saddle" "scape" "seed" "sense" "shade" "shake" "shed" "ship"
   "side" "skin" "smith" "some" "stage" "stake" "stall" "stand" "stead" "stick"
   "stone" "stress" "stroke" "surge" "take" "thrift" "tide" "time" "tire" "toe"
   "tone" "trap" "tree" "ward" "ware" "way" "ways" "weed" "wife" "wig" "wind"
   "wise" "woman" "wood" "word" "work" "works" "world" "wort" "wright" "yard")
 "Default nouns which might be found at end of words. And which may be added
  to the lexicon dynamically by the program.")

(defvar *Verbs_End_Word*
 '("bid" "bind""board" "bottle" "bound" "bow" "brained" "bred" "breed" "bud"
   "castle" "charge" "cycle" "edge" "eye" "eyed" "fall" "fame" "fare" "feed"
   "fend" "fest" "figure" "fish" "fold" "foot" "footed" "frame" "free" "geld"
   "gild" "ground" "guard" "hand" "handed" "handle" "head" "herd" "hold"
   "hound" "jack" "jawed" "lace" "land" "lay" "let" "like" "line" "mate" "mold"
   "nose" "nosed" "pan" "pipe" "place" "poll" "pot" "quake" "race" "right"
   "saddle" "seed" "sense" "shade" "shake" "shed" "ship" "side" "sized" "smith"
   "sparred" "stage" "stake" "stall" "stand" "stick" "stone" "stress" "stroke"
   "surge" "take" "time" "tire" "trap" "weed" "wood" "work" "works")
 "Default verbs which might be found at end of words. And which may be added
  to the lexicon dynamically by the program")

(defvar *Adjectives_End_Word*
 '("bound" "cade" "clad" "face" "fertile" "fest" "field" "footed" "fore" "free"
   "ground" "hand" "kind" "lay" "less" "mobile" "most" "proof" "red" "right"
   "side" "sized" "some" "time" "wise" "wood" "worthy")
 "Default adjectives which might be found at end of word. And which may be
  aded to the lexicon dynamically by the program.")

(defvar *Adverbs_End_Word* 
 '("fest" "free" "less" "most" "right" "some")
 "Default adverbs which might be found at end of word. And which may be added
  to the lexicon dynamically by the program.")

(defvar *Preps_End_Word*
 '("fore" "less")
 "Default prepositions which might be found at end of word. And which may be
  added to the lexicon dynamically by the program.")

(defvar *trace-ending* ""
 "This is used with tracing to output current suffix or word ending.")

(defun suffix (l)
 "suffix - args: l
  checks is word l contains a recognized suffix or word ending and then
  branches. Default case of cond is to return default lexicon entry that was 
  just added."
    (declare (special *Suffixes* *trace-ending*))
    (cond ((or (null l)(string= l "")) nil)
          ((and (null *Suffixes*) (init) 'nil))   
	  ((and (> (length l) 8)
		(string= "knowledge"  (subseq l (- (length l) 9) (length l)))
		(setq *trace-ending* "knowledge")
		(Six_7_9_Char_Suffix (remove_suffix l 9) (get_current_suff l 9))))
	  ((and (Has_Size_Ending l 7)
		(Six_7_9_Char_Suffix (remove_suffix l 7) (get_current_suff l 7))))
	  ((and (Has_Size_Ending l 6)
		(Six_7_9_Char_Suffix (remove_suffix l 6) (get_current_suff l 6))))
	  ((and (Has_Size_Ending l 5)
		(Five_Char_Suffix (remove_suffix l 5) (get_current_suff l 5))))
	  ((and (Has_Size_Ending l 4)
		(Four_Char_Suffix (remove_suffix l 4) (get_current_suff l 4))))
	  ((and (Has_Size_Ending l 3)
		(Three_Char_Suffix (remove_suffix l 3) (get_current_suff l 3))))
	  ((and (Has_Size_Ending l 2)
		(Two_Char_Suffix (remove_suffix l 2) (get_current_suff l 2))))
	  ((and (Has_Size_Ending l 1)
		(One_Char_Suffix (remove_suffix l 1) (get_current_suff l 1))))
	  (t (geth l))
     )
)

(defun LookFor_New_Ending (l list_endings)
 "LookFor_New_Ending - args: l is word without suffix
                             list_endings is list of possible new endings.
  Routine trys to identify root word by assigning various new endings to it
  and looking in lexicon."
  (let (features)
    (when (and (not (null list_endings)) (listp list_endings))
      (dolist (ending list_endings) 
	   (when (setq features (lexic (concatenate 'string l ending)))
	   (return features))
      )
    )
  )
)

(defvar *C_R_F* nil
 "*C_R_F* - Used only with 1 clause of routine Spelling_Rules for nested
  suffixes and in routine Three_Char_Suffix. I,e, concatenate 'e' to a word.")

(defun Spelling_Rules (without_suffix &optional list_endings)
 "Spelling_Rules - args l is word without suffix
                        list_endings is list of possible new endings.
  Routine checks if word without suffix is in dictionary, and if it's not it
  trys to make word in dictionary by concatenation with items in a list, if all
  else fails it trys applying spelling rules to try and determine if 
  without_suffix is in dictionary. let statement is used to prevent changing
  global symbol *Change_Root_Feature*."
  (declare (special *Change_Root_Feature* *Negative_Feature* *trace-ending*
		    *C_R_F*))
  (let (temp (Save_CRF *Change_Root_Feature*) (Save_N *Negative_Feature*)
             (save-trace-ending *trace-ending*))
    (cond ((or (null without_suffix) (string= "" without_suffix)) nil)
	  ((lexic without_suffix))
	  ((LookFor_New_Ending without_suffix list_endings))
	  ((lexic (concatenate 'string without_suffix "e")))
	  ((and (> (length without_suffix) 1) 
		(char-equal (char without_suffix (1- (length without_suffix)))
			    (char without_suffix (- (length without_suffix) 2)))
		 (lexic (subseq without_suffix 0 (1- (length without_suffix))))))
	  ((and (char-equal '#\k (char without_suffix (1- (length without_suffix))))
		 (lexic (subseq without_suffix 0 (1- (length without_suffix))))))
	  ((and (char-equal '#\s (char without_suffix (1- (length without_suffix))))
		(LookFor_New_Ending (subseq without_suffix 0 (1- (length without_suffix)))
				    '("de" "d" "t" "re"))))
	  ((and (char-equal '#\c (char without_suffix (1- (length without_suffix))))
		(LookFor_New_Ending (subseq without_suffix 0 (1- (length without_suffix)))
				    '("ke"))))
	  ((and (char-equal (char without_suffix (1- (length without_suffix))) '#\i)
		(let ((no_last_char (subseq without_suffix 0 (1- (length without_suffix)))))
		  (or
		   (LookFor_New_Ending no_last_char '("y" "ey" "ie"))
		   (suffix (concatenate 'string no_last_char "y"))))))
	  ((and (> (length without_suffix) 4)
		(member (subseq without_suffix (- (length without_suffix) 2))
			'("ag""in""ur""at""iv""iz") :test 'string=)
		(setq *C_R_F* 'Inner_Suffix)
		(progn
		 (setq temp (suffix (concatenate 'string without_suffix "e"))
		      *Change_Root_Feature* Save_CRF *Negative_Feature* Save_N
		      *trace-ending* save-trace-ending *C_R_F* nil) temp)))
	  (t  (setq *C_R_F* 'Inner_Suffix 
                    *Negative_Feature* Save_N *trace-ending* save-trace-ending
		    temp (suffix without_suffix)
		    *Change_Root_Feature* Save_CRF *Negative_Feature* Save_N
                    *trace-ending* save-trace-ending *C_R_F* nil)
	      temp)
   )
 )
)

(defun Spell_Rule_Up*_Parameters (without_suffix current_suffix)
 "Spell_Rule_Up*_Parameters calls used to get parameters from hash table 
  *Suffixes*, and to apply spelling rules to without_suffix."
  (let ((parameters (get_ending current_suffix)))
	    (up* (first parameters) (second parameters)
		(Spelling_Rules without_suffix (third parameters)))
  )
)

(defun One_Char_Suffix (without_suffix current_suffix)
 "One_Char_Suffix - args -without_suffix current_suffix are strings
  Routine handles processing of single letter suffixes"
  (declare (special *Change_Root_Feature*))
  (let (temp)
    (cond 
     ((string= "" without_suffix) nil)
     ((and (string= current_suffix "s") (s without_suffix)))
     ((string= current_suffix "y") (y without_suffix))
     ((member current_suffix '("o" "s") :test 'string=)
      (setq *Change_Root_Feature* t)
      (Spell_Rule_Up*_Parameters without_suffix current_suffix))
     ((member current_suffix '("a" "i" "x") :test 'string=)
      (setq *Change_Root_Feature* nil)
      (progn (setq temp (Spell_Rule_Up*_Parameters without_suffix current_suffix))
	     (setq *Change_Root_Feature* t)
	     temp))
    )
  )
)
  
(defun Two_Char_Suffix (without_suffix current_suffix)
 "Two_Char_Suffix - args: without_suffix and current_suffix
  Routine processes what appears to be a 2 letter suffix. First it trys
  applying spelling rules to without_suffix, then second the word might
  be a homograph so it trys to find definition of homograph."
 (declare (special *Change_Root_Feature* *C_R_F*))
 (when (not *C_R_F*)
       (setq *Change_Root_Feature* t))
 (EndWord_NotIn_Lex current_suffix)
 (when (member current_suffix '("ae" "es" "im") :test 'string=)
	  (setq *Change_Root_Feature* nil))
 (let (temp)
   (cond ((string= "" without_suffix) nil)
         ((Spell_Rule_Up*_Parameters without_suffix current_suffix))
	 ((member current_suffix '("ae" "im") :test 'string=)
	       (setq *Change_Root_Feature* t) 'nil)
	 ((string= current_suffix "es") 
	  (progn 
	    (setq temp
		  (append 
		   (up* '((num . plur)) '(n)
			(Spelling_Rules without_suffix '("is")))
		   (up* '((ctgy . v)) '(v n) 
			(Spelling_Rules without_suffix))
		   (when (char-equal '#\v (char without_suffix (1- (length without_suffix))))
			 (Two_Char_Suffix (concatenate 
					   'string (subseq without_suffix 0 
							   (1- (length without_suffix)))
					   "f") current_suffix))))
	    (setq *Change_Root_Feature* t) temp))
	 ((string= current_suffix "ed")
	  (setq *Change_Root_Feature* nil)
	  (append 
	   (up* '((ctgy . v) (tense . past)) '(v n adj adv prep) 
		(setq temp (Spelling_Rules without_suffix)))
	   (up* '((ctgy . v) (pprt . t)) '(v n adj adv prep) 
                temp)
	   (progn (setq *Change_Root_Feature* t) 
		  (up* '((ctgy . adj)) '(n) 
		       (Spelling_Rules without_suffix)))
	   )
	  )
	 ((string= current_suffix "er")
	  (append (up* '((ctgy . n)) '(n v adj)
		       (setq temp (Spelling_Rules without_suffix)))
		  (up* '((ctgy . adj)) '(n v adj)
		       temp)
		  (up* '((ctgy . adv)) '(n adj adv)
		       temp)
		  (progn 
		    (setq *Change_Root_Feature* nil)
		    (up* '((ctgy . v)) '(n v)
			 (Spelling_Rules without_suffix)))))
	 ((string= current_suffix "ar")
	  (up* '((ctgy . adj)) '(n) (Spelling_Rules without_suffix)))
	 ((string= current_suffix "ly")
	  (up* '((ctgy . adj)) '(n) (Spelling_Rules without_suffix)))
	 ((string= current_suffix "en")
	  (en without_suffix))
	 ((string= current_suffix "al")
	  (append 
	   (up* '((ctgy . adj)) '(n) 
		(setq temp (Spelling_Rules without_suffix '("a"))))
	   (up* '((ctgy . n)) '(n adj v)
		temp)))
	 ((string= current_suffix "id")
	  (append (up* '((ctgy . adj)) '(v) 
		       (Spelling_Rules without_suffix '("efy" "ify")))
		  (up* '((ctgy . n))   '(v) 
		       (Spelling_Rules without_suffix '("efy")))))
	 ((string= current_suffix "re") (lexic (concatenate 'string  without_suffix "er")))
	 )
   )
)

(defun Three_Char_Suffix (without_suffix current_suffix)
 "Three_Char_Suffix - without_suffix and current_suffix
  Routine processes what appears to be a 3 letter suffix or word ending."
  (declare (special *Change_Root_Feature* *C_R_F*))
  (EndWord_NotIn_Lex current_suffix)
  (when (not *C_R_F*)
	(if (member current_suffix '("ept" "ics") :test 'string=)
	    (setq *Change_Root_Feature* nil)
	  (setq *Change_Root_Feature* t)))
  (let (temp temp2)
    (cond ((string= "" without_suffix) nil)
          ((progn 
	     (setq temp (Spell_Rule_Up*_Parameters without_suffix current_suffix))
	     (when (string= current_suffix "ics")
		   (setq *Change_Root_Feature* t))
		temp))
	  ((string= current_suffix "ing")
	   (setq *Change_Root_Feature* nil)
	   (append (progn (setq temp
				(up* '((ctgy . v) (presprt . t)) '(v n)
				      (Spelling_Rules without_suffix)))
			  (setq *Change_Root_Feature* t)
			  temp)
		   (append (up* '((ctgy . adj)) '(v n) 
				(setq temp2 (Spelling_Rules without_suffix)))
			   (up* '((ctgy . n)) '(n adj) temp2))))
	  ((string= current_suffix "ate")
	   (append
	    (up* '((ctgy . adj)) '(n adj) 
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . n)) '(n v) temp)
	    (up* '((ctgy . v)) '(n v adj) temp)))
	  ((string= current_suffix "ent")
	   (append
	    (up* '((ctgy . adj)) '(v n) 
		 (setq temp (Spelling_Rules without_suffix '("ence"))))
	    (up* '((ctgy . n)) '(n) temp)))
	  ((string= current_suffix "ant")
	   (append
	    (up* '((ctgy . adj)) '(v n) 
		 (setq temp (Spelling_Rules without_suffix '("ance"))))
	    (up* '((ctgy . n)) '(v n) temp)))
	  ((string= current_suffix "ful")
	   (append
	    (up* '((ctgy . adj)) '(n v adj) 
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . n)) '(n v adj) temp)))
	  (t (B_Three_Char_Suffix without_suffix current_suffix))
        )
  )
)

(defun B_Three_Char_Suffix (without_suffix current_suffix)
 "B_Three_Char_Suffix - without_suffix and current_suffix
  Routine processes what appears to be a 3 letter suffix or word ending 
  homograph."
  (let (temp (Parameters (get_ending current_suffix)))
	(cond
	  ((string= current_suffix "ite")
	   (append
	    (up* '((ctgy . n)) '(n v adj) 
		 (setq temp (Spelling_Rules without_suffix '("y"))))
	    (up* '((ctgy . adj)) '(v) temp)))
	  ((string= current_suffix "ory")
	   (append
	    (up* '((ctgy . n)) '(n v) 
		 (setq temp (Spelling_Rules without_suffix '("y"))))
	    (up* '((ctgy . adj)) '(n v) temp)))
	  ((string= current_suffix "est")
	   (append
	    (up* '((ctgy . adv)) '(adj adv) 
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . adj)) '(n adj adv) temp)))
	  ((string= current_suffix "ine")
	   (append
	    (up* '((ctgy . n)) '(n) 
		 (setq temp (Spelling_Rules without_suffix '("y"))))
	    (up* '((ctgy . adj)) '(n adj) temp)))
	  ((string= current_suffix "ify")
	   (up* (first Parameters) (second Parameters) (y without_suffix)))
	  ((string= current_suffix "ile")
	   (append
	    (up* '((ctgy . n)) '(n adv) (setq temp (Spelling_Rules without_suffix '("y"))))
	    (up* '((ctgy . adj)) '(n v adj) temp)))
	  ((string= current_suffix "oid")
	   (append
	    (up* '((ctgy . n)) '(n) 
		 (setq temp (Spelling_Rules without_suffix '("a"))))
	    (up* '((ctgy . adj)) '(n adj) temp)))
	  ((string= current_suffix "ept")
	   (append
    	    (up* '((tense . past)) '(v n) 
		 (progn (setq temp (Spelling_Rules without_suffix '("eep"))
		              *Change_Root_Feature* t) temp))
	    (up* '((tense . pastp)) '(v n) temp)))
	  (t nil)
    )
  )
)

(defun Four_Char_Suffix (without_suffix current_suffix)
 "Four_Char_Suffix - args: without_suffix and current_suffix
  Routine processes what appears to be a 4 letter suffix or word ending."
  (declare (special *Change_Root_Feature* *C_R_F*))
  (when (not *C_R_F*)
	(if (string= "ices" current_suffix)
	    (setq *Change_Root_Feature* nil)
            (setq *Change_Root_Feature* t)))
  (EndWord_NotIn_Lex current_suffix)
  (let (temp)
    (cond ((string= "" without_suffix) nil)
          ((progn 
	     (setq temp (Spell_Rule_Up*_Parameters without_suffix current_suffix))
	     (when (string= "ices" current_suffix)
		   (setq *Change_Root_Feature* t))
	     temp))
	  ((and (string= current_suffix "ment")
		(char-equal `#\a 
			    (char without_suffix (1- (length without_suffix)))))
	   (Spell_Rule_Up*_Parameters 
	    (concatenate 
	     'string (subseq without_suffix 0 
			     (1- (length without_suffix))) "ey") current_suffix))
	  ((string= current_suffix "ough")
	   (lexic (concatenate 'string without_suffix "o")))
	  ((string= current_suffix "some")
	   (append
	    (up* '((ctgy . n)) '(n adj)
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . adj)) '(n v adj adv) temp)))
	  ((string= current_suffix "side")
	   (append
	    (up* '((ctgy . n)) '(n v adj adv prep)
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . prep)) '(prep adv) temp)
	    (up* '((ctgy . adv)) '(prep adv) temp)))
	  ((string= current_suffix "meal")
	   (append
	    (up* '((ctgy . n)) '(n)
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . adv)) '(n) temp)))
	  ((string= current_suffix "fold")
	   (append
	    (up* '((ctgy . v)) '(n adj adv prep)
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . adj)) '(n adj) temp)))
	  (t nil)
	 )
    )
)

(defun Five_Char_Suffix (without_suffix current_suffix)
 "Five_Char_Suffix - args: without_suffix and current_suffix.
  Routine processes what appears to be a 5 letter word ending or suffix."
  (declare (special *Change_Root_Feature* *C_R_F*))
  (when (not *C_R_F*)
	(setq *Change_Root_Feature* t))
  (EndWord_NotIn_Lex current_suffix)
  (cond ((string= "" without_suffix) nil)
        ((Spell_Rule_Up*_Parameters without_suffix current_suffix))
	((and (string= current_suffix "ation")
	      (char-equal `#\c (char without_suffix (1- (length without_suffix)))))
	 (Spell_Rule_Up*_Parameters (subseq without_suffix 0
					    (1- (length without_suffix))) current_suffix))
	((string= current_suffix "stand")
	 (let (temp)
	   (append 
	    (up* '((ctgy . n)) '(n v adj) 
		 (setq temp (Spelling_Rules without_suffix)))
	    (up* '((ctgy . v)) '(adj n prep adv v) temp))))
  )
)

(defun Six_7_9_Char_Suffix (without_suffix current_suffix)
 "Six_7_9_Char_Suffix - args: without_suffix and current_suffix.
  Routine processes what appears to be a 6 letter word ending or suffix."
  (declare (special *Change_Root_Feature* *C_R_F*))
  (when (not *C_R_F*)
	(setq *Change_Root_Feature* t))
  (EndWord_NotIn_Lex current_suffix)
  (if (string= "" without_suffix)
      nil
      (Spell_Rule_Up*_Parameters without_suffix current_suffix))
)

(defvar *TwosAndOnes*                  '(("ac"  (adj (n) ("a")))
 ("an"  (adj (n npr) ("a")))             ("ar"  (n (v)))
 ("cy"  (n (n adj) ("t""te")))           ("ee"  (n (n v adj))) 
 ("el"  (n (n adj v)))                   ("et"  (n (n adj v)))
 ("ey"  (adj (n adj)))                   ("fy"  (v (n)))
 ("ia"  (n (n adj)))                     ("ic"  (adj (n) ("y")))
 ("ie"  (n (n adj npr) ("y")))           ("io"  (n (v) ("y"))) 
 ("le"  (n (n adj v)))                   ("ly"  (adv (adj v) ("le")))
 ("mo"  (n (n)))                         ("o"   (n (n adj v adv)))
 ("or"  (n (v n)))                       ("s"   (adv (adj adv)))
 ("sy"  (n (n adj adv) ("t""te""tic")))  ("th"  (n (adj v n adv)))
 ("ty"  (n (n adj adv prep))))
 "Used to initialize hash table *Suffixes* for 2 and 1 letter suffixes.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffix ee is:
                             (up* '((ctgy . n)) '(n v adj) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . n)) in definition of 
   word if the word in the lexicon has a feature ((ctgy . n)) or ((ctgy . v))
   or ((ctgy . adj)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's ('ie')) that
   are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *Suff_Threes*                  '(("acy"  (n (n v adj)))
 ("ade"  (n (n adj)))                    ("age"  (n (n v)))
 ("and"  (n (n adj v) ("ate")))          ("ard"  (n (n v adj)))
 ("ary"  (adj (n adj)))                  ("dom"  (n (n v adj)))
 ("eer"  (n (n adj)))                    ("eme"  (n (n v)))
 ("ery"  (n (n v adj) ("er")))           ("ese"  (n (n v adj)))
 ("eth"  (n (n)))                        ("fic"  (adj (n) ("fy")))
 ("ial"  (adj (n v) ("y")))              ("ian"  (adj (n adj) ("y")))
 ("ier"  (n (n v) ("a""y")))             ("ify"  (v (adj n v) ("ic""y""ey")))
 ("ion"  (n (n v adj) ("y")))            ("ish"  (adj (n adj) ("y")))
 ("ism"  (n (n adj) ("y""ia")))          ("ist"  (n (n v adj) ("y""s""o")))
 ("ity"  (n (n adj v) ("y""ious")))      ("ive"  (adj (v n adj) ("y")))
 ("ize"  (v (adj v n) ("y")))            ("kin"  (n (n)))
 ("ock"  (n (n adj v)))                  ("ose"  (adj (n adj))) 
 ("our"  (n (n) ("or")))                 ("ous"  (adj (n adj v) ("ty""y""ety")))
 ("tic"  (adj (n) ("s""sis""tis")))      ("ual"  (adj (n v)))
 ("ule"  (n (n v)))                      ("ure"  (n (v n adj adv pron))))
 "*Suff_Threes* - Contains suffixes and has no words for default lexicon.
  Used to initialize hash table *Suffixes* for 3 letter suffixes.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffic age is:
                            (up* '((ctgy . n)) '(n v) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . n)) in definition of 
   word if the word in the lexicon has a feature ((ctgy . n)) or ((ctgy . v)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's nil by default)
   that are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *Threes*                       '(("bid"  (v (adj v n prep adv)))
 ("bow"  (n (n v adj adv)))              ("boy"  (n (n v)))
 ("bud"  (n (n adj)))                    ("dew"  (n (n adj v)))
 ("god"  (n (n adj adv prep)))           ("lay"  (v (prep adv adj n)))
 ("leg"  (n (n adj v adv)))              ("let"  (n (n v adj)))
 ("man"  (n (n v adj)))                  ("pot"  (n (n v adj)))
 ("toe"  (n (n v adj)))                  ("red"  (n (n v adj)))
 ("way"  (n (n v adj adv)))              ("wig"  (n (n v))))
 "*Threes* - Contains words that are in default lexicon, i.e. common word 
  endings.
  Used to initialize hash table *Suffixes* for 3 letter word endings.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffic bid is:
                            (up* '((ctgy . v)) '(adj v n prep adv) 
                                                       (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . v)) in definition of 
   word if the word in the lexicon has a feature ((ctgy.adj))...((ctgy.adv)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's nil by default)
   that are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *Suff_Fours*                  '(("able"  (adj (v n) ("ate")))
 ("ably"  (adv (v)))                    ("ally"  (adv (adj)))
 ("ance"  (n (v adj n) ("ant""ent")))   ("ancy"  (n (n adj)))
 ("ator"  (n (v n) ("ate")))            ("berg"  (n (n v)))
 ("elle"  (n (v n)))                    ("ence"  (n (v n adj) ("ant""ent""ency")))
 ("ency"  (n (v adj n) ("ent")))        ("eous"  (adj (n adv adj v) ("a")))
 ("etic"  (adj (n) ("y")))              ("etta"  (n (n) ("a")))
 ("ette"  (n (n adj v)))                ("ible"  (adj (n v adj) ("y")))
 ("ibly"  (adv (v n adj) ("ible")))     ("ical"  (adj (n adj)) ("y"))
 ("ious"  (adj (n adj) ("y")))          ("itic"  (adj (n) ("y""ite""itis")))
 ("ling"  (n (n v adj adv prep)))       ("logy"  (n (n adj)))
 ("ment"  (n (v n adj adv)))            ("ness"  (n (adj n adv v)))
 ("ogue"  (n (n) ("og")))               ("osis"  (n (adj n)))
 ("ster"  (n (n v adj)))                ("tain"  (v (v n adj prep)))
 ("tive"  (adj (n v adj) ("t")))        ("ture"  (n (n v adj)))
 ("uous"  (adj (n))))
 "*Suff_Fours* - Contains suffixes only and no words for default lexicon.
  Used to initialize hash table *Suffixes* for 4 letter suffixes.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffix able is:
                             (up* '((ctgy . adj)) '(v n) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . adj)) in definition of
   word if the word in the lexicon has a feature ((ctgy . n)) or ((ctgy . v)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's ('ie')) that
   are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *Fours*                       '(("band"  (n (n v adj)))
 ("bell"  (n (n v adj)))                ("bind"  (n (n v adj)))
 ("bird"  (n (n v)))                    ("bred"  (adj (n v adj adv prep)))
 ("bush"  (n (n v adj)))                ("cade"  (n (n v adj)))
 ("card"  (n (n v adj)))                ("clad"  (adj (n v adj adv prep)))
 ("cord"  (n (n v)))                    ("dame"  (n (n v adj)))
 ("eyed"  (adj (v n adj)))              ("face"  (n (n adj)))
 ("fall"  (n (n v adj)))                ("fame"  (v (v n adj adv prep)))
 ("fare"  (v (n adv)))                  ("feed"  (n (n v prep adj)))
 ("fend"  (v (prep)))                   ("fest"  (n (v n)))
 ("fish"  (n (n adj)))                  ("foot"  (n (adj adv n)))
 ("fore"  (adv (v adv)))                ("free"  (adj (n)))
 ("geld"  (n (n v adj)))                ("gild"  (v (v n adv adj prep)))
 ("hand"  (adj (n v adj prep adv)))     ("head"  (n (n v adj adv)))
 ("hold"  (n (n v adj prep adv)))       ("hole"  (n (n v)))
 ("hood"  (n (n v adj adv)))            ("jack"  (n (n v adj)))
 ("kind"  (n (n adj)))                  ("lace"  (n (n v)))
 ("land"  (n (n v adj adv)))            ("less"  (adj (n adj adv v)))
 ("like"  (adj (n v) ("a")))            ("line"  (n (n adj adv)))
 ("maid"  (n (n v prep adj)))           ("mold"  (n (n v)))
 ("most"  (adj (n adj adv prep v)))     ("nose"  (n (n v adv)))
 ("pate"  (n (n v adj)))                ("path"  (n (n v)))
 ("pipe"  (n (n v adj)))                ("poll"  (n (n v adj)))
 ("port"  (n (n v)))                    ("seed"  (n (n adj)))
 ("shed"  (n (n v adj)))                ("ship"  (n (n v)))
 ("skin"  (n (n adj v)))                ("take"  (v (adj prep)))
 ("tide"  (n (n adj adv v)))            ("tire"  (v (n adj adv prep)))
 ("tone"  (n (n adj adv v)))            ("trap"  (n (n v)))
 ("tree"  (n (n v)))                    ("ward"  (adv (n v adj adv prep)))
 ("ware"  (n (n adj v)))                ("ways"  (adv (n v adj adv pron)))     
 ("weed"  (n (n v adj)))                ("wife"  (n (n v adj)))
 ("wind"  (n (n v adj prep)))           ("wise"  (adv (n v adj prep adv)))
 ("wood"  (n (n adj v)))                ("word"  (n (n v adv prep)))
 ("work"  (n (n v adj)))                ("yard"  (n (n adj adv))))
 "*Fours* - Contains ending words that are in default lexicon.
  Used to initialize hash table *Suffixes* for 4 letter word endings.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffix bird is:
                             (up* '((ctgy . n)) '(n v) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . adj)) in definition of
   word if the word in the lexicon has a feature ((ctgy . n)) or ((ctgy . v)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's ('ie')) that
   are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *Suff_FivesMore*                  '(("ability"  (n (n v adj) ("able")))
 ("aceous"   (adj (n v adj)))               ("ation"    (n (v n) ("ate")))
 ("ative"    (adj (v n) ("y")))             ("atory"    (adj (v) ("ate")))
 ("ature"    (n (n v)))                     ("drome"    (n (n adj)))
 ("ergic"    (adj (adj pron n adv)))        ("escence"  (n (n adj v) ("escent""esce")))
 ("esque"    (adj (n v adj)))               ("ibility"  (n (n v adj) ("ible")))        
 ("iferous"  (adj  (n adj)))                ("istic"    (adj (n) ("y""ism""ist")))
 ("ition"    (n (v n) ("y")))               ("itious"   (adj (n v) ("y""ion""ition"))) 
 ("itive"    (adj (n v) ("y")))             ("manship"  (n (n v adj)))
 ("ology"    (n (n adj)))                   ("osity"    (n (adj) ("ous")))
 ("sicle"    (n (n v)))                     ("tious"    (adj (n v)))
 ("ulous"    (adj (n v)))                   ("ville"    (adj (adj n v))))
 "*Suff_FivesMore* - Contains suffixes only.
  Used to initialize hash table *Suffixes* for 5 & more letter suffixes.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have suffix ation is:
                             (up* '((ctgy . n)) '(v n) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . n)) in definition of 
   word if the word in the lexicon has a feature ((ctgy . n)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's ('ate')) that
   are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")

(defvar *FivesAndMore*                 '(("blood"     (n (n v adj)))
 ("bottle"    (n (n v adj)))             ("bound"     (adj (n v)))
 ("brained"   (adj (adj v n)))           ("breed"     (v (adv adj prep v n)))
 ("burger"    (n (n adj)))               ("castle"    (n (n adj prep)))
 ("charge"    (v (n v adj adv prep)))    ("cloth"     (n (n adj)))
 ("craft"     (n (n adj v)))             ("cycle"     (n (n adj prep)))
 ("fertile"   (adj (n adj adv prep)))    ("field"     (n (n v)))
 ("figure"    (v (adv adj prep n)))      ("footed"    (adj (n v adj pron)))
 ("frame"     (n (n v)))                 ("ground"    (n (n v adj)))
 ("handed"    (adj (adj n)))             ("handle"    (n (n v adj adv prep)))
 ("hound"     (n (n v)))                 ("horse"     (n (n v adj)))
 ("house"     (n (n v)))                 ("jawed"     (adj (prep adv adj n)))
 ("knowledge" (n (n adj prep)))          ("mobile"    (n (n v adj)))
 ("monger"    (n (n adj v)))             ("mouth"     (n (n v adj)))
 ("nosed"     (adj (n adj v prep)))      ("piece"     (n (n v adj adv)))
 ("people"    (n (n adv adj prep v)))    ("proof"     (adj (n v)))
 ("quake"     (n (n v)))                 ("right"     (adj (prep adv adj n v)))
 ("saddle"    (n (n adj v prep)))        ("scape"     (n (n v)))
 ("sense"     (n (adj n v)))             ("sized"     (adj (adv adj prep n)))
 ("smith"     (n (n v adj)))             ("sparred"   (adj (adj prep adv n)))
 ("stage"     (n (adv adj prep)))        ("stake"     (n (n v)))
 ("stall"     (n (n v)))                 ("stick"     (n (n v adj)))
 ("stone"     (n (n v)))                 ("stroke"    (n (n adj adv v)))
 ("surge"     (v (prep adv adj n pron))) ("thrift"    (n (v n adj)))
 ("woman"     (n (n v adj)))             ("world"     (n (n v adj adv prep)))
 ("worthy"    (adj (n v)))               ("wright"    (n (n v))))
 "*FivesAndMore* - Contains word endings for default lexicon. 
  Used to initialize hash table *Suffixes* for 5 & more letter word endings.
  i.e. each inner list will contain 2 of required parameters for routine up*
  followed by last list which elsewhere is concatenated to current word to try
  to find lexical entry in lexicon.
  Example call to routine up* when have word ending wright is:
                             (up* '((ctgy . n)) '(n v) (lexic <string>)).
   What routine up* does is substitutes feature ((ctgy . n)) in definition of 
   word if the word in the lexicon has a feature ((ctgy . n)) or ((ctgy . v)).
   The next part of the hash entry is sent to routine LookFor_New_Ending,
   which just concatenates in turn strings (in example it's ('ate')) that
   are in the last list of the entry, and then trys to find new string in
   the lexicon, by calling routine lexic.")
 
(defvar *Suff_spelling* '("aire""iere""itis")
 "*Suff_spelling* - contain only suffixes.
  Easy cases (Current Word is Noun From root Noun) added only to pass most 
  words through routine For Spelling_Rules. i.e. it is to do the same thing as 
  symbols *TwosAndOnes*, *Threes*, *Fours*, and *FivesAndMore*, in other words
  each of these is a key into the hash table *Suffixes*, and has exact same 
  format, but there are no lists to concatenate to current words when trying to
  locate word in lexicon.")

(defvar *Spelling* '("board""bone""bridge""buster""cake""edge""eye""girl"
  "guard""herd""hill""mate""pan""person""place""pole""race""shade""shake"
  "stead""stress""time""works""wort")
 "*Spelling* - Contains word endings that are part of default lexicon.
  Easy cases (Current Word is Noun From root Noun) added only to pass most 
  words through routine For_Spelling_Rules. i.e. it is to do the same thing as 
  symbols *TwosAndOnes*, *Threes*, *Fours*, and *FivesAndMore*, in other words
  each of these is a key into the hash table *Suffixes*, and has exact same 
  format, but there are no lists to concatenate to current words when trying to
  locate word in lexicon.")

(defvar *Not_ctgy_addto_Suf*
 '(("a"     (((ctgy . n) (num . plur)) (n adj) ("on" "um" "us")))
   ("ae"    (((num . plur)) (n) ("a")))
   ("es'"   (((ctgy . adj) (pos . t)) (n npr)))
   ("ess"   (((fem . t)) (n)))
   ("i"     (((ctgy . n) (num . plur)) (n adj v) ("o" "us")))
   ("ices"  (((num . plur)) (n) ("ex" "ix")))
   ("ics"   (((num . plur)) (n) ("y")))
   ("im"    (((num . plur)) (n)))
   ("n't"   (((neg . t)) (v)))
   ("ress"  (((fem . t)) (n) ("or""er")))
   ("s'"    (((ctgy . adj) (pos . t)) (n npr)))
   ("'s"    (((ctgy . adj) (pos . t)) (n npr)))
   ("trice" (((ctgy . n) (fem . t)) (v n) ("t""te")))
   ("'ve"   (((ctgy . contr)) (pron v n)))
   ("x"     (((num . plur)) (n))))
 "*Not_ctgy_addto_Suf* - contains suffixes.
  These are entries of hash table *Suffixes* that change features other than 
  only ctgy.")

(defvar *Suffixes* nil "Hash table that contains suffixes and word endings 
 which modify root word.")

(defun put_ending (ending_key parameters)
 "put_ending - args ending_key is string
                    parameters is the thing stored at key.
  Routine enters parameters in hash table *Suffixes* using key."
  (declare (special *Suffixes*))
  (setf (gethash ending_key *Suffixes*) parameters)
)

(defun get_ending (current_suffix)
 "get_ending - args current_suffix is suffix or word ending string."
  (declare (special *Suffixes*))
  (gethash current_suffix *Suffixes*)
)

(defun AddToGlobalVar (end)
 "AddToGlobalVar - arg - end is string representing word ending or suffix.
  Routine adds end to one of 6 global symbols, where each contains same size 
  of suffixes or word endings. Used so that future programers can add suffixes
  to this program by putting them only in one spot. Note assuming all suffixes
  are 1 to 7 letters in size.
  Case statement is set up so that test for largest group of word endings and 
  suffixes 1st etc.."
  (declare (special *Six_Char_End* *Five_Char_End* *Four_Char_End* 
		    *Three_Char_End* *Two_Char_End* *One_Char_End*))
  (case (length end)
	(4 (setq *Four_Char_End* (cons end *Four_Char_End*)))
	(3 (setq *Three_Char_End* (cons end *Three_Char_End*)))
	(5 (setq *Five_Char_End* (cons end *Five_Char_End*)))
	(6 (setq *Six_Char_End* (cons end *Six_Char_End*)))
	(2 (setq *Two_Char_End* (cons end *Two_Char_End*)))
	(7 (setq *Seven_Char_End* (cons end *Seven_Char_End*)))
	(1 (setq *One_Char_End* (cons end *One_Char_End*)))
   )
)

(defun initialize (Many_Modifying_Suffixes)
 "initialize - For each changing feature ctgy this routine is Used to 
  initialize hash table for each Many_Modifying_Suffixes symbol. And adds 
  suffix or word ending to a global list of them."
  (dolist (entry Many_Modifying_Suffixes)
	  (AddToGlobalVar (first entry))
	  (put_ending (first entry) 
		(cons (list (subst (caadr entry) 'value '(ctgy . value) :test 'eql))
		      (cdadr entry)
		)
	   )
  )
)

(defun duplicate_init (Many_Modifying_Suffixes)
 "duplicate_init - args -  Many_Modifying_Suffixes list symbol of entries and
  their keys for hash table. Example is it sets  a hash table entry to be: 
                           '(((ctgy . n)) (n)). And adds suffix or word ending
  to a global list of them."
  (dolist (entry Many_Modifying_Suffixes)
	  (AddToGlobalVar entry)
	  (put_ending entry
		'(((ctgy . n)) (n))
	   )
  )
)

(defun Add_oddities (Many_Modifying_Suffixes)
 "Add_oddities - args - Many_Modifying_Suffixes is a list of suffixes and word 
  endings.
  Used to initialize hash table for each Many_Modifying_Suffixes symbol. And
  adds suffix or word ending to a global list of them."
  (dolist (entry Many_Modifying_Suffixes)
	  (AddToGlobalVar (first entry))
	  (put_ending (first entry) (second entry))
  )
)

(defun Var_Suffixes (list_suffixes)
 "Var_Suffixes - args -list_suffixes is list of strings which are suffixes.
  Adds suffixes to list *Actual_Suffixes*. Routine used only to be able to add
  more suffixes to this program, by a programer (ie. programer only has to type
  in code for suffix only once - note: for word endings you also have to worry 
  about the default lexicon, homographs worry about *Actual_Suffixes* when 
  they are added [also 'ept'] et cetera)."
  (declare (special *Actual_Suffixes*))
  (dolist (suffix list_suffixes)
	  (setq *Actual_Suffixes* (cons (if (listp suffix)
					  (first suffix)
					  suffix)
				      *Actual_Suffixes*)
	  )
  )
)

(defun init ()
 "init - args - none
 Used to initialize hash table *Suffixes* and calls routines to initialize all
 symbols like *One_Char_End*. Does this only once because uses global symbol
 *Six_Char_End* as flag. Also initializes symbol *Actual_Suffixes*. This 
 routine initializes all symbols dealing with suffixes and word endings (not 
 counting default lexicon, homographs, or 'ept')."
   (declare (special *Suffixes* *TwosAndOnes* *Threes* *Fours* *FivesAndMore*
		     *Spelling* *Not_ctgy_addto_Suf* *Six_Char_End* 
		     *Suff_Threes* *Suff_Fours* *Suff_FivesMore* 
		     *Suff_spelling*))
   (when (null *Six_Char_End*)	      
         (setq *Suffixes* (make-hash-table :test 'equal))
	 (initialize *TwosAndOnes*)
	 (initialize *Suff_Threes*)
	 (initialize *Threes*)
	 (initialize *Suff_Fours*)
	 (initialize *Fours*)
	 (initialize *Suff_FivesMore*)
	 (initialize *FivesAndMore*)
	 (duplicate_init *Suff_spelling*)
	 (duplicate_init *Spelling*)
	 (Add_oddities *Not_ctgy_addto_Suf*)
	 (Var_Suffixes *TwosAndOnes*)
	 (Var_Suffixes *Suff_Threes*)
	 (Var_Suffixes *Suff_Fours*)
	 (Var_Suffixes *Suff_FivesMore*)
	 (Var_Suffixes *Suff_spelling*)
	 (Var_Suffixes *Not_ctgy_addto_Suf*)
   )
)
