;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: MULTI; Base: 10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: multi.lisp,v 1.6 1993/07/20 06:24:11 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :multi)


;;; THE MULTI PACKAGE
;;;
;;;

;(defvar evnts)
(defvar curnt%)

(defvar *high-priority-queue* (dequeue:new)
  "Holds processes with high priority, that have been at the front
of the EVNTS queue in the old scheme")

(defvar *low-priority-queue* (dequeue:new)
  "Holds processes with low priority, that have been at the back
of the EVNTS queue in the old scheme")

(defvar *use-one-queue-only* nil
  "When this flag is T multi does not give priority to processes with reports,
   enabling the correct use of recursive rules.
   Introduced by njm/cpf 06/20/89")

(defun clear-all-queues ()
  "Makes all event queues empty"
  (setq *high-priority-queue* (dequeue:new))
  (setq *low-priority-queue* (dequeue:new)))

;;;
;;;
;;; TRACING VARIABLES
;;;
(defvar *trace-initiated-processes* nil
  "Enable trace of initiated processes.
   Values:      NIL ----> Trace is disabled.
                T   ----> Trace all initiated processes.
          (<PROCS>) ----> Trace initiation of specified processes (<PROCS>)")

(defvar *trace-events* nil
  "Enable trace of before-after snapshots of events.
   Values:      NIL ----> Trace is disabled.
                  T ----> Trace all events before & after execution.
          (<PROCS>) ----> Trace specified events before & after execution.")

(defvar *trace-enabled* nil
  "Indicates that some trace (ev-trace/cs-trace) is enabled (value:T).")




;;;DP
;;;--
;;;This macro defines processes for MULTIP. The call expected is
;;;(DP <PROCESS-NAME> (<LIST OF REGISTERS>) <PROCESS-BODY>)
;;;where <PROCESS-NAME>       :is atomic and name of a process template
;;;      <LIST-OF-REGISTERS> :is the list of all local registers used
;;;                           within the <PROCESS-BODY> except *NAME*,
;;;                           which the system manages.
;;;      <PROCESS-BODY>      :consists of forms to be evaluated.
;;;e.g. -- (DP MPLUS (A1 A2 ANS)
;;;           (SETQ ANS (+ A1 A2)))

(defmacro dp (name arg_list &body body)
  "Defines a MULTI process"
  (let ((df (get_def (cons arg_list body)))
	(reg_sym `(*NAME* ,@arg_list)))
    `(prog (retval)
	   (cond ((functionp ',name)
		  (setq retval (list ',name 'redefined)))
		 (t (setq retval ',name)))
	   (setf (get ',name 'lregs%) ',reg_sym)
	   (defun ,name ,@df)
	   
	   (return retval)))) 

(defmacro has-reg (proc reg)
  "Returns T if the <process> has the given register, NIL otherwise."
  `(member ,reg (get (first (eval ,proc)) 'lregs%)))

;;;GET-DEF
;;;-------
;;;This a helper function for DP. It takes the process body template and
;;;adds the system register *NAME* and returns a completed
;;;process template.

(defun get_def (pform)
  "Constructs the body of the process template for the DP macro"
  (prog (arg_list body)
	(setq arg_list `(*NAME* ,@(car pform))
	      body (cdr pform))
	(return `(,arg_list (declare (special curnt%)) ,@body
		  (set curnt% (list ,@arg_list))))))

;;;REGFETCH
;;;--------
;;;This function returns the value of a particular register in a process.
;;;The form of the call is:
;;;       (REGFETCH <PROCESS> <REGISTER-NAME>)
;;;where <PROCESS> evaluates to a list of register values and
;;;      <REGISTER-NAME> is the name of a register as defined by the original\
;;;                      call to DP.
;;;...  
;;;RETURNS: The value of the specified register with the process,
;;;         or calls BREAK if the process has no such register.

(defun regfetch (process register)
  "Returns the value of the specified register in a process"
  (prog (regvals reglist remaining-reglist)
	(setq regvals (eval process))
	(setq reglist (get (car regvals) 'lregs%))
	(setq remaining-reglist (member register reglist :test #'eq))
	(cond ((null remaining-reglist)
	       (format t "~%USER-ERR from regfetch~%in process ~S register ~S~%OFFENDING-VAL = unknown register" process register)
	       (break))
	      (t (return (nth (- (length reglist) (length remaining-reglist))
			      regvals))))))

;;;REGSTORE
;;;--------
;;;This function stores a specified value in the register of the process. The
;;;for of the call is:
;;;     (REGFETCH <PROCESS> <REGISTER-NAME> <REGISTER-VALUE>)
;;;where <PROCESS> evaluates to a list of register values,
;;;      <REGISTER-NAME> is the name of a register in the call to DP
;;;                      for the type of process, and
;;;      <REGISTER-VALUE> is the value to be stored in the process.
;;;REGSTORE acts destructively upon the values of the process.
;;;...
;;;RETURNS: <REGISTER-VALUE>
;;;         or calls BREAK if the process has no such register.

(defun regstore (process register value)
  "Stores the VALUE in the REGISTER of the PROCESS"
  (prog (reglist regvals)
	(setq regvals (eval process))
	(setq reglist (get (car regvals) 'lregs%))
     loop
	(cond ((null (and reglist regvals))
	       (format t "USER-ERR from regstore~%~S~%UNKNOWN REGISTER" register)
	       (break))
	      (t (cond ((eq (car reglist) register)
			(rplaca regvals value)
			(return value))
		       (t
			(setq reglist (cdr reglist))
			(setq regvals (cdr regvals))
			(go loop)))))))
;;;NEW
;;;---
;;;NEW assigns as the value of a unique identifier the value of the list
;;;of register values passed to it. This implementation assumes that the
;;;arguments to new are in the same order as the definition of the process
;;;template. NEW also assumes that the *NAME* (or process template) is the first
;;;argument.
;;;If the number of registers supplied in the call differs from the definition
;;;of the process template, then BREAK is called.

(defun new (&rest arglist)
  "Creates a new process"
  (let ((p-def (copy-list arglist))
	(new-process (new-process-name)))
    (cond ((= (length p-def) (length (get (car p-def) 'lregs%)))
	   (if *trace-initiated-processes*
	       (format t "~%** New process ~S with id: ~a ~%" (car p-def) (symbol-name new-process)))
	   (set new-process p-def)
	   new-process)
	  (t
	   (format t "~%USER-ERR from new~%~S~%HAS WRONG NUMBER OF ARGUMENTS~%" p-def)
	   (break)))))

(defun new-process-name ()
  "Generates a new name for a multi process"
  (gentemp "p" (find-package 'multi)))

(defun is-process-name (thing)
  "Returns T if THING is a symbol bound to a multi process"
  (and (symbolp thing)
       (multiple-value-bind (sym type)
	   (find-symbol (string thing) 'multi)
	 (and (eq type :internal)
	      (boundp sym)
	      (eql (char (symbol-name sym) 0) #\p)))))
;
; =============================================================================
;
; INITIATE
; --------
;
;       arguments     : event - <process>
;
;       returns       : <process queue>
;
;       description   : initiate function required by Multi
;
;                                        written :  ???
;                                        modified:  njm/cpf 6/20/89
;
; change: If *use-one-queue-only* flag is true then only the *high-priority-queue*
;         is used. This was implemented to avoid dead cycles generated by
;         recursive rules.
;

(defun initiate (evnt)
  "Schedules the given process (EVNT)."
  (let* ((event-priority (regfetch evnt '*priority*))
	 (queue (cond (*use-one-queue-only* '*high-priority-queue*)
		      (t (case event-priority
			   (snip:high '*high-priority-queue*)
			   (snip:low  '*low-priority-queue*)
			   (t (error "Now queue for priority ~s. Can't initiate event ~s"
				     event-priority (car (eval evnt)))))))))
    (if (and *trace-initiated-processes*
	     (not (dequeue:in-queue evnt (eval queue)))
	     (or (eq *trace-initiated-processes* t)
		 (member (car (eval evnt)) *trace-initiated-processes* :test #'eq)))
	(format t "~%** Initiate process ~S with id: ~A - on ~S. ~%   Initiated by process: ~A. Length of ~S - ~S~%"
		(car (eval evnt)) (symbol-name evnt) queue (symbol-name curnt%)
		queue (1+ (dequeue:queue-length (eval queue)))))
    (set queue (schedule evnt (eval queue)))))


;;;SCHEDULE
;;;--------
;;;This function performs the scheduling of events.

;(defun schedule (evnt evnts)
 ; "Performs the scheduling of events"
  ;(dequeue:insert-rear evnt evnts))
;
;;;MULTIP
;;;------
;;;This function does the evaluation of processes. The list of processes
;;;created by NEW are stored in two queues, *high-priority-queue* and
;;;*low-priority-queue*. These are global variables that will be bound
;;;to the values of HPQ and LPQ on entry to multip. MULTIP loops until 
;;;no more events remain on either of the queues where events on the 
;;;low priority list are not processed at all until the high priority
;;;queue becomes empty. MULTIP expects processes to be represented
;;;as follows:
;;;      <ID> --value-- (<NAME> <OTHER> ... <REGISTERS>)
;;;where ID is an identifier associated with each process, and has
;;;as its initial value a list of register values of which the first
;;;must be the name of the process template (*NAME*),
;;;and the name is also a function definition (a lambda expression).

(defun multip (hpq lpq)
  "Evaluates processes"
  (let ((*high-priority-queue* hpq)
	(*low-priority-queue* lpq)
	highest-priority-non-empty-queue
	*NAME*)
    (declare (optimize (speed 3))
	     (special *NAME*))
    (prog (curnt% regvals)
       loop
	  (cond ((not (dequeue:empty *high-priority-queue*))
		 (setq highest-priority-non-empty-queue *high-priority-queue*))
		((not (dequeue:empty *low-priority-queue*))
		 (setq highest-priority-non-empty-queue *low-priority-queue*))
		(t (return)))
	  (setq curnt% (dequeue:front highest-priority-non-empty-queue))
	  (setq regvals (eval curnt%))
	  (setq *NAME* (car regvals))
	  (dequeue:delete-front highest-priority-non-empty-queue)
	  (if *trace-enabled*
	      (cond
		((or (eq *trace-events* t)
		     (member (car regvals) *trace-events*))
		 (print-regs curnt% "Entering"))))
	  (apply (symbol-function (car regvals)) regvals)
	  (if *trace-enabled*
	      (cond
		((or (eq *trace-events* t)
		     (member (car regvals) *trace-events*))
		 (print-regs curnt% "Leaving"))))
	  (go loop))))

;;;
;;;
;;; TRACING FUNCTIONS
;;;

(defsnepscom ev-trace ((&rest lfrms))
  "Enables event trace for multi (by name)"
  (setq *trace-enabled* t
	*trace-events*
	(record *trace-events*
		(mapcan #'(lambda (p)
			    (cond ((get p 'lregs%) (list p))
				  (t (format t "~&~A is not a process name.~%"
					     p)
				     nil)))
			lfrms))))

(defsnepscom unev-trace ((&rest lfrms))
  "Undoes the effect of EV-TRACE. Returns the current status of
   *trace-events* flag."
  (setq *trace-events* (forget *trace-events* lfrms))
  (when (null *trace-events*)
    (setq *trace-enabled* nil)))

(defsnepscom in-trace ((&rest lfrms))
  "Enables tracing of initiated processes."
  (setq *trace-initiated-processes*
    (record *trace-initiated-processes* lfrms)))

(defsnepscom unin-trace ((&rest lfrms))
  "Undoes the effect of IN-TRACE."
  (setq *trace-initiated-processes*
    (forget *trace-initiated-processes* lfrms)))

(defun record (trace-flag add-list)
  "Inserts ADD-LIST into TRACE-FLAG by union."
  (cond ((and (null add-list) (null trace-flag)) t)
	((And add-list (atom trace-flag)) add-list)
	(add-list (append add-list trace-flag))
	(t trace-flag)))

(defun forget (trace-flag rem-list)
  "Removes REM-LIST from TRACE-FLAG."
  (cond ((null rem-list) nil)
	((atom trace-flag) nil)
	((member (car trace-flag) rem-list) (forget (cdr trace-flag) rem-list))
	(t (cons (car trace-flag) (forget (cdr trace-flag) rem-list)))))

#+explorer
(defun pprin-value (value &optional (stream *standard-output*))
  (let* ((initial-indentation
	   (or (global:send-if-handles stream :read-cursorpos :character)
	       30))
	 (stream-width
	   (or (global:send-if-handles stream :size-in-characters)
	       80))
	 (sys:pp-line-length
	   (cond ((> stream-width 100) 100)
		 (t (- stream-width 5)))))
    (sys:output-pretty-object value nil initial-indentation)))

#-explorer
(defun pprin-value (value &optional (stream *standard-output*))
  (write value :pretty t :stream stream))

(defun print-regs (process &optional msg)
  "Prints the current bindings for PROCESS."
  (prog (bindings
	 (ind 5))
	(cond (msg (format t "~%>>>>>> ~A process id = ~A with bindings:~%" msg (symbol-name process)))
	      (t (format t "~%~V@TProcess id = ~A has bindings~%:" ind (symbol-name process))))
	(setq process (eval process))
	(setq bindings (pairlis (get (car process) 'lregs%)
			        process))
     loop
	(cond ((null bindings) (return))
	      (t (format t "~%~V@T~A = " ind (symbol-name (caar bindings)))
		 (pprin-value (cdar bindings))
		 (setq bindings (cdr bindings))
		 (go loop))))
  (format t "~%"))





