;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSUL; Base:10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: snactor21.lisp,v 1.4 1993/07/20 06:26:59 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snepsul)


;;; Definitions of SNACTOR
;;;
;;; Representation
;;;
;;;   act = action act-node
;;;         objecti ith-argument to act-node

(defvar *primitive-action-functions*
  (make-hash-table :test #'eq)
  "keys are the node accesses of primitive action nodes, values are functions of one argument.")

(defvar *control-actions* '("achieve" "believe" "snsequence" "snif" "sniterate" "noop" "say" "forget" "do-one" "do-all"))

(defsnepscom perform ((nodeset) (top) t)
  "Acting executive, using nodeset as the acting queue."
  (do ((first-act (choose.ns nodeset) (first snac-queue))
       (need-to-satisfy nil nil)
       (preconditions nil nil)
       (snac-queue (rest nodeset) (rest snac-queue)))
      ((null first-act))
      (declare (special snac-queue first-act))
      (if (not (control-action? first-act))
	  (plantrace "I intend to do " (cons first-act snac-queue) nil))
      (cond ((not (control-action? first-act))
	     (plantrace "About to do" (list first-act) nil)))
      (clear-infer)
      (cond ((multiple-value-setq (need-to-satisfy preconditions)
	       (get-preconditions first-act))
	     (setq snac-queue (schedule-preconditions first-act preconditions snac-queue)))
	    (t
	     (setq snac-queue 
		   (schedule-believe-effects-of first-act snac-queue)) 
	     (cond ((primact? first-act) 
		    (execute-primaction first-act)) 
		   (t (setq snac-queue 
			    (schedule-a-plan-for first-act snac-queue)))))
	    )))

(defun control-action? (act)
  (member (node-na (action-of-act act)) *control-actions* :test #'string=))

(defun get-preconditions (first-action)
  "Finds all the preconditions required to be satisfied for FIRST-ACTION,
   tests them, returns two values (multiple value return):
   NEED-TO-SATISFY is nil if all preconditions are satisfied, non-nil o/w
   PRECONDITIONS is the nodeset of preconditions for first-action."
  ;(declare (special first-action))
  (unless (and (* 'precondition) 
               (isvar.n (first (* 'precondition))))
    ($ 'precondition))
  (if (control-action? first-action)
      (values nil nil)
      (funcall #'(lambda (preconds)
		 (cond (preconds (plantrace "The act " (list first-action) nil)
				 (plantrace (format nil "has ~[no~;a~:;the following~] precondition~:p:"
						    (length preconds))
					    preconds nil)))
		 (test-preconditions preconds))
	     (eval `(find precondition-
			  (deduce precondition (* 'precondition)
				  act ,first-action))))))

(defun test-preconditions (preconditions)
  "Returns two values: NEED-TO-SATISFY is nil if all preconditions are satisfied,
                       PRECONDITIONS contains all the preconditions."
    (if preconditions
      (values (funcall #'(lambda (not-satisfied)
			   (if not-satisfied
			     (plantrace (format nil "~[~;It is~:;They are~] not satisfied." (length preconditions))
					nil nil)
			     (plantrace (format nil "~[~;It is~:;They are~] satisfied." (length preconditions))
					nil nil))
			   not-satisfied)
		     (not (every #'satisfied? preconditions)))
	      preconditions)
      (values nil preconditions)))

(defun satisfied? (prop)
  (let ((sneps::crntct 'sneps::default-defaultct))
	       (declare (special sneps::crntct))
	       (snip::deduce* nil (list prop))))

(defun schedule-preconditions (first-action preconditions to-do-queue)
  "Schedules the achieving of PRECONDITIONS before FIRST-ACTION is performed on the TO-DO-QUEUE."
  ;(declare (special first-action))
  (append (mapcan #'(lambda (n)
		      (eval `(build action (build lex "achieve") object1 ,n)))
		  preconditions)
	  (list first-action)
	  to-do-queue))

(defun schedule-believe-effects-of (first-action to-do-queue)
  "Schedules the believing of effects of doing ACT on the TO-DO-QUEUE."
  ;(declare (special first-action))
  (cond ((control-action? first-action)
	 to-do-queue)
	(t (unless (and (* 'effect) (isvar.n (first (* 'effect))))
	     ($ 'effect))
	   (append (mapcan #'(lambda (n)
						;(declare (special n))
			       (eval `(build action (build lex "believe")
					     object1 ,n)))
			   (eval `(let ((sneps::crntct 'sneps::default-defaultct))
				    (declare (special sneps::crntct))
				    (find effect-
					  (deduce effect (* 'effect)
						  act ,first-action)))))
		   to-do-queue))))

(defun schedule-a-plan-for (first-action to-do-queue)
  "Schedules a plan for doing FIRST-ACTION on the TO-DO-QUEUE."
  (declare (special first-action))
  (unless (and (* 'plan) (isvar.n (first (* 'plan))))
    ($ 'plan))
  (append
    (find plan-
	  (^ (funcall #'(lambda (some-plan)
			  (cond (some-plan (plantrace "I deduced " some-plan nil)))
			  some-plan)
		      (eval `(deduce plan (* 'plan)
				     act ,first-action)))))
    to-do-queue))




(defun well (x)
  (format t ">>>>>THIS IS WHAT WE FOUND TO BE BELIEVED:~S" x)
  x)

;;(setf (get 'perform '=topcommand) t)

(defun action-of-act (actnode)
  "Returns the action node of the given act node."
  (eval `(choose.ns (find lex- (find action- ,actnode)))))
;  (choose.ns (pathfrom '(action lex) actnode)))

(defun execute-primaction (act-node)
  "Applies the given primitive action to the given act node."
  (funcall (gethash (intern (format nil "~:@(~a~)"
				    ;; Print the symbol not the name so
				    ;; *print-circle* t won't cause problems
				    (node-na (action-of-act act-node)))
			    "SNEPSUL")
		    *primitive-action-functions*)
	   act-node))

(defmacro define-primaction (primaction vars &body forms)
  "Creates and stores the function definition of the primitive action."
  `(prog2
    (setf (gethash ',primaction *primitive-action-functions*)
	  (function (lambda ,vars (declare (special snac-queue)) ,@forms)))
    ',primaction))

(defvar infertrace nil)

(defun trace-on ()
  (setq infertrace t))

;;(setf (get 'trace-on '=topcommand) t)

(defun trace-off ()
  (setq infertrace nil))

;;(setf (get 'trace-off '=topcommand) t)

(defun primact? (act-node)
  "Returns T if act-node represents an act with a primitive action."
  (declare (special act-node))
  (eval
    `(findassert (member action-) ,act-node (class lex) "primitive")))

;;; Definitions of Primitive Control Structure Actions

(define-primaction snsequence (n)
  "Performs the object1 act, then the object2 act."
  (setq snac-queue (append (pathfrom '(object1) n)			   (append (pathfrom '(object2) n) snac-queue))))

(define-primaction snif (n)
  "Of the set of object1 act nodes,
   nondeterministically performs any one whose condition is true.
   If none are true, does nothing."
  (do.ns (pairnode (pathfrom '(object1) n))
     (when (let ((sneps::crntct 'sneps::default-defaultct))
	     (declare (special sneps::crntct))
	     (snip::deduce* nil (pathfrom '(condition) pairnode)))
       (setq snac-queue (append (pathfrom '(then) pairnode) snac-queue))
       (return))))

(define-primaction do-one (n)
  "Randomly picks one of a set of actions and schedules it to be performed."
  ;(format t "The actions to choose from are: ~S~%" (pathfrom '(object1) n))
  (let ((possible-actions (pathfrom '(object1) n)))
    (setq snac-queue (cons (nth (random (length possible-actions))
				possible-actions)
			   snac-queue))))

(define-primaction do-all (n)
  "Schedules the performing of all the actions on the queue."
  (setq snac-queue (append (pathfrom '(object1) n) snac-queue)))

(define-primaction sniterate (n)
  "If the condition of any object1 act is true,
   performs that act, and tries the entire sniteration again.
   If no condition is true, does nothing."
  (declare (special n))
  (do.ns (pairnode (pathfrom '(object1) n))
     (declare (special pairnode))
     (when (let ((sneps::crntct 'sneps::default-defaultct))
	     (declare (special sneps::crntct))
	     (snip::deduce* nil (pathfrom '(condition) pairnode)))
       (setq snac-queue
	     (append (eval `(build action (build lex "snsequence")
				   object1 (find then- ,pairnode)
				   object2 ,n))
		     snac-queue))
       (return))))
	  
(define-primaction achieve (n)
  "Tries to find plans to make the object1 proposition true.
   If it does, chooses one, and schedules it."
  (declare (special n))
  (let ((prop (choose.ns (pathfrom '(object1) n))))
    (declare (special prop))
    (plantrace "Want to achieve " (list prop) nil)
    ;(describe (^ prop))
    (cond ((node-assert prop)
	   (format t "~&~%Already Achieved.~%"))
	  (t (unless (and (* 'gplan) (isvar.n (first (* 'gplan))))
	       ($ 'gplan))
	     (setq snac-queue
		   (cons (choose-plan
			   (find plan-
				 (^ (funcall #'(lambda (some-plan)
						 (cond (some-plan (plantrace "I deduced " some-plan nil)))
						 some-plan)
					     (eval `(deduce plan (* 'gplan)
							    goal (find object1- ,n)))))))
			 snac-queue))))))

(defun choose-plan (plan-list)
  "Chooses an act from the list of act nodes, and returns it.
   If a no-op is in the list, chooses that.
   Otherwise, chooses arbitrarily."
  (or
   (find-if #'(lambda (n) (member (node 'noop) (pathfrom '(action lex) n)))
	    plan-list)
   (choose.ns plan-list)))

(define-primaction noop (n)
  "Does nothing."
  (declare (ignore n))
  (format t "Now doing: NOOP~%"))

;;; Definitions of Generally Useful Primitive Actions

(define-primaction say (n)
  "Prints the object1 node."
  (format t " ~A " (choose.ns (pathfrom '(object1) n))))

(define-primaction believe (believe-act)
  "Causes object1(believe-act) to be believed."
  (believe-prop (choose.ns (pathfrom '(object1) believe-act))))

(defun believe-prop (n)
  "Causes n to be believed."
  (declare (special n))
  (cond ((is-nor.n n)
	 (forget-prop (choose.ns (pathfrom '(arg) n))))
	((node-assert n)
	 (cond ((and (boundp '*plantrace*) *plantrace*)
                (format t "~&~%I already believe ")
		(if (equal 'surface *plantrace*)
		    (surface (^ n))
		    (describe (^ n))))))
	(t (clear-infer)
	   ;;(let ((not-n (choose.ns (findassert min 0 max 0 arg (^ n)))))
	   ;;  (when not-n  ; commented out until deduce can deal with and-ors
	   ;;    (setf (node-assert not-n) nil)))
	   (! (^ n))
	   (cond ((and (boundp '*plantrace*) *plantrace*)
		  (format t "~&~%Believe ")
		  (if (equal 'surface *plantrace*)
		      (surface (^ n))
		      (describe (^ n))))))))

(define-primaction forget (forget-act)
  "Causes object1(forget-act) to be disbelieved."
  (forget-prop (choose.ns (pathfrom '(object1) forget-act))))

(defun forget-prop (n)
  "Causes n to be disbelieved."
  (declare (special n))
  (cond ((is-nor.n n)
	 (mapc #'believe-prop (pathfrom '(arg) n)))
	((node-assert n)
	 (clear-infer)
	 ;;(setf (node-assert n) nil)
	 (sneps:chew-up-output (sneps:outunit)
	    (remove-from-context (^ n)))
	 ;;(assert min 0 max 0 arg (^ n)) commented out until deduce can handle and-ors
	 (cond ((and (boundp '*plantrace*) *plantrace*)
		(format t "~&~%Disbelieve ")
		(if (equal 'surface *plantrace*)
		    (surface (^ n))
		    (describe (^ n))))))
	(t (cond ((and (boundp '*plantrace*) *plantrace*)
		  (format t "~&~%I already didn't believe:~%")
		  (if (equal 'surface *plantrace*)
		      (surface (^ n))
		      (describe (^ n))))))))


(defun node-assert (n)
  (let ((sneps::crntct 'sneps::default-defaultct))
    (declare (special sneps::crntct))
    (isassert.n n)))
