;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSUL; Base:10 -*-

;; Copyright (C) 1984, 1988, 1989, 1993 Research Foundation of 
;;                                      State University of New York

;; Version: $Id: driver.lisp,v 1.2 1993/06/04 06:21:08 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State University of
;; New York at Buffalo, 226 Bell Hall, Buffalo, NY 14260, USA

(in-package :snepsul)


;; Lisp side of the ARC/LISP communication package. Contains functions
;; that generate appropriate nextcommand.aml files and send them to
;; the communication directory where they will be read and executed
;; by the ARC command loop.

;; Author:   Hans Chalupsky
;; Created:  May 31, 1990
;; Modified: July 17, 1990

(defvar *interface-directory* "/u0/grads/hans/interface"
  "This is the name of the ARC/LISP communication directory viewed
from the machine where LISP runs on.")

;; Pathnames of various files viewed from the machine LISP is running on
(defvar *nextcom* (format nil "~a/nextcommand.aml" *interface-directory*))
(defvar *exit* (format nil "~a/exit" *interface-directory*))

;; The command-output file has to be defined in 2 steps because its name
;; is needed also on the machine ARC is running on.
(defvar *comout-name* "comout.log")
(defvar *comout* (format nil "~a/~a" *interface-directory* *comout-name*))


(defun wait-for-completion ()
  "Waits until the file nextcommand.aml gets deleted by the
AML command loop running in the ARC process. Returns once
nextcommand.aml does not exist anymore."
  (loop (unless (probe-file *nextcom*)
	  (return))
	;; give system some time to breathe
	(sleep 0.1)))

(defun execute-command (command-string)
  "Takes a COMMAND-STRING and writes a file nextcommand.aml into the
communication interface directory. The command loop running in the ARC
process waits for this file, executes it and deletes it. The output
generated by the command gets written to a command output (watch) file.
This allows to get results back into the LISP process."
  (wait-for-completion)
  (with-open-file (nextcom *nextcom* :direction :output)
    ;; nextcommand.aml takes as an argument the name of the communication
    ;; directory viewed from the ARC process, into which ARC will
    ;; write the comout file
    (format nextcom

	    "&args interface~%~
             &watch %interface%/~a~%~
             ~a~%~
             &watch &off~%~
             &return~%"

	    *comout-name* command-string)
    )
  (wait-for-completion))

(defun get-file-as-string (file)
  "Opens FILE and returns its contents as a string. Used to
read ARC command execution results back into the LISP process."
  (cond ((probe-file file)
	 (with-output-to-string (out)
	   (with-open-file (in file :direction :input)
	     (loop
	      (let ((line (read-line in nil :eof nil)))
		(cond ((eq line :eof)
		       (return))
		      (t (format out "~a~%" line))))))))
	(t "")))

(defun print-result (&optional (stream *standard-output*))
  "Prints contents of current command output file to STREAM."
  (format stream "~&~a" (get-file-as-string *comout*)))

(defun stop-arc ()
  "Creates an exit file in the communication directory which will cause
the arc command loop to terminate."
  (with-open-file (e *exit* :direction :output :if-exists nil)
    (format e " ")))

(defun arc-loop ()
  "Simple read/execute/print loop that reads ARC commands in LISP,
executes them in the ARC process and prints the results. Used for
testing of the interface. The command ^^ terminates the loop."
  (let ((command "")
	(prompt "Arc: "))
    #+ibcl(read-line)
    (loop
     (format t "~&~a" prompt)
     (setq command (read-line))
     (cond ((equal command "^^")
	    (stop-arc)
	    (return 'bye))
	   (t (execute-command command)
	      (format t "~a" (get-file-as-string *comout*)))))))

   
