/* -*- Mode: Text -*- */

/*
 * Copyright 1987, 1988, 1989, by Geoff Kuenning, Manhattan Beach, CA
 * Permission for non-profit use is hereby granted.
 * All other rights reserved.
 * See "version.h" for a more complete copyright notice.
 */

/*
 * $Id: ispell.h,v 1.41 1991/12/09 00:41:05 geoff Exp $
 *
 * $Log: ispell.h,v $
 * Revision 1.41  1991/12/09  00:41:05  geoff
 * Explicitly initialize tempfile to the null string, since we depend on
 * this initialization.  (Not strictly necessary, but I prefer to make it
 * obvious that this initial value is required.)
 *
 * Revision 1.40  91/09/12  00:01:47  geoff
 * Add definitions and header-file fields for the new TeX special
 * characters.
 * 
 * Revision 1.39  91/09/11  23:22:46  geoff
 * Add TeX_comment.
 * 
 * Revision 1.38  91/08/10  16:38:30  geoff
 * Add vflag.  Remove the typecasts in the ctype-style stuff, since they
 * were unneeded and wrong.
 * 
 * Revision 1.37  91/08/10  14:09:36  geoff
 * Fix all of the ctype-style macros to properly typecast the character
 * before testing it.  Remove the lengthofchar macro.  Change the
 * *isstringch macros to allow stringchars to be only one character long.
 * 
 * Revision 1.36  91/07/15  19:27:03  geoff
 * Change all of the stringchar macros (*isstringch and lengthofchar) to
 * accept a "canonical" parameter and pass it to stringcharlen.
 * 
 * Revision 1.35  91/07/11  19:52:14  geoff
 * Include stdio.h for the definition of BUFSIZ.
 * 
 * Revision 1.34  91/07/05  19:51:45  geoff
 * Add a proper declaration of done(), and remove the unused vb variable.
 * 
 * Revision 1.33  91/07/03  18:20:50  geoff
 * Add the chartoichar macro for conversion between chars and ichar_t's.
 * Add the "ti" and "te" terminal capabilities.
 * 
 * Revision 1.32  91/05/27  21:47:58  geoff
 * Add data-structure support for the string-character type tables.
 * 
 * Revision 1.31  90/12/31  00:59:47  geoff
 * Reformat to follow a consistent convention throughout ispell
 * 
 * Revision 1.30  90/10/14  01:51:39  geoff
 * Add the "changes" and "readonly" variables.
 * 
 * Revision 1.29  90/04/26  22:44:06  geoff
 * Add fields and variables needed to support alternate string characters.
 * 
 * Revision 1.28  89/12/27  22:29:08  geoff
 * Add minword.
 * 
 * Revision 1.27  89/07/11  00:24:11  geoff
 * Add Amiga and REGEX_LOOKUP support from luis@rice.edu.
 * 
 * Revision 1.26  89/07/07  23:26:29  geoff
 * Add a complete set of external declarations, including using the
 * new INIT macro to handle initialized declarations.
 * 
 * Revision 1.25  89/06/09  15:54:45  geoff
 * Add some lint ifdefs, so that the various macros can be checked by
 * lint.  Move the isXXXX macros to ispell.h from config.h, since they
 * shouldn't be changed during configuration.  Add support for the
 * internal "character" type, ichar_t.
 * 
 * Revision 1.24  89/04/28  01:11:16  geoff
 * Change Header to Id;  nobody cares about my pathnames.
 * 
 * Revision 1.23  89/04/03  01:56:34  geoff
 * Add support for string characters.  Fix the magic number to be constant,
 * and put the compilations in the hash header separately.
 * 
 * Revision 1.22  89/02/22  23:16:18  geoff
 * Add "contextsize" to the list of globals.
 * 
 * Revision 1.21  89/02/20  22:10:36  geoff
 * Define ALLFLAGS even if CAPITALIZATION is turned off.
 * 
 * Revision 1.20  89/02/06  02:21:38  geoff
 * Fix the offsets of TEX's left and right curlies and parens to match
 * what the documentation specifies for the "texchars" statement.
 * 
 * Revision 1.19  88/12/26  02:29:52  geoff
 * Add a copyright notice.
 * 
 * Revision 1.18  88/11/25  19:52:38  geoff
 * Fix the SET/CLR/TSTMASKBIT macros to work correctly with masks of more
 * than 32 bits.
 * 
 * Revision 1.17  88/06/25  17:50:05  geoff
 * Add hash header fields for the defaults set in the affix file:  nroff
 * and troff control characters, compound word (missing space) flag, and
 * automatic affix-generation (try hard) flag.  Update the magic number
 * to reflect the change in hash formats.
 * 
 * Revision 1.16  88/04/30  22:14:12  geoff
 * Get rid of a bunch of unused termcap variables detected by lint.
 * 
 * Revision 1.15  88/02/20  23:12:48  geoff
 * Expand the handling of MASKBITS to provide more size options.  Redo the
 * capitalization handling structures.  Change the magic number and its
 * creation so that binary-incompatible configuration options will cause
 * differing magic numbers.  Add "struct success" to describe spelling
 * hits.  Add some new externals and delete some unused one.
 * 
 * Revision 1.14  87/09/30  23:30:51  geoff
 * Make the sort order a short so it can go past the range of a char.  Add
 * a bunch of globals that used to be scattered through .c files, plus
 * one or two new ones to support the "nopossibilities" stuff.
 * 
 * Revision 1.13  87/09/09  00:18:45  geoff
 * Add sortval and sortorder to the hash header.
 * 
 * Revision 1.12  87/09/03  23:17:30  geoff
 * Get rid of a spurious WORDLEN definition, and use a different magic
 * number for NO8BIT hash files.
 * 
 * Revision 1.11  87/07/20  23:22:44  geoff
 * Major changes to support table driving.  Highlights include macros
 * to support the new flag bits, removal of the X_flag fields from the
 * table entry and replacement with the lumped masks, addition of language
 * tables, and using a different magic number if CAPITALIZE is defined.
 * 
 * Revision 1.10  87/06/27  12:20:32  geoff
 * Add the "sg" termcap entry for braindamaged terminals.
 * 
 * Revision 1.9  87/04/19  22:52:54  geoff
 * Add fields for capitalization handling.  Update MAGIC so it will refuse to
 * work with uncapitalized hash files.  Make hashheader a global.
 * 
 * Revision 1.8  87/04/01  15:22:48  geoff
 * Integrate Joe Orost's V7/register changes into the main branch
 * 
 * Revision 1.7  87/03/22  23:56:32  geoff
 * Fix minor bugs found by Dave Mason
 * 
 * Revision 1.6  87/03/01  00:56:50  geoff
 * Get rid of struct node;  add keep flag to struct dent, and delete
 * some excess stuff caused by bad patching.
 * 
 * Revision 1.5  87/02/28  14:58:21  geoff
 * Add a full struct dent to the tree node structure.
 * 
 * Revision 1.4  87/02/26  00:31:07  geoff
 * Integrate McQueer's enhancements into the main branch
 * 
 * Revision 1.3  87/01/19  00:06:41  geoff
 * Move LIBDIR to where I keep it.
 * 
 * Revision 1.2  87/01/17  13:11:53  geoff
 * Add RCS ID keywords
 * 
 */

#include <stdio.h>

#ifdef NO8BIT
#define SET_SIZE	128
#else
#define SET_SIZE	256
#endif

#define MASKTYPE	long
#define MASKTYPE_WIDTH	(8 * sizeof (MASKTYPE))
#define MASKSIZE	(MASKBITS / MASKTYPE_WIDTH)

/* The following is really testing for MASKSIZE <= 1, but cpp can't do that */
#ifndef lint
#if MASKBITS <= 32
#define SETMASKBIT(mask, bit) ((mask)[0] |= 1 << (bit))
#define CLRMASKBIT(mask, bit) ((mask)[0] &= ~(1 << (bit)))
#define TSTMASKBIT(mask, bit) ((mask)[0] & (1 << (bit)))
#else
#define SETMASKBIT(mask, bit) \
		    ((mask)[(bit) / MASKTYPE_WIDTH] |= \
		      1 << ((bit) & (MASKTYPE_WIDTH - 1)))
#define CLRMASKBIT(mask, bit) \
		    ((mask)[(bit) / MASKTYPE_WIDTH] &= \
		      ~(1 << ((bit) & (MASKTYPE_WIDTH - 1))))
#define TSTMASKBIT(mask, bit) \
		    ((mask)[(bit) / MASKTYPE_WIDTH] & \
		      (1 << ((bit) & (MASKTYPE_WIDTH - 1))))
#endif
#endif /* lint */

#if MASKBITS > 64
#define FULLMASKSET
#endif

#if MASKBITS <= 32
# ifndef lint
#define BITTOCHAR(bit)	((bit) + 'A')
#define CHARTOBIT(ch)	((ch) - 'A')
# endif /* lint */
#define LARGESTFLAG	26	/* 5 are needed for flagfield below */
#define FLAGBASE	26
#else
# if MASKBITS <= 64
#  ifndef lint
#define BITTOCHAR(bit)	((bit) + 'A')
#define CHARTOBIT(ch)	((ch) - 'A')
#  endif /* lint */
#define LARGESTFLAG	(64 - 6) /* 5 are needed for flagfield below */
#define FLAGBASE	26
# else
#  ifndef lint
#define BITTOCHAR(bit)	(bit)
#define CHARTOBIT(ch)	(ch)
#  endif /* lint */
#define LARGESTFLAG	MASKBITS /* flagfield is a separate field */
#define FLAGBASE	0
# endif
#endif

/*
** Data type for internal word storage.  If necessary, we use shorts rather
** than chars so that string characters can be encoded as a single unit.
*/
#if (SET_SIZE + MAXSTRINGCHARS) <= 256
#ifndef lint
#define ICHAR_IS_CHAR
#endif /* lint */
#endif

#ifdef ICHAR_IS_CHAR
typedef unsigned char	ichar_t;	/* Internal character */
#define icharlen	strlen
#define icharcpy	strcpy
#define icharcmp	strcmp
#define icharncmp	strncmp
#define chartoichar(x)	((ichar_t) (x))
#else
typedef unsigned short	ichar_t;	/* Internal character */
#define chartoichar(x)	((ichar_t) (unsigned char) (x))
#endif

struct dent
    {
    struct dent *	next;
    char *		word;
    MASKTYPE		mask[MASKSIZE];
#ifdef FULLMASKSET
    char		flags;
#endif
    };

/*
** Flags in the directory entry.  If FULLMASKSET is undefined, these are
** stored in the highest bits of the last longword of the mask field.  If
** FULLMASKSET is defined, they are stored in the extra "flags" field.
#ifdef CAPITALIZATION
**
** If a word has only one capitalization form, and that form is not
** FOLLOWCASE, it will have exactly one entry in the dictionary.  The
** legal capitalizations will be indicated by the 2-bit capitalization
** field, as follows:
**
**	ALLCAPS		The word must appear in all capitals.
**	CAPITALIZED	The word must be capitalized (e.g., London).
**			It will also be accepted in all capitals.
**	ANYCASE		The word may appear in lowercase, capitalized,
**			or all-capitals.
**
** Regardless of the capitalization flags, the "word" field of the entry
** will point to an all-uppercase copy of the word.  This is to simplify
** the large portion of the code that doesn't care about capitalization.
** Ispell will generate the correct version when needed.
**
** If a word has more than one capitalization, there will be multiple
** entries for it, linked together by the "next" field.  The initial
** entry for such words will be a dummy entry, primarily for use by code
** that ignores capitalization.  The "word" field of this entry will
** again point to an all-uppercase copy of the word.  The "mask" field
** will contain the logical OR of the mask fields of all variants.
** A header entry is indicated by a capitalization type of ALLCAPS,
** with the MOREVARIANTS bit set.
**
** The following entries will define the individual variants.  Each
** entry except the last has the MOREVARIANTS flag set, and each
** contains one of the following capitalization options:
**
**	ALLCAPS		The word must appear in all capitals.
**	CAPITALIZED	The word must be capitalized (e.g., London).
**			It will also be accepted in all capitals.
**	FOLLOWCASE	The word must be capitalized exactly like the
**			sample in the entry.  Prefix (suffix) characters
**			must be rendered in the case of the first (last)
**			"alphabetic" character.  It will also be accepted
**			in all capitals.  ("Alphabetic" means "mentioned
**			in a 'casechars' statement".)
**	ANYCASE		The word may appear in lowercase, capitalized,
**			or all-capitals.
**
** The "mask" field for the entry contains only the affix flag bits that
** are legal for that capitalization.  The "word" field will be null
** except for FOLLOWCASE entries, where it will point to the
** correctly-capitalized spelling of the root word.
**
** It is worth discussing why the ALLCAPS option is used in
** the header entry.  The header entry accepts an all-capitals
** version of the root plus every affix (this is always legal, since
** words get capitalized in headers and so forth).  Further, all of
** the following variant entries will reject any all-capitals form
** that is illegal due to an affix.
**
** Finally, note that variations in the KEEP flag can cause a multiple-variant
** entry as well.  For example, if the personal dictionary contains "ALPHA",
** (KEEP flag set) and the user adds "alpha" with the KEEP flag clear, a
** multiple-variant entry will be created so that "alpha" will be accepted
** but only "ALPHA" will actually be kept.
#endif
*/
#ifdef FULLMASKSET
#define flagfield	flags
#else
#define flagfield	mask[MASKSIZE - 1]
#endif
#define USED		(1 << (FLAGBASE + 0))
#define KEEP		(1 << (FLAGBASE + 1))
#ifdef CAPITALIZATION
#define ANYCASE		(0 << (FLAGBASE + 2))
#define ALLCAPS		(1 << (FLAGBASE + 2))
#define CAPITALIZED	(2 << (FLAGBASE + 2))
#define FOLLOWCASE	(3 << (FLAGBASE + 2))
#define CAPTYPEMASK	(3 << (FLAGBASE + 2))
#define MOREVARIANTS	(1 << (FLAGBASE + 4))
#define ALLFLAGS	(USED | KEEP | CAPTYPEMASK | MOREVARIANTS)
#define captype(x)	((x) & CAPTYPEMASK)
#else /* CAPITALIZATION */
#define ALLFLAGS	(USED | KEEP)
#endif /* CAPITALIZATION */

/*
 * Language tables used to encode prefix and suffix information.
 */
struct flagent
    {
    ichar_t *		strip;			/* String to strip off */
    ichar_t *		affix;			/* Affix to append */
    short		flagbit;		/* Flag bit this ent matches */
    short		stripl;			/* Length of strip */
    short		affl;			/* Length of affix */
    short		numconds;		/* Number of char conditions */
    short		flagflags;		/* Modifiers on this flag */
    char		conds[SET_SIZE + MAXSTRINGCHARS]; /* Adj. char conds */
    };

/*
 * Bits in flagflags
 */
#define FF_CROSSPRODUCT	(1 << 0)		/* Affix does cross-products */

union ptr_union					/* Aid for building flg ptrs */
    {
    struct flagptr *	fp;			/* Pointer to more indexing */
    struct flagent *	ent;			/* First of a list of ents */
    };

struct flagptr
    {
    union ptr_union	pu;			/* Ent list or more indexes */
    int			numents;		/* If zero, pu.fp is valid */
    };

/*
 * Description of a single string character type.
 */
struct strchartype
    {
    char *		name;			/* Name of the type */
    char *		suffixes;		/* File suffixes, null seps */
    };

/*
 * Header placed at the beginning of the hash file.
 */
struct hashheader
    {
    short magic;    	    	    	    	/* Magic number for ID */
    short compileoptions;			/* How we were compiled */
    short maxstringchars;			/* Max # strchrs we support */
    short maxstringcharlen;			/* Max strchr len supported */
    int stringsize;				/* Size of string table */
    int lstringsize;				/* Size of lang. str tbl */
    int tblsize;				/* No. entries in hash tbl */
    int stblsize;				/* No. entries in sfx tbl */
    int ptblsize;				/* No. entries in pfx tbl */
    int sortval;				/* Largest sort ID assigned */
    int nstrchars;				/* No. strchars defined */
    int nstrchartype;				/* No. strchar types */
    int strtypestart;				/* Start of strtype table */
    char nrchars[5];				/* Nroff special characters */
    char texchars[13];				/* TeX special characters */
    char defspaceflag;				/* Default missingspace flag */
    char defhardflag;				/* Default tryveryhard flag */
    char flagmarker;				/* "Start-of-flags" char */
    unsigned short sortorder[SET_SIZE + MAXSTRINGCHARS]; /* Sort ordering */
    ichar_t lowerconv[SET_SIZE + MAXSTRINGCHARS]; /* Lower-conversion table */
    ichar_t upperconv[SET_SIZE + MAXSTRINGCHARS]; /* Upper-conversion table */
    char wordchars[SET_SIZE + MAXSTRINGCHARS]; /* NZ for chars found in wrds */
    char upperchars[SET_SIZE + MAXSTRINGCHARS]; /* NZ for uppercase chars */
    char lowerchars[SET_SIZE + MAXSTRINGCHARS]; /* NZ for lowercase chars */
    char boundarychars[SET_SIZE + MAXSTRINGCHARS]; /* NZ for boundary chars */
    char stringstarts[SET_SIZE];		/* NZ if char can start str */
    char stringchars[MAXSTRINGCHARS][MAXSTRINGCHARLEN + 1]; /* String chars */
    char stringdups[MAXSTRINGCHARS];		/* No. of "base" char */
    char dupnos[MAXSTRINGCHARS];		/* Dup char ID # */
    };

/* hash table magic number */
#define MAGIC			0x9601

/* compile options, put in the hash header for consistency checking */
#ifdef NO8BIT
# define MAGIC8BIT		0x01
#else
# define MAGIC8BIT		0x00
#endif
#ifdef CAPITALIZATION
# define MAGICCAPITALIZATION	0x02
#else
# define MAGICCAPITALIZATION	0x00
#endif
#if MASKBITS <= 32
# define MAGICMASKSET		0x00
#else
# if MASKBITS <= 64
#  define MAGICMASKSET		0x04
# else
#  if MASKBITS <= 128
#   define MAGICMASKSET		0x08
#  else
#   define MAGICMASKSET		0x0C
#  endif
# endif
#endif

#define COMPILEOPTIONS	(MAGIC8BIT | MAGICCAPITALIZATION | MAGICMASKSET)

/*
 * Structure used to record data about successful lookups; these values
 * are used in the ins_root_cap routine to produce correct capitalizations.
 */
struct success
    {
    struct dent *	dictent;	/* Header of dict entry chain for wd */
    struct flagent *	prefix;		/* Prefix flag used, or NULL */
    struct flagent *	suffix;		/* Suffix flag used, or NULL */
    };

/*
** Offsets into the nroff special-character array
*/
#define NRLEFTPAREN	hashheader.nrchars[0]
#define NRRIGHTPAREN	hashheader.nrchars[1]
#define NRDOT		hashheader.nrchars[2]
#define NRBACKSLASH	hashheader.nrchars[3]
#define NRSTAR		hashheader.nrchars[4]

/*
** Offsets into the TeX special-character array
*/
#define TEXLEFTPAREN	hashheader.texchars[0]
#define TEXRIGHTPAREN	hashheader.texchars[1]
#define TEXLEFTSQUARE	hashheader.texchars[2]
#define TEXRIGHTSQUARE	hashheader.texchars[3]
#define TEXLEFTCURLY	hashheader.texchars[4]
#define TEXRIGHTCURLY	hashheader.texchars[5]
#define TEXLEFTANGLE	hashheader.texchars[6]
#define TEXRIGHTANGLE	hashheader.texchars[7]
#define TEXBACKSLASH	hashheader.texchars[8]
#define TEXDOLLAR	hashheader.texchars[9]
#define TEXSTAR		hashheader.texchars[10]
#define TEXDOT		hashheader.texchars[11]
#define TEXPERCENT	hashheader.texchars[12]

/*
** The isXXXX macros normally only check ASCII range, and don't support
** the character sets of other languages.  These private versions handle
** whatever character sets have been defined in the affix files.
*/
#ifdef lint
extern ichar_t mytolower ();
extern ichar_t mytoupper ();
#else /* lint */
#define myupper(X)	(hashheader.upperchars[X])
#define mylower(X)	(hashheader.lowerchars[X])
#define myspace(X)	(((X) > 0)  &&  ((X) < 0x80) \
			  &&  isspace((unsigned char) (X)))
#define iswordch(X)	(hashheader.wordchars[X])
#define isboundarych(X) (hashheader.boundarychars[X])
#define isstringstart(X) (hashheader.stringstarts[(unsigned char) (X)])
#define mytolower(X)	(hashheader.lowerconv[X])
#define mytoupper(X)	(hashheader.upperconv[X])
#endif /* lint */

/*
** These macros are similar to the ones above, but they take into account
** the possibility of string characters.  Note well that they take a POINTER,
** not a character.
**
** The "l_" versions set "len" to the length of the string character as a
** handy side effect.  (Note that the global "laststringch" is also set,
** and sometimes used, by these macros.)
**
** The "l1_" versions go one step further and guarantee that the "len"
** field is valid for *all* characters, being set to 1 even if the macro
** returns false.  This macro is a great example of how NOT to write
** readable C.
*/
#define isstringch(ptr, canon)	(isstringstart (*ptr) \
				  &&  stringcharlen (ptr, canon) > 0)
#define l_isstringch(ptr, len, canon)	\
				(isstringstart (*ptr) \
				  &&  (len = stringcharlen (ptr, canon)) > 0)
#define l1_isstringch(ptr, len, canon)	\
				(len = 1, \
				  isstringstart (*ptr) \
				    &&  ((len = stringcharlen (ptr, canon)) \
					> 0 \
				      ? 1 : (len = 1, 0)))

#ifdef AMIGA
#ifndef BUILDHASH
#define printf		printcon
#undef putchar
#undef getchar
#define putchar 	putccon
#define getchar 	getccon
#endif /* BUILDHASH */
#endif /* AMIGA */

/*
 * termcap variables
 */
#ifdef MAIN
# define EXTERN /* nothing */
#else
# define EXTERN extern
#endif

extern SIGNAL_TYPE	done ();
extern char *		tgetstr ();
extern char *		ichartosstr ();
extern ichar_t *	strtosichar ();
extern char *		printichar ();
extern char *		getline ();
extern void		chupcase ();
extern void		upcase ();
extern void 		lowcase ();
#ifdef REGEX_LOOKUP
extern char *		do_regex_lookup ();
#endif /* REGEX_LOOKUP */

#ifdef AMIGA 
extern long		_Heapsize;
#endif /* AMIGA */

EXTERN char *	BC;	/* backspace if not ^H */
EXTERN char *	cd;	/* clear to end of display */
EXTERN char *	cl;	/* clear display */
EXTERN char *	cm;	/* cursor movement */
EXTERN char *	ho;	/* home */
EXTERN char *	nd;	/* non-destructive space */
EXTERN char *	so;	/* standout */
EXTERN char *	se;	/* standout end */
EXTERN int	sg;	/* space taken by so/se */
EXTERN char *	ti;	/* terminal initialization sequence */
EXTERN char *	te;	/* terminal termination sequence */
EXTERN int	li;	/* lines */
EXTERN int	co;	/* columns */

EXTERN int	contextsize;	/* number of lines of context to show */
EXTERN char	contextbufs[MAXCONTEXT][BUFSIZ]; /* Context of current line */
EXTERN char *	currentchar;	/* Location in contextbufs */
EXTERN char	ctoken[INPUTWORDLEN + MAXAFFIXLEN]; /* Current token as char */
EXTERN ichar_t	itoken[INPUTWORDLEN + MAXAFFIXLEN]; /* Ctoken as ichar_t str */

EXTERN char	termcap[1024];	/* termcap entry */
EXTERN char	termstr[1024];	/* for string values */
EXTERN char *	termptr;	/* pointer into termcap, used by tgetstr */

EXTERN int	numhits;	/* number of hits in dictionary lookups */
EXTERN struct success
		hits[MAX_HITS]; /* table of hits gotten in lookup */

EXTERN char *	hashstrings;	/* Strings in hash table */
EXTERN struct hashheader
		hashheader;	/* Header of hash table */
EXTERN struct dent *
		hashtbl;	/* Main hash table, for dictionary */
EXTERN int	hashsize;	/* Size of main hash table */

EXTERN char	hashname[MAXPATHLEN]; /* Name of hash table file */

EXTERN int	aflag;		/* NZ if -a or -A option specified */
EXTERN int	cflag;		/* NZ if -c (crunch) option */
EXTERN int	lflag;		/* NZ if -l (list) option */
EXTERN int	incfileflag;	/* whether xgets() acts exactly like gets() */
EXTERN int	nodictflag;	/* NZ if dictionary not needed */

EXTERN int	erasechar;	/* User's erase character, from stty */
EXTERN int	killchar;	/* User's kill character */

EXTERN int	laststringch;	/* Number of last string character */
EXTERN int	defdupchar;	/* Default duplicate string type */

EXTERN int	numpflags;		/* Number of prefix flags in table */
EXTERN int	numsflags;		/* Number of suffix flags in table */
EXTERN struct flagptr pflagindex[SET_SIZE + MAXSTRINGCHARS];
					/* Fast index to pflaglist */
EXTERN struct flagent *	pflaglist;	/* Prefix flag control list */
EXTERN struct flagptr sflagindex[SET_SIZE + MAXSTRINGCHARS];
					/* Fast index to sflaglist */
EXTERN struct flagent *	sflaglist;	/* Suffix flag control list */

EXTERN struct strchartype *		/* String character type collection */
		chartypes;

EXTERN FILE *	infile;			/* File being corrected */
EXTERN FILE *	outfile;		/* Corrected copy of infile */

EXTERN char *	askfilename;		/* File specified in -f option */

EXTERN int	changes;		/* NZ if changes made to cur. file */
EXTERN int	readonly;		/* NZ if current file is readonly */
EXTERN int	quit;			/* NZ if we're done with this file */

#define MAXPOSSIBLE	100	/* Max no. of possibilities to generate */

EXTERN char	possibilities[MAXPOSSIBLE][INPUTWORDLEN + MAXAFFIXLEN];
				/* Table of possible corrections */
EXTERN int	pcount;		/* Count of possibilities generated */
EXTERN int	maxposslen;	/* Length of longest possibility */
EXTERN int	easypossibilities; /* Number of "easy" corrections found */
				/* ..(defined as those using legal affixes) */

/*
 * The following array contains a list of characters that should be tried
 * in "missingletter."  Note that lowercase characters are omitted.
 */
EXTERN int	Trynum;		/* Size of "Try" array */
EXTERN ichar_t	Try[SET_SIZE + MAXSTRINGCHARS];

/*
 * Initialized variables.  These are generated using macros so that they
 * may be consistently declared in all programs.  Numerous examples of
 * usage are given below.
 */
#ifdef MAIN
#define INIT(decl, init)	decl = init
#else
#define INIT(decl, init)	extern decl
#endif

#ifdef MINIMENU
INIT (int minimenusize, 2);		/* MUST be either 2 or zero */
#else /* MINIMENU */
INIT (int minimenusize, 0);		/* MUST be either 2 or zero */
#endif /* MINIMENU */

INIT (int eflag, 0);			/* NZ for expand mode */
INIT (int dumpflag, 0);			/* NZ to do dump mode */
INIT (int fflag, 0);			/* NZ if -f specified */
#ifndef USG
INIT (int sflag, 0);			/* NZ to stop self after EOF */
#endif
INIT (int vflag, 0);			/* NZ to display characters as M-xxx */
INIT (int xflag, DEFNOBACKUPFLAG);	/* NZ to suppress backups */
INIT (int deftflag, DEFTEXFLAG);	/* NZ for TeX mode by default */
INIT (int tflag, DEFTEXFLAG);		/* NZ for TeX mode in current file */
INIT (int prefstringchar, -1);		/* Preferred string character type */

INIT (int terse, 0);			/* NZ for "terse" mode */
INIT (char *LibDict, NULL);		/* Pointer to name of $(LIBDIR)/dict */

INIT (char tempfile[MAXPATHLEN], "");	/* Name of file we're spelling into */

INIT (int minword, MINWORD);		/* Longest always-legal word */
INIT (int sortit, 1);			/* Sort suggestions alphabetically */
INIT (int missingspaceflag, -1);	/* Don't report missing spaces */
INIT (int tryhardflag, -1);		/* Always call tryveryhard */

INIT (char *currentfile, NULL);		/* Name of current input file */

/* Odd numbers for math mode in LaTeX; even for LR or paragraph mode */
INIT (int math_mode, 0);
/* P -- paragraph or LR mode
 * b -- parsing a \begin statement
 * e -- parsing an \end statement
 * r -- parsing a \ref type of argument.
 * m -- looking for a \begin{minipage} argument.
 */
INIT (char LaTeX_Mode, 'P');
INIT (int TeX_comment, 0);
