;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: GARNETDRAW; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   GARNET DRAW
;;;
;;;  Implemented by Vivek Gupta


;;; CHANGE LOG:
;;;
;;; 09/03/92  Andrew Mickish - GetFontFromMenu returns old font if no change
;;; 08/31/92  Andrew Mickish - Now fontfun always sets the tool's font
;;; 08/20/92  Andrew Mickish - "concatenate 'string" ---> "merge-pathnames"
;;; 08/12/92  Rajan Parthasarathy - Modified to use save/load gadgets.
;;;            Added s-value of :start-where of POLYGON-MAKER in Read-Sel-Fn.
;;;            The function Save-File-If-Wanted is now loaded with the
;;;            save-gadget.
;;;  07/7/92  Brad Myers - Moved multi-grow and select in region to
;;;                        multi-graphics-selection widget
;;; 05/20/92  Brad Myers - Add gridding, fixed text entry, use Modal-p windows
;;;                        removed load of prop-sheets
;;; 05/11/92  Ed Pervin - Moved bitmaps to lib/bitmaps directory.
;;;			  Added load of ps-loader.
;;; 05/01/92  Andrew Mickish - Added :except in :constant list of *Read-DB*,
;;;             Fixed interface to CURRENT-STATE's :feedback's :frfs slot,
;;;             NEW-ITEM and POLYGON-MAKER no longer create instances of NIL
;;;             filling-styles.
;;; 04/28/92  Andrew Mickish - Added schema-p check in Read-Sel-Fn, added
;;;             garnetdraw package name in Write-Draw-Agg
;;; 04/23/92  Ed Pervin      - Added proclaim statement
;;; 04/20/92  Andrew Mickish - Put into standard form
;;; 09/15/91  Vivek Gupta    - Started
;;;

(in-package "GARNETDRAW" :use '("LISP" "KR"))

(export '(DO-GO DO-STOP))

;; Load necessary files
;;
(defvar GARNETDRAW-INIT
  (progn
    ;;;  Load ps-loader.
    (load (merge-pathnames "ps-loader" user::Garnet-PS-PathName))
    ;;;  Load gadgets.
    (dolist (file '("multi-selection-loader" "polyline-creator-loader"
		    "arrow-line-loader" "menubar-loader" "error-gadget-loader"
		    "scrolling-labeled-box-loader"
		    "trill-device-loader" "save-gadget-loader"))
      (load (merge-pathnames file user::Garnet-Gadgets-PathName)
	    :verbose T))))

#|
====================================================================
VARIABLES

These are the variables used in different
parts of the file
====================================================================
|#

(proclaim '(special create-or-edit current-state draw-win mover-grower
		    moving-agg moving-arrowline moving-doublearrowline
		    moving-line moving-oval moving-rect moving-roundtangle
		    palette-feedback text-feedback
		    tools-menu top-draw-agg win PS-Read-WIN GRID-WIN SAVE-WIN))
(defvar *DRAW-AGG*)(defvar *Q-BOX*)
(defvar *SAVE-DB*)(defvar *Read-DB*)
(defvar *GRID-DB*)

(defparameter the-color-list NIL)

(create-instance 'GRID-OBJ NIL ; use an object so constraints to values
		 (:gridvis NIL) ; whether can see gridding or not
		 (:gridon NIL) ; whether gridding is in use or not
		 (:gridamt 10)) ; amount to grid by
(defparameter *Grid-Menu-Item* NIL) ; set with menu bar item for grid on/off
(defparameter *Grid-Vis-Item* NIL) ; set with menu bar item for grid vis on/off
(defparameter POLYGON-MAKER NIL)

(defparameter rgbvalues
  '((1.00 0.00 0.52) (1.00 0.00 0.82) (1.00 0.00 1.00) (0.82 0.00 1.00)
    (0.52 0.00 1.00) (0.00 0.00 1.00) (0.00 0.52 1.00) (0.00 0.82 1.00)
    (0.00 1.00 1.00) (0.00 1.00 0.82) (0.00 1.00 0.52) (0.00 1.00 0.00)
    (0.52 1.00 0.00) (0.82 1.00 0.00) (1.00 1.00 0.00) (1.00 0.82 0.00)
    (1.00 0.52 0.00) (1.00 0.00 0.00)))

(defvar *LINE-STYLE-TO-SWAP*
  (create-instance NIL opal:line-style
    (:foreground-color
     (create-instance NIL opal:color
       (:red 0)(:green 0)(:blue 0)))
    (:line-style :solid)
    (:line-thickness 4)))

(defvar *TEMP-POINTS* NIL)
(defvar *TEMP-LIST* NIL)
(defvar *CLIPBOARD* NIL)
(defvar *DOCUMENT-NAME* "Untitled")
(defvar *FEEDBACK-LINE-STYLE*
  (create-instance NIL opal:line-style
    (:constant T)
    (:line-thickness 3)))


#|
====================================================================
PROTOTYPES

Below we have the prototypes for all the objects which we are going
to draw.  The first is the one for grouping objects, the rest are
individual objects.
====================================================================
|#


(defun Create-Moving-Prototypes ()
  (create-instance 'moving-agg opal:aggregadget
    (:group-p t)
    (:grow-p t))

  (create-instance 'MOVING-LINE opal:line
    (:points (list 0 0 0 0))
    (:x1 (o-formula (first (gvl :points))))
    (:y1 (o-formula (second (gvl :points))))
    (:x2 (o-formula (third (gvl :points))))
    (:y2 (o-formula (fourth (gvl :points))))
    (:grow-p t)
    (:group-p NIL)
    (:line-p t)
    (:draw-function :xor)
    (:fast-redraw-p t)
    (:visible-p NIL)
    (:line-style opal:dashed-line))
  
  (create-instance 'moving-arrowline garnet-gadgets:double-Arrow-Line
    (:points (list 0 0 0 0))
    (:x1 (o-formula (first (gvl :points))))
    (:y1 (o-formula (second (gvl :points))))
    (:x2 (o-formula (third (gvl :points))))
    (:y2 (o-formula (fourth (gvl :points))))
    (:arrow-p t)
    (:line-p t)
    (:grow-p t)
    (:group-p NIL)
    (:arrowhead-p 2)
    (:visible-p NIL)
    (:filling-style NIL)
    (:line-style opal:dashed-line)
    (:open-p NIL)
    (:parts `((:line :modify (:fast-redraw-p t) (:draw-function :xor))
	      (:arrowhead1 :modify (:fast-redraw-p t) (:draw-function :xor))
	      (:arrowhead2 :modify (:fast-redraw-p t) (:draw-function :xor)))))

  (create-instance 'moving-doublearrowline garnet-gadgets:Double-Arrow-Line
    (:points (list 0 0 0 0))
    (:x1 (o-formula (first (gvl :points))))
    (:y1 (o-formula (second (gvl :points))))
    (:x2 (o-formula (third (gvl :points))))
    (:y2 (o-formula (fourth (gvl :points))))
    (:arrow-p t)
    (:arrowhead-p :both)
    (:grow-p t)
    (:group-p NIL)
    (:line-p t)
    (:visible-p NIL)
    (:filling-style NIL)
    (:line-style opal:dashed-line)
    (:open-p NIL)
    (:parts `((:line :modify (:fast-redraw-p t) (:draw-function :xor))
	      (:arrowhead1 :modify (:fast-redraw-p t) (:draw-function :xor))
	      (:arrowhead2 :modify (:fast-redraw-p t) (:draw-function :xor)))))
  
  (create-instance 'MOVING-RECT opal:rectangle
    (:box (list 0 0 0 0))
    (:left (o-formula (first (gvl :box))))
    (:top  (o-formula (second (gvl :box))))
    (:width (o-formula (third (gvl :box))))
    (:height (o-formula (fourth (gvl :box))))
    (:group-p NIL)
    (:grow-p t)
    (:filling-style NIL)
    (:line-p NIL)
    (:fast-redraw-p t)
    (:draw-function :xor)
    (:visible-p NIL)
    (:line-style opal:dashed-line))

  (create-instance 'MOVING-ROUNDTANGLE opal:roundtangle
    (:box (list 0 0 0 0))
    (:left (o-formula (first (gvl :box))))
    (:top (o-formula (second (gvl :box))))
    (:width (o-formula (third (gvl :box))))
    (:height (o-formula (fourth (gvl :box))))
    (:filling-style NIL)
    (:grow-p t)
    (:group-p NIL)
    (:visible-p NIL)
    (:line-p NIL)
    (:fast-redraw-p t)
    (:draw-function :xor)
    (:line-style opal:dashed-line))

  (create-instance 'moving-oval opal:oval
    (:box (list 0 0 0 0))
    (:left (o-formula (first (gvl :box))))
    (:top (o-formula (second (gvl :box))))
    (:width (o-formula (third (gvl :box))))
    (:height (o-formula (fourth (gvl :box))))
    (:filling-style NIL)
    (:grow-p t)
    (:group-p NIL)
    (:fast-redraw-p t)
    (:draw-function :xor)
    (:visible-p NIL)
    (:line-p NIL)
    (:line-style opal:dashed-line))

  )

(defun Create-Text-Feedback ()
  (create-instance 'TEXT-FEEDBACK opal:cursor-multi-text
    (:box (list 0 0 0 0))
    (:string "")
    (:visible (o-formula (gvl :obj-over)))
    (:group-p NIL)
    (:left (o-formula (first (gvl :box))))		 
    (:top (o-formula (second (gvl :box))))))

	      
			
#|
====================================================================
DIALOG BOX FUNCTIONS

====================================================================
|#

;; Since the same save gadget is used for BOTH creating ps files and
;; for saving, this function resets the :selection-function and the
;; :text of the save gadget after it is done.

(defun PS-Sel-Fn (g v)
  (declare (ignore v))
  (let ((filename (g-value g :file-input :value)))
    (opal:make-ps-file DRAW-WIN filename))
  (s-value g :selection-function 'Save-Sel-Fn)
  (s-value (g-value g :text) :string "Saving...")
  )

(defun Read-Sel-Fn (g v)
  (declare (ignore g))
  (let ((filename v))
    (if (probe-file filename)
	(progn
	  (gg:set-selection MOVER-GROWER NIL)
	  (if (schema-p *DRAW-AGG*) (opal:destroy *DRAW-AGG*))
	  (setf *document-name* filename)
	  (load *document-name*)  ;; This sets *DRAW-AGG*
	  (s-value MOVER-GROWER :start-where (list :element-of-or-none
						   *DRAW-AGG*))
	  (s-value CREATE-OR-EDIT :start-where
		   (list :element-of-or-none *DRAW-AGG*
			 :type opal:cursor-multi-text))
	  (s-value POLYGON-MAKER :start-where
		   (list :in *DRAW-AGG*))
	  
	  (let ((kr::*constants-disabled* t))
	    (opal:add-component TOP-DRAW-AGG *DRAW-AGG* :where :back))
	  (s-value WIN :title (file-namestring *document-name*)))
	(gg:display-query *Q-BOX* "There is no file by that name"
			  '("OK"))))
  )

(defun Read-File ()
  (gg:display-load-gadget-and-wait *Read-DB* *document-name*)
  )

(defun Write-Draw-Agg ()
  (with-open-file (*standard-output* *document-name*
		   :direction :output :if-exists :supersede)
    (format t "(in-package ~cUSER~c :use '(~cLISP~c ~cKR~c))~%~%" #\" #\" #\"
	    #\" #\" #\")
    (format t "(setf garnetdraw::*DRAW-AGG*~%")
    (opal:write-gadget *DRAW-AGG* T T)
    (format t ")~%")))

;; This is called by savefun and saveasfun.  If confirm-p is T, it
;; displays the save gadget.  Else, it simply saves.

(defun Save-File (&key confirm-p)
  (if (or confirm-p (equal *document-name* "Untitled"))
      (gg:display-save-gadget-and-wait *save-db* *document-name*)
      (Write-Draw-Agg)))

;; This is the selection function of the save gadget.  It sets the
;; document name to be the filename the gadget returns, v, and
;; calls Write-Draw-Agg, which does the saving.

(defun Save-Sel-Fn (g v)
  (declare (ignore g))
  (setq *document-name* v)
  (Write-Draw-Agg)
  )

#|
====================================================================
DIALOG BOXES

====================================================================
|#

(defun Create-Query-Gadget ()
  (setf *Q-BOX* (create-instance NIL garnet-gadgets:query-gadget
		  (:parent-window WIN))))

(defun Create-File-DB ()
  (setf *SAVE-DB*
	(create-instance NIL gg:save-gadget
	  (:min-gadget-width 285)
	  (:parent-window win)
	  (:modal-p T)
	  (:top 40)
	  (:button-panel-items '("OK" "Cancel"))
	  
	  (:selection-function 'Save-Sel-Fn)
	  (:parts `(:dir-input
		    :file-menu
		    :file-input
		    :message
		    :OK-Cancel-buttons
		    (:text ,opal:text
		     (:constant T  :except :string)
		     (:left ,(o-formula (gvl :parent :left)))
		     (:top 10)
		     (:font ,(opal:get-standard-font NIL :bold-italic :large))
		     (:string "Saving..."))
		    ))))
  (setf SAVE-WIN (g-value *save-db* :window))
)

(defun Create-Grid-DB ()
  (create-instance 'GRID-WIN inter:interactor-window
    (:parent WIN)
    (:modal-p T)
    (:visible NIL)
    (:left (o-formula (- (floor (gv WIN :width) 2)
			 (floor (gvl :width) 2))))
    (:top (o-formula (- (floor (gv WIN :height) 2)
			(floor (gvl :height) 2))))
    (:width 345)
    (:height 85))
  (setf *GRID-DB*
	(create-instance NIL opal:aggregadget
	  (:left 0) (:top 0)
	  (:parts
	   `((:text ,opal:text
	      (:constant T)
	      (:left 10) (:top 10)
	      (:font ,(opal:get-standard-font NIL :bold-italic :large))
	      (:string "New Grid Increment:"))
	     (:value ,gg:trill-device
	      (:constant T)
	      (:left 225) (:top 10))
	     (:ok-cancel ,gg:text-button-panel
	      (:constant T)
	      (:left 114) (:top 40)
	      (:items ("OK" "Cancel"))
	      (:final-feedback-p NIL)
	      (:gray-width 3) (:text-offset 2) (:shadow-offset 5)
	      (:direction :horizontal)
	      (:selection-function
	       ,#'(lambda (g v)
		    (s-value GRID-WIN :visible NIL)
		    (opal:update GRID-WIN)
		    (when (equal v "OK")
		      (s-value GRID-OBJ :gridamt
			       (g-value g :parent :value :value))))))))))
  (s-value GRID-WIN :aggregate *GRID-DB*)
  )

(defun Create-Read-DB ()
  (setf *Read-DB*
	(create-instance NIL gg:load-gadget
	  (:selection-function #'Read-Sel-Fn)
	  (:min-gadget-width 285)
	  (:modal-p T)
	  (:check-filenames-p NIL)
	  (:parent-window win)
	  (:top 40)
	  (:button-panel-items '("OK" "Cancel"))
	  (:parts
	   `(:dir-input
	     :file-menu
	     :file-input
	     :message
	     (:text ,opal:text
	      (:constant T :except :string)
	      (:left 10) (:top 10)
	      (:font ,(opal:get-standard-font NIL :bold-italic :large))
	      (:string "Reading..."))
	     
	     (:OK-cancel-buttons :modify
		     (:top ,(o-formula (+ (gvl :parent :file-input :top)
					  (gvl :parent :file-input :height)
					  20))))))))
	     
  (setf PS-Read-WIN (g-value *Read-DB* :window))
  )


#|
====================================================================
MENU FUNCTIONS AND MENUBAR

These functions are the necessary functions for the Menubar to act
properly on any action by the user.  Additional functions must be
added here for cut, paste, copy, and various font functions.
====================================================================
|#

(defun clearfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (opal:remove-component TOP-DRAW-AGG *DRAW-AGG*)
  (with-constants-disabled
      (dolist (comp (copy-list (g-value *DRAW-AGG* :components)))
	(opal:remove-component *DRAW-AGG* comp)))
  (opal:add-component TOP-DRAW-AGG *DRAW-AGG*)
  (garnet-gadgets:set-selection MOVER-GROWER NIL))

(defun quitfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (unless (eq :cancel (gg:Save-File-If-Wanted *save-db* *document-name*))
    (do-stop)))

(defun psfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (s-value *Save-DB* :selection-function #'PS-Sel-Fn)
  (s-value (g-value *Save-DB* :text) :string "PS File...")
  (gg:display-save-gadget-and-wait *Save-DB* *document-name*)
  )

(defun openfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (unless (eq :cancel (gg:save-file-if-wanted *save-db* *document-name*))
    (Read-File)))

(defun newfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (unless (eq :cancel (gg:Save-File-If-Wanted *save-db* *document-name*))
	(gg:set-selection MOVER-GROWER NIL)
	(opal:remove-component TOP-DRAW-AGG *DRAW-AGG*)
	(with-constants-disabled
	    (dolist (comp (copy-list (g-value *DRAW-AGG* :components)))
	      (opal:destroy comp)))
	(opal:add-component TOP-DRAW-AGG *DRAW-AGG*)))

(defun saveasfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (Save-File :confirm-p T))

(defun savefun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (Save-File))

(defun sort-objs-display-order (objs)
  (let ((sorted-list (copy-list objs))
	(reference-objs (g-value *DRAW-AGG* :components)))
    (sort sorted-list #'(lambda (o1 o2)
			  (< (position o1 reference-objs)
			     (position o2 reference-objs))))))

(defun cutfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (with-constants-disabled
      (dolist (item (copy-list *Clipboard*))
	(opal:destroy item)))
  (setq *Clipboard*
	(sort-objs-display-order (copy-list (g-value MOVER-GROWER :value))))
  (with-constants-disabled
      (dolist (item (copy-list (g-value MOVER-GROWER :value)))
	(opal:remove-component *DRAW-AGG* item)))
  (garnet-gadgets:set-selection MOVER-GROWER NIL))

(defun copyfun (gadget menu-item submenu-item)  
  (declare (ignore gadget menu-item submenu-item))
  (let ((kr::*constants-disabled* t))
    (dolist (item (copy-list *Clipboard*))
      (opal:destroy item))
#|    
    (setq *Clipboard* (sort-objs-display-order (copy-list (g-value MOVER-GROWER :value))))))
|#
    (setq *Clipboard* NIL)
    (dolist (item (reverse (sort-objs-display-order
			    (copy-list (g-value MOVER-GROWER :value)))))
      (setq *Clipboard* (cons (opal:copy-gadget item NIL) *Clipboard*)))))
    
(defun paste-helperfun (group-item)
  (with-constants-disabled
    (dolist (item (copy-list (get-values group-item :components)))
      (if (g-value item :group-p)
	  (paste-helperfun item)
	  (if (g-value item :arrow-p)
	      (s-value item :parts 
		       `((:line :modify
			  (:fast-redraw-p NIL)
			  (:draw-function :copy))
			 (:arrowhead1 :modify
			  (:fast-redraw-p NIL)
			  (:draw-function :copy))
			 (:arrowhead2 :modify
			  (:fast-redraw-p NIL)
			  (:draw-function :copy))))
	      (progn
		(s-value item :draw-function :copy)
		(s-value item :fast-redraw-p NIL)))))))

(defun install-quality-copies (proto inst)
  (if (g-value inst :group-p)
      (let ((proto-comps (g-value proto :components))
	    (inst-comps (g-value inst :components)))
	(dotimes (n (length proto-comps))
	  (install-quality-copies (nth n proto-comps) (nth n inst-comps))))
      (progn
	(s-value inst
		 :line-style
		 (opal:copy-gadget (g-value proto :line-style) NIL))
	(if (g-value proto :text-p)
	    (s-value inst :font
		     (opal:copy-gadget (g-value proto :font) NIL))))))

(defun my-copy-gadget (proto inst-name)
  (let ((inst (opal:copy-gadget proto inst-name)))
    (install-quality-copies proto inst)
    inst))

(defun pastefun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (setq *temp-list* NIL)
  (let ((kr::*constants-disabled* t))
    (dolist (obj *clipboard*)
      (let ((new-component (my-copy-gadget obj NIL)))
	(progn
	  (setf *temp-list* (cons new-component *temp-list*))
	  (if (g-value new-component :group-p)
	      (paste-helperfun new-component)
	      (if (g-value new-component :arrow-p)
		  (s-value new-component :parts 
			   `((:line :modify
			      (:fast-redraw-p NIL)
			      (:draw-function :copy))
			     (:arrowhead1 :modify
			      (:fast-redraw-p NIL)
			      (:draw-function :copy))
			     (:arrowhead2 :modify
			      (:fast-redraw-p NIL)
			      (:draw-function :copy))))
		  
		  (progn
		    (s-value new-component :fast-redraw-p NIL)
		    (s-value new-component :draw-function :copy)))))
	(with-constants-disabled
	    (opal:add-component *DRAW-AGG* new-component))))
    (garnet-gadgets:set-selection MOVER-GROWER (copy-list *temp-list*))
    (setq *temp-list* NIL)))
  
(defun totopfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (let ((kr::*constants-disabled* t)
	(objs (copy-list (g-value MOVER-GROWER :value))))
    (if objs
	(dolist (obj (sort-objs-display-order objs))
	  (opal:move-component *DRAW-AGG* obj :where :front)))))

(defun tobottomfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (let ((kr::*constants-disabled* t)
    (objs (copy-list (g-value MOVER-GROWER :value))))
	
    (if objs
	(dolist (obj (reverse (sort-objs-display-order objs)))
	  (opal:move-component *DRAW-AGG* obj :where :back)))))
 
(defun refreshfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (opal:update DRAW-WIN t))
  
(defun groupfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (let ((objs (copy-list (g-value MOVER-GROWER :value))))
    (if (> (list-length objs) 1)
	(let* ((new-agg (create-instance NIL moving-agg)))
	  (progn
	    (with-constants-disabled
		(opal:add-component *DRAW-AGG* new-agg)
	      (dolist (obj (sort-objs-display-order objs))
		(opal:remove-component (g-value obj :parent) obj)
		(opal:add-component new-agg obj :where :front)))
	    (garnet-gadgets:set-selection MOVER-GROWER new-agg))))))

(defun ungroupfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (setq *temp-list* NIL)
  (let ((objs (sort-objs-display-order
	       (copy-list (g-value MOVER-GROWER :value)))))
    (if objs
	(progn
	  (dolist (grp-item objs)
	    (let ((location
		   (position grp-item (g-value *DRAW-AGG* :components))))
	      (if (g-value grp-item :group-p)
		  (with-constants-disabled
		    (opal:remove-component *DRAW-AGG* grp-item)
		    (dolist (item (reverse (copy-list
					    (g-value grp-item :components))))
		      (opal:remove-component grp-item item)
		      (setq *temp-list* (cons item *temp-list*))
		      (opal:add-component *DRAW-AGG* item :at location)))
		  (setq *temp-list* (cons grp-item *temp-list*)))))
	  (garnet-gadgets:set-selection MOVER-GROWER *temp-list*)))))
	  
			     
(defun GetFontFromMenu (submenu-item old-font)
  (cond
    ((equal submenu-item " Fixed ")
     (opal:get-standard-font :fixed 
			     (g-value old-font :face)
			     (g-value old-font :size)))
    ((equal submenu-item " Serif ")
     (opal:get-standard-font :serif
			     (g-value old-font :face)
			     (g-value old-font :size)))
    ((equal submenu-item " Sans-Serif ")
     (opal:get-standard-font :sans-serif
			     (g-value old-font :face)
			     (g-value old-font :size)))

    ((equal submenu-item " Roman ")
     (opal:get-standard-font (g-value old-font :family)
			     :roman
			     (g-value old-font :size)))
    ((equal submenu-item " Bold ")
     (opal:get-standard-font (g-value old-font :family)
			     :bold
			     (g-value old-font :size)))
    ((equal submenu-item " Italic ")
     (opal:get-standard-font (g-value old-font :family)
			     :italic
			     (g-value old-font :size)))
    ((equal submenu-item " Bold-Italic ")
     (opal:get-standard-font (g-value old-font :family)
			     :bold-italic
			     (g-value old-font :size)))
    ((equal submenu-item " Small ")
     (opal:get-standard-font (g-value old-font :family)
			     (g-value old-font :face)
			     :small))
    ((equal submenu-item " Medium ")
     (opal:get-standard-font (g-value old-font :family)
			     (g-value old-font :face)
			     :medium))
    ((equal submenu-item " Large ")
     (opal:get-standard-font (g-value old-font :family)
			     (g-value old-font :face)
			     :large))
    ((equal submenu-item " Very-Large ")
     (opal:get-standard-font (g-value old-font :family)
			     (g-value old-font :face)
			     :very-large))
    (t old-font)))

(defun fontfun (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item))
  (if (g-value MOVER-GROWER :value)
      (dolist (item (g-value MOVER-GROWER :value))
	(when (g-value item :text-p)
	  (with-constants-disabled
		(s-value item :font 
		       (GetFontFromMenu submenu-item (g-value item :font)))))))
  ;; Always set the global state
  (let* ((text-state (g-value TOOLS-MENU :text-tool :text-state))
	 (new-font (GetFontFromMenu submenu-item
				    (g-value text-state :font))))
    (s-value text-state :font new-font)))

(defun gridtoggle (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (if (g-value GRID-OBJ :gridon) 
      (progn ; turn if off, and make menu so it will turn it on
	(s-value GRID-OBJ :gridon NIL)
	(gg:menubar-set-title *Grid-Menu-Item* " Turn Grid On "))
      (progn; turn if on, and make menu so it will turn it off
	(s-value GRID-OBJ :gridon T)
	(gg:menubar-set-title *Grid-Menu-Item* " Turn Grid Off "))))

(defun gridvisible (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (if (g-value GRID-OBJ :gridvis) 
      (progn ; turn if off, and make menu so it will turn it on
	(s-value GRID-OBJ :gridvis NIL)
	(gg:menubar-set-title *Grid-Vis-Item* " Show Grid Dots "))
      (progn; turn if on, and make menu so it will turn it off
	(s-value GRID-OBJ :gridvis T)
	(gg:menubar-set-title *Grid-Vis-Item* " Hide Grid Dots "))))

(defun setgrid (gadget menu-item submenu-item)
  (declare (ignore gadget menu-item submenu-item))
  (s-value (g-value *GRID-DB* :value) :value (g-value GRID-OBJ :gridamt))
  (s-value GRID-WIN :visible T)
  (opal:raise-window GRID-WIN)
  (opal:update GRID-WIN))

(defun Create-Main-Menubar ()
  (create-instance 'MAIN-MENU garnet-gadgets:MENUBAR
    (:left 60) (:top 0)
    (:title-font (opal:get-standard-font :sans-serif :bold :large))
    (:item-font (opal:get-standard-font :sans-serif :roman :medium))
    (:items 
     '(("  File  " NIL 
	((" Open " openfun) (" New " newfun) (" Save " savefun)
	 (" Save As " saveasfun) (" Create PS " psfun) (" Quit " quitfun)))
       ("  Edit  " NIL 
	((" Cut " cutfun) (" Copy " copyfun) (" Paste " pastefun)
	 (" Clear All " clearfun) (" To Top " totopfun)
	 (" To Bottom " tobottomfun) (" Group " groupfun)
	 (" Ungroup " ungroupfun) (" Refresh " refreshfun)))
       ("  Font  " fontfun
	((" Fixed ")(" Serif ")(" Sans-Serif ")(" Roman ")(" Bold ")
	 (" Italic ")(" Bold-Italic ")(" Small ")(" Medium ")
	 (" Large ")(" Very-Large ")))
       ("  Options  " NIL
	((" Turn Grid On " gridtoggle) (" Set Grid... " setgrid)
	 (" Show Grid Dots " gridvisible)
	 ))
       )))
  )


#|
====================================================================
TOOL PALETTE

This is the list of tools available for drawing objects.  Each
contains a bitmap representation of the tool and the location
where it goes.
====================================================================
|#

(defun Create-Tool-Palette ()

  (create-instance 'TOOL-FEEDBACK opal:rectangle
    (:left (o-formula (gvl :obj-over :left)))
    (:top (o-formula (gvl :obj-over :top)))
    (:width (o-formula (gvl :obj-over :width)))
    (:height (o-formula (gvl :obj-over :height)))
    (:visible (o-formula (gvl :obj-over)))
    (:line-style opal:line-4))

  (create-instance 'TOOLS-MENU opal:aggregadget
    (:top (+ 2 (g-value MAIN-MENU :height)))
    (:selected NIL)
    (:parts
     `((:line-tool ,opal:aggregadget
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (gvl :parent :top)))
	(:feedback-object ,MOVING-LINE)
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/line.bm"
				       user::Garnet-Bitmap-Pathname)))))))
       (:rect-tool ,opal:aggregadget
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :line-tool :top))))
	(:feedback-object ,MOVING-RECT)
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/rectangle.bm"
				       user::Garnet-Bitmap-Pathname)))))))
       (:roundrect-tool ,opal:aggregadget
	(:left 5) (:top 84) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :rect-tool :top))))
	(:feedback-object ,MOVING-ROUNDTANGLE)
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/roundrect.bm"
				       user::Garnet-Bitmap-Pathname)))))))
       (:oval-tool ,opal:aggregadget
	(:feedback-object ,moving-oval)
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :roundrect-tool :top))))
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/oval.bm"
				       user::Garnet-Bitmap-Pathname)))))))
				       
       (:text-tool ,opal:aggregadget
	(:feedback-object ,TEXT-FEEDBACK)
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :oval-tool :top))))
	(:parts
	 ((:background ,opal:rectangle
		       (:left 5)(:width 32)(:height 32)
		       (:top ,(o-formula (gvl :parent :top)))
		       (:filling-style ,opal:white-fill)
		       (:line-style ,opal:line-0))
	  (:text-state ,opal:text
	   (:left ,(o-formula (opal:gv-center-x-is-center-of (gvl :parent))))
	   (:top ,(o-formula (opal:gv-center-y-is-center-of (gvl :parent))))
	   (:string "T")
	   (:line-style ,(create-instance NIL opal:default-line-style))
	   (:font ,(opal:get-standard-font NIL NIL NIL))))))
       (:polygon-tool ,opal:aggregadget
	(:feedback-object ,MOVING-RECT)
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :text-tool :top))))
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/polygon.bm"
				       user::Garnet-Bitmap-Pathname)))))))
				       
       (:arrowline-tool ,opal:aggregadget
	(:feedback-object ,moving-arrowline)
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :polygon-tool :top))))
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/linearrow.bm"
				       user::Garnet-Bitmap-Pathname)))))))
       (:doublearrowline-tool ,opal:aggregadget
	(:feedback-object ,MOVING-DOUBLEARROWLINE)
	(:left 5) (:height 32) (:width 32)
	(:top ,(o-formula (+ 31 (gvl :parent :arrowline-tool :top))))
	(:parts
	 ((:bm ,opal:bitmap
	       (:left 5) (:top ,(o-formula (gvl :parent :top)))
	       (:image
		,(opal:read-image
		      (merge-pathnames "garnetdraw/doublelinearrow.bm"
				       user::Garnet-Bitmap-Pathname)))))))))
    (:interactors
     `((:tool-interactor ,inter:button-interactor
	(:window ,(o-formula (gvl :operates-on :window)))
	(:start-event :any-mousedown)
	(:final-feedback-obj ,TOOL-FEEDBACK)
	(:how-set :set)
	(:start-where ,(o-formula (list :element-of (gvl :operates-on))))))))
					
  (s-value TOOLS-MENU :selected (car (get-values TOOLS-MENU :components)))
  (s-value TOOL-FEEDBACK :obj-over (car (get-values TOOLS-MENU :components))))


#|
====================================================================
LINE PALETTE

This is the line palette and the function which sets the value of
the line style slot of all selected object, if a new value is
selected in the line palette.
====================================================================
|#

(defun Create-Line-Palette ()

  (create-instance 'LINE-FEEDBACK opal:rectangle
    (:obj-over NIL)
    (:left 2)
    (:top (o-formula (gvl :obj-over :top)))
    (:width 52)
    (:height (o-formula (gvl :obj-over :height)))
    (:filling-style NIL)
    (:visible (o-formula (gvl :obj-over)))
    (:line-style opal:line-4))

  (create-instance 'LINE-PALETTE opal:aggregadget
    (:left 2) (:top 321) (:width 52) (:height 106)
    (:selected NIL)
    (:parts
     `((:background ,opal:rectangle
	(:left 2) (:top 321) (:width 52) (:height 106)
	(:filling-style ,opal:white-fill)
	(:line-style ,opal:line-0))
       (:line0 ,opal:line
	(:x1 11) (:y1 332) (:x2 43) (:y2 332)
	(:top 322) (:height 20)
	(:line-thick 0)
	(:line-style ,opal:line-0)
	(:hit-threshold 5))
       (:line1 ,opal:line
	(:x1 11)(:y1 353) (:x2 43) (:y2 353)
	(:top 343) (:height 20)
	(:line-thick 1)
	(:line-style ,opal:line-1)
	(:hit-threshold 5))
       (:line2 ,opal:line
	(:x1 11)(:y1 374) (:x2 43) (:y2 374)
	(:top 364) (:height 20)
	(:line-thick 2)
	(:line-style ,opal:line-2)
	(:hit-threshold 5))
       (:line4 ,opal:line
	(:x1 11)(:y1 395) (:x2 43) (:y2 395)
	(:top 385) (:height 20)
	(:line-thick 4)
	(:line-style ,opal:line-4)
	(:hit-threshold 5))
       (:line8 ,opal:line
	(:x1 11)(:y1 416) (:x2 43) (:y2 416)
	(:top 406) (:height 20)
	(:line-thick 8)
	(:line-style ,opal:line-8)
	(:hit-threshold 5))))
    (:interactors
     `((:line-interactor ,inter:button-interactor
	(:window ,(o-formula (gvl :operates-on :window)))
	(:start-event :any-mousedown)
	(:how-set :set)
	(:final-feedback-obj ,line-feedback)
	(:final-function
	 ,#'(lambda (an-interactor final-obj-over)
	      (declare (ignore an-interactor final-obj-over))
	      (when (g-value MOVER-GROWER :value)
		(dolist (thing (copy-list (g-value MOVER-GROWER :value)))
		  (let ((old-line-style (g-value thing :line-style))
			(new-line-style *line-style-to-swap*))
		    (if (not (g-value thing :group-p))
			(progn
			  (setf *line-style-to-swap* old-line-style)
			  (s-value new-line-style :line-thickness 
				   (g-value line-palette
					    :selected :line-thick))
			  (s-value new-line-style :foreground-color
				   (g-value old-line-style :foreground-color))
			  (s-value thing :line-style new-line-style))))))))
	(:start-where ,(o-formula (list :element-of (gvl :operates-on)
					:type opal:line)))))))

  (let ((line0 (g-value line-palette :line0)))
    (s-value line-palette :selected line0)
    (s-value line-feedback :obj-over line0))
  )


#|
====================================================================
SELECTED FUNCTION

This is used to change the line style or filling color of a selected
object using the current state aggregadget.
====================================================================
|#

(defun Selectedfun (an-interactor final-obj-over)
  (declare (ignore an-interactor final-obj-over))
  (when (g-value MOVER-GROWER :value)
    (if (g-value CURRENT-STATE :selectable-objs :frame :selected)
	(dolist (thing (g-value MOVER-GROWER :value))
	  (unless (g-value thing :group-p)
	    (let ((old-line-style (g-value thing :line-style))
		  (new-line-style *line-style-to-swap*))
	      (setf *line-style-to-swap* old-line-style)
	      (s-value new-line-style :line-thickness 
		       (g-value old-line-style :line-thickness))
	      (s-value new-line-style :foreground-color
		       (g-value PALETTE-FEEDBACK :obj-over :line-hold
				:foreground-color))
	      (s-value thing :line-style new-line-style))))
	  
	(dolist (thing (g-value MOVER-GROWER :value))
	  (unless (g-value thing :group-p)
	    (s-value thing :filling-style
		     (g-value PALETTE-FEEDBACK :obj-over :filling-style))))
	))

  (let* ((selectable-objs (g-value CURRENT-STATE :selectable-objs))
	 (frame (g-value selectable-objs :frame))
	 (filler (g-value selectable-objs :filler))
	 (obj-over (g-value PALETTE-FEEDBACK :obj-over)))
    (if (g-value frame :selected)
	(progn
	  (s-value frame :line-hold (g-value obj-over :line-hold))
	  (s-value frame :filling-style (g-value obj-over :filling-style)))
	(s-value filler :filling-style (g-value obj-over :filling-style))))
  )

#|
====================================================================
PATTERN AND COLOR PALETTES

The first one is for the pattern palette, this contains in the items
slot the necessary numbers for creating the different halftone
values.  The second one uses the values in the color list generated
earlier.
====================================================================
|#

(defun Create-Palette-Feedback ()
  (create-instance 'PALETTE-FEEDBACK opal:aggregadget
    (:obj-over NIL)
    (:left (o-formula (gvl :obj-over :left)))
    (:top (o-formula (gvl :obj-over :top)))
    (:width (o-formula (gvl :obj-over :width)))
    (:height (o-formula (gvl :obj-over :height)))
    (:visible (o-formula (gvl :obj-over)))
    (:fast-redraw-filling-style NIL)
    (:parts
     `((:black-rect ,opal:rectangle
	(:left ,(o-formula (+ 1 (gvl :parent :left))))
	(:top ,(o-formula (+ 1 (gvl :parent :top))))
	(:width ,(o-formula (- (gvl :parent :width) 2)))
	(:height ,(o-formula (- (gvl :parent :height) 2)))
	(:line-style ,*FEEDBACK-LINE-STYLE*)
	(:fast-redraw-p :rectangle)
	(:fast-redraw-filling-style
	 ,(o-formula (gvl :parent :fast-redraw-filling-style))))
       (:thin-white-rect ,opal:rectangle
	(:left ,(o-formula (+ 4 (gvl :parent :left))))
	(:top ,(o-formula (+ 4 (gvl :parent :top))))
	(:width ,(o-formula (- (gvl :parent :width) 8)))
	(:height ,(o-formula (- (gvl :parent :height) 8)))
	(:line-style ,opal:white-line)
	(:fast-redraw-p :rectangle)
	(:fast-redraw-filling-style
	 ,(o-formula (gvl :parent :fast-redraw-filling-style)))))))
  )

(defun Create-Stipple-Palette ()
  (create-instance 'PALETTE-ITEM opal:rectangle
    (:left (o-formula (gvl :parent :left)))
    (:top (o-formula (gvl :parent :top)))
    (:width 32)
    (:line-hold
     (create-instance NIL opal:line-style
       (:foreground-color
	(create-instance NIL opal:color (:red 0)(:blue 0)(:green 0)))
       (:line-thickness 0)))
    (:height 32)
    (:filling-style (o-formula (opal:halftone (nth (gvl :rank)
						   (gvl :parent :items))))))
					     
  (create-instance 'STIPPLE-PALETTE opal:aggregadget
    (:left 90)
    (:top (o-formula (if (g-value opal:color :color-p) 442 477)))
    (:selected NIL)
    (:items '(0 5 10 15 25 30 35 40 50 55 60 65 75 80 85 90 100))
    (:parts
     `((:background ,opal:rectangle
	(:left 87)
	(:top ,(o-formula (if (g-value opal:color :color-p) 439 474)))
	(:width 616)
	(:height ,(if (g-value opal:color :color-p) 73 38))
	(:filling-style ,opal:white-fill)
	(:line-style ,opal:line-0))
       (:patterns-agg ,opal:aggrelist
	(:left ,(o-formula (gvl :parent :left)))
	(:top ,(o-formula (gvl :parent :top)))
	(:selected NIL)
	(:items ,(o-formula (gvl :parent :items)))
	(:h-spacing 2)
	(:direction :horizontal)
	(:item-prototype ,palette-item))
       (:NIL-text ,opal:text
	(:left 674)
	(:top ,(o-formula (if (g-value opal:color :color-p) 450 485)))
	(:string "NIL"))))
    (:interactors
     `((:palette-interactor ,inter:button-interactor
	(:window ,(o-formula (gvl :operates-on :window)))
	(:start-event :any-mousedown)
	(:how-set :set)
	(:final-feedback-obj ,PALETTE-FEEDBACK)
	(:final-function ,#'Selectedfun)
	(:start-where ,(o-formula (list :element-of
					(gvl :operates-on :patterns-agg))))
	(:start-action
	 ,#'(lambda (inter obj)
	      (let ((fill (g-value PALETTE-FEEDBACK :obj-over :filling-style)))
		(s-value PALETTE-FEEDBACK :fast-redraw-filling-style fill)
		(s-value (g-value CURRENT-STATE :feedback)
			 :fast-redraw-filling-style (or fill opal:white-fill)))
	      (call-prototype-method inter obj)))))))

  (s-value (car (g-value stipple-palette :patterns-agg :components))
	   :line-hold
	   (create-instance NIL opal:line-style
	     (:foreground-color
	      (create-instance NIL opal:color
		(:red 1) (:green 1) (:blue 1)))
	     (:line-thickness 0)))

  ;; This is the NIL filling style.
  (let* ((kr::*constants-disabled* T)
	 (patterns-agg (g-value STIPPLE-PALETTE :patterns-agg))
	 (line-hold (g-value (car (g-value patterns-agg :components))
			     :line-hold)))
    (opal:add-component patterns-agg
         (create-instance 'NIL-PALETTE-ITEM opal:rectangle
	   (:left 638) (:width 32) (:height 32)
	   (:top (if (g-value opal:color :color-p) 442 477))
	   (:line-hold line-hold)
	   (:filling-style NIL))))
  )

(defun Create-Color-Palette ()
  (create-instance 'COLOR-PALETTE-ITEM opal:rectangle
    (:left (o-formula (gvl :parent :left)))
    (:top (o-formula (gvl :parent :top)))
    (:width 32)(:height 32)
    (:line-hold (o-formula
		 (car (cdr (nth (gvl :rank) (gvl :parent :items))))))
    (:filling-style (o-formula (car (nth (gvl :rank) (gvl :parent :items))))))


  (create-instance 'COLOR-PALETTE opal:aggregadget
    (:left 90) (:top 477)		 
    (:selected NIL)
    (:items (copy-list the-color-list))
    (:parts
     `((:colors-agg ,opal:aggrelist
	(:left ,(o-formula (gvl :parent :left)))
	(:top ,(o-formula (gvl :parent :top)))
	(:selected NIL)
	(:items ,(o-formula (gvl :parent :items)))
	(:h-spacing 2)
	(:direction :horizontal)
	(:item-prototype ,color-palette-item))))
    (:interactors
     `((:palette-interactor ,inter:button-interactor
	(:window ,(o-formula (gvl :operates-on :window)))
	(:start-event :any-mousedown)
	(:how-set :set)
	(:final-feedback-obj ,PALETTE-FEEDBACK)
	(:final-function ,#'Selectedfun)
	(:start-where ,(o-formula (list :element-of
					(gvl :operates-on :colors-agg))))
	(:start-action
	 ,#'(lambda (inter obj)
	      (let ((fill (g-value PALETTE-FEEDBACK :obj-over :filling-style)))
		(s-value PALETTE-FEEDBACK :fast-redraw-filling-style fill)
		(s-value (g-value CURRENT-STATE :feedback)
			 :fast-redraw-filling-style (or fill opal:white-fill)))
	      (call-prototype-method inter obj)))))))
  )

(defun Create-Fill-Palettes ()
  (Create-Palette-Feedback)       ; Creates PALETTE-FEEDBACK
  (Create-Stipple-Palette)        ; Creates STIPPLE-PALETTE
  (if (g-value opal:color :color-p)
      (Create-Color-Palette))     ; Creates COLOR-PALETTE
  
  ;; Set the initial selected values
  (s-value (g-value stipple-palette :patterns-agg) :selected
	   (nth 16 (g-value (g-value stipple-palette :patterns-agg) :components)))
  (s-value palette-feedback :obj-over
	   (nth 16 (g-value (g-value stipple-palette :patterns-agg) :components)))
  )

#|
====================================================================
 the-color-list is used to store the list of tuples, made up of
 line-styles and filling-styles.  These are created from the
 list of defined rgbvalues.  This is used to create the color
 palette on screens which can display color.  The function
 Create-Color-List takes care of the actual creation of the
 list to be stored in the-color-list.
====================================================================
|#

(defun Create-Color-List ()
  (let ((val 0) l2 l)
    (dotimes (i 18)
      (let* ((triplet (nth val rgbvalues))
	     (red (first triplet))
	     (green (second triplet))
	     (blue (third triplet)))
	(push (create-instance NIL opal:line-style
			       (:line-thickness 2)
			       (:foreground-color
				(create-instance
				 NIL opal:color
				 (:red red) (:green green) (:blue blue))))
	      l)
	(push (create-instance NIL opal:filling-style
			       (:foreground-color
				(create-instance
				 NIL opal:color
				 (:red red) (:green green) (:blue blue))))
	      l)
	(push l l2)
	(setq l NIL)
	(incf val 1)))
    (setq the-color-list l2)))


#|
====================================================================
CURRENT STATE

The Current-State menu shows the current colors selected for the
line-styles and filling-styles.
====================================================================
|#

(defun Create-Current-State ()		 
  (create-instance 'CURRENT-STATE opal:aggregadget
    (:left 2) (:top 446) (:width 52) (:height 52)
    (:parts
     `((:decoration ,opal:aggregadget
	(:left 2) (:top 446) (:width 52) (:height 52)
	(:parts
	 ((:background ,opal:rectangle
		       (:left 2) (:top 446) (:width 52) (:height 62)
		       (:filling-style ,opal:white-fill))
	  (:line-text ,opal:text
		      (:left 4) (:top 491)
		      (:string "LINE")
		      (:font ,(opal:get-standard-font
			       :sans-serif :roman :small)))
	  (:fill-text ,opal:text
		      (:left 31) (:top 491)
		      (:string "FILL")
		      (:font ,(opal:get-standard-font
			       :sans-serif :roman :small))))))
       (:selectable-objs ,opal:aggregadget
	(:left 4) (:top 442) (:width 51) (:height 40)
	(:parts
	 ((:frame ,opal:rectangle
		  (:left 4)(:top 448)(:width 24)(:height 40)
		  (:filling-style ,opal:black-fill))
	  (:filler ,opal:rectangle
		   (:left 27)(:top 448)(:width 24)(:height 40)))))
       (:feedback ,opal:aggregadget
	(:obj-over NIL)
	(:left ,(o-formula (gvl :obj-over :left)))
	(:top ,(o-formula (gvl :obj-over :top)))
	(:width ,(o-formula (gvl :obj-over :width)))
	(:height ,(o-formula (gvl :obj-over :height)))
	(:visible ,(o-formula (gvl :obj-over)))
	(:fast-redraw-filling-style NIL)
	(:parts
	 ((:black-rect ,opal:rectangle
		       (:left ,(o-formula (+ 1 (gvl :parent :left))))
		       (:top ,(o-formula (+ 1 (gvl :parent :top))))
		       (:width ,(o-formula (- (gvl :parent :width) 2)))
		       (:height ,(o-formula (- (gvl :parent :height) 2)))
		       (:line-style ,*FEEDBACK-LINE-STYLE*)
		       (:fast-redraw-p :rectangle)
		       (:fast-redraw-filling-style
			,(o-formula (gvl :parent :fast-redraw-filling-style))))
	  (:thin-white-rect ,opal:rectangle
			    (:left ,(o-formula (+ 4 (gvl :parent :left))))
			    (:top ,(o-formula (+ 4 (gvl :parent :top))))
			    (:width ,(o-formula (- (gvl :parent :width) 8)))
			    (:height ,(o-formula (- (gvl :parent :height) 8)))
			    (:line-style ,opal:white-line)
			    (:fast-redraw-p :rectangle)
			    (:fast-redraw-filling-style
			     ,(o-formula (gvl :parent :fast-redraw-filling-style)))))))))
    (:interactors
     `((:state-interactor ,inter:button-interactor
	(:window ,(o-formula (gvl :operates-on :window)))
	(:start-event :any-mousedown)
	(:final-feedback-obj ,(o-formula (gvl :operates-on :feedback)))
	(:how-set :set)
	(:start-where ,(o-formula
			(list :element-of
			      (gvl :operates-on :selectable-objs))))
	(:start-action
	 ,#'(lambda (inter obj)
	      (let* ((feedback-obj (g-value inter :operates-on :feedback))
		     (fill (g-value feedback-obj :obj-over :filling-style))) 
		(s-value feedback-obj :fast-redraw-filling-style
			 (or fill opal:white-fill)))
	      (call-prototype-method inter obj)))))))
							  

  (let* ((selectable-objs (g-value CURRENT-STATE :selectable-objs))
	 (frame (g-value selectable-objs :frame))
	 (filler (g-value selectable-objs :filler)))
    (s-value selectable-objs :selected frame)
    (s-value (g-value CURRENT-STATE :feedback) :obj-over frame)
    (s-value frame :selected t)
    (s-value frame :line-hold
	     (g-value palette-feedback :obj-over :line-hold))
    (s-value frame :filling-style
	     (g-value palette-feedback :obj-over :filling-style))
    (s-value filler :filling-style
	     (g-value (car (g-value stipple-palette :patterns-agg :components))
		      :filling-style)))
  )

(defun Create-Main-Window ()
  
  ;; This is the main window.
  (create-instance 'WIN inter:interactor-window
    (:left 500) (:top 20) (:width 750) (:height 512)
    (:position-by-hand NIL)
    (:title "Garnet Draw ver. 1.0")
    (:background-color opal:black))

  ;; This aggregate is where the tool and paint palette are stored,
  ;; and the feedback current selection of the palettes are stored
  ;; here.

  (s-value WIN :aggregate (create-instance 'TOP-AGG opal:aggregate))

  (Create-Moving-Prototypes) ; Creates MOVING-RECT, MOVING-OVAL, etc.
  (Create-Text-Feedback)     ; Creates TEXT-FEEDBACK
  (Create-Main-Menubar)      ; Creates MAIN-MENU
  (Create-Line-Palette)      ; Creates LINE-PALETTE and LINE-FEEDBACK
  (Create-Tool-Palette)      ; Creates TOOLS-MENU and TOOL-FEEDBACK
  (Create-Fill-Palettes)     ; Creates STIPPLE-PALETTE, COLOR-PALETTE, and
                             ; PALETTE-FEEDBACK
  (Create-Color-List)

  (Create-Current-State)     ; Creates CURRENT-STATE
  
  (opal:add-components TOP-AGG
		       (create-instance 'BACKGROUND-RECT opal:rectangle
			 (:left 0)
			 (:top (g-value MAIN-MENU :height))
			 (:width 750)
			 (:height (- (g-value WIN :height)
				     (g-value MAIN-MENU :height)))
			 (:line-style NIL)
			 (:filling-style opal:gray-fill))
		       TOOLS-MENU TOOL-FEEDBACK
		       CURRENT-STATE line-palette line-feedback
		       stipple-palette)

  (if (g-value opal:color :color-p)
      (opal:add-component TOP-AGG color-palette))
  (opal:add-component TOP-AGG palette-feedback)

  (opal:update win)

  ;; These functions must be called after WIN is updated!
  (opal:add-component TOP-AGG main-menu)
  (opal:notice-items-changed MAIN-MENU)

  (opal:update win)
  (setf *Grid-Menu-Item* (gg:find-submenu-component main-menu "  Options  "
						    " Turn Grid On "))
  (setf *Grid-Vis-Item* (gg:find-submenu-component main-menu "  Options  "
						    " Show Grid Dots "))
  (gg:menubar-disable-component *Grid-Vis-Item*) ; not implemented yet
  )


#|
====================================================================
MOVER GROWER GADGET

This is the gadget used to move and scale different graphical objects.
====================================================================
|#

(defun Create-MOVER-GROWER ()
  (create-instance 'MOVER-GROWER garnet-gadgets:multi-Graphics-Selection
    (:input-filter (o-formula (if (gv GRID-OBJ :gridon)
				  (gv GRID-OBJ :gridamt)
				  NIL)))
    (:start-where (list :element-of-or-none *DRAW-AGG*))
    (:start-event :rightdown)
    (:check-line T)
    (:check-polygon T)
    (:check-group T)
    (:check-grow-p T)
    (:multiple-select t)
    (:movegrow-boxes-p t)
    (:movegrow-lines-p t)
    (:value NIL)
    (:running-where t))
  
  (garnet-gadgets:set-selection MOVER-GROWER NIL)
  )

#|
====================================================================
POLYGON GADGET

Used to make polygons when the polygon tool is the one being used.
====================================================================
|#

(defun Create-Polygon-Maker ()
  (create-instance 'POLYGON-MAKER garnet-gadgets:polyline-creator
    (:input-filter (o-formula (if (gv GRID-OBJ :gridon)
				  (gv GRID-OBJ :gridamt)
				  NIL)))
    (:start-event :rightdown)
    (:start-where `(:in ,*DRAW-AGG*))
    (:close-enough-value 3)
    (:active-p (o-formula (gv (nth 5 (g-value TOOLS-MENU :components))
			      :selected)))
    (:running-where t)
    (:selection-function
     #'(lambda (gadget new-point-list)
	 (declare (ignore gadget))
	 (garnet-gadgets:set-selection MOVER-GROWER NIL)
	 (let* ((selectable-objs (g-value CURRENT-STATE :selectable-objs))
		(fill (g-value selectable-objs :filler :filling-style))
		(frame (g-value selectable-objs :frame))
		(line-hold (g-value frame :line-hold))
		(new-obj
		 (create-instance NIL opal:polyline
		   (:point-list (copy-list new-point-list))
		   (:line-p NIL)
		   (:polygon-p t)
		   (:grow-p t)
		   (:group-p NIL)
		   (:filling-style (if fill
				       (create-instance NIL fill)))
		   (:line-style
		    (create-instance NIL opal:line-style
		      (:line-thickness (g-value line-palette :selected
						:line-thick))
		      (:line-style :solid)
		      (:foreground-color
		       (create-instance NIL (g-value line-hold
						     :foreground-color))))))))
	   (with-constants-disabled
	       (opal:add-component *DRAW-AGG* new-obj))
	   (garnet-gadgets:set-selection MOVER-GROWER new-obj)))))
  )



(defun Create-Draw-Window ()
  
  ;; This is the window in which the drawings are done.
  (create-instance 'DRAW-WIN inter:interactor-window
    (:left 55)
    (:top (+ 2 (g-value MAIN-MENU :height)))
    (:width 680)
    (:height (- (g-value WIN :height) (g-value MAIN-MENU :height)
		(g-value STIPPLE-PALETTE :height) 7))
    (:border-width 2)
    (:omit-title-bar t)
    (:parent WIN))

  ;; This aggregate is where the feedback objects for the different
  ;; objects are stored.

  (s-value DRAW-WIN :aggregate
	   (create-instance 'TOP-DRAW-AGG opal:aggregate))
                          
  ;; The *DRAW-AGG* is used to actually store the actual drawings.

  (setf *DRAW-AGG* (create-instance NIL opal:aggregadget
		     (:left 0) (:top 0)
		     (:width (o-formula (gv DRAW-WIN :width)))
		     (:height (o-formula (gv DRAW-WIN :height)))))
  (setf (kr::schema-name *DRAW-AGG*) 'DRAW-AGG)

  ;; MOVING-RECT, MOVING-OVAL, etc. were created in Create-Main-Window
  ;; TEXT-FEEDBACK was created in Create-Main-Window
  (Create-MOVER-GROWER)      ; Creates MOVER-GROWER
  (Create-Polygon-Maker)     ; Creates POLYGON-MAKER

  (opal:add-components TOP-DRAW-AGG
		       *DRAW-AGG* MOVER-GROWER MOVING-LINE MOVING-RECT
		       moving-oval MOVING-ROUNDTANGLE TEXT-FEEDBACK 
		       moving-arrowline moving-doublearrowline polygon-maker)
  )


(defun Create-Interactors ()

  #|
  ====================================================================
  TWO POINT INTERACTOR

  This is the interactor used to give information necessary for drawing
  a new object.
  ====================================================================
  |#

  (create-instance 'NEW-ITEM inter:two-point-interactor
    (:window DRAW-WIN)
    (:active (o-formula (and (not (gv (nth 5 (g-value TOOLS-MENU :components))
				      :selected))
			     (not (gv (nth 4 (g-value TOOLS-MENU :components))
				      :selected)))))
    (:input-filter (o-formula (if (gv GRID-OBJ :gridon)
				  (gv GRID-OBJ :gridamt)
				  NIL)))
    (:start-event :rightdown)
    (:start-where T)
    (:final-function
     #'(lambda (an-interactor points-list)
	 (garnet-gadgets:set-selection MOVER-GROWER NIL)
	 (when points-list
	   (let* ((selectable-objs (g-value CURRENT-STATE :selectable-objs))
		  (feedback-obj (g-value an-interactor :feedback-obj))
		  (frame-color (g-value selectable-objs :frame :line-hold
					:foreground-color))
		  (filler-fill (g-value selectable-objs :filler
					:filling-style))
		  (line-thick (g-value LINE-PALETTE :selected :line-thick))
		  (obj
		   (if (g-value an-interactor :line-p)
		       (if (g-value feedback-obj :arrow-p)
			   (create-instance NIL feedback-obj
			     (:points (copy-list points-list))
			     (:filling-style (if filler-fill
						 (create-instance NIL filler-fill)))
			     (:line-style (create-instance NIL opal:line-style
					    (:line-thickness line-thick)
					    (:line-style :solid)
					    (:foreground-color
					     (create-instance NIL frame-color))))
			     (:parts
			      `((:line :modify
				 (:fast-redraw-p NIL)
				 (:draw-function :copy))
				(:arrowhead1 :modify
				 (:fast-redraw-p NIL)
				 (:draw-function :copy))
				(:arrowhead2 :modify
				 (:fast-redraw-p NIL)
				 (:draw-function :copy)))))

			   (create-instance NIL MOVING-LINE
			     (:points (copy-list points-list))
			     (:fast-redraw-p NIL)
			     (:draw-function :copy)
			     (:line-style
			      (create-instance NIL opal:line-style
				(:line-thickness line-thick)
				(:line-style :solid)
				(:foreground-color
				 (create-instance NIL frame-color))))))
		       (create-instance NIL feedback-obj
			 (:box (copy-list points-list))
			 (:draw-function :copy)
			 (:fast-redraw-p NIL)
			 (:filling-style (if filler-fill
					     (create-instance NIL filler-fill)))
			 (:line-style (create-instance NIL opal:line-style
					(:line-thickness line-thick)
					(:line-style :solid)
					(:foreground-color
					 (create-instance NIL frame-color))))))))
	     (with-constants-disabled
		 (opal:add-component *DRAW-AGG* obj))
	     (garnet-gadgets:set-selection MOVER-GROWER obj)
	     obj))))
    (:outside-action :last)
    (:arrow-p (o-formula (gv TOOLS-MENU :selected :feedback-object)))
    (:feedback-obj (o-formula (gv TOOLS-MENU :selected :feedback-object)))
    (:line-p (o-formula (gv TOOLS-MENU :selected :feedback-object :line-p)))
    (:Min-length 0)
    (:Min-height 0)
    (:Min-width 0))

  #|
  ====================================================================
  TEXT EDITING

  This is used to decide if a newly selected object is text, if so
  then we must set it up so it is editable.  This is the interactor
  used to do text editing.
  ====================================================================
  |#

  (create-instance 'CREATE-OR-EDIT inter:text-interactor
    (:feedback-obj (o-formula (if (eq :none (gvl :first-obj-over))
				  TEXT-FEEDBACK)))
    (:active (o-formula (gv (nth 4 (get-values TOOLS-MENU :components))
			    :selected)))
    (:start-where `(:element-of-or-none ,*DRAW-AGG*
		    :type ,opal:cursor-multi-text))
    (:input-filter (o-formula (if (gv GRID-OBJ :gridon)
				  (gv GRID-OBJ :gridamt)
				  NIL)))
    (:window DRAW-WIN)
    (:start-event :any-rightdown)
    (:stop-event '(:any-mousedown :control-\j))
    (:start-action #'(lambda (inter obj ev)
		       ;; make sure the font of the feedback object is correct
		       (let ((feed (g-value inter :feedback-obj)))
			 (when feed
			   (let ((current-font (g-value TOOLS-MENU :text-tool
							:text-state :font)))
			     (s-value feed :font
				      (opal:get-standard-font 
				       (g-value current-font :family)
				       (g-value current-font :face)
				       (g-value current-font :size))))
			   (s-value feed :string "")))
		       (call-prototype-method inter obj ev)))
    (:final-function
     #'(lambda (an-interactor obj-being-edited stop-event final-string x y)
	 (declare (ignore an-interactor stop-event))
	 (garnet-gadgets:set-selection MOVER-GROWER NIL)
	 (when (eq :none obj-being-edited)
	   (let* ((current-font (g-value TOOLS-MENU :text-tool
					 :text-state :font))
		  (current-color (g-value CURRENT-STATE :selectable-objs
					  :frame :line-hold
					  :foreground-color))
		  (new-str
		   (create-instance NIL opal:cursor-multi-text
		     (:box (list x y 0 0))
		     (:string (copy-seq final-string))
		     (:left (o-formula (first (gvl :box))))
		     (:top (o-formula (second (gvl :box))))
		     (:text-p t)
		     (:font (opal:get-standard-font 
			     (g-value current-font :family)
			     (g-value current-font :face)
			     (g-value current-font :size)))
		     (:line-style (create-instance NIL opal:line-style
				    (:line-thickness (g-value line-palette 
							      :selected
							      :line-thick))
				    (:line-style :solid)
				    (:foreground-color
				     (create-instance NIL current-color)))))))
	     (with-constants-disabled
		 (opal:add-component *DRAW-AGG* new-str))
	     (garnet-gadgets:set-selection MOVER-GROWER new-str)
	     )))))

  (create-instance 'DELETE inter:button-interactor
    (:continuous NIL)
    (:start-where T)
    (:start-event #\rubout)
    (:final-function
     #'(lambda (an-interactor final-obj-over)
	 (declare (ignore an-interactor final-obj-over))
	 (setf *temp-list* NIL)
	 (dolist (item (copy-list (g-value MOVER-GROWER :value)))
	   (with-constants-disabled
	       (opal:remove-component *DRAW-AGG* item))
	   (setf *temp-list* (cons item *temp-list*)))
	 (garnet-gadgets:set-selection MOVER-GROWER NIL)
	 (dolist (item (copy-list *temp-list*))
	   (opal:destroy item))
	 (setf *temp-list* NIL)))
    (:window `(,draw-win ,WIN)))
  )


(defun Do-Go (&optional dont-enter-main-event-loop)
  (Create-Color-List)
  (Create-Main-Window)
  (Create-Draw-Window)
  (Create-Query-Gadget)      ; Creates *Q-BOX*
  (Create-File-DB)           ; Creates SAVE-WIN, *SAVE-DB*
  (Create-Grid-DB)           ; Creates GRID-WIN, *GRID-DB*
  (Create-Read-DB)           ; Creates PS-Read-DB, *Read-DB*
  (Create-Interactors)

  (opal:update WIN T)
  
  (Format T "~%GARNET DRAW v1.0:

To Draw an object:
^^^^^^^^^^^^^^^^^
  1. Select the type of object to be created from the Tools-Menu
     on the left side of the screen.

  2. Press and drag the mouse button to the desired size for the
     object.  If text then just depress mouse button and release,
     to end editing type ctrl-j or click with the mouse elsewhere.
     For polygon, depress right mouse button, release and click
     again for next point, etc. till either you have added enough
     points or you don't wish to add any more points.  If you don't
     want to add any more points then depress any other button to
     stop.


To select objects:
^^^^^^^^^^^^^^^^^
  1. Use the left button to select a single item or the middle mouse
     button to select multiple objects.

  2. Depress the middle mouse button and select the region within
     which you want all objects to be selected.


To change line or filling color of a(n) object(s):
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  1. Select the object(s) to be changed, choose either fill or line
     and then select the appropriate palette pattern or color.

  2. To change the default line and filling colors, make sure no
     objects are selected.  Then select line/fill and select the
     palette item of your choice.


To change size, family, or face of text objects:
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  1. Select text object and use Font menu to change item.

  2. To change the default font, deselect all items, then choose
     appropriate menu items from Font menu.  (NOTE:  You can
     see what the default font is in the text tools slot of the
     tools menu.)~%")
  
  (unless dont-enter-main-event-loop #-cmu (inter:main-event-loop))
  )

(defun Do-Stop ()
  (opal:destroy WIN)
  
  ;; These interactors were probably not active when WIN was destroyed, so
  ;; we must destroy them explicitly.
  (if (and (boundp 'NEW-ITEM) (schema-p NEW-ITEM))
      (opal:destroy NEW-ITEM))
  (if (and (boundp 'CREATE-OR-EDIT) (schema-p CREATE-OR-EDIT))
      (opal:destroy CREATE-OR-EDIT))
  
  ;; These prototype objects were not added to a window themselves, so we
  ;; must destroy them explicitly.
  (if (and (boundp 'PALETTE-ITEM) (schema-p PALETTE-ITEM))
      (opal:destroy PALETTE-ITEM))
  (if (and (boundp 'COLOR-PALETTE-ITEM) (schema-p COLOR-PALETTE-ITEM))
      (opal:destroy COLOR-PALETTE-ITEM))
  (if (and (boundp 'MOVING-AGG) (schema-p MOVING-AGG))
      (opal:destroy MOVING-AGG)))

#|
Things to fix:

* Align-to-grid command

|#
