/*
 * Karn (MDC) encryption
 * Based on Phil Karn, sci.crypt, 13 Feb 1992
 * See also his comments from sci.crypt, 23 Mar 1992.
 * The method is a variant of that described in
 * Zheng, Matsumoto and Imai, Crypto 89.
 * See also, "A New Class of Cryptosystems Based on
 * Interconnection Networks" by
 * michaelp@terpsichore.informatic.rwth-aachen.de
 *
 * A method for turning a hash function, here MD5, into a fast
 * secret-key encryption.
 *
 * This does triple hashing with nondistinct keys.
 *
 * adapted to a Windows DLL by Andy Brown 1994
 */

#include <windows.h>   
#include "mdc.h"

/* prototypes */

static VOID mdc(LPDWORD out1,LPDWORD out2,LPDWORD in1,LPDWORD in2,
								LPDWORD key1,LPDWORD key2);
static VOID Transform(LPDWORD buf,LPDWORD in);
int FAR PASCAL LibMain(HANDLE hInstance,WORD wDataSeg,WORD wHeapSize,LPSTR lpszCmdLine);

/* F, G, H and I are basic MD5 functions */

#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

/* ROTATE_LEFT rotates x left n bits */

#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

/* FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4 */
/* Rotation is separate from addition to prevent recomputation  */

#define FF(a, b, c, d, x, s, ac) \
	{(a) += F ((b), (c), (d)) + (x) + (DWORD)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define GG(a, b, c, d, x, s, ac) \
	{(a) += G ((b), (c), (d)) + (x) + (DWORD)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define HH(a, b, c, d, x, s, ac) \
	{(a) += H ((b), (c), (d)) + (x) + (DWORD)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

#define II(a, b, c, d, x, s, ac) \
	{(a) += I ((b), (c), (d)) + (x) + (DWORD)(ac); \
   (a) = ROTATE_LEFT ((a), (s)); \
   (a) += (b); \
  }

/* Initial values for MD5 Transform hash function */
static const DWORD ihash[4] = {
	0x67452301L, 0xefcdab89L, 0x98badcfeL, 0x10325476L };


/* Basic transform for Karn encryption.  Take two 16-byte
  half-buffers, two 48-byte keys (which must be distinct), and use
  the MD5 Transform algorithm to produce two 16-byte output
  half-buffers.

  This is reversible: If we get out1 and out2 from in1, in2, key1, key2,
  then we can get in2 and in1 from out2, out1, key1, key2.

  in1, in2, out1, and out2 should point to 16-byte buffers.
  By convention, in1 and in2 are two halves of a 32-byte input
  buffer, and out1 and out2 are two halves of a 32-byte output
  buffer.

  key1 and key2 should point to 48-byte buffers with different contents.
 */

VOID mdc(LPDWORD out1,LPDWORD out2,LPDWORD in1,LPDWORD in2,
				 LPDWORD key1,LPDWORD key2)
{
int i;
DWORD buf[16];
DWORD hash[4];
DWORD temp[4];

  hmemcpy(hash,ihash,sizeof(hash));
  hmemcpy(buf,in1,16);
  hmemcpy(buf+4,key1,48);
  Transform(hash,buf);

  for (i=0;i<4;++i)
    temp[i]=buf[i]=in2[i]^hash[i];

  hmemcpy(hash,ihash,sizeof(hash));
  hmemcpy(buf+4,key2,48);
  Transform(hash,buf);

  for(i=0;i<4;++i)
    out2[i]=buf[i]=in1[i]^hash[i];

  hmemcpy(hash,ihash,sizeof(hash));
  hmemcpy(buf+4,key1,48);
  Transform(hash,buf);

  for (i=0;i<4;++i)
    out1[i]=temp[i]^hash[i];
}
   
/**********************************************/
/* Basic MD5 step. Transforms buf based on in */ 
/**********************************************/

static VOID Transform (LPDWORD buf,LPDWORD in)
{
DWORD a = buf[0], b = buf[1], c = buf[2], d = buf[3];

  /* Round 1 */
#define S11 7
#define S12 12
#define S13 17
#define S14 22
  FF ( a, b, c, d, in[ 0], S11, 3614090360); /* 1 */
  FF ( d, a, b, c, in[ 1], S12, 3905402710); /* 2 */
  FF ( c, d, a, b, in[ 2], S13,  606105819); /* 3 */
  FF ( b, c, d, a, in[ 3], S14, 3250441966); /* 4 */
  FF ( a, b, c, d, in[ 4], S11, 4118548399); /* 5 */
  FF ( d, a, b, c, in[ 5], S12, 1200080426); /* 6 */
  FF ( c, d, a, b, in[ 6], S13, 2821735955); /* 7 */
  FF ( b, c, d, a, in[ 7], S14, 4249261313); /* 8 */
  FF ( a, b, c, d, in[ 8], S11, 1770035416); /* 9 */
  FF ( d, a, b, c, in[ 9], S12, 2336552879); /* 10 */
  FF ( c, d, a, b, in[10], S13, 4294925233); /* 11 */
  FF ( b, c, d, a, in[11], S14, 2304563134); /* 12 */
  FF ( a, b, c, d, in[12], S11, 1804603682); /* 13 */
  FF ( d, a, b, c, in[13], S12, 4254626195); /* 14 */
  FF ( c, d, a, b, in[14], S13, 2792965006); /* 15 */
  FF ( b, c, d, a, in[15], S14, 1236535329); /* 16 */

  /* Round 2 */
#define S21 5
#define S22 9
#define S23 14
#define S24 20
  GG ( a, b, c, d, in[ 1], S21, 4129170786); /* 17 */
  GG ( d, a, b, c, in[ 6], S22, 3225465664); /* 18 */
  GG ( c, d, a, b, in[11], S23, 643717713); /* 19 */
  GG ( b, c, d, a, in[ 0], S24, 3921069994); /* 20 */
  GG ( a, b, c, d, in[ 5], S21, 3593408605); /* 21 */
  GG ( d, a, b, c, in[10], S22,  38016083); /* 22 */
  GG ( c, d, a, b, in[15], S23, 3634488961); /* 23 */
  GG ( b, c, d, a, in[ 4], S24, 3889429448); /* 24 */
  GG ( a, b, c, d, in[ 9], S21, 568446438); /* 25 */
  GG ( d, a, b, c, in[14], S22, 3275163606); /* 26 */
  GG ( c, d, a, b, in[ 3], S23, 4107603335); /* 27 */
  GG ( b, c, d, a, in[ 8], S24, 1163531501); /* 28 */
  GG ( a, b, c, d, in[13], S21, 2850285829); /* 29 */
  GG ( d, a, b, c, in[ 2], S22, 4243563512); /* 30 */
  GG ( c, d, a, b, in[ 7], S23, 1735328473); /* 31 */
  GG ( b, c, d, a, in[12], S24, 2368359562); /* 32 */

  /* Round 3 */
#define S31 4
#define S32 11
#define S33 16
#define S34 23
  HH ( a, b, c, d, in[ 5], S31, 4294588738); /* 33 */
  HH ( d, a, b, c, in[ 8], S32, 2272392833); /* 34 */
  HH ( c, d, a, b, in[11], S33, 1839030562); /* 35 */
  HH ( b, c, d, a, in[14], S34, 4259657740); /* 36 */
  HH ( a, b, c, d, in[ 1], S31, 2763975236); /* 37 */
  HH ( d, a, b, c, in[ 4], S32, 1272893353); /* 38 */
  HH ( c, d, a, b, in[ 7], S33, 4139469664); /* 39 */
  HH ( b, c, d, a, in[10], S34, 3200236656); /* 40 */
  HH ( a, b, c, d, in[13], S31, 681279174); /* 41 */
  HH ( d, a, b, c, in[ 0], S32, 3936430074); /* 42 */
  HH ( c, d, a, b, in[ 3], S33, 3572445317); /* 43 */
  HH ( b, c, d, a, in[ 6], S34,  76029189); /* 44 */
  HH ( a, b, c, d, in[ 9], S31, 3654602809); /* 45 */
  HH ( d, a, b, c, in[12], S32, 3873151461); /* 46 */
  HH ( c, d, a, b, in[15], S33,  530742520); /* 47 */
  HH ( b, c, d, a, in[ 2], S34, 3299628645); /* 48 */

  /* Round 4 */
#define S41 6
#define S42 10
#define S43 15
#define S44 21
  II ( a, b, c, d, in[ 0], S41, 4096336452); /* 49 */
  II ( d, a, b, c, in[ 7], S42, 1126891415); /* 50 */
  II ( c, d, a, b, in[14], S43, 2878612391); /* 51 */
  II ( b, c, d, a, in[ 5], S44, 4237533241); /* 52 */
  II ( a, b, c, d, in[12], S41, 1700485571); /* 53 */
  II ( d, a, b, c, in[ 3], S42, 2399980690); /* 54 */
  II ( c, d, a, b, in[10], S43, 4293915773); /* 55 */
  II ( b, c, d, a, in[ 1], S44, 2240044497); /* 56 */
  II ( a, b, c, d, in[ 8], S41, 1873313359); /* 57 */
  II ( d, a, b, c, in[15], S42, 4264355552); /* 58 */
  II ( c, d, a, b, in[ 6], S43, 2734768916); /* 59 */
  II ( b, c, d, a, in[13], S44, 1309151649); /* 60 */
  II ( a, b, c, d, in[ 4], S41, 4149444226); /* 61 */
  II ( d, a, b, c, in[11], S42, 3174756917); /* 62 */
  II ( c, d, a, b, in[ 2], S43,  718787259); /* 63 */
  II ( b, c, d, a, in[ 9], S44, 3951481745); /* 64 */

  buf[0] += a;
  buf[1] += b;
  buf[2] += c;
  buf[3] += d;
}
                                                

/**********************************************/
/* Encrypt a 256 bit block with a 768 bit key */
/**********************************************

in and out need to be distinct */

VOID FAR PASCAL _export mdc_encrypt(LPBYTE in,LPBYTE out,LPBYTE key)
{
	mdc((LPDWORD)out,(LPDWORD)&out[16],(LPDWORD)in,(LPDWORD)&in[16],
			(LPDWORD)key,(LPDWORD)&key[48]);
}


/**********************************************/
/* Decrypt a 256 bit block with a 768 bit key */
/**********************************************/

VOID FAR PASCAL _export mdc_decrypt(LPBYTE in,LPBYTE out,LPBYTE key)
{                                            
	mdc((LPDWORD)&out[16],(LPDWORD)out,(LPDWORD)&in[16],(LPDWORD)in,
			(LPDWORD)key,(LPDWORD)&key[48]);
}


/******************************/
/* LibMain DLL initialisation */
/******************************/

int FAR PASCAL LibMain(HANDLE hInstance,WORD wDataSeg,WORD wHeapSize,LPSTR lpszCmdLine)
{
	if(wHeapSize>0) UnlockData(0);
	return 1;
}
