/* Data Envelopment Analysis (DEA)
 *
 * DEA quantifies the relative efficiency of decision making units (DMUs) by
 * finding the efficient frontier in multiple input multiple output data.  The
 * inputs are resources (eg. number of employees, available machines, ...),
 * the outputs are productive outputs (eg. contracts made, total sales, ...).
 * The method is non-parametric.  More details are available in the paper
 * below.
 *
 * Models according to: Seiford, Threall, "Recent developments in DEA", 1990.
 *
 * Implementation: Sebastian Nowozin <nowozin@gmail.com>
 */

### SETS ###

set dmus;       # Decision Making Units (DMU)
set inputs;     # Input parameters
set outputs;    # Output parameters


### PARAMETERS ###

param input_data{dmus,inputs} >= 0;
param output_data{dmus,outputs} >= 0;


### PROGRAM ###

var theta{dmus} >= 0;
var lambda{dmus,dmus} >= 0;

minimize inefficiency: sum{td in dmus} theta[td];

s.t. output_lower_limit{o in outputs, td in dmus}:
    sum{d in dmus} lambda[d,td]*output_data[d,o] >= output_data[td,o];
s.t. input_upper_limit{i in inputs, td in dmus}:
    sum{d in dmus} lambda[d,td]*input_data[d,i] <= theta[td]*input_data[td,i];

    s.t. PI1{td in dmus}:
        sum{d in dmus} lambda[d,td] = 1;
/*
possibilities:
      i) (no constraint)
     ii) s.t. PI1{td in dmus}:
        sum{d in dmus} lambda[d,td] <= 1;
    iii) s.t. PI1{td in dmus}:
        sum{d in dmus} lambda[d,td] >= 1;
*/


### SOLVE AND PRINT SOLUTION ###

solve;

printf "DMU\tEfficiency\n";
for {td in dmus} {
    printf "%s\t%1.4f\n", td, theta[td];
}

### DATA ###

data;

set dmus := 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20
    21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40
    41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60
    61 62 63 64 65 66 67 68 69 ;
set inputs := AvgInventory LaborCost OperatingCost Population ;
set outputs := PrescrVol kDollarValue ;

param input_data default 0.0 :

        AvgInventory LaborCost OperatingCost Population :=

1 8000 17030 1280 1410
2 9000 25890 2779 1523
3 13694 29076 2372 1354
4 4250 17506 1385 822
5 6500 23208 639 746
6 7000 12946 802 1281
7 4500 18001 1130 1016
8 5000 14473 1097 1070
9 27000 31760 5559 1694
10 21560 50972 15010 1910
11 15000 39523 4799 1745
12 8500 13076 3489 1353
13 35000 35427 1704 500
14 18000 27554 2882 1016
15 59750 53848 14208 2500
16 19200 38253 1480 2293
17 40000 109404 83016 2718
18 8466 18198 1278 2877
19 16000 40891 7599 4150
20 10000 45444 5556 4421
21 25000 35623 2121 3883
22 14000 20192 5515 3519
23 12500 34973 10475 32366
24 17260 32284 14498 3393
25 7000 17920 7585 4489
26 14000 42094 3742 2217
27 16400 35422 14236 4641
28 13000 19100 3529 5968
29 30000 72167 8656 8715
30 12530 19970 1714 5968
31 31500 39183 4919 5607
32 10000 32048 3483 7324
33 22000 68877 12279 8685
34 10000 29812 3332 8685
35 16000 47686 2507 5420
36 10000 33415 4738 7703
37 9000 12359 4603 4665
38 16439 23614 2989 6317
39 14500 36069 1793 31839
40 39000 76307 9539 15619
41 24927 40706 12661 30213
42 13858 39267 4609 34719
43 33375 29509 11323 31839
44 29044 44482 5542 34719
45 32257 61365 20550 32366
46 8800 49671 3306 43561
47 47000 40425 10396 31263
48 12000 33034 4915 31263
49 28000 69163 4688 15173
50 13300 28931 16735 73064
51 13500 29758 4260 62309
52 24000 40927 8285 23166
53 16000 40403 2131 99836
54 17000 38730 2539 60348
55 25000 35978 2502 99836
56 16000 37509 6278 99836
57 20000 46950 10715 85925
58 14000 35966 3144 85925
59 22000 68318 8015 108987
60 21879 69537 7778 108987
61 15000 25425 2812 201404
62 10000 19508 2454 201404
63 20000 28191 3367 201404
64 18000 37073 8624 108987
65 19051 23763 3496 201404
66 15000 28642 3366 201404
67 10000 35919 3868 201404
68 24000 54653 26494 108987
69 1800 6276 3413 60348
        ;

param output_data default 0.0 :

        PrescrVol kDollarValue :=

1 12293 61.00
2 18400 92.00
3 16789 92.65
4 10700 45.00
5 9800 50.00
6 6500 29.00
7 8200 56.00
8 8680 45.00
9 33800 183.00
10 23710 156.00
11 24000 120.00
12 17500 75.00
13 25000 130.00
14 26000 122.00
15 26830 178.513
16 16600 106.00
17 90000 450.00
18 11140 73.624
19 25868 136.00
20 32700 191.295
21 29117 152.864
22 18000 100.00
23 11100 60.00
24 23030 137.778
25 10656 58.00
26 24682 152.095
27 26908 120.00
28 16464 80.00
29 57000 321.00
30 17532 94.747
31 30035 168.00
32 16000 100.00
33 63700 277.00
34 18000 90.00
35 27339 139.134
36 19500 116.00
37 13000 80.00
38 15370 102.00
39 18446 90.00
40 56000 260.00
41 73845 364.951
42 28600 145.00
43 27000 243.00
44 52423 279.816
45 73759 363.388
46 20500 80.00
47 27100 115.00
48 15000 110.00
49 50895 277.852
50 19707 128.00
51 17994 78.80
52 36135 167.222
53 30000 153.00
54 26195 125.00
55 28000 216.00
56 24658 152.551
57 36850 190.00
58 29250 183.69
59 50000 250.00
60 40078 265.443
61 20200 110.00
62 12500 75.00
63 30890 195.00
64 31000 175.00
65 31277 192.992
66 11500 75.00
67 30000 175.668
68 38383 190.00
69 2075 8.650
        ;

end;
