

/*

________________________________________________________________

        sunraster2biff.c
        $Id: sunraster2biff.c,v 1.32 1997/01/24 15:17:33 svein Exp $
        Copyright 1990, Blab, UiO
        Image processing lab, Department of Informatics
        University of Oslo
        E-mail: blab@ifi.uio.no
________________________________________________________________
  
  Permission to use, copy, modify and distribute this software and its
  documentation for any purpose and without fee is hereby granted, 
  provided that this copyright notice appear in all copies and that 
  both that copyright notice and this permission notice appear in supporting
  documentation and that the name of B-lab, Department of Informatics or
  University of Oslo not be used in advertising or publicity pertaining 
  to distribution of the software without specific, written prior permission.

  B-LAB DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL B-LAB
  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
  OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 

*/

#ifndef lint

static char *Id = "$Id: sunraster2biff.c,v 1.32 1997/01/24 15:17:33 svein Exp $, Blab, UiO";

#endif



#include <xite/includes.h>
#include <xite/biff.h>
#include <xite/sunraster.h>
#include XITE_STDIO_H
#include XITE_UNISTD_H
#include <xite/color.h>
#include <xite/message.h>
#include <xite/readarg.h>
#include <xite/utils.h>
#include XITE_TYPES_H
#include XITE_FILE_H
#include XITE_FCNTL_H
#include XITE_MALLOC_H
#include "rast.h"


#define ERR_BIFF 1
#define ERR_MALLOC 2
#define ERR_OPEN 2
#define ERR_READ 3
#define ERR_WRITE 4
#define ERR_MAGIC 5
#define ERR_STD 6

char *sunraster2biff_error[] =
{
  "Ok",
  "Cannot open BIFF-file",
  "Sunraster file open error",
  "Read error on sunraster file",
  "Write error on BIFF file",
  "Bad magic number",
  "Not standard format",
  "Biff file must have 1 band",
};




/*L*

________________________________________________________________

		uncompress
________________________________________________________________

Name:		uncompress
Syntax:		
Description:
Return value:
Author:		Otto Milvang
________________________________________________________________

*/

#ifndef FUNCPROTO
static void uncompress(band, bit_data, depth)
IBAND band;
unsigned char *bit_data;
int depth;
#else /* FUNCPROTO */
static void uncompress(IBAND band, unsigned char *bit_data, int depth)
#endif /* FUNCPROTO */
{
  unsigned char *byte_data;
  unsigned short *bit;
  register int count, w;
  register unsigned short pix;
  int xsize, ysize, y;

  xsize = Ixsize(band);
  ysize = Iysize(band);
  bit = (unsigned short *) bit_data;
  for(y=1; y<= ysize; y++)
    {
      byte_data = &band[y][1];
      w = xsize;
      if (depth == 1)
	{
	  count = 16;
	  pix = *bit++;
	  while (w--)
	    { 
	      *byte_data++ =  (pix & 0x8000) ? 0xff : 0;
	      pix = pix + pix;
	      if ((--count == 0) && w)
		{
		  count = 16; 
		  pix = *bit++;
		}
	    }
	} else {
	  while (w--) *byte_data++ = *bit_data++;
	}
    }
}

#ifndef FUNCPROTO
static int _Iread_(fd, ptr, size)
int fd, size;
unsigned char *ptr;
#else /* FUNCPROTO */
static int _Iread_(int fd, unsigned char *ptr, int size)
#endif /* FUNCPROTO */
{
  int blokk, total = 0;
  for(;;)
  {
    blokk = read(fd, ptr, size);
    if (blokk <= 0) break;
    ptr = (unsigned char*) (ptr+blokk);
    size -= blokk;
    total += blokk;
    if(size == 0) break;
    }
  return(total);
}



/*F:read_sunraster*

________________________________________________________________

		read_sunraster
________________________________________________________________

Name:		read_sunraster - read sunraster file to BIFF image

Syntax:         | #include <xite/sunraster.h>
                | #include <xite/color.h>
		| 
                | int read_sunraster( IMAGE* img, Color_tab tab,
                |    char* filename, int* depth );

Description:	Read a sunraster-file into a BIFF-image (and colortab).
                'filename' is the name of a sunrasterfile, 'img' will
		return a pointer to a new BIFF image and 'tab' will
	        return the colormap. 'tab' may be NULL.
		'depth' will return the depth of the raster image.

Restriction:	Accept only 'standard' sun raster.

Return value:	| 0 - Ok
                | 1 - BIFF - error
		| 2 - Cannot open Sunrasterfile
		| 3 - Read error
		| 4 - Write error
		| 5 - Bad magic number (not a sunrasterfile)
		| 6 - Bad format (Not standard format)

Author:		Otto Milvang
________________________________________________________________

*/

#ifndef FUNCPROTO
int  read_sunraster(img, tab, filename, depth)
IMAGE *img;
Color_tab tab;
char *filename;
int *depth;
#else /* FUNCPROTO */
int  read_sunraster(IMAGE *img, Color_cell *tab, char *filename, int *depth)
#endif /* FUNCPROTO */
{
  struct rasterfile ras;
  int fd;
  int len, i;
  char *col, *bit;
  IBAND b;

  col = NULL; 
  fd = open(tilde_expand(filename), O_RDONLY | FNDELAY, 0);
  if (fd == -1) return(ERR_OPEN);
 
  len = _Iread_(fd, (unsigned char *) &ras, sizeof(struct rasterfile));
  if (len != sizeof(struct rasterfile)) return(ERR_READ);
  if (ras.ras_magic != RAS_MAGIC) return(ERR_MAGIC);
  if (ras.ras_type != RT_STANDARD) return(ERR_STD);

  if (ras.ras_maplength)
    {
      col = malloc( (unsigned) ras.ras_maplength);
      if (col == NULL) return(ERR_MALLOC);
      len  = _Iread_(fd, (unsigned char *) col, ras.ras_maplength);
      if (len != ras.ras_maplength) return(ERR_READ);
    }

  if (ras.ras_length)
    {
      bit =  malloc( (unsigned) ras.ras_length);
      if (bit == NULL) return(ERR_MALLOC);
      len  = _Iread_(fd, (unsigned char *) bit, ras.ras_length);
      if (len != ras.ras_length) return(ERR_READ);
    }
  (void) close(fd);

  *img = Imake_image(1, "Converted from Sun Raster", 3, 
		    ras.ras_width, ras.ras_height );
  if (*img == NULL) return(ERR_BIFF);
  b = (*img)[1];
  
  *depth = ras.ras_depth;
  uncompress(b, (unsigned char *) bit, *depth);  
  if (tab && (*depth == 8))
    for(i=0; i< 256; i++)
      {
	tab[i].pixel = RgbPixel(i);
	tab[i].red   = col ? (col[i ] << 8   ) : (i << 8);
	tab[i].green = col ? (col[i+256] << 8) : (i << 8);
	tab[i].blue  = col ? (col[i+512] << 8) : (i << 8);
      }
  return(0);
}
   



#ifdef MAIN

/*P:sunraster2biff*

________________________________________________________________

		sunraster2biff
________________________________________________________________

Name:		sunraster2biff - convert sunraster-file to BIFF-file

Syntax:		sunraster2biff <sunrasterfile> <BIFF-file> [<BIFF colorfile>]

Description:	Convert a 'sunrasterfile' to a 'BIFF-file'.
                If raster-depth=8 and 'BIFF colorfile' is specified,
		a colorfile is written.

Restrictions:	Accept onlys standard sunraster-files.

Return value:	| 0 - Ok
                | 1 - BIFF - error
		| 2 - Cannot open Sunrasterfile
		| 3 - Read error
		| 4 - Write error
		| 5 - Bad magic number (not a sunrasterfile)
		| 6 - Bad format (Not standard format)
		| 7 - Illegal command line

Author:		Otto Milvang

Examples:	| sunraster2biff sun-raster8/andy.im8 andy.img andy.col
                | sunraster2biff sun-raster1/andy.im1 andy.img

Id: 		$Id: sunraster2biff.c,v 1.32 1997/01/24 15:17:33 svein Exp $
________________________________________________________________

*/


#ifndef FUNCPROTO
int main(argc,argv)
int argc;
char *argv[];
#else /* FUNCPROTO */
int main(int argc, char **argv)
#endif /* FUNCPROTO */
{
  IMAGE img;
  Color_tab tab;   /* This is non-NULL (it refers to the first element of a
		    * a vector). */
  int stat, depth; 

  InitMessage(&argc, argv, xite_app_std_usage_text(
   "Usage: %s <sunraster-file> <BIFF-file> [<BIFF colorfile>]\n"));

  if (argc == 1) Usage(1, NULL);
  if (argc != 3 && argc != 4) Usage(7, "Illegal number of arguments.\n");

  stat = read_sunraster(&img, tab, argv[1], &depth);
  if (stat) Error(stat, "%s\n", sunraster2biff_error[stat]);

  if ((argc == 4) && (depth == 8))
    {
      color_write(argv[3], tab, "Converted from Sun Raster", 256);
    }

  stat = Iwrite_image(img, argv[2]);
  if(stat) Error(stat, "%s\n", sunraster2biff_error[ERR_BIFF]);

  return(0);
}

#endif
