/*
 * sipp - SImple Polygon Processor
 *
 *  A general 3d graphic package
 *
 *  Copyright Jonas Yngvesson  (jonas-y@isy.liu.se) 1988/89/90/91
 *            Inge Wallin      (ingwa@isy.liu.se)         1990/91
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You can receive a copy of the GNU General Public License from the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * See the README for more information.
 *
 * Revision history:
 *
 * 910527  Ver. 2.0.2. A major bug in traverse_object_tree() fixed.
 *         The invers transformation of the eyepoint was not correct, 
 *         causing polygons erroneously to be marked as backfacing.
 *
 * 910412  Ver. 2.0.1. Some minor bugfixes. Division by zero in
 *         create_edges() if an edge was horizontal in both x and y.
 *         Free-list became corrupted since the y_bucket was freed
 *         twice. Wrong parameters to (D)noise() in bumpy.c. Two lines
 *         interchanged in mat_rotate(). A few other minor changes.
 *
 * 901219  At last! 2.0 is out of the bag. And right in time for X-mas :-)
 *         *Major* rewrite. A new level in the object hierarchy introduced.
 *         An object is now a collection of surfaces and other objects
 *         (subobjects). This allows creation of complex composite objects.
 *         Objects in the hierarchy have coupled transformations.
 *         The old "Object" is now, more appropriately, called Surface.
 *         A fatal bug in the viewing transformation fixed.
 *         Objects and surfaces have internal reference counters so deletion
 *         won't leave dangling references in an hierachy.
 *         The code is now almost readable.
 *         Images format changed to PPM since much more applications
 *         exists that can handle that format.
 *         The shader package is extended and a package with functions
 *         to create geometric primitives as objects is provided.
 *
 * 901030  More general shader interface. The simple internal shader
 *         was moved out of sipp into its own file. Several shaders
 *         provided together with it in a "shader package".
 *
 * 900712  Major code beautifying, quite a lot left to do though...
 *         Typedefs instead of raw structures,
 *         much better typenames, lots of comments added (probably
 *         not enough though). Split into a few header files and
 *         a source file. More portable interface.
 *
 * 891124  Made the object representation independent of the
 *         illumination model used. A user can now supply his
 *         own surface description and shading function.
 *         Added support for texture mapping (both solid and 2d).
 *         No texture mapping is built into the internal shader though.
 *
 * 891120  Added colour. New illumination model. Simple antialiasing
 *         with double oversampling and a box filter.
 *
 * 891028  Converted from sippuz to sipp. Z-buffer changed 
 *         to scan-line z-buffer.
 *         Phong-shading.
 *         The stack notion introduced.
 *         Changed fixed viewpoint to a user defined one.
 *         World coordinates and picture resolution can be chosen
 *         freely.
 *
 * 881128  Converted from Pascal to C. Gouraud shading. Redesign of the
 *         data structures.
 *
 * 88????  Original program "sippuz - simple polygon processing using a
 *         z-buffer", written in Hedrick Pascal on a DEC-20 system
 *         running TOPS-20. Greyscales only. Flat shading.
 */


#include <stdio.h>
#include <math.h>
#include <malloc.h>
#ifndef NOMEMCPY
#include <memory.h>
#endif
#include <xalloca.h>

#include <sipp.h>
#include <sipptypes.h>
#include <geometric.h>


#define VERSION "2.0.2"

#define ZCLIPF 100.0        /* Magic number used when defining hither & yon */



/*
 * Global variables.
 */
static Vertex       *vertex_tree;     /* Vertex tree for current object. */
static Vertex_ref   *vertex_stack;    /* Vertex stack for current polygon. */
static Vertex_ref   *vstack_bottom;   /* Last entry in vertex stack. */
static Polygon      *poly_stack;      /* Polygon stack for current object. */
static Inst_object  *object_db;       /* Object database. */
static Lightsource  *lightsrc_stack;  /* Lightsource list. */
static Edge        **y_bucket;        /* Y-bucket for edge lists. */
static double        dist_limit;      /* Minimal distance between two      */
                                      /* vertices without them being       */
                                      /* considered to be the same vertex. */
static int           first_vertex;    /* Used when determining if we are   */
                                      /* installing the first vertex in an */
                                      /* object. *Not* a boolean!          */

/*
 * Stack of transformation matrices used
 * when traversing an object hierarchy.
 */
static struct tm_stack_t {
    Transf_mat         mat;
    struct tm_stack_t *next;
} *tm_stack;

static Transf_mat      curr_mat;     /* Current transformation matrix */

static Transf_mat      ident_matrix = {{        /* Unit tranfs. matrix */
                           { 1.0, 0.0, 0.0 },
                           { 0.0, 1.0, 0.0 },
                           { 0.0, 0.0, 1.0 },
                           { 0.0, 0.0, 0.0 }
                       }};


/*
 * Viewpoint definition
 */
static struct {
    double x0, y0, z0;    /* viewpoint position */
    double x, y, z;       /* point to look at */
    Vector vec;           /* vector from point to eye, used in shading calc. */
    Vector up;            /* Up direction in the view */ 
    double focal_ratio;
} camera;



/*======================== "Safe" malloc ================================*/
static char *
smalloc(size)
    int size;
{
    char *p;

    p = (char *)malloc(size);
    if (p == NULL) {
        fprintf(stderr, "Out of virtual memory.\n");
        exit(1);
    }

    return p;
}



/*================ Functions that handles lightsources ==================*/

/*
 * Define a new lightsource in the scene.
 */
void
lightsource_push(x, y, z, intensity)
    double  x, y, z, intensity;
{
    double       norm;
    Lightsource *lp;

    norm = sqrt(x * x + y * y + z * z);
    lp = (Lightsource *)smalloc(sizeof(Lightsource));
    lp->dir.x = x / norm;
    lp->dir.y = y / norm;
    lp->dir.z = z / norm;
    lp->intensity = intensity;
    lp->next = lightsrc_stack;
    lightsrc_stack = lp;
}



/*================ Functions that handles the viewpoint ==================*/

/*
 * Calculate the vector from the point of interest
 * to the viewpoint. The shaders needs this vector normalized
 * while the view coordinate transformation needs it
 * non normalized, therefore we need this routine to
 * recalculate the non normalized value.
 */
static void
view_vec_eval()
{
    MakeVector(camera.vec, 
               camera.x0 - camera.x, 
               camera.y0 - camera.y, 
               camera.z0 - camera.z);
}


    
/*
 * Define the viewpoint.
 */
void
view_from(x0, y0, z0)
    double x0, y0, z0;
{
    camera.x0 = x0;
    camera.y0 = y0;
    camera.z0 = z0;
    view_vec_eval();
}


/*
 * Define the point that we are looking at.
 */
void
view_at(x, y, z)
    double x, y, z;
{
    camera.x = x;
    camera.y = y;
    camera.z = z;
    view_vec_eval();
}


/*
 * Define the "up" direction of the view (well, rather the y-z plane).
 */
void
view_up(x, y, z)
    double x, y, z;
{
    MakeVector(camera.up, x, y, z);
}


/*
 * Set the focal ratio for the "camera".
 */
void
view_focal(ratio)
    double ratio;
{
    camera.focal_ratio = ratio;
}


/*
 * Set all viewpoint parameters in one call.
 */
void
viewpoint(x0, y0, z0, x, y, z, ux, uy, uz, ratio)
    double x0, y0, z0, x, y, z, ux, uy, uz, ratio;
{
    camera.x0 = x0;
    camera.y0 = y0;
    camera.z0 = z0;
    camera.x = x;
    camera.y = y;
    camera.z = z;
    MakeVector(camera.up, ux, uy, uz);
    camera.focal_ratio = ratio;
    view_vec_eval();
}



/*============= Functions that handles the object database ================*/

/*
 * Search for a vertex in a vertex tree. Vertices are asumed
 * to be equal if they differ less than dist_limit in all directions.
 *
 * If the vertex is not found, install it in the tree.
 */
static Vertex *
vertex_lookup(x, y, z, u, v, w, p)
    double   x, y, z, u, v, w;
    Vertex **p;
{
    double  xdist, ydist, zdist;

    if (*p == NULL) {
        *p = (Vertex *)smalloc(sizeof(Vertex));
        (*p)->x = x;
        (*p)->y = y;
        (*p)->z = z;
        (*p)->a = 0;
        (*p)->b = 0;
        (*p)->c = 0;
        (*p)->u = u;
        (*p)->v = v;
        (*p)->w = w;
        (*p)->big = NULL;
        (*p)->sml = NULL;
        return *p;
    } else if ((xdist = x - ((*p)->x)) > dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->big)));
    } else if (xdist < -dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->sml)));
    } else if ((ydist = y - ((*p)->y)) > dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->big)));
    } else if (ydist < -dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->sml)));
    } else if ((zdist = z - ((*p)->z)) > dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->big)));
    } else if (zdist < -dist_limit) {
        return (vertex_lookup(x, y, z, u, v, w, &((*p)->sml)));
    } else {
        return *p;
    }
}


/*
 * Push a vertex on the vertex stack (without texture coordinates).
 */
void
vertex_push(x, y, z)
    double  x, y, z;
{
    vertex_tx_push(x, y, z, (double)0.0, (double)0.0, (double)0.0);
}


/*
 * Push a vertex on the vertex stack (with texture coordinates).
 */
void
vertex_tx_push(x, y, z, u, v, w)
    double  x, y, z, u, v, w;
{
    Vertex_ref *vref;

   /* 
    * To get a reasonable dist_limit we use the following "heuristic" 
    * value:
    * The distance between the first two vertices installed in
    * the scene, multiplied by the magic number 1e-10, unless
    * they are the same vertex. In that case 1e-10 is used until
    * we get a vertex that differs from the first.
    */
    if (!first_vertex)
        first_vertex++;
    else if (first_vertex == 1) {
        dist_limit = sqrt((x - vertex_tree->x) * (x - vertex_tree->x)
                        + (y - vertex_tree->y) * (y - vertex_tree->y)
                        + (z - vertex_tree->z) * (z - vertex_tree->z))
                   * 1e-10;                      /* Magic!!! */
        if (dist_limit != 0.0)
            first_vertex++;
        else
            dist_limit = 1e-10;                  /* More Magic */
    }
    vref = (Vertex_ref *)smalloc(sizeof(Vertex_ref));
    if (vertex_stack == NULL) {
        vertex_stack = vref;
    } else {
        vstack_bottom->next = vref;
    }
    vstack_bottom = vref;
    vref->vertex = vertex_lookup(x, y, z, u, v, w, &vertex_tree);
    vref->next = NULL;
}


/*
 * Push a polygon on the polygon stack. Empty the vertex stack afterwards.
 */
void
polygon_push()
{
    Polygon *polyref;

    if (vertex_stack != NULL) {
        polyref = (Polygon *)smalloc(sizeof(Polygon));
        polyref->vertices = vertex_stack;
        polyref->backface = 0;
        polyref->next = poly_stack;
        poly_stack = polyref;
        vertex_stack = NULL;
    }
}


/*
 * Create a surface of all polygons in the polygon stack.
 * Empty the polygon stack afterwards.
 */
Surface *
surface_create(surf_desc, shader)
    void   *surf_desc;
    Shader *shader;
{
    Surface *surfref;
    
    if (poly_stack != NULL) {
        surfref = (Surface *)smalloc(sizeof(Surface));
        surfref->vertices = vertex_tree;
        surfref->polygons = poly_stack;
        surfref->surface = surf_desc;
        surfref->shader = shader;
        surfref->ref_count = 0;
        surfref->next = NULL;
        vertex_tree = NULL;
        poly_stack = NULL;
        first_vertex = 0;
        return surfref;
    } else
        return NULL;
}


/*
 * Create a surface to be shaded with the simple shader.
 */
Surface *
surface_basic_create(ambient, red, grn, blu, specular, c3)
    double ambient, red, grn, blu, specular, c3;
{
    Surf_desc *surf_desc;

    surf_desc = (Surf_desc *)smalloc(sizeof(Surf_desc));
    surf_desc->ambient = ambient;
    surf_desc->color.red = red;
    surf_desc->color.grn = grn;
    surf_desc->color.blu = blu;
    surf_desc->specular = specular;
    surf_desc->c3 = c3;
    return surface_create(surf_desc, basic_shader);
}


/*
 * Set SURFACE to be shaded with the shading function SHADER
 * using the surface description SURF_DESC.
 */
void
surface_set_shader(surface, surf_desc, shader)
    Surface *surface;
    void    *surf_desc;
    Shader  *shader;
{
    
    if (surface != NULL) {
        surface->surface = surf_desc;
        surface->shader = shader;
    }
}


/*
 * Set SURFACE to be shaded with the simple shader.
 */
void
surface_basic_shader(surface, ambient, red, grn, blu, specular, c3)
    Surface *surface;
    double  ambient, red, grn, blu, specular, c3;
{
    Surf_desc *surf_desc;

    surf_desc = (Surf_desc *)smalloc(sizeof(Surf_desc));
    surf_desc->ambient = ambient;
    surf_desc->color.red = red;
    surf_desc->color.grn = grn;
    surf_desc->color.blu = blu;
    surf_desc->specular = specular;
    surf_desc->c3 = c3;
    surface_set_shader(surface, surf_desc, basic_shader);
}



/*
 * Copy a vertex tree.
 */
static Vertex *
copy_vertices(vp)
    Vertex *vp;
{
    Vertex *tmp;

    if (vp == NULL)
        return NULL;
    tmp = (Vertex *)smalloc(sizeof(Vertex));
    *tmp = *vp;
    tmp->big = copy_vertices(vp->big);
    tmp->sml = copy_vertices(vp->sml);
    return tmp;
}


/*
 * We have a list of vertes references, each pointing into a certain
 * vertex tree. Create a new list with pointers into a copy of the
 * first vertex tree.
 */
static Vertex_ref *
copy_vlist(vp, surface)
    Vertex_ref *vp;
    Surface    *surface;
{
    Vertex_ref *tmp;
    
    if (vp == NULL)
        return NULL;
    tmp = (Vertex_ref *)smalloc(sizeof(Vertex_ref));
    tmp->vertex = vertex_lookup(vp->vertex->x, vp->vertex->y, vp->vertex->z,
                                vp->vertex->u, vp->vertex->v, vp->vertex->w,
                                &(surface->vertices));
    tmp->next = copy_vlist(vp->next, surface);
    return tmp;
}


/*
 * Copy a list of polygons.
 */
static Polygon *
copy_polygons(pp, surface)
    Polygon *pp;
    Surface *surface;
{
    Polygon *tmp;

    if (pp == NULL)
        return NULL;
    tmp = (Polygon *)smalloc(sizeof(Polygon));
    tmp->vertices = copy_vlist(pp->vertices, surface);
    tmp->next = copy_polygons(pp->next, surface);
    return tmp;
}


/*
 * Copy a list of surfaces. All polygons and vertices are copied but
 * the shader and surface descriptions are the same as in the
 * original surfaces.
 */
static Surface *
surface_copy(surface)
    Surface  *surface;
{
    Surface  *newsurf;

    if (surface != NULL) {
        newsurf = (Surface *)smalloc(sizeof(Surface));
        if (newsurf == NULL) {
            return NULL;
        }
        memcpy(newsurf, surface, sizeof(Surface));
        newsurf->vertices = copy_vertices(surface->vertices);
        newsurf->polygons = copy_polygons(surface->polygons, newsurf);
        newsurf->ref_count = 1;
        newsurf->next = surface_copy(surface->next);
        return newsurf;
    } else {
        return NULL;
    }
}


/*
 * Delete a vertex tree.
 */
static void
delete_vertices(vtree)
    Vertex **vtree;
{
    if (*vtree != NULL) {
        delete_vertices(&((*vtree)->big));
        delete_vertices(&((*vtree)->sml));
        free(*vtree);
        *vtree = NULL;
    }
}


/*
 * Delete a surface list.
 */
static void
surface_delete(surface)
    Surface *surface;
{
    Vertex_ref *vref1, *vref2;
    Polygon    *polyref1, *polyref2;

    if (surface != NULL) {
        if (--surface->ref_count == 0) {
            if (surface->next != NULL) {
                surface_delete(surface->next);
            }
            polyref2 = surface->polygons;
            while (polyref2 != NULL) {
                vref2 = polyref2->vertices;
                while (vref2 != NULL) {
                    vref1 = vref2;
                    vref2 = vref2->next;
                    free(vref1);
                }
                polyref1 = polyref2;
                polyref2 = polyref2->next;
                free(polyref1);
            }
            delete_vertices(&(surface->vertices));
            free(surface);
        }
    }
}


/*
 * Install an object in the rendering database.
 */
static void
r_object_install(obj, obj_tree)
    Object       *obj;
    Inst_object **obj_tree;
{
    if (obj != NULL) {
        obj->ref_count++;
        if (*obj_tree == NULL) {
            *obj_tree = (Inst_object *)smalloc(sizeof(Inst_object));
            (*obj_tree)->object = obj;
            (*obj_tree)->big = NULL;
            (*obj_tree)->sml = NULL;
        } else if (obj > (*obj_tree)->object) {
            r_object_install(obj, &(*obj_tree)->big);
        } else if (obj < (*obj_tree)->object) {
            r_object_install(obj, &(*obj_tree)->sml);
        }
    }
}


/*
 * Interface to r_object_install(). (Why is there no
 * subfunctions in C?...)
 */
void
object_install(obj)
    Object *obj;
{
    r_object_install(obj, &object_db);
}



/*
 * Subfunction to r_object_uninstall.
 */
static void
r_del(r, q)
    Inst_object **r;
    Inst_object  *q;
{
    if ((*r)->big != NULL) {
        r_del(&((*r)->big), q);
    } else {
        q->object = (*r)->object;
        q = *r;
        *r = (*r)->sml;
        free(q);
    }
}


/*
 * Delete an object from the rendering database.
 * The object itself is not deleted of course.
 */
static void
r_object_uninstall(obj, root)
    Object       *obj;
    Inst_object **root;
{
    Inst_object *ptr;

    if (*root == NULL) {
        return;            /* Object is not in the tree */
    } else if (obj < (*root)->object) {
        r_object_uninstall(obj, &(*root)->sml);
    } else if (obj > (*root)->object) {
        r_object_uninstall(obj, &(*root)->big);
    } else {
        obj->ref_count--;
        ptr = *root;
        if (ptr->big == NULL) {
            *root = ptr->sml;
        } else if (ptr->sml == NULL) {
            *root = ptr->big;
        } else {
            r_del(&ptr->sml, ptr);
        }
    }
}
    
    
/*
 * Interface to r_object_uninstall.
 */
void
object_uninstall(obj)
    Object *obj;
{
    r_object_uninstall(obj, &object_db);
}



/*
 * Create an empty object. Before it is rendered it
 * must get a surface or a subobject. 
 */
Object *
object_create()
{
    Object *obj;

    obj = (Object *)smalloc(sizeof(Object));
    obj->surfaces = NULL;
    obj->sub_obj = NULL;
    MatCopy(&obj->transf, &ident_matrix);
    obj->ref_count = 0;
    obj->next = NULL;

    return obj;
}



/*
 * Copy the top object in an object hierarchy.
 * The new object will reference the same
 * subobjects and surfaces as the original object.
 * if REF_COUNT_UPDATE is true, the reference counts
 * in the surfaces and subobjects will be incremented.
 */
static Object *
object_copy(object, ref_count_update)
    Object *object;
    bool    ref_count_update;
{
    Object *newobj;

    if (object == NULL) {
        return NULL;
    }

    if ((newobj = (Object *)smalloc(sizeof(Object))) != NULL) {
        memcpy(newobj, object, sizeof(Object));
        if (ref_count_update) {
            if (newobj->sub_obj != NULL) {
                newobj->sub_obj->ref_count++;
            }
            if (newobj->surfaces != NULL) {
                newobj->surfaces->ref_count++;
            }
        }
        MatCopy(&newobj->transf, &ident_matrix);
        newobj->ref_count = 0;
        newobj->next = NULL;
    }

    return newobj;
}


/*
 * Copy a list of objects. If SURF_COPY is true
 * the surfaces in the objects will be copied too.
 */
static Object *
object_list_copy(object, surf_copy)
    Object *object;
    bool    surf_copy;
{
    Object *newobj;

    if (object == NULL) {
        return NULL;
    }

    if ((newobj = (Object *)smalloc(sizeof(Object))) != NULL) {
        memcpy(newobj, object, sizeof(Object));
        newobj->ref_count = 0;
    } else {
        return NULL;
    }

    if (surf_copy) {
        newobj->surfaces = surface_copy(object->surfaces);
    } else if (newobj->surfaces != NULL){
        newobj->surfaces->ref_count++;
    }

    newobj->sub_obj = object_list_copy(object->sub_obj, surf_copy);
    if (newobj->sub_obj != NULL) {
        newobj->sub_obj->ref_count++;
    }
    newobj->next = object_list_copy(object->next, surf_copy);
    if (newobj->next != NULL) {
        newobj->next->ref_count++;
    }

    return newobj;
}
    


/*
 * Copy the top node of an object hierarchy. The
 * subobjects and surface references will be the
 * same as in the original.
 */
Object *
object_instance(obj)
    Object *obj;
{
    return object_copy(obj, TRUE);
}



/*
 * Copy an object hierarchy. The objects in
 * the new hierarchy will reference the same
 * surfaces as the object in
 * the old hierarchy, but all object nodes
 * will be duplicated.
 */
Object *
object_dup(object)
    Object *object;
{
    Object *newobj;

    if ((newobj = object_copy(object, FALSE)) == NULL) {
        return NULL;
    }

    newobj->sub_obj = object_list_copy(object->sub_obj, FALSE);
    newobj->next = NULL;

    return newobj;
}
 


/*
 * Copy an object hierarchy. All object nodes
 * and surfaces in the old hierarchy
 * will be duplicated.
 */
Object *
object_deep_dup(object)
    Object *object;
{
    Object *newobj;

    if ((newobj = object_copy(object, FALSE)) == NULL) {
        return NULL;
    }

    newobj->surfaces = surface_copy(object->surfaces);
    newobj->sub_obj = object_list_copy(object->sub_obj, TRUE);
    newobj->next = NULL;

    return newobj;
}



/*
 * Recursively delete an object hierarchy. Reference
 * counts are decremented and if the result is zero
 * the recursion continues and the memory used is freed.
 */
static void
r_object_delete(object)
    Object * object;
{
    if (object != NULL) {
        if (--object->ref_count == 0) {
            surface_delete(object->surfaces);
            r_object_delete(object->sub_obj);
            r_object_delete(object->next);
            free(object);
        }
    }
}



/*
 * Delete an object hierarchy. This is only possible
 * to do on a top level object.
 */
void
object_delete(object)
    Object * object;
{
    if (object != NULL) {
        if (object->ref_count == 0) {         /* Is it a top level object? */
            surface_delete(object->surfaces);
            r_object_delete(object->sub_obj);
            r_object_delete(object->next);
            free(object);
        }
    }
}



/*
 * Add SUBOBJ as a subobject in OBJECT. SUBOBJ is appended
 * on the *end* of OBJECT's subobject list, 
 * so that if SUBOBJ, for some obscure reason, 
 * were the head of an object list, we don't loose
 * the rest of that list. 
 * Remove SUBOBJ from the rendering database since it is no
 * longer a root object in an hierarchy.
 */
void
object_add_subobj(object, subobj)
    Object *object, *subobj;
{
    Object *oref;

    if (object == NULL || subobj == NULL) {
        return;
    }

    if (object->sub_obj == NULL) {
        object->sub_obj = subobj;
    } else {
        oref = object->sub_obj;
        while (oref->next != NULL) {
            oref = oref->next;
        }
        oref->next = subobj;
    }

    subobj->ref_count++;
    object_uninstall(subobj);
}



/*
 * Add SURFACE to the list of surfaces belonging
 * to OBJECT. SURFACE is appended on the *end* of the
 * list for the same reasons as in object_add_subobj.
 */
void
object_add_surface(object, surface)
    Object  *object;
    Surface *surface;
{
    Surface *sref;

    if (object == NULL || surface == NULL) {
        return;
    }

    if (object->surfaces == NULL) {
        object->surfaces = surface;
    } else {
        sref = object->surfaces;
        while (sref->next != NULL) {
            sref = sref->next;
        }
        sref->next = surface;
    }

    surface->ref_count++;
}
    
    
    
    
/*============= Functions that handles object transformations==============*/

/*
 * Set the transformation matrix of OBJ to MATRIX.
 */
void
object_set_transf(obj, matrix)
    Object     *obj;
    Transf_mat *matrix;
{
    MatCopy(&obj->transf, matrix);
}


/*
 * Set the transformation matrix of OBJ to the identity matrix.
 */
void
object_clear_transf(obj)
    Object *obj;
{
    MatCopy(&obj->transf, &ident_matrix);
}


/*
 * Post multiply MATRIX into the transformation matrix of OBJ.
 */
void
object_transform(obj, matrix)
    Object     *obj;
    Transf_mat *matrix;
{
    mat_mul(&obj->transf, &obj->transf, matrix);
}


/*
 * Rotate the object OBJ ANG radians about the x-axis.
 */
void
object_rot_x(obj, ang)
    Object *obj;
    double  ang;
{
    mat_rotate_x(&obj->transf, ang);
}


/*
 * Rotate the object OBJ ANG radians about the y-axis.
 */
void
object_rot_y(obj, ang)
    Object *obj;
    double  ang;
{
    mat_rotate_y(&obj->transf, ang);
}


/*
 * Rotate the object OBJ ANG radians about the z-axis.
 */
void
object_rot_z(obj, ang)
    Object *obj;
    double  ang;
{
    mat_rotate_z(&obj->transf, ang);
}


/*
 * Rotate the object OBJ ANG radians about the line defined
 * by POINT and VEC.
 */
void
object_rot(obj, point, vec, ang)
    Object *obj;
    Vector *point;
    Vector *vec;
    double  ang;
{
    mat_rotate(&obj->transf, point, vec, ang);
}


/*
 * Scale the object OBJ with respect to the origin.
 */
void
object_scale(obj, xscale, yscale, zscale)
    Object *obj;
    double  xscale, yscale, zscale;
{
    mat_scale(&obj->transf, xscale, yscale, zscale);
}


/*
 * Translate the object OBJ.
 */
void
object_move(obj, dx, dy, dz)
    Object *obj;
    double  dx, dy, dz;
{
    mat_translate(&obj->transf, dx, dy, dz);
}




/*============= Functions that handles rendering of the scene==============*/


/*
 * Calculate the normal vector for all polygons in the polygon list PSTART.
 *
 * Check if the polygon is backfacing with respect to the current
 * viewpoint.
 *
 * The normalized normal is added to a normal kept at each vertex
 * in the polygon. This will produce, at each vertex, an average of the
 * normals of the adjectent plygons.
 */
static void
calc_normals(pstart, eyepoint)
    Polygon *pstart;    /* Head of polygon list */
    Vector   eyepoint;  /* Viewpoint transformed to local coordinate system */
{
    Vector      normal;
    Vertex_ref *vref1, *vref2;
    Polygon    *polyref;
    double      plane_const;

    for (polyref = pstart; polyref != NULL; polyref = polyref->next) {
        vref1 = polyref->vertices;
        vref2 = vref1->next;

        normal.x = normal.y = normal.z = 0.0;
        do {
            normal.x += ((vref1->vertex->y - vref2->vertex->y)
                         * (vref1->vertex->z + vref2->vertex->z));
            normal.y += ((vref1->vertex->z - vref2->vertex->z)
                         * (vref1->vertex->x + vref2->vertex->x));
            normal.z += ((vref1->vertex->x - vref2->vertex->x)
                         * (vref1->vertex->y + vref2->vertex->y));
            vref1 = vref1->next;
            vref2 = ((vref2->next == NULL)?polyref->vertices:vref2->next);
        } while (vref1 != NULL);
        vecnorm(&normal);

        plane_const = -(normal.x * vref2->vertex->x
                        + normal.y * vref2->vertex->y
                        + normal.z * vref2->vertex->z);
        if (VecDot(eyepoint, normal) + plane_const <= 0.0) {
            polyref->backface = TRUE;
        } else {
            polyref->backface = FALSE;
        }
            
        for (vref1 = polyref->vertices; vref1 != NULL; vref1 = vref1->next) {
            vref1->vertex->a += normal.x;
            vref1->vertex->b += normal.y;
            vref1->vertex->c += normal.z;
        }
    }
}



/*
 * Walk around a polygon, create the surrounding
 * edges and sort them into the y-bucket.
 * Clip polygons in y at the same time.
 */
static void
create_edges(view_vert, yres, polygon, surface)
    View_coord *view_vert;
    int         yres;
    int         polygon;
    Surface    *surface;
{
    Edge       *edge;
    View_coord *view_ref, *last;
    int         nderiv, y1, y2, ymax;
    int         clip1, clip2;
    double      deltay, ratio;
    double      x1, x2, xstep;
    double      z1, z2, zstep;
    double      nx1, nx2, nxstep;
    double      ny1, ny2, nystep;
    double      nz1, nz2, nzstep;
    double      u1, u2, ustep;
    double      v1, v2, vstep;
    double      w1, w2, wstep;

    view_ref = last = view_vert;
    ymax = yres - 1;
    do {
        view_ref = view_ref->next;
        x1 = view_ref->x;
        x2 = view_ref->next->x;
        y1 = view_ref->y;
        y2 = view_ref->next->y;
        z1 = view_ref->z;
        z2 = view_ref->next->z;
        nx1 = view_ref->nx;
        nx2 = view_ref->next->nx;
        ny1 = view_ref->ny;
        ny2 = view_ref->next->ny;
        nz1 = view_ref->nz;
        nz2 = view_ref->next->nz;
        u1 = view_ref->u;
        u2 = view_ref->next->u;
        v1 = view_ref->v;
        v2 = view_ref->next->v;
        w1 = view_ref->w;
        w2 = view_ref->next->w;
        clip1 = (y1 < 0) + ((y1 > ymax) << 1);
        clip2 = (y2 < 0) + ((y2 > ymax) << 1);
        if (!(clip1 & clip2)) {
            if (clip1 != 0) {
                if (clip1 == 1)
                    ratio = (0.0 - (double)y1) / (double)(y2 - y1);
                else 
                    ratio = (double)(ymax - y1) / (double)(y2 - y1);
                x1 = x1 + ratio * (x2 - x1);
                y1 = y1 + ratio * (y2 - y1);
                z1 = z1 + ratio * (z2 - z1);
                nx1 = nx1 + ratio * (nx2 - nx1);
                ny1 = ny1 + ratio * (ny2 - ny1);
                nz1 = nz1 + ratio * (nz2 - nz1);
                u1 = u1 + ratio * (u2 - u1);
                v1 = v1 + ratio * (v2 - v1);
                w1 = w1 + ratio * (w2 - w1);
            }
            if (clip2 != 0) {
                if (clip2 == 1)
                    ratio = (0.0 - (double)y2) / (double)(y1 - y2);
                else 
                    ratio = (double)(ymax - y2) / (double)(y1 - y2);
                x2 = x2 + ratio * (x1 - x2);
                y2 = y2 + ratio * (y1 - y2);
                z2 = z2 + ratio * (z1 - z2);
                nx2 = nx2 + ratio * (nx1 - nx2);
                ny2 = ny2 + ratio * (ny1 - ny2);
                nz2 = nz2 + ratio * (nz1 - nz2);
                u2 = u2 + ratio * (u1 - u2);
                v2 = v2 + ratio * (v1 - v2);
                w2 = w2 + ratio * (w1 - w2);
            }
            deltay = (double)(y2 - y1);
            if (deltay > 0.0)
	        nderiv = 1;
	    else if (deltay < 0.0)
	        nderiv = -1;
	    else
	        nderiv = 0;
            if (nderiv) {
	        deltay = fabs(deltay);
                xstep = (x2 - x1) / deltay;
                zstep = (z2 - z1) / deltay;
                nxstep = (nx2 - nx1) / deltay;
                nystep = (ny2 - ny1) / deltay;
                nzstep = (nz2 - nz1) / deltay;
                ustep = (u2 - u1) / deltay;
                vstep = (v2 - v1) / deltay;
                wstep = (w2 - w1) / deltay;
                edge = (Edge *)smalloc(sizeof(Edge));
                if (nderiv > 0) {
                    edge->y = y2;
                    edge->y_stop = y1;
                    edge->x = x2;
                    edge->z = z2;
                    edge->nx = nx2;
                    edge->ny = ny2;
                    edge->nz = nz2;
                    edge->u = u2;
                    edge->v = v2;
                    edge->w = w2;
                    edge->xstep = -xstep;
                    edge->zstep = -zstep;
                    edge->nxstep = -nxstep;
                    edge->nystep = -nystep;
                    edge->nzstep = -nzstep;
                    edge->ustep = -ustep;
                    edge->vstep = -vstep;
                    edge->wstep = -wstep;
                } else {
                    edge->y = y1;
                    edge->y_stop = y2;
                    edge->x = x1;
                    edge->z = z1;
                    edge->nx = nx1;
                    edge->ny = ny1;
                    edge->nz = nz1;
                    edge->u = u1;
                    edge->v = v1;
                    edge->w = w1;
                    edge->xstep = xstep;
                    edge->zstep = zstep;
                    edge->nxstep = nxstep;
                    edge->nystep = nystep;
                    edge->nzstep = nzstep;
                    edge->ustep = ustep;
                    edge->vstep = vstep;
                    edge->wstep = wstep;
                }
            } else {
                if (x2 != x1) {
                    zstep = (z2 - z1) / fabs(x2 - x1);
                } else {
                    zstep = z2 - z1;
                }
                edge = (Edge *)smalloc(sizeof(Edge));
                edge->y = y2;
                edge->y_stop = y1;
                if (x1 < x2) {
                    edge->x = x1;
                    edge->z = z1;
                    edge->xstep = x2;
                    edge->zstep = zstep;
                } else {
                    edge->x = x2;
                    edge->z = z2;
                    edge->xstep = x1;
                    edge->zstep = -zstep;
                }
            }
            edge->polygon = polygon;
            edge->surface = surface;
            edge->next = y_bucket[edge->y];
            y_bucket[edge->y] = edge;
        }
    } while (view_ref != last);
}


/*
 * Transform vertices into view coordinates. The transform is
 * defined in MATRIX. Store the transformed vertices in a
 * temporary list, create edges in the y_bucket.
 */
static void
transf_vertices(vertex_list, surface, matrix, tr_mat, xsiz, ysiz)
    Vertex_ref *vertex_list;
    Surface    *surface;
    double      matrix[4][4];
    Transf_mat *tr_mat;
    double      xsiz, ysiz;
{
    static int  polygon = 0;        /* incremented for each call to provide */
                                    /* unique polygon id numbers */
    Vertex_ref *vref;
    View_coord *view_ref, *nhead, *ntail;
    double      minsize;
    double      w, tmp;
    bool        first, last;

    first = TRUE;
    last = FALSE;
    vref = vertex_list;
    nhead = NULL;

    minsize = ((xsiz > ysiz) ? ysiz : xsiz);

    while ((vref != NULL) && !last) {

        view_ref = (View_coord *)alloca(sizeof(View_coord));
        last = (vref->next == NULL);

        view_ref->nx = (vref->vertex->a * tr_mat->mat[0][0] 
                        + vref->vertex->b * tr_mat->mat[1][0] 
                        + vref->vertex->c * tr_mat->mat[2][0]);
        view_ref->ny = (vref->vertex->a * tr_mat->mat[0][1] 
                        + vref->vertex->b * tr_mat->mat[1][1] 
                        + vref->vertex->c * tr_mat->mat[2][1]);
        view_ref->nz = (vref->vertex->a * tr_mat->mat[0][2] 
                        + vref->vertex->b * tr_mat->mat[1][2] 
                        + vref->vertex->c * tr_mat->mat[2][2]);

        w = vref->vertex->x * matrix[0][3] + vref->vertex->y * matrix[1][3] +
	    vref->vertex->z * matrix[2][3] + matrix[3][3];
        view_ref->x = ((vref->vertex->x * matrix[0][0] 
                        + vref->vertex->y * matrix[1][0] 
                        + vref->vertex->z * matrix[2][0]
                        + matrix[3][0]) * minsize / w + xsiz);
        tmp         = ((vref->vertex->x * matrix[0][1] 
                        + vref->vertex->y * matrix[1][1] 
                        + vref->vertex->z * matrix[2][1]
                        + matrix[3][1]) * minsize / w + ysiz) ;

        view_ref->y = (int)(tmp + 0.5);
        view_ref->z = (vref->vertex->x * matrix[0][2] + vref->vertex->y *
                       matrix[1][2] + vref->vertex->z * matrix[2][2] +
                       matrix[3][2]) / w; 

        view_ref->u = vref->vertex->u;
        view_ref->v = vref->vertex->v;
        view_ref->w = vref->vertex->w;
        view_ref->next = nhead;
        nhead = view_ref;

        if (first) {
            ntail = view_ref;
            first = FALSE;
        }
        if (!last)
            vref = vref->next;
    }

    ntail->next = nhead;
    create_edges(nhead, (int)ysiz << 1, polygon++, surface);
}


/*
 * Initialize the scanline z-buffer and the actual picture
 * scanline buffer.
 */
static void
init_buffers(res, z_buffer, scanline)
    int            res;
    double        *z_buffer;
    unsigned char *scanline;
{
    int i;
    
#ifdef NOMEMCPY
    bzero(scanline, res * 3);
#else
    memset(scanline, 0, res * 3);
#endif
    for (i = 0; i < res; i++)
        z_buffer[i] = 2.0;
}
    

/*
 * Read edge pairs from the edge list EDGE_LIST. Walk along the scanline
 * and interpolate z value, texture coordinates and normal vector as 
 * we go. Call the shader and write into scanline buffer according to 
 * result on z-buffer test.
 */
static void
render_line(res, z_buffer, scanline, edge_list)
    int            res;
    double        *z_buffer;
    unsigned char *scanline;
    Edge          *edge_list;
{
    double z, zstep;
    double nx, nxstep;
    double ny, nystep;
    double nz, nzstep;
    double u, ustep;
    double v, vstep;
    double w, wstep;
    double ratio;
    Color  color;
    int    i, j, x, xstop;
    Edge  *edgep, *next;
    
    edgep = edge_list;
    next = NULL;
    while (edgep != NULL) {
        if (edgep->y != edgep->y_stop) {
            next = edgep->next;
            while (next->y == next->y_stop)
                next = next->next;
            x = (int)(edgep->x + 0.5);
            xstop = (int)(next->x + 0.5);
            z = edgep->z;
            nx = edgep->nx;
            ny = edgep->ny;
            nz = edgep->nz;
            u = edgep->u;
            v = edgep->v;
            w = edgep->w;
            if (x < xstop) {
                ratio = (double)(xstop - x);
                zstep = (next->z - z) / ratio;
                nxstep = (next->nx - nx) / ratio;
                nystep = (next->ny - ny) / ratio;
                nzstep = (next->nz - nz) / ratio;
                ustep = (next->u - u) / ratio;
                vstep = (next->v - v) / ratio;
                wstep = (next->w - w) / ratio;
            } else {
                zstep = 0.0;
                nxstep = nystep = nzstep = 0.0;
                ustep = vstep = wstep = 0.0;
            }
            for (i = x, j = i * 3; i <= xstop; i++) {
                if ((i >= 0) && (i < res) && (z >= 0.0) && (z <= 1.0)
                    && (z < z_buffer[i])) {
                    (*edgep->surface->shader)
                        (nx, ny, nz, u, v, w,
                         camera.vec, lightsrc_stack, 
                         edgep->surface->surface, &color);
                    scanline[j++] = (unsigned char)(color.red * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(color.grn * 255.0 + 0.5);
                    scanline[j++] = (unsigned char)(color.blu * 255.0 + 0.5);
                    z_buffer[i] = z;
                } else if (i >= res) {
                    break;
                } else {
                    j += 3;
                }
                z += zstep;
                nx += nxstep;
                ny += nystep;
                nz += nzstep;
                u += ustep;
                v += vstep;
                w += wstep;
            }
        }
        edgep = edgep->next;
        if ((edgep == next) && (next != NULL))
            edgep = edgep->next;
    }
}



/*
 * Insert an edge into an edge list. Edges belonging to the same
 * polygon must be inserted sorted in x, so that edge pairs are
 * created.
 */
static Edge *
insert_edge(edge_list, edge, poly_found)
    Edge *edge_list, *edge;
    bool  poly_found;
{
    if (edge_list == NULL) {
        edge_list = edge;
	edge->next = NULL;
    } else if (edge_list->polygon == edge->polygon) {
        if (edge_list->x > edge->x) {
	    edge->next = edge_list;
	    edge_list = edge;
	} else if ((((int)(edge_list->x + 0.5)) == ((int)(edge->x + 0.5)))
                   && (edge_list->xstep > edge->xstep)) {
	    edge->next = edge_list;
	    edge_list = edge;
        } else {
	    edge_list->next = insert_edge(edge_list->next, edge, TRUE);
        }
    } else if (poly_found) {
        edge->next = edge_list;
        edge_list = edge;
    } else {
        edge_list->next = insert_edge(edge_list->next, edge, FALSE);
    }

    return edge_list;
}
        


/*
 * Merge two edge lists.
 */
static Edge *
merge_edge_lists(list1, list2)
    Edge *list1, *list2;
{
    Edge *eref1, *eref2, *next;
    
    if (list2 == NULL)
        return NULL;
    eref1 = list1;
    eref2 = list2;
    do {
        next = eref2->next;
        eref1 = insert_edge(eref1, eref2, FALSE);
	eref2 = next;
    } while (eref2 != NULL);
    return eref1;
}



/*
 * Allocate the needed buffers. Create a list of active edges and
 * move down the y-bucket, inserting and deleting edges from this
 * active list as we go. Call render_line for each scanline and
 * do an average filtering before writing the scanline to the result
 * file descriptor.
 */
static void
scan_and_render(xres, yres, image_file)
    int  xres, yres;
    FILE *image_file;
{
    Edge          *active_list, *edgep, *edgep2;
    double        *z_buffer;
    unsigned char *scanline1, *scanline2, *stmp;
    int            i, y, next_edge;
    
    z_buffer = (double *)calloc(xres, sizeof(double));
    scanline1 = (unsigned char *)calloc(xres * 3, sizeof(unsigned char));
    scanline2 = (unsigned char *)calloc(xres * 3, sizeof(unsigned char));

    fprintf(image_file, "P6\n");
    fprintf(image_file, "#Image rendered with SIPP %s\n", VERSION);
    fprintf(image_file, "%d\n%d\n255\n", xres >> 1, yres >> 1);
 
    y = yres - 1;
    active_list = NULL;
    stmp = scanline1;
 
    while (y >= 0) {
        active_list = merge_edge_lists(active_list, y_bucket[y]);
        next_edge = y - 1;
        while (next_edge >=0 && y_bucket[next_edge] == NULL)
            next_edge--;
        while (y > next_edge) {
            init_buffers(xres, z_buffer, stmp);
            render_line(xres, z_buffer, stmp, active_list);
            if (stmp == scanline1)
                stmp = scanline2;
            else {
                for (i = 0; i < (xres * 3); i += 6) {
                    scanline1[i >> 1] = (scanline1[i] +
                                         scanline1[i + 3] +
                                         scanline2[i] +
                                         scanline2[i + 3]) >> 2;
                    scanline1[(i >> 1) + 1] = (scanline1[i + 1] +
                                               scanline1[i + 4] +
                                               scanline2[i + 1] +
                                               scanline2[i + 4]) >> 2;
                    scanline1[(i >> 1) + 2] = (scanline1[i + 2] +
                                               scanline1[i + 5] +
                                               scanline2[i + 2] +
                                               scanline2[i + 5]) >> 2;
                }
                fwrite(scanline1, (xres >> 1) * 3, 1, image_file);
                fflush(image_file);
                stmp = scanline1;
            }
	    if (active_list != NULL) {
	        edgep2 = active_list;
	        edgep = active_list->next;
	        while (edgep != NULL)
	            if (edgep->y <= (edgep->y_stop + 1)) {
                        edgep2->next = edgep->next;
		        free(edgep);
	                edgep = edgep2->next;
		    } else {
		        edgep2 = edgep;
		        edgep = edgep->next;
		    }
  	        if (active_list->y <= (active_list->y_stop + 1)) {
	            edgep = active_list;
		    active_list = active_list->next;
	            free(edgep);
	        }
	        edgep = active_list;
	        while (edgep != NULL) {
	            edgep->y--;
		    edgep->x += edgep->xstep;
		    edgep->z += edgep->zstep;
		    edgep->nx += edgep->nxstep;
		    edgep->ny += edgep->nystep;
		    edgep->nz += edgep->nzstep;
		    edgep->u += edgep->ustep;
		    edgep->v += edgep->vstep;
		    edgep->w += edgep->wstep;
		    edgep = edgep->next;
	        }
	    }
	    y--;
	}
    }
    free(z_buffer);
    free(scanline1);
    free(scanline2);
}



/*
 * Reset the averaged normals in the vertex tree P.
 */
static void
reset_normals(vref)
    Vertex *vref;
{
    if (vref != NULL) {
        vref->a = 0;
        vref->b = 0;
        vref->c = 0;
        reset_normals(vref->big);
        reset_normals(vref->sml);
    }
}



/*
 * Build a transformation matrix for transformation
 * into view coordinates. Perpective transformation
 * is also included
 */
static void
get_view_transf(mat)
    double mat[4][4];
{
    Vector tmp;
    double transl[3];
    double vy, vz;
    double hither, yon;
    double alfa, beta;
    int i, j;
    
    /*
     * First we need a translation so the origo
     * of the view coordinat system is placed
     * in the viewpoint.
     */
    transl[0] = -camera.x0;
    transl[1] = -camera.y0;
    transl[2] = -camera.z0;

    /*
     * Then we need a rotation that makes the
     * up-vector point up, and alignes the sightline
     * with the z-axis.
     * This code might seem magic but the algebra behind
     * it can be found in Jim Blinn's Corner in IEEE CG&A July 1988
     */
    VecCopy(tmp, camera.vec);
    VecNegate(tmp);
    vecnorm(&tmp);
    vecnorm(&camera.up);
    vz = VecDot(tmp, camera.up);
    if ((vz * vz) > 1.0) {        /* this should not happen, but... */
        vz = 1.0;
    } else {
        vy = sqrt(1.0 - vz * vz);
        if (vy == 0.0) {          /* oops, the world collapses... */
            vy = 1.0e10;
            vz = 1.0;
        } else {
            vy = 1.0 / vy;
        }
    }

    mat[0][2] = tmp.x;
    mat[1][2] = tmp.y;
    mat[2][2] = tmp.z;

    VecScalMul(tmp, vz, tmp);
    VecSub(tmp, camera.up, tmp);
    mat[0][1] = tmp.x * vy;
    mat[1][1] = tmp.y * vy;
    mat[2][1] = tmp.z * vy;

    mat[0][0] = mat[1][1] * mat[2][2] - mat[1][2] * mat[2][1];
    mat[1][0] = mat[2][1] * mat[0][2] - mat[2][2] * mat[0][1];
    mat[2][0] = mat[0][1] * mat[1][2] - mat[0][2] * mat[1][1];

    /*
     * Install the translation into the matrix.
     * Note that it is PRE-multiplied into the matrix.
     */
    for (i = 0; i < 3; i++) {
        mat[3][i] = 0.0;
        for (j = 0; j < 3; j++) {
            mat[3][i] += transl[j] * mat[j][i];
        }
    }
     
    /*
     * Include the perspective transformation
     * using heuristic values for hither and yon.
     */
    hither = VecLen(camera.vec) / ZCLIPF;
    yon    = VecLen(camera.vec) * ZCLIPF;
    alfa = camera.focal_ratio / (1.0 - hither / yon);
    beta = -hither * alfa;

    mat[0][3] = mat[0][2] * camera.focal_ratio;
    mat[0][2] *= alfa;
    mat[1][3] = mat[1][2] * camera.focal_ratio;
    mat[1][2] *= alfa;
    mat[2][3] = mat[2][2] * camera.focal_ratio;
    mat[2][2] *= alfa;
    mat[3][3] = mat[3][2] * camera.focal_ratio;
    mat[3][2] = mat[3][2] * alfa + beta;

    /*
     * Since the screen coordinates are defined in a left handed
     * coordinate system, we must switch the sign of the first
     * column in the matrix to get appropriate signs of x-values.
     */
    mat[0][0] = -mat[0][0];
    mat[1][0] = -mat[1][0];
    mat[2][0] = -mat[2][0];
    mat[3][0] = -mat[3][0];
}



/*
 * Push the current transformation matrix on the matrix stack.
 */
static void
matrix_push()
{
    struct tm_stack_t *new_tm;

    new_tm = (struct tm_stack_t *)smalloc(sizeof(struct tm_stack_t));
    MatCopy(&new_tm->mat, &curr_mat);
    new_tm->next = tm_stack;
    tm_stack     = new_tm;
}


/*
 * Pop the top of the matrix stack and make
 * it the new current transformation matrix.
 */
static void
matrix_pop()
{
    struct tm_stack_t *tmp;

    MatCopy(&curr_mat, &tm_stack->mat);
    tmp = tm_stack;
    tm_stack = tm_stack->next;
    free(tmp);
}



/*
 * Traverse an object hierarchy, transform each object
 * according to its transformation matrix.
 * Transform all polygons in the object to view coordinates.
 * Build the edge lists in y_bucket.
 */
static void
traverse_object_tree(object, glob_view_mat, xres, yres)
    Object *object;
    double  glob_view_mat[4][4];
    int     xres, yres;
{
    Object  *objref;
    Surface *surfref;
    Polygon *polyref;
    Vector   eyepoint, tmp;
    double   loc_view_mat[4][4];
    double   m[3][4], dtmp;
    int      i, j;


    if (object == NULL) {
        return;
    }

    for (objref = object; objref != NULL; objref = objref->next) {

        matrix_push();
        mat_mul(&curr_mat, &objref->transf, &curr_mat);
        mat_mul34(loc_view_mat, &curr_mat, glob_view_mat);

        tmp.x = camera.x0;
        tmp.y = camera.y0;
        tmp.z = camera.z0;

        /*
         * Do an inverse transformation of the viewpoint to use
         * when doing backface culling (in calc_normals()).
         */
        tmp.x -= curr_mat.mat[3][0];
        tmp.y -= curr_mat.mat[3][1];
        tmp.z -= curr_mat.mat[3][2];
        m[0][0] = curr_mat.mat[0][0] ; m[0][1] = curr_mat.mat[1][0];
        m[0][2] = curr_mat.mat[2][0] ; m[0][3] = tmp.x;
        m[1][0] = curr_mat.mat[0][1] ; m[1][1] = curr_mat.mat[1][1];
        m[1][2] = curr_mat.mat[2][1] ; m[1][3] = tmp.y;
        m[2][0] = curr_mat.mat[0][2] ; m[2][1] = curr_mat.mat[1][2];
        m[2][2] = curr_mat.mat[2][2] ; m[2][3] = tmp.z;

        if (m[0][0] == 0.0) {
            if (m[1][0] != 0.0)
                j = 1;
            else
                j = 2;
            for (i = 0; i < 4; i++) {
                dtmp     = m[0][i];
                m[0][i] = m[j][i];
                m[j][i] = dtmp;
            }
        }
        
        for (j = 1; j < 3; j++) {
            m[j][0] /= (-m[0][0]);
            for (i = 1; i < 4; i++)
                m[j][i] += m[0][i] * m[j][0];
        }
        
        if (m[1][1] == 0.0)
            for (i = 1; i < 4; i++) {
                dtmp     = m[1][i];
                m[1][i] = m[2][i];
                m[2][i] = dtmp;
            }
        
        if (m[1][1] != 0.0) {
            m[2][1] /= (-m[1][1]);
            m[2][2] += m[1][2] * m[2][1];
            m[2][3] += m[1][3] * m[2][1];
        }
        
        eyepoint.z = m[2][3] / m[2][2];
        eyepoint.y = (m[1][3] - eyepoint.z * m[1][2]) / m[1][1];
        eyepoint.x = (m[0][3] - eyepoint.z * m[0][2] 
                      - eyepoint.y * m[0][1]) / m[0][0];

        for (surfref = objref->surfaces; surfref != NULL; 
             surfref = surfref->next) {

            calc_normals(surfref->polygons, eyepoint);

            for (polyref = surfref->polygons; polyref != NULL; 
                 polyref = polyref->next) {

                if (!polyref->backface) {
                    transf_vertices(polyref->vertices, surfref, loc_view_mat, 
                                    &curr_mat, (double)(xres >> 1),
                                    (double)(yres >> 1));
                }

            }
            reset_normals(surfref->vertices);

        }
        traverse_object_tree(objref->sub_obj, glob_view_mat, xres, yres);
        matrix_pop();
    }
}



/*
 * Recursively traverse the rendering database (preorder).
 * Call traverse_object_tree for each object.
 */
static void
traverse_object_db(obj_root, view_mat, xres, yres)
    Inst_object *obj_root;
    double       view_mat[4][4];
    int          xres, yres;
{
    if (obj_root != NULL) {
        traverse_object_tree(obj_root->object, view_mat, xres, yres);
        traverse_object_db(obj_root->sml, view_mat, xres, yres);
        traverse_object_db(obj_root->big, view_mat, xres, yres);
    }
}




/*
 * "Main" function in rendering. Allocate y-bucket, transform vertices
 * into viewing coordinates, make edges and sort them into the y-bucket.
 * Call scan_and_render to do the real work.
 */
void
render_image(xres, yres, image_file)
    int   xres, yres;
    FILE *image_file;
{
    double      matrix[4][4];
    int         i;

    y_bucket = (Edge **)calloc(yres << 1, sizeof(Edge *));

    get_view_transf(matrix);
    MatCopy(&curr_mat, &ident_matrix);
    
    traverse_object_db(object_db, matrix, xres << 1, yres << 1);

    vecnorm(&camera.vec);
    scan_and_render(xres << 1, yres << 1, image_file);
    view_vec_eval();
}



/*============= Functions that handles global initializations==============*/

/*
 * Necessary initializations.
 */
void
sipp_init()
{
    vertex_tree    = NULL;
    vertex_stack   = NULL;
    poly_stack     = NULL;
    object_db      = NULL;
    lightsrc_stack = NULL;
    first_vertex   = 0;
    viewpoint(0.0, 0.0, 10.0,  0.0, 0.0, 0.0,  0.0, 1.0, 0.0,  0.25);
}
