/*
 * File: geometric.h
 *
 * All kinds of stuff with matrixes, transformations, coordinates
 * a.s.o.
 */

/* Make sure no multiple including */
#ifndef _GEOMETRIC_H_
#define _GEOMETRIC_H_

#include <math.h>
#ifndef NOMEMCPY
#include <memory.h>
#endif


/* #define PI    3.1415926535897932384626 */


/*
 * Define a homogenous transformation matrix. The first row (vector) 
 * is the new X axis, i.e. the X axis in the transformed coordinate 
 * system. The second row is the new Y axis, and so on. The last row
 * is the translation, for a transformed point.
 *
 * The reason we make surround the rows with a struct is that we
 * don't want to say (Transf_mat *) &foo[0] instead of &foo when
 * sending an address to a matrix as a parameter to a function.
 * Alas, arrays are not first class objects in C.
 */



/*
 * NOTE:
 * Capitalized types denote Vectors and other aggregates.
 * Lower case denote scalars.
 */

/* P = Pointd(x, y) */
#define MakePoint2d(P, xx, yy)	 { (P)[0]=(xx); \
                                   (P)[1]=(yy); }

/* P = Point3d(x, y, z) */
#define MakePoint3d(P, xx, yy, zz) { (P)[0]=(xx); \
                                     (P)[1]=(yy); \
                                     (P)[2]=(zz); }

/* V = Vec(x, y, z) */
#define MakeVector(V, xx, yy, zz)  { (V).x=(xx); \
                                     (V).y=(yy); \
                                     (V).z=(zz); }

/* A = -A */
#define VecNegate(A)	         { (A).x=0-(A).x; \
			           (A).y=0-(A).y; \
			           (A).z=0-(A).z; }

/* return A . B */
#define VecDot(A, B)	((A).x*(B).x+(A).y*(B).y+(A).z*(B).z)

/* return length(A) */
#define VecLen(A)	(sqrt((double)VecDot(A, A)))

/* B = A */
#define VecCopy(B, A)	((B) = (A))

/* C = A + B */
#define VecAdd(C, A, B)	 { (C).x=(A).x+(B).x; \
			   (C).y=(A).y+(B).y; \
			   (C).z=(A).z+(B).z; }

/* C = A - B */
#define VecSub(C, A, B)	 { (C).x=(A).x-(B).x; \
			   (C).y=(A).y-(B).y; \
			   (C).z=(A).z-(B).z; }

/* C = a*A */
#define VecScalMul(C, a, A)	 { (C).x=(a)*(A).x; \
				   (C).y=(a)*(A).y; \
				   (C).z=(a)*(A).z; }

/* C = a*A + B */
#define VecAddS(C, a, A, B)	 { (C).x=(a)*(A).x+(B).x; \
				   (C).y=(a)*(A).y+(B).y; \
				   (C).z=(a)*(A).z+(B).z; }

/* C = a*A + b*B */
#define VecComb(C, a, A, b, B)	 { (C).x=(a)*(A).x+(b)*(B).x; \
				   (C).y=(a)*(A).y+(b)*(B).y; \
			 	   (C).z=(a)*(A).z+(b)*(B).z; }

/* C = A X B */
#define VecCross(C, A, B)   	 { (C).x=(A).y*(B).z-(A).z*(B).y; \
                                   (C).y=(A).z*(B).x-(A).x*(B).z; \
			           (C).z=(A).x*(B).y-(A).y*(B).x; }


/*-------------------------- Matrix operations -------------------------*/


/* *A = *B    N.b. A and B are pointers! */
#define MatCopy(A, B)		 (*A) = (*B)


/*--------------------- Transf_mat-Vector operations -------------------*/


/* Store a Vector into a row in a Matrix */
/* NOTE: This implementation depends on that a Vector is the same */
/*       type as a row in a Matrix!! */
#define Vec2Row(mat, row, vec)	((mat).rows[row] = vec)


/*----------------------------------------------------------------------*/


/* Function declarations for the functions in geometric.c */

extern void          vecnorm(/* Vector */); /* Normalize a vector */
extern Transf_mat  * new_matrix(/* Matrix * */);
extern void          mat_translate(/* Matrix *, double, double, double */);
extern void          mat_rotate_x(/* Matrix *, double */);
extern void          mat_rotate_y(/* Matrix *, double */);
extern void          mat_rotate_z(/* Matrix *, double */);
extern void          mat_rotate(/* Matrix *, Point3d *, Vector *, double */);
extern void          mat_scale(/* Matrix *, double, double, double */);
extern void          mat_mirror_plane(/* Matrix *, Point3d *, Vector * */);
extern void          mat_mul(/* Matrix *, Matrix *, Matrix * */);
extern void          mat_mul34(/* double[4][4] *, Matrix *, double[4][4] */);
extern void          point_transform(/* Point3d *, Point3d *, Matrix * */);


#endif  /* _GEOMETRIC_H_ */
