/**********************************************************************/
/* raycube.c                                                          */
/*                                                                    */
/* Ray / cube intersection routines                                   */
/*                                                                    */
/* Modified from Optik v(1.2e) (C) 1987 John Amanatides & Andrew Woo  */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#include <math.h>
#include "geo.h"
#include "io.h"
#include "misc.h"
#include "struct.h"
#include "ray.h"

extern RayStats_Type RayStats;
extern OptionType Option;

/**********************************************************************/
/* Find ray intersect of ray (if any) with cube.                      */
/* If checking form factors, return 1st hit closer than maximum hit   */
/* distance. */
/**********************************************************************/
void RayCube(ray, hit, optr)
     register Ray *ray;
     register HitData *hit;
     register Objectt *optr;
{
  register int X, Y, Z;
  double t, x, y, z;

  RayStats.rayCube++;
  if (Option.debug)
    printf("\t\tRay-Cube: (%g,%g,%g) > (%g,%g,%g)\n",
	   ray->origin.x, ray->origin.y, ray->origin.z, 
	   ray->dir.x, ray->dir.y, ray->dir.z);
  
  /* Check direction of ray */
  X = BSGN(ray->dir.x);
  Y = BSGN(ray->dir.y);
  Z = BSGN(ray->dir.z);
  
  /* Check right face of cube */
  if (X != ZERO) {
    t= (1.0-ray->origin.x)/ray->dir.x;
    if (t > MIN_DISTANCE && t < hit->distance) {
      y= ray->origin.y + t*ray->dir.y;
      if (y > -1.0 && y < 1.0) {
	z= ray->origin.z + t*ray->dir.z;
	
	/* Store hit data */
	if (z > -1.0 && z < 1.0) {
	  RayStats.intCube++;
	  if (Option.visibility == FORM_FACTOR)
	    ray->visible = 0.0;
	  Store_HitInter(hit,optr,t,1.0,y,z);
	  if (!ray->shadow) {
	    Store_HitNorm(hit,1.0,0.0,0.0);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }

  /* check left face */
  if (Option.visibility == FORM_FACTOR)
    if (ray->visible == 0.0)
      return;
  if (X != ZERO) {
    t= (-1.-ray->origin.x)/ray->dir.x;
    if (t > MIN_DISTANCE && t < hit->distance) {
      y= ray->origin.y + t*ray->dir.y;
      if (y > -1.0 && y < 1.0) {
	z= ray->origin.z + t*ray->dir.z;

	/* Store hit data */
	if (z > -1.0 && z < 1.0) {
	  RayStats.intCube++;
	  if (Option.visibility == FORM_FACTOR)
	    ray->visible = 0.0;
	  Store_HitInter(hit, optr, t, -1., y, z);
	  if (!ray->shadow) {
	    Store_HitNorm(hit, -1., 0., 0.);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }

  /* check back face */
  if (Option.visibility == FORM_FACTOR)
    if (ray->visible == 0.0)
      return;
  if (Y != ZERO) {
    t= (1.-ray->origin.y)/ray->dir.y;
    if (t > MIN_DISTANCE && t < hit->distance) {
      x= ray->origin.x + t*ray->dir.x;
      if (x > -1. && x < 1.) {
	z= ray->origin.z + t*ray->dir.z;
	
	/* Store hit data */
	RayStats.intCube++;
	if (Option.visibility == FORM_FACTOR)
	  ray->visible == 0.0;
	if (z > -1.0 && z < 1.0) {
	  Store_HitInter(hit, optr, t, x, 1.0, z);
	  if (!ray->shadow) {
	    Store_HitNorm(hit, 0., 1. ,0.);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }


  /* check front face */
  if (Option.visibility == FORM_FACTOR)
    if (ray->visible == 0.0)
      return;
  if (Y != ZERO) {
    t= (-1.-ray->origin.y)/ray->dir.y;
    if (t > MIN_DISTANCE && t < hit->distance) {
      x= ray->origin.x + t*ray->dir.x;
      if (x > -1. && x < 1.) {
	z= ray->origin.z + t*ray->dir.z;

	/* Store hit data */
	RayStats.intCube++;
	if (z > -1 && z < 1.) {
	  if (Option.visibility == FORM_FACTOR)
	    ray->visible == 0.0;
	  Store_HitInter(hit, optr, t, x, -1. ,z);
	  if (!ray->shadow) {
	    Store_HitNorm(hit, 0., -1., 0.);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }

  /* check top face */
  if (Option.visibility == FORM_FACTOR)
    if (ray->visible == 0.0)
      return;
  if (Z != ZERO) {
    t= (1.-ray->origin.z)/ray->dir.z;
    if (t > MIN_DISTANCE && t < hit->distance) {
      x= ray->origin.x + t*ray->dir.x;
      if (x > -1. && x < 1.) {
	y= ray->origin.y + t*ray->dir.y;
	if (y > -1 && y < 1.0) {

	  /* Store hit data */
	  RayStats.intCube++;
	  if (Option.visibility == FORM_FACTOR)
	    ray->visible == 0.0;
	  Store_HitInter(hit, optr, t, x, y, 1.);
	  if (!ray->shadow) {
	    Store_HitNorm(hit, 0. , 0. ,1.);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }

  /* check bottom face */
  if (Option.visibility == FORM_FACTOR)
    if (ray->visible == 0.0)
      return;
  if (Z != ZERO) {
    t= (-1.-ray->origin.z)/ray->dir.z;
    if (t > MIN_DISTANCE && t < hit->distance) {
      x= ray->origin.x + t*ray->dir.x;
      if (x > -1.0 && x < 1.0) {
	y= ray->origin.y + t*ray->dir.y;
	if (y > -1.0 && y < 1.0) {

	  /* Store hit data */
	  RayStats.intCube++;
	  if (Option.visibility == FORM_FACTOR)
	    ray->visible == 0.0;
	  Store_HitInter(hit, optr, t, x, y, -1.);
	  if (!ray->shadow) {
	    Store_HitNorm(hit, 0., 0., -1.);
	    hit->texture= hit->intersect;
	  }
	}
      }
    }
  }
}

