/** @file tree.h
 * Directory tree handling and file system utility functions
 * @author Marko Mkel <msmakela@nic.funet.fi>
 */

/* Copyright  2003 Marko Mkel.

   This file is part of PHOTOMOLO, a program for generating
   thumbnail images and HTML files for browsing digital photographs.

   PHOTOMOLO is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   PHOTOMOLO is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   The GNU General Public License is often shipped with GNU software, and
   is generally kept in a file called COPYING or LICENSE.  If you do not
   have a copy of the license, write to the Free Software Foundation,
   59 Temple Place, Suite 330, Boston, MA 02111 USA. */

#include <sys/types.h>

/** Image file entry */
struct imagefile
{
  /** file name */
  char* name;
  /** modification time */
  time_t mtime;
};

/** Directory tree */
struct dirtree
{
  /** path name of the directory */
  char* name;
  /** modification time of last image contained in the subtree */
  time_t mtime;
  /** number of subdirectories */
  unsigned n_dirs;
  /** number of files */
  unsigned n_files;
  /** array of subdirectories */
  struct dirtree* dirs;
  /** array of files */
  struct imagefile* files;
};

/** Initialize a directory tree structure
 * @param tree	the directory tree object
 * @param name	the path name of the directory
 */
void
inittree (struct dirtree* tree,
	  const char* name);

/** Read a directory tree
 * @param tree	the destination tree (with the path name filled in)
 * @param dir_p	callback for pruning directories by name
 * @return	zero on success, nonzero on failure
 */
int
readtree (struct dirtree* tree,
	  int (*dir_p) (const char*));

/** Deallocate a directory tree
 * @param tree	the directory tree
 */
void
freetree (struct dirtree* tree);

/** Determine if the specified file exists
 * @param name	the path name of the file
 * @param mtime	the modification time
 * @return	nonzero if the file exists with the given date stamp
 */
int
exists_mtime (const char* name,
	      time_t mtime);

/** Touch a file to the specified modification time
 * @param name	the path name of the file
 * @param mtime	the modification time
 * @return	zero on success, nonzero on failure
 */
int
touch_mtime (const char* name,
	     time_t mtime);

/** Create a directory
 * @param path	the path name of the directory
 * @return	zero on success, nonzero on failure
 */
int
make_directory (const char* name);

/** Remove a directory tree
 * @param path	the path name of the directory
 * @return	zero on success, nonzero on failure
 */
int
remove_directory (const char* name);
