

/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: yexparse.c
*
*  Purpose: To read and parse user functions in algebraic form
*           for evolver constraints.  Expressons are given
*           in algebraic form and parsed with yacc (see express.yac
*           and y.tab.c).
*           See express.c for evaluation functions.
*              +,-,*,/  (replace top two stack entries with result)
*              ^<n>     (integer power)
*              sin,cos,tan,log,exp,sqr,sqrt,acos,asin,atan
*              pow      (real power)
*              
*/

#include "include.h" 
#include "express.h"
#include "lex.h"
#include "ytab.h"
  

/*****************************************************************
*
*  Function exparse()
*
*  Purpose: reading, parsing, checking of algebraic function definition.
*           Takes tokens from yylex() until non-expression token.
*           Function value is size in bytes of tree.
*
*/

#define LISTMAX 500

struct treenode *list;   /* tree */
NTYPE listtop;  /* first spot reserved for root */
NTYPE maxp;

int exparse(maxparam,root)
int maxparam;  /* maximum number of parameters allowed */
struct treenode **root;   /* pointer to storage pointer */
{
  int retval;
  char *c;

  maxp = (NTYPE)maxparam;
  *root = list = (struct treenode *)mycalloc(LISTMAX,sizeof(struct treenode));
  listtop = 1;
  parse_error_flag = 0;
  parens = 0;
  retval = yyparse();  /* 0 for accept, 1 for error */
 
  if ( tok != 0 )
    {
      /* push back last token */
      for ( c = yytext ; *c ; c++ ); /* end of token */
      while ( c != yytext ) { --c; unput(*c); }
      switch ( tok ) /* these tokens are at start of line */
       { case LEAD_INTEGER_ :
         case ENVECT_:
         case BCOORD_:
         case CONVECT_:
          unput('\n'); /* so will recognize start of line */
          break;
       }
    }

  if ( (retval == 1) || parse_error_flag  || (listtop == 1) )
    { free((char *)*root); return -1; }

  list[0] = list[listtop-1];  /* root also in first spot */
  list[0].left += listtop - 1;
  list[0].right += listtop - 1;

  /* put DONE marker after root */
  list[listtop++].type = FINISHED;

  /* free excess list */
  *root = (struct treenode *)kb_realloc((char *)list,
                                      listtop*sizeof(struct treenode));
  
#ifdef TREE_EVAL
  /* fold constants */
  ex_fold(*root);
#endif

  return listtop*sizeof(struct treenode);

}

/*********************************************************************
*
*  Function: free_expr()
*
*  Purpose:  Deallocate lists of expression.
*
*/

void free_expr(ex)
struct expnode *ex;
{
  if ( ex == NULL ) return;
  if ( ex->root ) free((char *)ex->root);
  if ( ex->evallist ) free((char *)ex->evallist);
  ex->root = NULL;
  ex->evallist = NULL;
}

/*********************************************************************
*
*  Function:  makenode()
*
*  Purpose: Add nodes to parse tree when called by yyparse.
*
*  Return:  Index of created node.
*/

int makenode(type,left,right)
NTYPE type;  /* type of node */
NTYPE left;  /* left son, if any */
NTYPE right; /* right son, if any */
{
  NTYPE n;

  switch ( type )
    {

      case PUSHCONST:
          list[listtop].type = PUSHCONST;
          list[listtop].value = real_val;
          break;

      case PUSHPI:
          list[listtop].type = type;
          list[listtop].value = M_PI;
          break;

      case PUSHE:
          list[listtop].type = type;
          list[listtop].value = M_E;
          break;

      case PUSHG:
          list[listtop].type = type;
          break;

      case EXP:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = exp(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case LOG:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = log(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case ABS:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = fabs(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case SIN:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = sin(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case ASIN:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = asin(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case COS:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = cos(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case ACOS:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = acos(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case TAN:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = tan(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case ATAN:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = atan(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case SQR:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = list[listtop].value*list[listtop].value;
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case SQRT:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = sqrt(list[listtop].value);
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case CHS:
          if ( is_constant(left) )
            { /* fold constants */
              listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].value = -list[listtop].value;
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case POW:
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = pow(list[left].value,list[right].value);
              list[listtop].type = PUSHCONST;
              break;
            }
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;

      case '+':
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = list[left].value + list[right].value;
              list[listtop].type = PUSHCONST;
            }
          else
            {
              list[listtop].right = right - listtop;
              list[listtop].left = left - listtop;
              list[listtop].type = PLUS;
            }
          break;

      case '-':
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = list[left].value - list[right].value;
              list[listtop].type = PUSHCONST;
            }
          else
            {
              list[listtop].right = right - listtop;
              list[listtop].left = left - listtop;
              list[listtop].type = MINUS;
            }
          break;

      case '=':
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = list[left].value - list[right].value;
              list[listtop].type = PUSHCONST;
            }
          else
            {
              list[listtop].right = right - listtop;
              list[listtop].left = left - listtop;
              list[listtop].type = EQUATE;
            }
          break;

      case '*':
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = list[left].value * list[right].value;
              list[listtop].type = PUSHCONST;
            }
          else
            {
              list[listtop].right = right - listtop;
              list[listtop].left = left - listtop;
              list[listtop].type = TIMES;
            }
          break;

      case '/':
          if ( is_constant(right) && is_constant(left) )
            { /* fold constants */
              
              listtop -= 2;
              if ( (left != listtop) || (right != listtop+1) )
                error("Constant folding not working.",UNRECOVERABLE);
              list[listtop].value = list[left].value / list[right].value;
              list[listtop].type = PUSHCONST;
            }
          else
            {
              list[listtop].right = right - listtop;
              list[listtop].left = left - listtop;
              list[listtop].type = DIVIDE;
            }
          break;

      case '^':
          n = (NTYPE)atoi(yytext+1);
          if ( n == 0 )
            { list[listtop].type = REPLACECONST;
              list[listtop].value = 1.0;
              break;
            }
          else if ( n == 1 )
            { /* leave alone */
              listtop--;
              break;
            }
          if ( is_constant(left) )
            { /* fold constants */
              int k;
              REAL x;

              listtop--;
              list[listtop].type = PUSHCONST;
              if ( left != listtop )
                error("Constant folding not working.",UNRECOVERABLE);
              x = list[left].value;
              for ( k = 1 ; k < abs(n) ; k++ )
                list[left].value *= x;
              if ( n < 0 )
                list[left].value = 1/list[left].value;
              break;
            }
          list[listtop].left = left - listtop;
          list[listtop].inx = n;
          list[listtop].type = INTPOW;
          break;

      case PUSHPARAM:
          if ( maxp == 0 )
            {
              error("Constant expression required.\n",EXPRESSION_ERROR);
            }
          list[listtop].type = PUSHPARAM;
          list[listtop].inx = n = (NTYPE)int_val-1;
          if ( (n < 0) || (n >= maxp) )
            {
              sprintf(errmsg,"Invalid parameter number: %d\n",n+1);
              error(errmsg,EXPRESSION_ERROR);
            }
          break;

      case USERFUNC:
	  list[listtop].type = USERFUNC;
	  list[listtop].inx = (NTYPE)int_val-1;
	  if ( int_val < 1 || (userfunc[int_val-1] == NULL) )
	    { sprintf(errmsg,"Invalid user function number: %d\n",int_val);
	      error(errmsg,EXPRESSION_ERROR);
	    }    
	  break;

      case COORD_:
	  list[listtop].type = COORD_;
	  list[listtop].inx = (NTYPE)int_val-1;
	  break;

      case PUSHADJUSTABLE:
          list[listtop].type = PUSHADJUSTABLE;
          for ( n = 0 ; n < (NTYPE)web.paramcount ; n++ )
            if ( strncmp(yytext,web.params[n].name,31) == 0 ) break;
          if ( n == (NTYPE)web.paramcount )
            {
              sprintf(errmsg,"Invalid identifier: %s\n",yytext);
              error(errmsg,EXPRESSION_ERROR);
            }
          list[listtop].inx = n;
          break;

       case ATTRIBUTE:
	   list[listtop].type = (NTYPE)tok;
	   break;

       case EQ_:
       case NE_:
       case GE_:
       case LE_:
       case '>':
       case '<':
       case AND_:
       case OR_:
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = type;
	  break;

       case NOT_:
          list[listtop].left = left - listtop;
          list[listtop].type = type;
          break;
     }

  if ( listtop > LISTMAX-2 )
         error("Expression too long",UNRECOVERABLE);

  return listtop++;
}

/******************************************************************
*
*  Function: is_constant()
*
*  Purpose:  See if tree node type is a constant value.
*/

int is_constant(node)
int node;
{
  switch(list[node].type)
    {  case PUSHPI:
       case PUSHE:
       case PUSHCONST:
         return 1;
       default:
         return 0;
    }
}


/*****************************************************************
*
*  Function print_express()
*
*  Purpose: print expression in algebraic format
*
*/

static char *pos;  /* current position in string */
static char vch;
static int  strsize;
static char *strstart;

/* precedence levels for knowing how to parenthesize */
#define PREC_POW     50
#define PREC_UMINUS  45
#define PREC_MUL     40
#define PREC_DIV     40
#define PREC_ADD     35
#define PREC_SUB     35
#define PREC_COMP    30
#define PREC_NOT     25
#define PREC_AND     20
#define PREC_OR      15
#define PREC_ASSIGN  10
#define PREC_ARG      0

void print_express(node,str,c,size)
struct expnode *node;      /* expression tree */
char *str;  /* remaining string */
int c;    /* character for parameters */
int  size; /* max length of string */
{
  pos = strstart = str;
  vch = (char)c;
  strsize = size;

  if ( node == NULL ) { strcpy(str,"NULL"); return;}

  exprint_recur(node->root,0);
}

void exprint_recur(node,prec_parent)
struct treenode *node;
int prec_parent;  /* precedence level of parent, for parenthesizing */
{
    /* check room remaining */
    if ( (pos - strstart) > (strsize - 32) )
      error("Expression string too long.\n",RECOVERABLE);

    switch ( node->type )
      {
        case PUSHCONST:
          sprintf(pos,"%1.15g",node->value);
          pos += strlen(pos);
          return;

        case PUSHPI:
          sprintf(pos,"PI");
          pos += strlen(pos);
          return;

        case PUSHE:
          sprintf(pos,"E");
          pos += strlen(pos);
          return;

        case PUSHG:
          sprintf(pos,"G");
          pos += strlen(pos);
          return;

        case PUSHPARAM:
          sprintf(pos,"%c%d",vch,node->inx+1);
          pos += strlen(pos);
          return;

        case PUSHADJUSTABLE:
          sprintf(pos,"%s",web.params[node->inx].name);
          pos += strlen(pos);
          return;

        case USERFUNC:
          sprintf(pos,"usr%d",node->inx+1);
          pos += strlen(pos);
          return;

        case PLUS:
          if ( prec_parent > PREC_ADD ) 
             { sprintf(pos,"(");
               pos += strlen(pos);
             }
          exprint_recur(node+node->left,PREC_ADD);
          sprintf(pos," + ");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_ADD);
          if ( prec_parent > PREC_ADD )
            { sprintf(pos,")");
              pos += strlen(pos);
            }
          return;

        case MINUS:
          if ( prec_parent > PREC_SUB )
            { sprintf(pos,"(");
              pos += strlen(pos);
            }
          exprint_recur(node+node->left,PREC_SUB);
          sprintf(pos," - ");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_SUB+1);
          if ( prec_parent > PREC_SUB )
            { sprintf(pos,")");
              pos += strlen(pos);
            }
          return;

        case EQUATE:
          exprint_recur(node+node->left,PREC_ASSIGN);
          sprintf(pos," = ");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_ASSIGN+1);
          return;

        case TIMES:
          if ( prec_parent > PREC_MUL )
            { sprintf(pos,"(");
              pos += strlen(pos);
            }
          exprint_recur(node+node->left,PREC_MUL);
          sprintf(pos,"*");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_MUL);
          if ( prec_parent > PREC_MUL )
            { sprintf(pos,")");
              pos += strlen(pos);
            }
          return;

        case DIVIDE:
          if ( prec_parent > PREC_DIV )
            { sprintf(pos,"(");
              pos += strlen(pos);
            }
          exprint_recur(node+node->left,PREC_DIV);
          sprintf(pos,"/");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_DIV+1);
          if ( prec_parent > PREC_DIV )
           { sprintf(pos,")");
             pos += strlen(pos);
           }
          return;

        case INTPOW:
          exprint_recur(node+node->left,PREC_POW);
          sprintf(pos,"^%1d",node->inx);
          pos += strlen(pos);
          return;

        case POW:
          if ( prec_parent > PREC_POW )
            { sprintf(pos,"(");
              pos += strlen(pos);
            }
          exprint_recur(node+node->left,PREC_POW);
          sprintf(pos,"**");
          pos += strlen(pos);
          exprint_recur(node+node->right,PREC_POW);
          if ( prec_parent > PREC_POW )
            { sprintf(pos,")");
              pos += strlen(pos);
            }
          return;

        case SQR:
          sprintf(pos,"sqr(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case SQRT:
          sprintf(pos,"sqrt(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case ABS:
          sprintf(pos,"abs(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case SIN:
          sprintf(pos,"sin(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case COS:
          sprintf(pos,"cos(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case TAN:
          sprintf(pos,"tan(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case EXP:
          sprintf(pos,"exp(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case LOG:
          sprintf(pos,"log(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case ASIN:
          sprintf(pos,"asin(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case ACOS:
          sprintf(pos,"acos(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;

        case ATAN:
          sprintf(pos,"atan(");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_ARG);
          sprintf(pos,")");
          pos += strlen(pos);
          return;
        
        case CHS:
          sprintf(pos,"-");
          pos += strlen(pos);
          exprint_recur(node+node->left,PREC_UMINUS);
          return;
        
	default: 
	  sprintf(pos,"(???)");
	  pos += strlen(pos);
	  sprintf(errmsg,"Printing of expression node type %d unimplemented.\n",
	     node->type);
	  error(errmsg,WARNING);
	  return;
    }
  return ; /* shouldn't happen */
}

#ifdef TREE_EVAL
/*****************************************************************
*
*  Function ex_fold()
*
*  Purpose: Constant folding in expressions.  Replaces all
*           nodes whose descendents don't involve parameters
*           with evaluated constants.
*
*/

void ex_fold(node)
struct treenode *node;      /* expression tree */
{
  int flag = 0;
 
  fold_recur(node,&flag);
}
#endif

#ifdef TREE_EVAL
void fold_recur(node,flag)
struct treenode *node;      /* expression tree */
int *flag;     /* set to 1 when node is not constant */
{
  int lflag = 0;

  if ( node == NULL ) { *flag = 1; return; }

    switch ( node->type )
      {
        case PUSHCONST:
        case PUSHPI:
        case PUSHE:
          break;

        case PUSHG:
        case PUSHPARAM:
        case PUSHADJUSTABLE:
	case COORD_:
	case LENGTH_:
	case VALENCE_:
	case AREA_:
	case VOLUME_:
	case DENSITY_:
	case ID_:
	case TAG_:
	case ORIGINAL_:
	case FIXED_:
          lflag = 1;
          break;

       case EQ_:
       case NE_:
       case GE_:
       case LE_:
       case '>':
       case '<':
       case AND_:
       case OR_:
        case PLUS:
        case MINUS:
        case EQUATE:
        case TIMES:
        case DIVIDE:
        case POW:
          fold_recur(node+node->left,&lflag);
          fold_recur(node+node->right,&lflag);
          break;

        case INTPOW:
        case SQR:
        case SQRT:
	case ABS:
        case SIN:
        case COS:
        case TAN:
        case EXP:
        case LOG:
        case ASIN:
        case ACOS:
        case ATAN:
        case CHS:
        case INV:
          fold_recur(node+node->left,&lflag);
          break;
       
	default:
	  fprintf(stderr,"fold_recur: Invalid node type: %d\n",node->type);
	  exit(2);
	  break;

    }
     
  if ( lflag == 0 )
    { 
          node->value = tree_eval(node,NULL);
          node->type = PUSHCONST;
    }
  else *flag = 1;

}
#endif
