/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/********************************************************************
*
*  File: storagex.h
*
*  Purpose:  Header file defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (for inclusion in other source files that need to know)
*            or in storage.c (purely private details).
*
*     This version has element ids typed as longs.
*     Also implements id as offset from element base.
*     Elements allocated in permanent blocks.
*/

#define NUMELEMENTS 5

/* these values used for identification and as index into skeleton info */
#define  VERTEX  0
#define  EDGE    1
#define  FACET   2
#define  BODY    3
#define  FACETEDGE    4 

/*****************************************************************
*
*  Universal element identifier.  Don't want to use straight
*  pointer since base changes when realloced.
*  (not actually used; just for documentation)
*/

typedef struct xelement_id {
    unsigned int  type : 3;   /* see enum below */
    unsigned int  valid: 1;   /* valid id bit */
    unsigned int  sign : 1;   /* set for reverse orientation */
    unsigned int  offset: 27;   /* offset from block start */
    } xelement_id;


/* masks for fields */
#define TYPEMASK   0xE0000000
#define VALIDMASK  0x10000000
#define SIGNMASK   0x08000000
#define OFFSETMASK 0x07FFFFFF

/* shifts for fields */
#define TYPESHIFT  29
#define VALIDSHIFT 28
#define SIGNSHIFT  27

#define NULLID 0L 

/* to get type of an element */
#define id_type(id)  ((int)(((id)&TYPEMASK)>>TYPESHIFT))

/* to give switched orientation of first if that of second is inverted */
#define same_sign(id1,id2)    ((id1) ^ ((id2) & SIGNMASK))

/* number of elements to allocate memory for at one time */
#define BATCHSIZE 100

/* outside storage.*, element_id structure is not visible; acts like long */    
typedef long
     element_id, vertex_id, edge_id, facet_id, body_id, facetedge_id; 

/* macros for getting structure pointer from id */
#define vptr(v_id) ((struct vertex *)((v_id)&OFFSETMASK))
#define eptr(e_id) ((struct edge   *)((e_id)&OFFSETMASK))
#define fptr(f_id) ((struct facet  *)((f_id)&OFFSETMASK))
#define bptr(b_id) ((struct body   *)((b_id)&OFFSETMASK))
#define feptr(fe_id) ((struct facetedge *)((fe_id)&OFFSETMASK))

/* id attr bits */
#define VALID_BIT   0x0001
#define INVERSE_BIT 0x0001

#define edge_inverse(id)  inverse_id(id)
#define facet_inverse(id)  inverse_id(id)
#define fe_inverse(id)  inverse_id(id)
#define invert(id)     ((id) ^= SIGNMASK)
#define equal_id(a,b)  ((a)==(b))
#define equal_element(a,b)  (((a)|SIGNMASK) == ((b)|SIGNMASK))
#define valid_id(id)   ((id)&VALIDMASK)
#define inverted(id)   ((id)&SIGNMASK)
#define inverse_id(id) ((id) ^ SIGNMASK)
typedef int ORDTYPE;                /* element numbering type */

/* base structure for a skeleton of a particular dimension */
struct skeleton {
    int             type;   /* type of element, see defines above */
    char           *base;   /* to chain of structure block */
    long             maxcount; /* elements allocated          */
    element_id      free;   /* start of free list        */
    element_id      used;   /* start of in-use elements  */
    element_id      last;   /* end of in-use chain, for adding on */
    element_id      discard; /* start of discard list */
    long            count;  /* number active             */
    element_id      current; /* current element in generation */
    ORDTYPE         max_ord; /* highest ordinal */
  } ;

