
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: pixgraph.c
*
*  Purpose: Creates data file for Pixar from current triangulation.
*           Hooks directly to graphgen().
*
*/

#include "include.h"
#include "gobject.h"


static FILE *pfd = NULL;
static  REAL **verts; /* for adjusted triangle vertices */

/****************************************************************************
*
*  Function: pix_start()
*
*  Purpose:  Initialize Pixar output file.
*/


void pix_start()
{
  char name[50];       /* base picture name  */
  char quadname[60];   /* quadrilateral file */
  char response[120];
  char *tptr;

  prompt("Enter name of picture: ",name);
  prompt("Do normal interpolation? ",response);
  if ( toupper(response[0]) == 'Y' ) normflag = 1;
    else normflag = 0;
  prompt("Do inner, outer, or all surfaces? (i,o,a)",response);
  switch ( response[0] )
    {
      case 'i' : innerflag = 1; outerflag = 0; break;
      case 'o' : innerflag = 0; outerflag = 1; break;
      default  : innerflag = 1; outerflag = 1; break;
    }
  prompt("Do body colors? ",response);
  if ( toupper(response[0]) == 'Y' )
     colorflag = 1;
  else colorflag = 0;

  thickness = overall_size/1000;
  sprintf(msg,"Thicken(n | y [thickness(%g)])? ",thickness);
  prompt(msg,response);
  if ( toupper(response[0]) == 'Y' )
    { thickenflag = 1;
      strtok(response," \t,;:");
      tptr = strtok(NULL," \t;:,");
      if ( tptr )
         thickness = atof(tptr);
    }
  else thickenflag = 0;

  strcpy(quadname,name);
  strcat(quadname,".quad");
  pfd = fopen(quadname,"w"); 
  if (pfd == NULL) { perror(quadname); return; } 

  if ( normflag ) 
       { fprintf(pfd,"CNPOLY");
       }
  else 
       { fprintf(pfd,"CPOLY");
       }
  fprintf(pfd,"\n");
  verts = dmatrix(0,2,0,2);
}

/************************************************************************
*
*  Function: pix_facet()
*
*  Purpose:  Accepts facets from graphgen() and plots them.
*/

void pix_facet(gdata,f_id)
struct graphdata *gdata;
facet_id f_id;
{
  int i,j;
  REAL map[4];  /* color of this vertex */
  int color = get_facet_color(f_id);

  if ( color == CLEAR ) return;
  if ( colormap )
     for ( i = 0 ; i < 4 ; i++ ) map[i] = colormap[gdata[0].color][i];
  else  {
	  map[0] = iris_colors[color].r;
	  map[1] = iris_colors[color].g;
	  map[2] = iris_colors[color].b;
	  map[3] = facet_alpha;
	}
          
  if ( thickenflag )
    for ( i = 0 ; i < FACET_VERTS ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
        gdata[i].x[j] += gdata[i].norm[j]*thickness;

  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { 
      fprintf(pfd,"%17.15f %17.15f %17.15f  ",gdata[i].x[0],
         gdata[i].x[1],gdata[i].x[2]); 
      if ( normflag )
        { fprintf(pfd,"%5.3f %5.3f %5.3f    ",gdata[i].norm[0],
                 gdata[i].norm[1],gdata[i].norm[2]);
        }
       fprintf(pfd,"%5.3f %5.3f %5.3f %5.3f   ",
               map[0],map[1],map[2],map[3]);
    }

  i = FACET_VERTS-1;  /* repeat last point */ 
  fprintf(pfd,"%17.15f %17.15f %17.15f  ",gdata[i].x[0],
     gdata[i].x[1],gdata[i].x[2]); 
  if ( normflag )
    { fprintf(pfd,"%5.3f %5.3f %5.3f    ",gdata[i].norm[0],
             gdata[i].norm[1],gdata[i].norm[2]);
    }
  fprintf(pfd,"%5.3f %5.3f %5.3f %5.3f   ",
           map[0],map[1],map[2],map[3]);

  fprintf(pfd,"\n");

  if ( thickenflag )
   { /* do other orientation */
    for ( i = 0 ; i < FACET_VERTS ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
        gdata[i].x[j] -= gdata[i].norm[j]*2*thickness;

    for ( i = FACET_VERTS-1 ; i >=  0 ; i-- )
     { 
      fprintf(pfd,"%17.15f %17.15f %17.15f  ",gdata[i].x[0],
         gdata[i].x[1],gdata[i].x[2]); 
      if ( normflag )
        { fprintf(pfd,"%5.3f %5.3f %5.3f    ",-gdata[i].norm[0],
                 -gdata[i].norm[1],-gdata[i].norm[2]);
        }
      fprintf(pfd,"%5.3f %5.3f %5.3f %5.3f   ",map[0],map[1],map[2],map[3]);
     }

    i = 0;  /* repeat last point */ 
    fprintf(pfd,"%17.15f %17.15f %17.15f  ",gdata[i].x[0],
       gdata[i].x[1],gdata[i].x[2]); 
    if ( normflag )
      { fprintf(pfd,"%5.3f %5.3f %5.3f    ",-gdata[i].norm[0],
             -gdata[i].norm[1],-gdata[i].norm[2]);
      }
    fprintf(pfd,"%5.3f %5.3f %5.3f %5.3f   ",map[0],map[1],map[2],map[3]);
   } /* end thickenflag */

  fprintf(pfd,"\n");
}

/*******************************************************************
*
*  Function: pix_end()
*
*  Purpose:  Finish off Pixar file.
*
*/

void pix_end()
{
  fclose(pfd);
  free_matrix(verts);
}





