/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/*************************************************************
*
*    file:      dump.c
*
*    Contents:  Functions for making ASCII dump file of surface.
*/

/********************************************************
*
*  Function: dump()
*
*  Purpose: Dumps configuration to ASCII file in format
*           suitable for read-in.
*
*/

#include "include.h"

/* max length of expression printout */
#define MAXEXPRSIZE 1000

void dump()
{
  char name[70];

  prompt("Enter name of dump file: ",name);
  if ( name[0] == '\0' ) return;
  do_dump(name);
}

void top_dump(fd)
FILE *fd;
{
  int i,j;

  /* model info dump */
  if ( web.simplex_flag ) fprintf(fd,"SIMPLEX_REPRESENTATION\n");
  if ( web.sdim != 3 ) 
     fprintf(fd,"SPACE_DIMENSION %d\n",web.sdim);
  if ( web.dimension == STRING ) fprintf(fd,"STRING\n\n");
  else if ( web.dimension == SOAPFILM ) fprintf(fd,"SOAPFILM\n\n");
  else fprintf(fd,"SURFACE_DIMENSION %d\n\n",web.dimension);
  if ( web.modeltype == LINEAR ) fprintf(fd,"LINEAR\n\n");
  else fprintf(fd,"QUADRATIC\n\n");
  if ( web.symmetry_flag && !web.torus_flag )
    fprintf(fd,"SYMMETRY_GROUP \"%s\"\n\n",symmetry_name);
  if ( web.full_flag ) fprintf(fd,"TORUS_FILLED\n");
  else if ( web.torus_flag ) fprintf(fd,"TORUS\n");
  if ( web.torus_flag || web.inverse_periods )
    { fprintf(fd,"PERIODS\n");
      for ( i = 0 ; i < web.sdim ;i++ )
        { for ( j = 0 ; j < web.sdim ; j++ )
            fprintf(fd,"%17.15f  ",web.torus_period[i][j]);

          fprintf(fd,"\n");
        }
      fprintf(fd,"\n");
    }
  if ( web.meritfactor != 1.0 )
    fprintf(fd,"MERIT_FACTOR: %20.15g\n\n",web.meritfactor);
  if ( web.gravflag )
    fprintf(fd,"GRAVITY_CONSTANT: %20.15g\n\n",web.grav_const);
  if ( web.diffusion_const != 0.0 )
    fprintf(fd,"DIFFUSION: %20.15g\n\n",web.diffusion_const);
  if ( autochop_flag )
    fprintf(fd,"AUTOCHOP %20.15g\n\n",autochop_size);
  if ( autopop_flag )
    fprintf(fd,"AUTOPOP\n\n");
  if ( effective_area_flag )
    fprintf(fd,"EFFECTIVE_AREA\n\n");
  if ( total_time > 0.0 )
    fprintf(fd,"TOTAL_TIME %f\n\n",total_time);
  if ( runge_kutta_flag )
    fprintf(fd,"RUNGE_KUTTA\n\n");
  if ( square_curvature_flag )
    fprintf(fd,"SQUARE_CURVATURE: %20.15g\n\n",
	 web.params[square_curvature_param].value);
  if ( web.wulff_flag )
    fprintf(fd,"WULFF: \n     %s\n\n",web.wulff_name);
  if ( phase_flag )
    fprintf(fd,"PHASEFILE \"%s\"\n\n",phase_file_name);
  if ( web.spring_constant != 1.0 )
    fprintf(fd,"GAP_CONSTANT: %20.15g\n\n",web.spring_constant);
  if ( web.motion_flag )
    fprintf(fd,"SCALE: %20.15g    FIXED\n\n",web.scale);
  else if ( web.scale != 0.1 )
    fprintf(fd,"SCALE: %20.15g\n\n",web.scale);
  if ( web.area_norm_flag )
    fprintf(fd,"AREA_NORMALIZATION \n\n");
  if ( web.jiggle_flag )
    fprintf(fd,"JIGGLE\n\n");
  if ( web.temperature != 0.05 )
    fprintf(fd,"TEMPERATURE: %20.15g\n\n",web.temperature);
  if ( web.pressure != 0.0 )
    fprintf(fd,"PRESSURE: %20.15g\n\n",web.pressure);
  if ( web.maxscale != 1.0 )
    fprintf(fd,"SCALE_LIMIT: %17.15g\n\n",web.maxscale);
  if ( web.gauss_order != 2 )
    fprintf(fd,"INTEGRAL_ORDER: %d\n\n",web.gauss_order);
  if ( web.concount )
    fprintf(fd,"CONSTRAINT_TOLERANCE: %17.15g\n\n",web.tolerance);
  if ( web.symmetric_content )
    fprintf(fd,"SYMMETRIC_CONTENT\n\n");
  if ( ordinal(web.zoom_v) != 0 ) 
    fprintf(fd,"ZOOM_VERTEX  %d\n\n",ordinal(web.zoom_v)+1);
  if ( web.zoom_radius < 9000.0 )
    fprintf(fd,"ZOOM_RADIUS  %17.15g\n\n",web.zoom_radius);

  /* adjustable parameters */
  for ( i = 0 ; i < web.paramcount ; i++ )
    { if ( square_curvature_flag && (i == square_curvature_param) ) continue;
      fprintf(fd,"PARAMETER %s ",web.params[i].name);
      if ( web.params[i].flags & FILE_VALUES )
	{ if ( level == 0 )
	    fprintf(fd,"PARAMETER_FILE \"%s\" ",web.params[i].value_file);
	  else  fprintf(fd,"PARAMETER_FILE \"%s\" ","not dumped");
	}
      else
	fprintf(fd,"= %17.15g",web.params[i].value);
      fprintf(fd,"\n");
    }

  if ( web.conformal_flag )
    { char str[MAXEXPRSIZE];
      fprintf(fd,"CONFORMAL_METRIC\n");
      /* metric expression */
      print_express(&web.metric[0][0],str,'X',sizeof(str));
      fprintf(fd," %s\n\n",str);
    }
  else if ( web.metric_flag )
    { fprintf(fd,"METRIC\n");
      /* metric expressions */
        for ( i = 0 ; i < web.sdim ; i++ )
          { for ( j = 0 ; j < web.sdim ; j++ )
              { 
                char str[MAXEXPRSIZE];
	        print_express(&web.metric[i][j],str,'X',sizeof(str));
                fprintf(fd," %s,",str);
              }
	    fprintf(fd,"\n");
	  }
       fprintf(fd,"\n");
    }

  /* quantity integrand info */
  for ( i = 0 ; i < QUANTMAX ; i++ )
    { struct quantity *quan = web.quants + i;
      char str[MAXEXPRSIZE];

      if ( quan->quanvect[0] == NULL ) continue;
      fprintf(fd,"\nQUANTITY %d  ",i);
      if ( quan->attr & GLOBAL      ) fprintf(fd,"  GLOBAL");
      if ( quan->attr & QFIXED      )
         fprintf(fd,"  FIXED = %17.15g",quan->target);
      fprintf(fd,"\n");
      for ( j = 0 ; j < web.sdim ; j++ )
            { print_express(quan->quanvect[j],str,'X',sizeof(str));
              fprintf(fd,"Q%1d: %s \n",j+1,str);
            }
     } 

  /* print boundary info */
  for ( i = 0 ; i < BDRYMAX ; i++ )
    { struct boundary * bdry = web.boundaries + i;
      if ( bdry->pcount == 0 ) continue;
      fprintf(fd,"\nBOUNDARY %d  PARAMETERS %d",i,bdry->pcount);
      if ( bdry->attr & B_CONVEX ) fprintf(fd,"   CONVEX");
      fprintf(fd,"\n");
      for ( j = 0 ; j < web.sdim ; j++ )
        { char str[MAXEXPRSIZE];
          print_express(bdry->coordf[j],str,'P',sizeof(str));
          fprintf(fd,"X%1d: %s \n",j+1,str);
        }
      /* boundary integral dumps go here when implemented */
    }

  /* print constraint info */
  for ( i = 0 ; i < CONSTRMAX ; i++ )
    { struct constraint *con = web.constraints + i;
      char str[MAXEXPRSIZE];
      int k;

      if ( con->formula == NULL ) continue;
      fprintf(fd,"\nCONSTRAINT %d  ",i);
      if ( con->attr & B_CONVEX ) fprintf(fd,"  CONVEX");
      if ( con->attr & NONNEGATIVE ) fprintf(fd,"  NONNEGATIVE");
      if ( con->attr & NONPOSITIVE ) fprintf(fd,"  NONPOSITIVE");
      if ( con->attr & GLOBAL      ) fprintf(fd,"  GLOBAL");
      fprintf(fd,"\n");
      print_express(con->formula,str,'X',sizeof(str));
      fprintf(fd,"FUNCTION:  %s \n",str);
      if ( con->attr & BDRY_ENERGY )
        {
          fprintf(fd,"ENERGY \n");

          for ( j = 0 ; j < con->compcount ; j++ )
            { print_express(con->envect[j],str,'X',sizeof(str));
              fprintf(fd,"E%1d: %s \n",j+1,str);
            }
        }
      if ( con->attr & BDRY_CONTENT )
        {
          fprintf(fd,"CONTENT \n");
          for ( j = 0 ; j < con->compcount ; j++ )
            { print_express(con->convect[j],str,'X',sizeof(str));
              fprintf(fd,"C%1d: %s \n",j+1,str);
            }
        }

      /* quantity integrand info */
      for ( k = 0 ; k < QUANTMAX ; k++ )
        {
          if ( !(con->quantity_map & (1<<k)) ) continue;
          fprintf(fd,"\nQUANTITY %d  \n",k);
          for ( j = 0 ; j < con->compcount ; j++ )
            { print_express(con->quanvect[k][j],str,'X',sizeof(str));
              fprintf(fd,"Q%1d: %s \n",j+1,str);
            }
         } 
     } 

  /* surface energy integrand info */
  for ( i = 0 ; i < SURFENMAX ; i++ )
    { struct surf_energy *surfen = web.surfen + i;
      char str[MAXEXPRSIZE];

      if ( surfen->envect[0] == NULL ) continue;
      fprintf(fd,"\nSURFACE_ENERGY %d  ",i);
      if ( surfen->attr & GLOBAL      ) fprintf(fd,"  GLOBAL");
      fprintf(fd,"\n");
      for ( j = 0 ; j < web.sdim ; j++ )
            { print_express(surfen->envect[j],str,'X',sizeof(str));
              fprintf(fd,"E%1d: %s \n",j+1,str);
            }
     } 
}

void vertex_dump(v_id,fd)
vertex_id v_id;
FILE *fd;
{ int i;
  REAL *x;
      
      if ( !valid_id(v_id) ) return;
      x = get_coord(v_id);
      if ( get_vattr(v_id) & BOUNDARY )
        { struct boundary *bdry = get_boundary(v_id);
          REAL *param = get_param(v_id);

          fprintf(fd,"%3d",1+ordinal(v_id));
          for ( i = 0 ; i < bdry->pcount ; i++ )
            fprintf(fd,"  %17.15g",param[i]);
          fprintf(fd,"     boundary %d ",bdry-web.boundaries);
          fprintf(fd," /* (");
          for ( i = 0; i < web.sdim ; i++)
             fprintf(fd," %17.15g",x[i]);
          fprintf(fd,") */");
        }
      else
        {
          fprintf(fd,"%3d ",1+ordinal(v_id));
          for ( i = 0; i < web.sdim ; i++)
             fprintf(fd," %17.15g",x[i]);
        }
      if ( get_vattr(v_id) & CONSTRAINT )
        { MAP conmap = get_v_constraint_map(v_id);
          fprintf(fd,"   constraints ");
          for ( i = 0 ; i < web.concount ; i++,conmap>>=1 )
            if ( (conmap & 1) && !(get_constraint(i)->attr & GLOBAL) )
              fprintf(fd,"%d ",i);
        }
      if ( get_vattr(v_id)&FIXED ) fprintf(fd, "  fixed ");
      fprintf(fd,"\n");
}

void edge_dump(e_id,fd)
edge_id e_id;
FILE *fd;
   { int i;

     if ( !valid_id(e_id) ) return;

     fprintf(fd,"%3d    ",1+ordinal(e_id));
     if ( web.simplex_flag )
	{ vertex_id *v = get_edge_vertices(e_id);
	  for ( i = 0 ; i <= web.dimension-1 ; i++ )
	    fprintf(fd,"%3d  ",ordinal(v[i])+1);
	}
     else
       fprintf(fd,"%3d  %3d   ",
         1+ordinal(get_edge_tailv(e_id)),1+ordinal(get_edge_headv(e_id)));

     if ( web.torus_flag )
       { WRAPTYPE wrap = get_edge_wrap(e_id);
         for ( i = 0 ; i < web.sdim ; i++, wrap >>= TWRAPBITS )
           switch ( wrap & WRAPMASK  )
           {
              case  NEGWRAP : fprintf(fd," -"); break;
              case  0       : fprintf(fd," *"); break;
              case  POSWRAP : fprintf(fd," +"); break;
              default : fprintf(fd," bad"); break;
           }
       }
     if ( web.symmetry_flag && !web.torus_flag )
       { WRAPTYPE wrap = get_edge_wrap(e_id);
	 fprintf(fd," wrap %ld ",wrap);
       }
     if ( web.modeltype == QUADRATIC )
        fprintf(fd,"  /*midpoint %3d*/ ",1+ordinal(get_edge_midv(e_id)));
     if ( (get_eattr(e_id) & DENSITY) || ((web.dimension==STRING) &&
	       (get_edge_density(e_id) != 1.0)))
           fprintf(fd,"   density %1.15g ",get_edge_density(e_id));
     if ( get_eattr(e_id) & BOUNDARY )
        { struct boundary *bdry = get_edge_boundary(e_id);

          if ( get_eattr(e_id) & NEGBOUNDARY )
            fprintf(fd,"     boundary %d ",-(bdry-web.boundaries));
          else
            fprintf(fd,"     boundary %d ",bdry-web.boundaries);
        }               
      if ( get_eattr(e_id) & CONSTRAINT )
        { 
	  MAP conmap = get_e_constraint_map(e_id);
          fprintf(fd,"   constraints ");
          for ( i = 0 ; i < web.concount ; i++,conmap>>=1 )
            { if ( !(conmap & 1) ) continue; 
              if ( get_eattr(e_id) & NEGBOUNDARY )
                fprintf(fd,"%d ",-i);
              else
                fprintf(fd,"%d ",i);
            }
        }
     fprintf(fd," %s",get_eattr(e_id)&FIXED ? "fixed " : "      ");
     if ( get_edge_color(e_id)  != DEFAULT_COLOR )
       fprintf(fd," COLOR %d ",get_edge_color(e_id));
     if ( (fd == outfd) && !web.simplex_flag )
            { calc_edge(e_id);
	      fprintf(fd," length %g ",get_edge_length(e_id));
	    }
     fprintf(fd,"\n");
    }


void facet_dump(f_id,fd)
facet_id f_id;
FILE *fd;
    { int i;
      int per_line = 0; 
      MAP surfenmap;
      MAP quantmap;
      edge_id e_id;
      facetedge_id fe;

      if ( !valid_id(f_id) ) return;
      fprintf(fd,"%3d  ",1+ordinal(f_id));
      if ( web.simplex_flag )
	{ vertex_id *v = get_facet_vertices(f_id);
	  for ( i = 0 ; i <= web.dimension ; i++ )
	    fprintf(fd," %3d",ordinal(v[i])+1);
	}
      else
      {
       generate_facet_fe_init();
       while ( generate_facet_fe(f_id,&fe) ) 
        { e_id = get_fe_edge(fe);
          fprintf(fd," %3d",inverted(e_id) ? -1-ordinal(e_id) :
              1+ordinal(e_id) );
          per_line++;
          if ( per_line >= 10 )
            { fprintf(fd," \\\n           ");
              per_line = 0;
            }
        }
      }
      if ( get_fattr(f_id) & BOUNDARY )
        { struct boundary *bdry = get_facet_boundary(f_id);

          fprintf(fd,"     boundary %d ",bdry-web.boundaries);
        }               
      if ( get_fattr(f_id) & CONSTRAINT )
        { 
	  MAP conmap = get_f_constraint_map(f_id);
          fprintf(fd,"   constraints ");
          for ( i = 0 ; i < web.concount ; i++,conmap >>= 1 )
	    if ( conmap & 1 )
              fprintf(fd,"%d ",i);
        }
      surfenmap = get_f_surfen_map(f_id) & ~web.surf_global_map;
      if ( surfenmap )
        { 
          fprintf(fd,"   energy ");
          for ( i = 0 ; i < web.surfen_count ; i++,surfenmap >>= 1 )
	    if ( surfenmap & 1 )
              fprintf(fd,"%d ",i);
        }
      quantmap = get_f_quant_map(f_id) & ~web.quant_global_map;
      if ( quantmap )
        { 
          fprintf(fd," quantity ");
          for ( i = 0 ; i < web.quantity_count ; i++,quantmap >>= 1 )
	    if ( quantmap & 1 )
              fprintf(fd,"%d ",i);
        }
      if ( get_fattr(f_id) & DENSITY )
           fprintf(fd,"   density %1.15g ",get_facet_density(f_id));
      if ( get_fattr(f_id)&FIXED ) fprintf(fd, "  fixed " );
      if ( get_fattr(f_id) & NODISPLAY )
        fprintf(fd,"  NODISPLAY");
      if ( get_tag(f_id) ) fprintf(fd," tag %d ",get_tag(f_id));
      if ( phase_flag && (web.dimension == STRING) )
       fprintf(fd," PHASE %d ",get_f_phase(f_id));
      if ( get_facet_color(f_id) != DEFAULT_COLOR  )
       fprintf(fd," COLOR %d ",get_facet_color(f_id));
      if ( fd == outfd )
	 fprintf(fd,"  area %g", get_facet_area(f_id));
      fprintf(fd,"\n");
   }


void body_dump(b_id,fd)
body_id b_id;
FILE *fd;
    { 
      REAL den;
      int per_line = 0;
      facet_id f_id;

      if ( !valid_id(b_id) ) return;
      fprintf(fd,"%3d     ",1+ordinal(b_id));
      FOR_ALL_FACETS(f_id)
        { if ( equal_id(get_facet_body(f_id),b_id) )
            { fprintf(fd," %3d",1+ordinal(f_id));
              per_line++;
            }
          if ( equal_id(get_facet_body(facet_inverse(f_id)),b_id) )
            { fprintf(fd," %3d",-1-ordinal(f_id));
              per_line++;
            }
          
          if ( per_line >= 10 )
            { fprintf(fd," \\\n        ");
              per_line = 0;
            }
        }
     if ( get_battr(b_id) & FIXEDVOL )
       { fprintf(fd,"    volume %1.15g  /*actual: %1.15g*/",get_body_fixvol(b_id),
          get_body_volume(b_id));
       }
     if ( (get_body_volconst(b_id) != 0.0) & !web.torus_flag )
        fprintf(fd,"  volconst %1.15g ",get_body_volconst(b_id));
     if ( get_battr(b_id) & PRESSURE ) fprintf(fd,"    pressure %1.15g",
          get_body_pressure(b_id));
     den = get_body_density(b_id);
     if ( den != 0.0 )
       fprintf(fd,"     density %1.15g ",den);
     if ( phase_flag && (web.dimension != STRING) )
       fprintf(fd," PHASE %d ",get_b_phase(b_id));
     fprintf(fd,"\n");
    }

void do_dump(name)
char *name;
{
  FILE *fd;
  vertex_id v_id;
  edge_id e_id;
  facet_id f_id;
  body_id b_id;

  fd = fopen(name,"w");

  if ( fd == NULL )
   { 
     sprintf(errmsg,"Cannot open file %s.\n",name);
     error(errmsg,RECOVERABLE);
   }

  fprintf(fd,"// %s: Dump of structure.\n\n",name);

  top_dump(fd);

  /* vertex dump */
  fputs("\nvertices      /*  coordinates  */   \n",fd);
  FOR_ALL_VERTICES(v_id)
    vertex_dump(v_id,fd);
    
  /* edge dump */
     fputs("\nedges  \n",fd);
     FOR_ALL_EDGES(e_id)
       edge_dump(e_id,fd);

  /* facet dump */
  if ( web.simplex_flag )
    fputs("\nfaces   /* vertex set */     \n",fd);
  else
    fputs("\nfaces   /* edge loop */     \n",fd);
  FOR_ALL_FACETS(f_id)
    facet_dump(f_id,fd);

  /* body dump */
  calc_content();
  fputs("\nbodies  /* facets */\n",fd);
  FOR_ALL_BODIES(b_id)
    body_dump(b_id,fd);

  fputs("\n\n",fd);
  fclose(fd);

}

void facetedge_dump(fe,fd)
facetedge_id fe;
FILE *fd;
{ edge_id e_id;
  facet_id f_id;

     e_id = get_fe_edge(fe);
     f_id = get_fe_facet(fe);
     fprintf(fd,"%05d   %5d  %5d    %08d  %08d    %08d  %08d\n",
     1+ordinal(fe), inverted(e_id) ? -1-ordinal(e_id) : 1+ordinal(e_id),
         inverted(f_id) ? -1-ordinal(f_id) : 1+ordinal(f_id),
         1+ordinal(get_prev_edge(fe)), 1+ordinal(get_next_edge(fe)),
         1+ordinal(get_prev_facet(fe)), 1+ordinal(get_next_facet(fe)));
}


/*********************************************************************
*
*  Function:  dump_force()
*
*  Purpose:   List forces at vertices for debugging purposes.
*/

void dump_force()
{
  char name[40];

  FILE *fd;
  vertex_id v_id;
  int i;

  prompt("Enter name of dump file: ",name);
  if ( name[0] == '\0' ) return;

  fd = fopen(name,"w");

  if ( fd == NULL )
   { 
     sprintf(errmsg,"Cannot open file %s.\n",name);
     error(errmsg,RECOVERABLE);
   }

  fprintf(fd,"%s: Dump of force.\n\n",name);

  /* vertex dump */
  fputs("\nvertices           coordinates     \n",fd);
  FOR_ALL_VERTICES(v_id)
    { REAL *x,*f,mag;
      x = get_coord(v_id);
      fprintf(fd,"%3d  %17.15f %17.15f %17.15f  |x| = %17.15f ",
          1+ordinal(v_id),x[0],x[1],x[2],sqrt(dot(x,x,web.sdim)));
      if ( get_vattr(v_id) & CONSTRAINT )
        { MAP conmap = get_v_constraint_map(v_id);
          fprintf(fd,"   constraints ");
          for ( i = 0 ; i < web.concount ; i++,conmap>>=1 )
            if ( (conmap & 1) && !(get_constraint(i)->attr & GLOBAL) )
              fprintf(fd,"%d ",i);
        }
      if ( get_vattr(v_id)&FIXED ) fprintf(fd, "  fixed ");
      fprintf(fd,"\n");

      f = get_force(v_id);
      mag = sqrt(dot(f,f,web.sdim));
      fprintf(fd,"f:   %17.15f %17.15f %17.15f  |f| = %f  A = %f\n\n",
        f[0],f[1],f[2],mag,get_vertex_star(v_id));
    }

  fclose(fd);
}
