/* ScianButtons.c:	John R. Murray, 3-30-90
		creates and deals with button objects
*/
#include "Scian.h"
#include "ScianTypes.h"
#include "ScianColors.h"
#include "ScianIDs.h"
#include "ScianMethods.h"
#include "ScianLists.h"
#include "ScianErrors.h"
#include "ScianIcons.h"
#include "ScianScripts.h"
#include "ScianArrays.h"
#include "ScianButtons.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianControls.h"
#include "ScianEvents.h"
#include "ScianFontSystem.h"
#include "ScianStyle.h"

/* globals */
ObjPtr	buttonClass;			/* class of buttons */
ObjPtr	iconButtonClass;		/* class of icon buttons */
ObjPtr	iconLabeledButtonClass;		/* class of labeled icon buttons */
ObjPtr	radioButtonClass;		/* class of radio buttons */
ObjPtr	oldRadioButtonClass;		/* class of radio buttons */
ObjPtr	checkBoxClass;			/* class of check boxes */
ObjPtr	radioGroupClass;		/* class of '90.. er, um.. */

/* temporary defs to get something going */

/* none */

/* local constants */
#define BUTTONDEPTH	4	/* button "height"=width (pixels) of border) */
#define LABELFONTFUDGE	1	/* fudge strings up (or <0 = down) to center */
#define RADIOSPACER	((int) BUTTONFONTSIZE)
				/* distance in pixels from button to text */

/* values of object variables */
#define MARKERX		0	/* value of var MARKERTYPE for mark = 'X' */
#define MARKERCHECK	1	/* value of var MARKERTYPE for mark = check */
#define MARKERBLOT	2	/* value of var MARKERTYPE for mark = dot */
#define MARKERSPLAT	3	/* value of var MARKERTYPE for mark = splat */

/*Draws the frame and face of a button */
#ifdef PROTO
void	DrawButtonFrame(int left, int right,
		int bottom, int top, int depth, int topcolor)
#else
void	DrawButtonFrame(left, right, bottom, top, depth, topcolor)
int left, right, bottom, top, depth, topcolor;
#endif
{

#ifdef GRAPHICS
    Coord v[4][2];

    /*Draw bottom edge*/
    v[0][0] = left + depth - 0.5;
    v[0][1] = bottom + depth - 0.5;
    v[1][0] = right - depth + 0.5;
    v[1][1] = bottom + depth - 0.5;
    v[2][0] = right + 0.5;
    v[2][1] = bottom - 0.5;
    v[3][0] = left - 0.5;
    v[3][1] = bottom - 0.5;
    SetUIColor(UIBOTTOMEDGE);
    polf2(4, v);

    /*Draw left edge*/
    v[0][0] = left - 0.5;
    v[0][1] = top + 0.5;
    v[1][0] = left + depth - 0.5;
    v[1][1] = top - depth + 0.5;
    v[2][0] = left + depth - 0.5;
    v[2][1] = bottom + depth - 0.5;
    v[3][0] = left - 0.5;
    v[3][1] = bottom - 0.5;
    SetUIColor(UILEFTEDGE);
    polf2(4, v);

    /*Draw right edge*/
    v[0][0] = right - depth + 0.5;
    v[0][1] = top - depth + 0.5;
    v[1][0] = right + 0.5;
    v[1][1] = top + 0.5;
    v[2][0] = right + 0.5;
    v[2][1] = bottom - 0.5;
    v[3][0] = right - depth + 0.5;
    v[3][1] = bottom + depth - 0.5;
    SetUIColor(UIRIGHTEDGE);
    polf2(4, v);

    /*Draw top edge*/
    v[0][0] = left - 0.5;
    v[0][1] = top + 0.5;
    v[1][0] = right + 0.5;
    v[1][1] = top + 0.5;
    v[2][0] = right - depth + 0.5;
    v[2][1] = top - depth + 0.5;
    v[3][0] = left + depth - 0.5;
    v[3][1] = top - depth + 0.5;
    SetUIColor(UITOPEDGE);
    polf2(4, v);

    /* draw button surface */
    v[0][0] = left + depth - 0.5;
    v[0][1] = bottom + depth - 0.5;
    v[1][0] = right - depth + 0.5;
    v[1][1] = bottom + depth - 0.5;
    v[2][0] = right - depth + 0.5;
    v[2][1] = top - depth + 0.5;
    v[3][0] = left + depth - 0.5;
    v[3][1] = top - depth + 0.5;
    SetUIColor(topcolor);
    polf2(4, v);
#endif /* GRAPHICS */
}

#if 0
void GreyOut(left, right, bottom, top)
{
    int i;

    SetUIColor(UIBACKGROUND);

    if (right-left < top-bottom)
    {
	for(i=0; i<right-left; i+=2)
	{
	    move2(left + i, bottom);
	    draw2(left, bottom + i);
	}

	for( ; i < top-bottom; i+=2)
	{
	    move2(right, bottom + i - (right - left));
	    draw2(left, bottom + i);
	}
	for ( ; i < (top-bottom) + (right-left); i += 2)
	{
	    move2(right, bottom + i - (right - left));
	    draw2(left + i - (top - bottom), top);
	}
    }
    else
    {
	for(i=0;i<top-bottom;i+=2)
	{
	    move2(left + i, bottom);
	    draw2(left, bottom + i);
	}

	for( ; i < right-left; i+=2)
	{
	    move2(left + i, bottom);
	    draw2(left + i - (top - bottom), top);
	}
	for ( ; i < (top-bottom) + (right-left); i += 2)
	{
	    move2(right, bottom + i - (right - left));
	    draw2(left + i - (top - bottom), top);
	}
    }
}
#endif

static ObjPtr DrawButtonObj(theButton, theButtonOwner)
ObjPtr	theButton, theButtonOwner;
{

#ifdef GRAPHICS
    int left, right, bottom, top;
    int		depth;
    Bool	hilight, active;
    ObjPtr	theLabel;
    char	*label;
    ObjPtr	theColor;
    int		color;

    if (!Get2DIntBounds(theButton, &left, &right, &bottom, &top))
    {
	return NULLOBJ;
    }

    hilight = GetPredicate(theButton, HIGHLIGHTED);
    active = GetPredicate(theButton, ACTIVATED);

    theLabel = GetVar(theButton, NAME);
    if (theLabel)
    {
	label = GetString(theLabel);
    }
    else
    {
	label = (char *) NIL;
    }

    theColor = GetVar(theButton, COLOR);
    if (!theColor)
    {
	color = UIBACKGROUND;
    }
    else
    {
	color = GetPredicate(theButton, VALUE) ? GetInt(theColor) : UIBACKGROUND;
    }

    if (!active)
    {
	setpattern(GREYPAT);
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : color);
	setpattern(SOLIDPAT);
    }
    else
    {
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : color);
    }

    if (label)
    {
	if (active)
	    SetUIColor(UITEXT);
	else
	    SetUIColor(UIGREYTEXT);

	SetupFont(BUTTONFONT, BUTTONFONTSIZE);
	DrawString((left + right) / 2 - StrWidth(label) / 2,
		   (top+bottom)/2 + LABELFONTFUDGE - ((int) BUTTONFONTSIZE/2),
		   label);
    }
    return ObjTrue;
#endif /* GRAPHICS */
}

ObjPtr	NewIconButton(left, right, bottom, top, icon, color, name)
int	left, right, bottom, top;
int	icon, color;
char *name;
/*Makes a new icon button with bounds left, right, bottom, top,
   with icon icon, and color color when on (otherwise drawn in UITEXT) */
{
    ObjPtr	retVal;
    ObjPtr	iconPtr;
    ObjPtr	hilightPtr;


    retVal = NewObject(iconButtonClass, 0);
    iconPtr = NewInt(icon);
    SetVar(retVal, WHICHICON, iconPtr);
    Set2DIntBounds(retVal, left, right, bottom, top);
    hilightPtr = NewInt(false);
    SetVar(retVal, HIGHLIGHTED, hilightPtr);
    SetVar(retVal, VALUE, NewInt(false));
    SetVar(retVal, COLOR, NewInt(color));
    SetVar(retVal, ACTIVATED, NewInt(true));
    if (name)
    {
	SetVar(retVal, NAME, NewString(name));
    }
    return retVal;
}

/* DRAW method for icon buttons */

ObjPtr	DrawIconButtonObj(theButton, theButtonOwner)
ObjPtr	theButton, theButtonOwner;
{

#ifdef GRAPHICS
    int left, right, bottom, top;
    int		depth;
    Bool	hilight, active;
    ObjPtr	theIcon;
    int		label;
    ObjPtr	boundsArray;
    real	bounds[4];
    char	buttonLabel[128];
    int		value;
    ObjPtr	theColor;
    int		color;
    int		style;
    FuncTyp	extraDraw;

    if (!Get2DIntBounds(theButton, &left, &right, &bottom, &top))
    {
	return NULLOBJ;
    }

    hilight = GetPredicate(theButton, HIGHLIGHTED);
    active = GetPredicate(theButton, ACTIVATED);
    value = GetInt(GetIntVar("DrawIconButtonObj", theButton, VALUE));

    if (GetVar(theButton, COLOR))
    {
	color = GetInt(GetIntVar("DrawIconButtonObj" ,theButton, COLOR));
    }
    else
    {
	color = UIGREEN;
    }

    if (!active) /* button is not activated, grey it out.. */
    {
	setpattern(GREYPAT);
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : UIBACKGROUND);
	setpattern(SOLIDPAT);
	DrawIcon((left + right) / 2, (top + bottom) / 2,
	    GetInt(GetIntVar("DrawIconButtonObj", theButton, WHICHICON)),
	    (char *) NIL, (char *) NIL,
	    value ? color : (hilight ? UIHIBACKGROUND : UIBACKGROUND),
	    DI_DRAWFORE | DI_DRAWBACK | DI_SMALLTEXT | DI_GREY);
    }
    else
    {
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : UIBACKGROUND);
	DrawIcon((left + right) / 2, (top + bottom) / 2,
	    GetInt(GetIntVar("DrawIconButtonObj", theButton, WHICHICON)),
	    (char *) NIL, (char *) NIL,
	    value ? color : (hilight ? UIHIBACKGROUND : UIBACKGROUND),
	    DI_DRAWFORE | DI_DRAWBACK | DI_SMALLTEXT);

	extraDraw = GetMethod(theButton, ICONEXTRADRAW);
	if (extraDraw)
	{
	    (* extraDraw) (theButton, (left + right) / 2, (top + bottom) / 2);
	}
    }

    return ObjTrue;
#endif /* GRAPHICS */
}

ObjPtr	NewIconLabeledButton(left, right, bottom, top, icon, color, name)
int	left, right, bottom, top;
int	icon, color;
char *name;
/*Makes a new icon button with bounds left, right, bottom, top,
   with icon icon, color color when on (otherwise drawn in UITEXT)
   and label name (name is also the name of the object) */
{
    ObjPtr	retVal;
    ObjPtr	iconPtr, hilightPtr;

    retVal = NewObject(iconLabeledButtonClass, 0);
    Set2DIntBounds(retVal, left, right, bottom, top);
    iconPtr = NewInt(icon);
    SetVar(retVal, WHICHICON, iconPtr);
    hilightPtr = NewInt(false);
    SetVar(retVal, HIGHLIGHTED, hilightPtr);
    SetVar(retVal, VALUE, NewInt(false));
    SetVar(retVal, ACTIVATED, NewInt(true));
    SetVar(retVal, COLOR, NewInt(color));
    if (name)
    {
	SetVar(retVal, NAME, NewString(name));
	SetVar(retVal, LABEL, NewString(name));
    }
    else
    {
	SetVar(retVal, LABEL, NewString(""));
    }
    return retVal;
}

ObjPtr	DrawIconLabeledButtonObj(theButton, theButtonOwner)
ObjPtr	theButton, theButtonOwner;
{

#ifdef GRAPHICS
    int left, right, bottom, top;
    int		depth;
    Bool	hilight, active;
    int		icon;
    int		offset;
    char	*label;
    char	buttonLabel[128];
    int		value;
    ObjPtr	theColor;
    int		color;
    int		style;
    FuncTyp	extraDraw;

    if (!Get2DIntBounds(theButton, &left, &right, &bottom, &top))
    {
        return NULLOBJ;
    }

    hilight = GetPredicate(theButton, HIGHLIGHTED);
    value = GetInt(GetIntVar("DrawIconLabeledButtonObj", theButton, VALUE));
    active = GetPredicate(theButton, ACTIVATED);

    if (GetVar(theButton, COLOR))
    {
	color = GetInt(GetIntVar("DrawIconLabeledButtonObj" ,theButton, COLOR));
    }
    else
    {
	color = UIGREEN;
    }

    if (GetVar(theButton, WHICHICON))
    {
	icon = GetInt(GetIntVar("DrawIconLabeledButtonObj",theButton, WHICHICON));
    }
    else
    {
	icon = ICONQUESTION;
    }

    if (GetVar(theButton, LABEL))
    {
	label = GetString(GetStringVar("DrawIconLabeledButtonObj",theButton, LABEL));
	offset = ICONTEXTOFFSET / 2;
    }
    else
    {
	label = (char *) NIL;
	offset = 0;
    }

    if (!active) /* button is not activated, grey it out */
    {
	setpattern(GREYPAT);
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : UIBACKGROUND);
	setpattern(SOLIDPAT);
	DrawIcon((left + right) / 2, (top + bottom) / 2 + offset,
	    icon, label, (char *) NIL,
	    value ? color : (hilight ? UIHIBACKGROUND : UIBACKGROUND),
	    DI_DRAWFORE | DI_DRAWBACK | DI_SMALLTEXT | DI_GREY);
    }
    else
    {
	DrawButtonFrame(left, right, bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : UIBACKGROUND);
	DrawIcon((left + right) / 2, (top + bottom) / 2 + offset,
	    icon, label, (char *) NIL,
	    value ? color : (hilight ? UIHIBACKGROUND : UIBACKGROUND),
	    DI_DRAWFORE | DI_DRAWBACK | DI_SMALLTEXT);

	extraDraw = GetMethod(theButton, ICONEXTRADRAW);
	if (extraDraw)
	{
	    (* extraDraw) (theButton, (left + right) / 2, (top + bottom) / 2);
	}
    }

    return ObjTrue;

#endif /* GRAPHICS */
}

static ObjPtr SetButtonValuePretend(theButton, theValue)
ObjPtr	theButton, theValue;
{
    FuncTyp	changer;

    /* value is ignored, button just gets pressed */

    InhibitLogging(true);
    ChangedValue(theButton);
    InhibitLogging(false);

    ImInvalid(theButton);
    if (logging) LogControl(theButton);
    return ObjTrue;
}

static ObjPtr SetButtonValueReally(theButton, theValue)
ObjPtr	theButton, theValue;
{
    FuncTyp	changer;

    if (IsInt(theValue))
    {
	SetVar(theButton, VALUE, theValue);
    }
    else if (IsReal(theValue))
    {
	SetVar(theButton, VALUE, NewInt((int) GetReal(theValue)));
    }
    else
    {
	return ObjFalse;
    }
    InhibitLogging(true);
    ChangedValue(theButton);
    InhibitLogging(false);

    ImInvalid(theButton);
    if (logging) LogControl(theButton);
    return ObjTrue;
}

#ifdef PROTO
Bool	MakeButtonToggled(ObjPtr theButton, Bool flag)
#else
Bool	MakeButtonToggled(theButton, flag)
ObjPtr theButton;
Bool flag;
#endif
{
    if (flag)
    {
	return SetMethod(theButton, SETVAL, SetButtonValueReally);
    }
    else
    {
	return SetMethod(theButton, SETVAL, SetButtonValuePretend);
    }
}

Bool ToggleButtonActivated(ObjPtr theButton)
{
    SetVar(theButton, ACTIVATED, NewInt(!GetPredicate(theButton, ACTIVATED)));
    ImInvalid(theButton);
    return true;
}

#ifdef PROTO
Bool ActivateButton(ObjPtr theButton, Bool flag)
#else
Bool ActivateButton(theButton, flag)
ObjPtr theButton;
Bool flag;
#endif
{
    if (flag == GetInt(GetIntVar("ActivateButton", theButton, ACTIVATED)))
    {
	return false;
    }
    SetVar(theButton, ACTIVATED, NewInt(flag));
    ImInvalid(theButton);
    return true;
}

/* SetRadioButtonGroupValue: Radio button group SETVAL method */

static ObjPtr SetRadioButtonGroupValue(theGroup, theValue)
ObjPtr	theGroup, theValue;
{
    ObjPtr	buttonList;	/* the Radio's list of buttons */
    ThingListPtr	runner;		/* list runner */
    int		buttonNo;	/* counts things as runner runs */
    ObjPtr	oneThing;	/* one thing in the list */
    int		value;		/* value of theValue */

    if (IsInt(theValue))
    {
	value = GetInt(theValue);
    }
    else if (IsReal(theValue))
    {
	value = GetReal(theValue);
    }
    else
    {
	return ObjFalse;
    }
    SetVar(theGroup, VALUE, NewInt(value));

    /* run down list of buttons and set all but one false. */
    buttonList = GetListVar("SetRadioButtonGroupValue", theGroup, CONTENTS);
    if (!buttonList)
    {
	return ObjFalse;
    }
    
    runner = LISTOF(buttonList);
    buttonNo = 0;

    /* find the button in the CONTENTS list */
    while (runner && buttonNo < value)
    {
	runner = runner -> next;
	++buttonNo;
    }

    if (!runner)
    {
	ReportError("SetRadioButtonGroupValue","tried to set to illegal value");
	SetVar(theGroup, VALUE, NewInt(-1));
	ChangedValue(theGroup);
	ImInvalid(theGroup);
	return ObjFalse;
    }

    oneThing = runner -> thing;
    if (!InClass(oneThing, buttonClass))
    {
	ReportError("SetRadioButtonGroupValue",
	    "non-button in radio button group contents!\n");
    }
    else
    {
	InhibitLogging(true);
	ButtonPressed(oneThing);
	InhibitLogging(false);
    }

    InhibitLogging(true);
    ChangedValue(theGroup);
    InhibitLogging(false);

    if (logging) LogControl(theGroup);

    ImInvalid(theGroup);
    return ObjTrue;
}

ObjPtr	NewButton(left, right, bottom, top, label)
int	left, right, bottom, top;
char *label;
/*Makes a new panel with bounds left, right, bottom, top, and label label*/
{
    ObjPtr	retVal;
    ObjPtr	labelPtr;
    ObjPtr	hilightPtr;

    retVal = NewObject(buttonClass, 0);
    if(retVal)
    {
	labelPtr = NewString(label);
	Set2DIntBounds(retVal, left, right, bottom, top);
	hilightPtr = NewInt(false);
	SetVar(retVal, NAME, labelPtr);
	SetVar(retVal, HIGHLIGHTED, hilightPtr);
	SetVar(retVal, VALUE, NewInt(false));
	SetVar(retVal, ACTIVATED, NewInt(true));
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}

void	DrawMarkerSplat(left, right, bottom, top)
{

#ifdef GRAPHICS
    Coord v[4][2];

    SetUIColor(UITEXT);
    /*Draw part of button '*', the left-bottom to top-right part */
    v[0][0] = left + 2.5;
    v[0][1] = bottom + 3.5;
    v[1][0] = left + 3.5;
    v[1][1] = bottom + 2.5;
    v[2][0] = right - 2.5;
    v[2][1] = top - 3.5;
    v[3][0] = right - 3.5;
    v[3][1] = top - 2.5;
    polf2(4, v);

    /*Draw rest of button '*', the left-top to right-bottom part*/
    v[0][0] = left + 2.5;
    v[0][1] = top - 3.5;
    v[1][0] = left + 3.5;
    v[1][1] = top - 2.5;
    v[2][0] = right - 2.5;
    v[2][1] = bottom + 3.5;
    v[3][0] = right - 3.5;
    v[3][1] = bottom + 2.5;
    polf2(4, v);

    /*Draw rest of button '*', the left-to-right part*/
    v[0][0] = left + 1.5;
    v[0][1] = (top+bottom)/2.0 - 0.5;
    v[1][0] = left + 1.5;
    v[1][1] = (top+bottom)/2.0 + 0.5;
    v[2][0] = right - 1.5;
    v[2][1] = (top+bottom)/2.0 + 0.5;
    v[3][0] = right - 1.5;
    v[3][1] = (top+bottom)/2.0 - 0.5;
    polf2(4, v);

    /*Draw rest of button '*', the top-to-bottom part*/
    v[0][0] = (left+right)/2.0 - 0.5;
    v[0][1] = top - 1.5;
    v[1][0] = (left+right)/2.0 + 0.5;
    v[1][1] = top - 1.5;
    v[2][0] = (left+right)/2.0 + 0.5;
    v[2][1] = bottom + 1.5;
    v[3][0] = (left+right)/2.0 - 0.5;
    v[3][1] = bottom + 1.5;
    polf2(4, v);

#endif /* GRAPHICS */
}

void    DrawMarkerBlot(left, right, bottom, top)
int     left, right, bottom, top;
{
#ifdef GRAPHICS
    SetUIColor(UITEXT);
    rectf(left+2.5, bottom+2.5, right-2.5, top-2.5);
#endif /*GRAPHICS*/
}

void	DrawMarkerCheck(left, right, bottom, top)
int	left, right, bottom, top;
{
#ifdef GRAPHICS
    Coord v[4][2];

    SetUIColor(UITEXT);
    /*Draw part of checkmark*/
    v[0][0] = left + 1.5;
    v[0][1] = (top + bottom) / 2 - 0.5;
    v[1][0] = left + 2.5;
    v[1][1] = (top + bottom) / 2 + 1.5;
    v[2][0] = left + (right - left) / 3 + 1.5;
    v[2][1] = bottom + 1.5;
    v[3][0] = left + (right - left) / 3 - 0.5;
    v[3][1] = bottom + 1.5;
    polf2(4, v);

    /*Draw rest of checkmark*/
    v[0][0] = left + (right - left) / 3 + 1.5;
    v[0][1] = bottom + 1.5;
    v[1][0] = left + (right - left) / 3 - 0.5;
    v[1][1] = bottom + 1.5;
    v[2][0] = right - 1.5;
    v[2][1] = top - 1.5;
    polf2(3, v);
#endif /*GRAPHICS */
}

void	DrawMarkerX(left, right, bottom, top)
int	left, right, bottom, top;
{
#ifdef GRAPHICS
    Coord v[4][2];

    SetUIColor(UITEXT);
    /*Draw part of button 'X'*/
    v[0][0] = left + 1.5;
    v[0][1] = bottom + 2.5;
    v[1][0] = left + 2.5;
    v[1][1] = bottom + 1.5;
    v[2][0] = right - 1.5;
    v[2][1] = top - 2.5;
    v[3][0] = right - 2.5;
    v[3][1] = top - 1.5;
    polf2(4, v);

    /*Draw rest of button 'X'*/
    v[0][0] = left + 1.5;
    v[0][1] = top - 2.5;
    v[1][0] = left + 2.5;
    v[1][1] = top - 1.5;
    v[2][0] = right - 1.5;
    v[2][1] = bottom + 2.5;
    v[3][0] = right - 2.5;
    v[3][1] = bottom + 1.5;
    polf2(4, v);
#endif /*GRAPHICS */
}

static ObjPtr DrawRadioButtonObj(theButton)
ObjPtr	theButton;
{
#ifdef GRAPHICS
    int left, right, bottom, top;
    Bool	hilight, active;
    Bool	value;
    int		marker;
    char	*label;

    if (!Get2DIntBounds(theButton, &left, &right, &bottom, &top))
    {
        return NULLOBJ;
    }

    hilight = GetPredicate(theButton, HIGHLIGHTED);
    active = GetPredicate(theButton, ACTIVATED);
    marker = GetInt(GetIntVar("DrawRadioButtonObj", theButton, MARKERTYPE));
    label = GetString(GetStringVar("DrawRadioButtonObj",theButton, NAME));
    value = GetInt(GetIntVar("DrawRadioButtonObj", theButton, VALUE));

    if (!active)
	setpattern(GREYPAT);

    DrawButtonFrame(left, left + (top - bottom), bottom, top, BUTTONDEPTH,
			hilight ? UIHIBACKGROUND : UIBACKGROUND);
    if (value)
    {
	switch(marker)
	{
	    case MARKERX:
		DrawMarkerX(left+BUTTONDEPTH, left+(top-bottom) - BUTTONDEPTH,
				bottom + BUTTONDEPTH, top - BUTTONDEPTH);
		break;
	    case MARKERCHECK:
		DrawMarkerCheck(left+BUTTONDEPTH, left+(top-bottom)-BUTTONDEPTH,
				bottom + BUTTONDEPTH, top - BUTTONDEPTH);
		break;
	    case MARKERBLOT:
		DrawMarkerBlot(left+BUTTONDEPTH,left+(top-bottom)-BUTTONDEPTH,
                                bottom + BUTTONDEPTH, top - BUTTONDEPTH);
                break;
	    case MARKERSPLAT:
		DrawMarkerSplat(left+BUTTONDEPTH,left+(top-bottom)-BUTTONDEPTH,
                                bottom + BUTTONDEPTH, top - BUTTONDEPTH);
                break;
	}
    }
    if (!active)
	setpattern(SOLIDPAT);

    if (label)
    {
	if (active)
	    SetUIColor(UITEXT);
	else
	    SetUIColor(UIGREYTEXT);

	SetupFont(BUTTONFONT, BUTTONFONTSIZE);
        DrawString(left + (top - bottom) + RADIOSPACER,
                   (top+bottom)/2 + LABELFONTFUDGE - (int) BUTTONFONTSIZE/2,
                   label);
    }
#endif /* GRAPHICS */
}

#ifdef PROTO
static Bool TrackMouse(ObjPtr obj, int whichButton)
#else
static Bool TrackMouse(obj, whichButton)
ObjPtr	obj;
int	whichButton;
#endif
{
    int		mX, mY;		/* track mouse coordinates */
    Bool	isDown;		/* return value from Mouse, true if down */
    int		left, right, bottom, top; /* non-array version */

    if (!Get2DIntBounds(obj, &left, &right, &bottom, &top))
    {
        return false;
    }

    /* the buggy version? */
    while (Mouse(&mX, &mY))
    {
	if (mX < left || mX > right || mY < bottom || mY > top)
	{
	    /* mouse outside rectangle */
	    if (GetPredicate(obj, HIGHLIGHTED))
	    {
		SetVar(obj, HIGHLIGHTED, NewInt(false));/* turn highlight off */
		DrawMe(obj);		/* ..and redraw */
	    }
	}
	else
	{
	    /* mouse inside rectangle */
	    if (!GetPredicate(obj, HIGHLIGHTED))
	    {
		SetVar (obj, HIGHLIGHTED, NewInt(true));/* turn highlight on */
		DrawMe(obj);		/* ..and redraw */
	    }
	}
    }
    /* always turn highlight off and redraw on exit */
    SetVar(obj, HIGHLIGHTED, NewInt(false));
    ImInvalid(obj);
    /* is this obtuse or what? return true iff mouseup while in rectangle */
    return !(mX < left || mX > right || mY < bottom || mY > top);
}

static ObjPtr ButtonPressedButton(theButton)
/* ButtonPressedButton: BUTTONPRESSED Method for vanilla buttons.
	It only toggles the VALUE token of the button. returns true if
	successful, false if not.
*/
ObjPtr	theButton;
{
    ObjPtr	theValue;	/* pointer to object's VALUE int */
    int		value;		/* value of it's VALUE int */

    if (!GetPredicate(theButton, ACTIVATED))
    {
	return ObjFalse;
    }

    theValue = GetIntVar("ButtonPressedButton",theButton, VALUE);
    if (!theValue)
    {
	return ObjFalse;
    }
    value = GetInt(theValue);
    theValue = NewInt(value ? false : true);
    SetValue(theButton, theValue);

    return ObjTrue;
}

static ObjPtr ButtonPressedRadioButton(theButton)
ObjPtr theButton;
{
    ObjPtr	buttonList;	/* the Radio's list of buttons */
    ObjPtr	parent;		/* button's parent */
    ThingListPtr	runner;		/* list runner */
    ObjPtr	oneThing;	/* ont thing in the list */
    int		buttonNo;

    if (!GetPredicate(theButton, ACTIVATED))
    {
	return ObjFalse;
    }

    /* run down list of buttons and set them all false. */
    parent = GetVar(theButton, PARENT);
    if (!parent)
    {
	ReportError("ButtonPressedRadioButton","Radio button has no PARENT!\n");
	return ObjFalse;
    }
    buttonList = GetListVar("ButtonPressedRadioButton", parent, CONTENTS);
    if (!buttonList)
    {
	return ObjFalse;
    }
    else
    {
	runner = LISTOF(buttonList);
	SetVar(parent, VALUE, NewInt(-1));
	buttonNo = 0;
	while (runner)
	{
    	    ThingPtr oneThing;

	    oneThing = runner -> thing;
	    if (!InClass(oneThing, buttonClass))
	    {
		ReportError("ButtonPressedRadioButton","non-button in radio button group contents!\n");
	    }
	    else
	    {
		if (GetPredicate(oneThing, VALUE))
		{
		    SetValue(oneThing, NewInt(false));
		    /* note there's a quirk here if old button = new button */
		}
	    }
	    if (runner -> thing == theButton)
	    {
		SetVar(parent, VALUE, NewInt(buttonNo));
	    }
	    ++buttonNo;
	    runner = runner -> next;
	}
    }

    SetValue(theButton, NewInt(true));
    ChangedValue(parent);
    return ObjTrue;
}

/* PressButton: button PRESS method. if mouse down in my rectangle, track
	the mouse and (if up in my rectangle) change the value of the button.
	returns true if mouse down in my rectangle, regardless of whether mouse
	up happened in it.
*/
#ifdef PROTO
static ObjPtr PressButton(ObjPtr theButton, int mouseX,
			int mouseY, int flags)
#else
static ObjPtr PressButton(theButton, mouseX, mouseY, flags)
ObjPtr	theButton;
int	mouseX, mouseY;
int	flags;
#endif

{

#ifdef INTERACTIVE
    int left, right, bottom, top;

    if (!Get2DIntBounds(theButton, &left, &right, &bottom, &top))
    {
        return ObjFalse;
    }

    /* test if mouse in my rectangle */
    if (mouseX < left || mouseX > right ||
	mouseY < bottom || mouseY > top)
    {
	/* mouse out of my rectangle, do nothing, return */
	return ObjFalse;
    }

    if (TOOL(flags) == T_HELP)
    {
	ContextHelp(theButton);
	return ObjTrue;
    }

    if (!GetPredicate(theButton, ACTIVATED))
    {
	return ObjFalse;
    }

    if (!TrackMouse(theButton, flags))
    {
	/* mouse up outside of this button, do nothing but draw me */
	ImInvalid(theButton);
	return ObjTrue; /*return true if mousedown in me,regardless of mouseup*/
    }

    InhibitLogging(true);
    ButtonPressed(theButton);
    InhibitLogging(false);

    if (logging) LogControl(theButton);
    return ObjTrue;
#endif /* INTERACTIVE */
}

ObjPtr	NewRadioButton(left, right, bottom, top, label)
int	left, right, bottom, top;
char	*label;
/*Makes a new panel with bounds left, right, bottom, top, and label label,
	beloging to buttonGroup */
{
    ObjPtr	retVal;
    ObjPtr	labelPtr;
    ObjPtr	hilightPtr;

    retVal = NewObject(radioButtonClass, 0);
    if (retVal)
    {
	Set2DIntBounds(retVal, left, right, bottom, top);
	labelPtr = NewString(label);
	SetVar(retVal, NAME, labelPtr);
	hilightPtr = NewInt(false);
	SetVar(retVal, HIGHLIGHTED, hilightPtr);
	SetVar(retVal, VALUE, NewInt(false));
	SetVar(retVal, ACTIVATED, NewInt(true));
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}

/* Makes a new check box with bounds left, right, bottom, top, and label label*/

#ifdef PROTO
ObjPtr	NewCheckBox(int left, int right, int bottom, int top, char *label,
		    Bool initValue)
#else
ObjPtr	NewCheckBox(left, right, bottom, top, label, initValue)
int	left, right, bottom, top;
Bool	initValue;
char *label;
#endif
{
    ObjPtr	retVal;
    ObjPtr	labelPtr;
    ObjPtr	hilightPtr;

    retVal = NewObject(checkBoxClass, 0);
    if (retVal)
    {
	Set2DIntBounds(retVal, left, right, bottom, top);
	labelPtr = NewString(label);
	SetVar(retVal, NAME, labelPtr);
	hilightPtr = NewInt(false);
	SetVar(retVal, HIGHLIGHTED, hilightPtr);
	SetVar(retVal, VALUE, NewInt(initValue));
	SetVar(retVal, ACTIVATED, NewInt(true));
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}

static ObjPtr ReshapeRadioButtonGroup(object, ol, or, ob, ot, left, right, bottom, top)
ObjPtr object;
int ol, or, ob, ot;
int left, right, bottom, top;
/*Reshapes object, which used to exist within owner with edges ol, or, ob, ot
  to one which exists within owner with edges left, right, bottom, top.*/
{
	ObjPtr boundsArray;
	ObjPtr stickyInt;
	real bounds[4];
	real oldWidth, oldHeight;	/*Old width and height*/
	Bool sideLocked[4];		/*True iff side is locked*/
	Bool xStretch, yStretch;	/*Stretchiness in x and y*/
	int stickiness;			/*Side stickiness of the object*/
	real oldBounds[4];		/*Old bounds of the object*/
	ObjPtr contents;		/*Contents of the object, if any*/
	real wr, hr;			/*Width and height ratios*/

	wr = ((real) (right - left)) / ((real) (or - ol));
	hr = ((real) (top - bottom)) / ((real) (ot - ob));

	boundsArray = GetVar(object, BOUNDS);
	if (!boundsArray || !IsArray(boundsArray) || RANK(boundsArray) != 1 ||
	    DIMS(boundsArray)[0] != 4)
	{
	    return;
	}
	Array2CArray(bounds, boundsArray);
	Array2CArray(oldBounds, boundsArray);
	oldWidth = bounds[1] - bounds[0];
	oldHeight = bounds[3] - bounds[2];

	/*Get the object's stickiness*/
	stickyInt = GetVar(object, STICKINESS);
	if (stickyInt && IsInt(stickyInt))
	{
	    stickiness = GetInt(stickyInt);
	}
	else
	{
	    stickiness = 0;
	}

	if ((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT))
	{
	    if (stickiness & FLOATINGLEFT)
	    {
		bounds[0] = (bounds[0] - ol) * wr + left;
	    }
	    else
	    {
		bounds[0] += left - ol;
	    }
	    if (!((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT)))
	    {
		bounds[1] = bounds[0] + oldWidth;
	    }
	}
	if ((stickiness & STICKYRIGHT) || (stickiness & FLOATINGRIGHT))
	{
	    if (stickiness & FLOATINGRIGHT)
	    {
		bounds[1] = (bounds[1] - ol) * wr + left;
	    }
	    else
	    {
		bounds[1] += right - or;
	    }
	    if (!((stickiness & STICKYLEFT) || (stickiness & FLOATINGLEFT)))
	    {
		bounds[0] = bounds[1] - oldWidth;
	    }
	}

	if ((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM))
	{
	    if (stickiness & FLOATINGBOTTOM)
	    {
		bounds[2] = (bounds[2] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[2] += bottom - ob;
	    }
	    if (!((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP)))
	    {
		bounds[3] = bounds[2] + oldHeight;
	    }
	}
	if ((stickiness & STICKYTOP) || (stickiness & FLOATINGTOP))
	{
	    if (stickiness & FLOATINGTOP)
	    {
		bounds[3] = (bounds[3] - ob) * hr + bottom;
	    }
	    else
	    {
		bounds[3] += top - ot;
	    }
	    if (!((stickiness & STICKYBOTTOM) || (stickiness & FLOATINGBOTTOM)))
	    {
		bounds[2] = bounds[3] - oldHeight;
	    }
	}

	/*We've got a new bounds, put it back*/
	boundsArray = NewRealArray(1, 4L);
	CArray2Array(boundsArray, bounds);
	SetVar(object, BOUNDS, boundsArray);

	/*If there are some contents to this, do the reshape recursively*/
	contents = GetVar(object, CONTENTS);
	if (contents && IsList(contents))
	{
	    ReshapeList(LISTOF(contents),
		    (int) oldBounds[0], (int) oldBounds[1] ,
		    (int) oldBounds[2], (int) oldBounds[3],
		    (int) bounds[0], (int) bounds[1],
		    (int) bounds[2], (int) bounds[3]);
	}
}


static ObjPtr DrawRadioButtonGroup(theGroup)
ObjPtr	theGroup;
{
#ifdef GRAPHICS
    ObjPtr	theList;

    theList = GetListVar("DrawRadioButtonGroup", theGroup, CONTENTS);
    if (!theList)
    {
	return NULLOBJ;
    }

    DrawList(theList);
    return NULLOBJ;
#endif /* GRAPHICS */
}

#ifdef PROTO
static ObjPtr PressRadioButtonGroup(ObjPtr theGroup, int mX, int mY, int flags)
#else
static ObjPtr PressRadioButtonGroup(theGroup, mX, mY, flags)
ObjPtr theGroup;
int mX, mY;
int flags;
#endif
{
#ifdef INTERACTIVE
    int		left, right, bottom, top;
    ObjPtr	theList;
    ObjPtr	valuePtr;		/* radio group value ptr */
    int		oldValue;		/* group's old value */
    FuncTyp	bpmethod;		/* BUTTONPRESSED method */
    ObjPtr	pressVal;		/* thing returned by PressList */

    theList = GetListVar("PressRadioButtonGroup", theGroup, CONTENTS);
    if (!theList)
    {
	return ObjFalse;
    }

    if (!Get2DIntBounds(theGroup, &left, &right, &bottom, &top))
    {
        return ObjFalse;
    }

    valuePtr = GetIntVar("PressRadioButtonGroup", theGroup, VALUE);
    if (!valuePtr)
    {
	return ObjFalse;
    }
    oldValue = GetInt(valuePtr);

    /* test if mouse in my rectangle */
    if (mX < left || mX > right ||
	mY < bottom || mY > top)
    {
	/* mouse out of my rectangle, do nothing, return */
	return ObjFalse;
    }

    InhibitLogging(true);
    pressVal = PressList(theList, mX, mY, flags);
    InhibitLogging(false);

    if(IsTrue(pressVal))
    {
	/* if really changed */
	if (oldValue != GetInt(GetIntVar("PressRadioButtonGroup",theGroup, VALUE)))
	{
	    InhibitLogging(true);
	    ButtonPressed(theGroup);
	    InhibitLogging(false);
	}
	if (logging) LogControl(theGroup);
	return ObjTrue;
    }
    else
    {
	if (TOOL(flags) == T_HELP)
	{
	    /* it was a help-press in the group bounds (that missed buttons) */
	    ContextHelp(theGroup);
	    return ObjTrue;
	}
	else
	{
	    return ObjFalse;
	}
    }
#endif /* INTERACTIVE */
}

ObjPtr	NewRadioButtonGroup(name)
char *name;
/* creates an empty radio button group */
{
    ObjPtr	retVal;

    retVal = NewObject(radioGroupClass, 0);
    if (retVal)
    {
	SetVar(retVal, CONTENTS, NewList());
	SetVar(retVal, VALUE, NewInt(-1));
	SetVar(retVal, NAME, NewString(name));
	SetVar(retVal, ACTIVATED, NewInt(true));
	return retVal;
    }
    else
    {
	return ObjFalse;
    }
}

void	AddRadioButton(theGroup, theButton)
ObjPtr	theGroup, theButton;
/* adds a radio button object to an existing radio button group */
{
    ObjPtr	groupBoundsArray;
    int		bounds[4], groupBounds[4];
    ObjPtr	theList;
    int		i;	/* temporary? loop counter */

    if (!Get2DIntBounds(theButton,&bounds[0],&bounds[1],&bounds[2],&bounds[3]))
    {
	return;
    }

    theList = GetListVar("AddRadioButton", theGroup, CONTENTS);
    if(!theList)
    {
	return;
    }
    groupBoundsArray = GetVar(theGroup, BOUNDS);
    if(!groupBoundsArray)
    {
	/*Try to make bounds. Initial value is bounds of first button*/
	Set2DIntBounds(theGroup, bounds[0], bounds[1], bounds[2], bounds[3]);
    }
    else
    {
	Get2DIntBounds(theGroup, &groupBounds[0], &groupBounds[1],
			&groupBounds[2], &groupBounds[3]);
	/* if necessary, modify the BOUNDS of the button group to include
	   the new button
	*/
	groupBounds[0] = bounds[0]<groupBounds[0] ? bounds[0] : groupBounds[0];
	groupBounds[1] = bounds[1]>groupBounds[1] ? bounds[1] : groupBounds[1];
	groupBounds[2] = bounds[2]<groupBounds[2] ? bounds[2] : groupBounds[2];
	groupBounds[3] = bounds[3]>groupBounds[3] ? bounds[3] : groupBounds[3];
	Set2DIntBounds(theGroup, groupBounds[0], groupBounds[1],
			groupBounds[2], groupBounds[3]);
    }
    PostfixList(theList, theButton);
    SetVar(theButton, PARENT, theGroup);
    SetMethod(theButton, BUTTONPRESSED, ButtonPressedRadioButton);
    SetMethod(theButton, SETVAL, SetButtonValueReally);
}

void	InitButtons()
/* sets up button stuff */
{
    buttonClass = NewObject(controlClass, 0);
    AddToReferenceList(buttonClass);
    SetMethod(buttonClass, DRAW, DrawButtonObj);
    SetMethod(buttonClass, PRESS, PressButton);
    SetMethod(buttonClass, BUTTONPRESSED, ButtonPressedButton);
    SetMethod(buttonClass, SETVAL, SetButtonValuePretend);
    SetVar(buttonClass, TYPESTRING, NewString("button"));
    SetVar(buttonClass, HELPSTRING, NewString(
"To press the button, click on it with the left mouse button. If you move the \
mouse away from the button before releasing, the button will not be pressed."));

    iconButtonClass = NewObject(buttonClass, 0);
    AddToReferenceList(iconButtonClass);
    SetMethod(iconButtonClass, DRAW, DrawIconButtonObj);
    SetMethod(iconButtonClass, SETVAL, SetButtonValuePretend);
    SetVar(iconButtonClass, TYPESTRING, NewString("icon button"));

    iconLabeledButtonClass = NewObject(iconButtonClass, 0);
    AddToReferenceList(iconLabeledButtonClass);
    SetMethod(iconLabeledButtonClass, DRAW, DrawIconLabeledButtonObj);
    SetVar(iconLabeledButtonClass,TYPESTRING,NewString("icon button"));

    radioGroupClass = NewObject(controlClass, 0);
    AddToReferenceList(radioGroupClass);
    SetMethod(radioGroupClass, DRAW, DrawRadioButtonGroup);
    SetMethod(radioGroupClass, PRESS, PressRadioButtonGroup);
    SetMethod(radioGroupClass, SETVAL, SetRadioButtonGroupValue);
    SetMethod(radioGroupClass, RESHAPE, ReshapeRadioButtonGroup);
    SetVar(radioGroupClass, TYPESTRING, NewString("radio button group"));
    SetVar(radioGroupClass, HELPSTRING, NewString(
"A radio button group provides a way to select one choice from among several. \
Selecting one of the buttons automatically deselects the others in the group."));

    radioButtonClass = NewObject(buttonClass, 0);
    AddToReferenceList(radioButtonClass);
    SetMethod(radioButtonClass, DRAW, DrawRadioButtonObj);
    SetVar(radioButtonClass, MARKERTYPE, NewInt(MARKERSPLAT));
    SetVar(radioButtonClass, TYPESTRING, NewString("radio button"));

    oldRadioButtonClass = NewObject(radioButtonClass, 0);
    AddToReferenceList(oldRadioButtonClass);
    SetVar(oldRadioButtonClass, MARKERTYPE, NewInt(MARKERX));

    checkBoxClass = NewObject(buttonClass, 0);
    AddToReferenceList(checkBoxClass);
    SetMethod(checkBoxClass, DRAW, DrawRadioButtonObj);
    SetMethod(checkBoxClass, SETVAL, SetButtonValueReally);
    SetVar(checkBoxClass, MARKERTYPE, NewInt(MARKERCHECK));
    SetVar(checkBoxClass, TYPESTRING, NewString("check box"));
}

void KillButtons()
/* kills button stuff */
{
    DeleteThing(checkBoxClass);
    DeleteThing(radioButtonClass);
    DeleteThing(radioGroupClass);
    DeleteThing(iconButtonClass);
    DeleteThing(buttonClass);
}

