/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* Message parsing */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/lib/RCS/parse.c,v $ */

static char rcsid[] = "parse.c $Revision: 2.1.1.2 $ $Date: 91/11/15 13:35:26 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include <stdio.h>

#include "mbus.h"
#include "api.h"

/* This is a function to be called when raw data is being received. It is
 * provided so that programs can be more efficient if they want to be.
 * The arguments are a buffer, size of buffer, bytes remaining.
 * The filter will be called before any data with (NULL, 0, <full count>)
 * before any data for any setup the function needs. When the function is
 * called with a remaining count of 0, it _must_ return a t_sexp that will
 * be put in the S-exp parse results in place of the raw expression.
 */
#ifdef __STDC__
struct mb_object *(*MBraw_filter)(char *, int, int) = NULL;
#else
struct mb_object *(*MBraw_filter)() = NULL;
#endif

char *MBparse_state_name[] =
{
  ":normal", ":string", ":complete",
  ":macro-start", ":macro-length", ":macro-recover",
  ":raw-read",
} ;
/* ------------------------------------------------------------------------- */
void
MBInitializeParseState(p) struct mb_parse_state *p;
{
  p->state = MB_PARSE_NORMAL;
  p->sexp = &(p->base);
  p->base.type = MB_CONS;
  p->base.next = NULL;			/* avoid GC */
  p->base.object.cons.car = p->base.object.cons.cdr = NULL;
  p->escape = p->error = p->delimiter = 0;
  p->depth = 0;
  p->open = 0;
}
/* ------------------------------------------------------------------------- */
void
MBResetParseState(p) struct mb_parse_state *p;
{
  if (p->base.object.cons.car != NULL) MBfree(p->base.object.cons.car);
  MBInitializeParseState(p);
}
/* ------------------------------------------------------------------------- */
static void
ExtendParseList(p) struct mb_parse_state *p;
{
  struct mb_object *obj;

  obj = MBGetCons();
  /* Two choices - if the open flag is set, extend the list down
   * otherwise, extend horizontally
   */
  if (p->open)
    {
      p->depth += 1;
      /* make back pointer */
      obj->object.cons.cdr = p->sexp;
      /* connect to existing list */
      p->sexp->object.cons.car = obj;
      /* make new sublist the current list */
      p->sexp = obj;
    }
  else					/* assume p->delimiter is set */
    {
      /* move the back pointer from current cons to new */
      obj->object.cons.cdr = p->sexp->object.cons.cdr;
      /* link the new cons into the current list */
      p->sexp->object.cons.cdr = obj;
      /* make the new cons the current one */
      p->sexp = obj;

    }
  p->open = 0;				/* clear pending open flag */
  p->delimiter = 0;			/* extended the list -> clear flag */
}
      
/* ------------------------------------------------------------------------- */
static void
ParseChar(c, p)
     char c;
     struct mb_parse_state *p;
{
  struct mb_object *obj;

  if (p->escape)			/* always put next char into a chunk */
    {
      if (p->delimiter) ExtendParseList(p);
      obj = p->sexp->object.cons.car;
      if (NULL == obj)			/* no chunk right now */
	{
	  obj = MBGetName();		/* get a new chunk */
	  p->sexp->object.cons.car = obj;
	}
      MBChunkPutChar(&(obj->object.chunk),c);
      p->escape = 0;
    }
  else if (0 == p->depth && !p->open && c != L_PAREN)
    return;
  else
    switch (p->state)
      {
      case MB_PARSE_COMPLETE:
	if (MBLogLevel > 0)
	  fprintf(stderr, "Parse complete - dropped char %c\n", c);
	break;
      case MB_PARSE_MACRO_RECOVER:
      case MB_PARSE_NORMAL:
	switch (c)
	  {
	  case L_PAREN:			/* start of new sexp */
	    if (p->depth || p->open) ExtendParseList(p); /* not at top level */
	    p->open = p->delimiter = 1;
	    break;
	  case R_PAREN:			/* end of sexp */
	    if (! p->open)		/* no pending open */
	      {
		obj = p->sexp->object.cons.cdr; /* get back pointer */
		p->sexp->object.cons.cdr = NULL; /* terminate list */
		p->sexp = obj;		/* go back to previous level */
		p->depth -= 1;		/* update nesting counter */
	      }
	    p->delimiter = 1;		/* counts as delimiter */
	    p->open = 0;		/* close deals with previous open */
	    if (p->depth == 0) p->state = MB_PARSE_COMPLETE;
	    break;
	  case '"':			/* start of string */
	    ExtendParseList(p);
	    obj = MBGetString();
	    p->sexp->object.cons.car = obj;
	    p->state = MB_PARSE_STRING;
	    break;
	  case ' ':
	  case '\t':
	  case '\n':
	  case '\f':			/* add new element to list */
	    p->delimiter = 1;
	    break;
	  case '\\':
	    p->escape = 1;
	    break;
	  case MB_MACRO_CHAR:
	    if (p->delimiter && MB_PARSE_MACRO_RECOVER != p->state)
	      {
		p->state = MB_PARSE_MACRO_START;
		break;
	      }
	    /* else FALL THROUGH to default (!) */
	  default:
	    if (p->delimiter) ExtendParseList(p);
	    if (NULL == p->sexp->object.cons.car)
	      p->sexp->object.cons.car = MBGetName();
	    MBChunkPutChar(&(p->sexp->object.cons.car->object.chunk),c);
	    break;
	  }
	/* if we're in macro recovery mode, then we've dealt the leading bogus
	 * char, so switch back into normal parse state if we haven't already
	 * switched the state
	 */
	if (MB_PARSE_MACRO_RECOVER == p->state) p->state = MB_PARSE_NORMAL;
	break;

      case MB_PARSE_STRING:
	switch (c)
	  {
	  case '"':
	    p->delimiter = 1;
	    p->state = MB_PARSE_NORMAL;
	    break;
	  case '\\':
	    p->escape = 1;
	    break;
	  default:
	    MBChunkPutChar(&(p->sexp->object.cons.car->object.chunk),c);
	    break;
	  }
	break;

      case MB_PARSE_MACRO_START:	/* found leading macro char */
	p->count = 0;
	/* FALL THROUGH */
      case MB_PARSE_MACRO_LENGTH:	/* found length digits */
	if ('0' <= c && c <= '9')
	  {
	    p->count = p->count * 10 + c - '0';
	    p->state = MB_PARSE_MACRO_LENGTH;
	  }
	else if (MB_MACRO_RAW_CHAR == c)
	  {
	    p->state = MB_PARSE_RAW_READ;
	    /* extend the parse list and put a new RAW object there to
	     * put the data into.
	     */
	    ExtendParseList(p);
	    if (NULL == MBraw_filter) p->sexp->object.cons.car = MBGetRaw();
	    else MBraw_filter(NULL, 0, p->count);
	  }
	else				/* bad macro, recover */
	  {
	    int digits = MB_PARSE_MACRO_LENGTH == p->state;
	    /* What we need to do is re-parse the characters that have
	     * been eaten by the length reader. Setting the state
	     * prevents recursing on the macro char, and the rest are
	     * just digits.  If the state was not MACRO_LENGTH, then
	     * there weren't any leading digits, so we skip that. We
	     * have to store that in a local variable (digits) since
	     * we're going to change state before dumping the leading
	     * digits.
	     */
	    p->state = MB_PARSE_MACRO_RECOVER;
	    ParseChar(MB_MACRO_CHAR, p);
	    if (digits)
	      {
		int n;
		char *s, buff[16];

		sprintf(buff, "%d", p->count);
		for ( n = strlen(buff), s = buff; n > 0 ; --n, ++s)
		  ParseChar(*s, p);
	      }
	    ParseChar(c, p);		/* redo current character */
	  }
	break;
      }
}
	      
/* ------------------------------------------------------------------------- */
int
MBParseChunk(s,p)
     struct mb_chunk *s;
     struct mb_parse_state *p;
{
  int c, n;
  char *buff;

  /* if we're reading a raw object, we want to grab the characters in large
   * groups for efficiency. So, we'll see how many chars are directly
   * available in the current block of the chunk, and send as many as we can
   * to the raw processing function
   */
  while (!MBisChunkEmpty(s) && MB_PARSE_COMPLETE != p->state)
    {
      if (MB_PARSE_RAW_READ == p->state)
	{
	  if (p->count < 0)
	    {
	      fprintf(stderr,"Mangled raw chunk\n");
	      p->state = MB_PARSE_NORMAL;
	      break;
	    }
	  if (NULL == MBraw_filter)
	    n = MBChunkMoveN(&(p->sexp->object.cons.car->object.chunk),
			     s, p->count);
	  else
	    {
	      /* find the first span, pass that to the filter, and
	       * then drop the characters
	       */
	      buff = (char *)MBChunkFindSpan(s, &n);
	      /* if buff is NULL, we are well and truly screwed */
	      if (n >= p->count)
		{
		  n = p->count;
		  p->sexp->object.cons.car = MBraw_filter(buff, n, 0);
		}
	      else
		MBraw_filter(buff, n, p->count - n);
	      MBChunkDropChars(s, n);	/* been processed, remove */
	    }

	  /* processed the data, do clean up */
	  if ((p->count -= n) <= 0)		/* all done */
	    {
	      /* go back to a normal parsing state */
	      p->state = MB_PARSE_NORMAL;
	      /* the raw object counts as a delimiter */
	      p->delimiter = 1;
	    }
	}
      else
	ParseChar(MBChunkGetChar(s), p);
    }

  return MB_PARSE_COMPLETE == p->state;
}
	      
/* ------------------------------------------------------------------------- */
struct mb_object *
MBExtractSexp(p) struct mb_parse_state *p;
{
  struct mb_object *val = NULL;

  if (MB_PARSE_COMPLETE == p->state)
    {
      val = p->base.object.cons.car;
      p->base.object.cons.car = p->base.object.cons.cdr = NULL;
      p->sexp = &(p->base);
      p->state = MB_PARSE_NORMAL;
    }
  return val;
}
/* ------------------------------------------------------------------------- */
t_sexp
MBparse_Cstring(str) char *str;
{
  t_sexp s_str, e;
  struct mb_parse_state ps;

  if (NULL == str) return NULL;
  MBInitializeParseState(&ps);
  s_str = MBput_Cstring(MBGetString(), str);
  MBParseChunk(&(s_str->object.chunk), &ps);
  e = MBExtractSexp(&ps);
  MBfree(s_str);
  MBResetParseState(&ps);		/* free up any dangling stuff */
  return e;
}
/* ------------------------------------------------------------------------- */
