/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Mark Allender (allender@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*
 *  This file contains the code that will communicate with the
 *  widget server.  There will probably be quite a bit of code
 *  simply because of the length of the widget server messages.
*/

#include <stdio.h>
#include <mbus/api.h>
#include "graph.h"
#include "elision.h"
#include "extern.h"
#include "callback.h"			/* we use this header for structure */

extern t_mb_connection conn;

#define ADD 0
#define REMOVE 1

/*
 *  Some routines that send out very common things between all message,
 *  namely header, and prompt information.
*/
void
TransmitHeader()
{
  extern char *cb_ws_domain, *cb_user, *cb_ui_domain, *cb_base_domain;
  char buf[1000];

  sprintf (buf, "(\"ws\" \"%s.%s.%s.%s\" ",
	   cb_ws_domain, cb_user, cb_ui_domain, cb_base_domain);
  MBtransmit_Cstring(conn, buf);
  MBtransmit_Cstring(conn, "(shell browser ");
}

void
TransmitRadioId(label, key, value, test_value)
     char *label, *key;
     int value, test_value;
{
  MBtransmit_Cstring(conn, "(id \"");
  MBtransmit_Cstring(conn, label);
  MBtransmit_Cstring(conn, "\" \"");
  MBtransmit_Cstring(conn, key);
  MBtransmit_Cstring(conn, "\"");
  if (value == test_value)
    MBtransmit_Cstring(conn, " :select t");
  MBtransmit_Cstring(conn, ")");
}


void
TransmitPrompt()
{
  MBtransmit_Cstring(conn, "(prompt id ((transmit \"Transmit\"");
  MBtransmit_Cstring(conn, ":action (:transmit :delete-top))");
  MBtransmit_Cstring(conn, "(cancel \"Cancel\" :action :delete-top)))");
}

void
TransmitFooter(tag, values, args, title)
     char *tag, *values, *args, *title;
{
  extern char *cb_ws_domain, *cb_user, *cb_ui_domain, *cb_base_domain;
  extern char *cb_browser_domain;
  char buf[1000];

  MBtransmit_Cstring(conn, ":edtibale t :type :prompt :tag ");
  MBtransmit_Cstring(conn, tag);
  if (NULL != values) {
    MBtransmit_Cstring(conn, " :values (");
    MBtransmit_Cstring(conn, values);
    MBtransmit_Cstring(conn, ")");
  }
  if (NULL != args) {
    MBtransmit_Cstring(conn, " :args (");
    MBtransmit_Cstring(conn, args);
    MBtransmit_Cstring(conn, ")");
  }
  if (NULL != title) {
    MBtransmit_Cstring(conn, " :title \"");
    MBtransmit_Cstring(conn, title);
    MBtransmit_Cstring(conn, "\"");
  }
  MBtransmit_Cstring(conn, ":header \"\\\"browser\\\" ");
  sprintf (buf, "\\\"%s.%s.%s.%s\\\"\" ",
	   cb_browser_domain, cb_user, cb_ui_domain, cb_base_domain);
  MBtransmit_Cstring(conn, buf);
  MBtransmit_Cstring(conn, " ))");
}

/*
 *  The routine AddTypetoList takes a list of types, and adds a new
 *  type to the list if one currently does not exist in the list
*/
void
AddTypetoList(list, type)
     struct FunctionList **list;
     char *type;
{
  struct FunctionList *ptr, *prev, *new_type;

  prev = *list;
  for (ptr = *list; ptr; ptr = ptr->next) {
    if (!strcmp(ptr->function, type))
      return;
    prev = ptr;
  }
  new_type = (struct FunctionList *)malloc(sizeof(struct FunctionList));
  new_type->function = type;
  new_type->next = NULL;
  if (prev == NULL)
    *list = new_type;
  else
    prev->next = new_type;
}

/*
 *  The function QueryforActiveType sets up a prompt label to query the user
 *  for an active type to select.  Note that we are abusing the structure
 *  FunctionList.  I needed a linked list of strings, and since one already
 *  existed, I didn't want to create another one.
*/
void
QueryforActiveType(graph, node)
     Graph *graph;
     Vertex *node;
{
  char buf[1000], *type;
  Vertex *this_node;
  struct FunctionList *type_list, *this_type;

  type_list = NULL;
  type = "No type";
  AddTypetoList(&type_list, type);
  if (node)
    type = node->values->type;

  for (this_node = graph->v; this_node; this_node = this_node->next)
    AddTypetoList(&type_list, this_node->values->type);

  TransmitHeader();
  MBtransmit_Cstring(conn, "(array id (");
  MBtransmit_Cstring(conn, "(radio id :active_type (");
  while (this_type = type_list) {
    type_list = this_type->next;
    MBtransmit_Cstring(conn, "(id \"");
    MBtransmit_Cstring(conn, this_type->function);
    MBtransmit_Cstring(conn, "\" \"");
    MBtransmit_Cstring(conn, this_type->function);
    MBtransmit_Cstring(conn, "\"");
    if (!strcmp(type, this_type->function))
      MBtransmit_Cstring(conn, " :select t");
    MBtransmit_Cstring(conn, ")");
    free(type_list);
  }
  MBtransmit_Cstring(conn, ") :title \"List of Node Types\" )");
  TransmitPrompt();
  MBtransmit_Cstring(conn, ") :width 175 :height 300 )");
  sprintf (buf, "\"%s\" () () :ws t", LookupId(graph));
  TransmitFooter("\"graph\"", NULL, buf, "Select Active Graph");
}
/*
 *  The function QueryforNode will take default values for node labels
 *  and type, and then send a message to the widget server to query
 *  for new values.
*/
void
QueryforNode(i_name, label, type, short_l, long_l, graph)
     char *i_name, *label, *type, *short_l, *long_l;
     Graph *graph;
{
  char buf[1000];

  TransmitHeader();
  MBtransmit_Cstring(conn, "(array id (");
  sprintf (buf, "(field id \"Label\" \"%s\" :key :label)", label);
  MBtransmit_Cstring(conn, buf);
  sprintf (buf, "(field id \"Type\" \"%s\" :key :type)", type);
  MBtransmit_Cstring(conn, buf);
  sprintf (buf, "(field id \"Short label\" \"%s\" :key :short)", short_l);
  MBtransmit_Cstring(conn, buf);
  sprintf (buf, "(field id \"Long label\" \"%s\" :key :long)", long_l);
  MBtransmit_Cstring(conn, buf);
  TransmitPrompt();
  MBtransmit_Cstring(conn, "))");
  sprintf (buf, "\"%s\" :ws t", LookupId(graph));
  TransmitFooter(i_name, NULL, buf, "Select Node Values");
}

/*
 *  this is the routine that will ask the widet server to create the
 *  elision control panel so that the user may choose what the
 *  current elision policy should be.  There will be quite a few options
 *  on this board, so the routine may look kind of gross.
*/
void
ElisionControlPanel(widget, graph, call_data)
     Widget *widget;
     caddr_t graph;
     caddr_t call_data;
{
  char buf[5000];
  short epolicy;

/*
 *  Create the message
*/
  TransmitHeader();
  MBtransmit_Cstring(conn, "(array id (");
  MBtransmit_Cstring(conn, "(radio id :inbound_policy (");
/*
 *  Set up the radio buttons for the inbound edge policies.
*/
  epolicy = ((Graph *)graph)->elision_policy & INBOUND;
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Unhide Node", "unhide_node", epolicy, UNHIDE_NODE);
  TransmitRadioId("Unhide Subgraph", "unhide_subgraph", epolicy, UNHIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Inbound Mechanisms\" )");
  MBtransmit_Cstring(conn, "(radio id :inbound_final (");
  epolicy = (((Graph *)graph)->elision_policy & INFINAL) >> 9;
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to Supernode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  MBtransmit_Cstring(conn, ") :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Final Inbound Edges\" )");
/*
 *  set up the buttons for outbound edge policies.
*/
  epolicy = (((Graph *)graph)->elision_policy & OUTBOUND) >> 3;
  MBtransmit_Cstring(conn, "(radio id :outbound_policy (");
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Hide Node", "hide_node", epolicy, HIDE_NODE);
  TransmitRadioId("Hide Subgraph", "hide_subgraph", epolicy, HIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Outbound Mechanisms\" )");
  MBtransmit_Cstring(conn, "(radio id :outbound_final (");
  epolicy = (((Graph *)graph)->elision_policy & OUTFINAL) >> 10;
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to Supernode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  MBtransmit_Cstring(conn, ") :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Final Outbound Edges\" )");
/*
 * Set up the buttons for undirected edge policies.
*/
  epolicy = (((Graph *)graph)->elision_policy & UNDIRECTED) >> 6;
  MBtransmit_Cstring(conn, "(radio id :undirected_policy (");
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Conenct to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Unhide Node", "unhide_node", epolicy, UNHIDE_NODE);
  TransmitRadioId("Unhide Subgraph", "unhide_subgraph", epolicy, UNHIDE_SUB);
  TransmitRadioId("Hide Node", "hide_node", epolicy, HIDE_NODE);
  TransmitRadioId("Hide Subgraph", "hide_subgraph", epolicy, HIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal");
  MBtransmit_Cstring(conn, " :title \"Undirected Edge Mechanisms\" )");
  MBtransmit_Cstring(conn, "(radio id :undirected_final (");
  epolicy = (((Graph *)graph)->elision_policy & UNFINAL) >> 11;
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to Supernode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  MBtransmit_Cstring(conn, ") :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Final Undirected Edges\" )");

  MBtransmit_Cstring(conn, "(radio id :label_policy (");
  TransmitRadioId("Prompt for Label", "query_for_label",
		  ((Graph *)graph)->label_policy, QUERY_FOR_LABEL);
  TransmitRadioId("Assign Root Node Values", "root_is_label",
		  ((Graph *)graph)->label_policy, ROOT_IS_LABEL);
  MBtransmit_Cstring(conn, ") :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Label Policy for Node Elision\" )");
  TransmitPrompt();
  MBtransmit_Cstring(conn, "))");	/* close off array */
  sprintf (buf, "\"%s\" () () :ws t", LookupId((Graph *)graph));
  TransmitFooter("\"graph\"", NULL, buf, "Change Elision Policy");
}

/*
 *  This code will pop up a dialog that allows the user to change the
 *  antielision policies.   Similar to the above code.
*/
void
AntiElisionControlPanel(widget, graph, call_data)
     Widget *widget;
     caddr_t graph;
     caddr_t call_data;
{
  char buf[5000];
  short epolicy;

/*
 *  Create the message
*/
  TransmitHeader();
  MBtransmit_Cstring(conn, "(array id (");
/*
 *  Set up the radio buttons for the inbound edge policies.
*/
  MBtransmit_Cstring(conn, "(radio id :anti_inbound_policy (");
  epolicy = ((Graph *)graph)->unelision_policy & 0x7;
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Unhide Node", "unhide_node", epolicy, UNHIDE_NODE);
  TransmitRadioId("Unhide Subgraph", "unhide_subgraph", epolicy, UNHIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Inbound Mechanisms\" )");
/*
 *  set up the buttons for outbound edge policies.
*/
  epolicy = (((Graph *)graph)->unelision_policy & 0x38) >> 3;
  MBtransmit_Cstring(conn, "(radio id :anti_outbound_policy (");
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Connect to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Hide Node", "hide_node", epolicy, HIDE_NODE);
  TransmitRadioId("Hide Subgraph", "hide_subgraph", epolicy, HIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal ");
  MBtransmit_Cstring(conn, ":title \"Outbound Mechanisms\" )");
/*
 * Set up the buttons for undirected edge policies.
*/
  epolicy = (((Graph *)graph)->unelision_policy & 0x1c0) >> 6;
  MBtransmit_Cstring(conn, "(radio id :anti_undirected_policy (");
  TransmitRadioId("Hide Edge", "hide_edge", epolicy, HIDE_EDGE);
  TransmitRadioId("Conenct to SuperNode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Unhide Node", "unhide_node", epolicy, UNHIDE_NODE);
  TransmitRadioId("Unhide Subgraph", "unhide_subgraph", epolicy, UNHIDE_SUB);
  TransmitRadioId("Hide Node", "hide_node", epolicy, HIDE_NODE);
  TransmitRadioId("Hide Subgraph", "hide_subgraph", epolicy, HIDE_SUB);
  MBtransmit_Cstring(conn, ") :columns 2 :orientation :horizontal");
  MBtransmit_Cstring(conn, " :title \"Undirected Edge Mechanisms\" )");
/*
 *  Set up the radio boxes for the unelision mechansism that deals with
 *  an edge between two nodes in different elision sets.
*/
  epolicy = (((Graph *)graph)->unelision_policy & 0xe00) >> 9;
  MBtransmit_Cstring(conn, "(radio id :anti_diff_set_policy (");
  TransmitRadioId("Connect to Supernode", "connect_to_sn", epolicy,
		  CONNECT_TO_SUPERNODE);
  TransmitRadioId("Hide in From Node", "hide_in_from_node", epolicy,
		  HIDE_IN_FROM_NODE);
  TransmitRadioId("Hide in To Node", "hide_in_to_node", epolicy,
		  HIDE_IN_TO_NODE);
  TransmitRadioId("Connect and Hide in From Node",
		  "connect & hide_in_from_node",
		  epolicy, (HIDE_IN_FROM_NODE | CONNECT_TO_SUPERNODE));
  TransmitRadioId("Connect and Hide in To Node",
		  "connect & hide_in_to_node",
		  epolicy, (HIDE_IN_TO_NODE | CONNECT_TO_SUPERNODE));
  MBtransmit_Cstring(conn, ") :columns 3 :orientation :horizontal");
  MBtransmit_Cstring(conn, " :title \"Edge Between Diff. Elision Sets\" )");
  TransmitPrompt();
  MBtransmit_Cstring(conn, "))");
  sprintf (buf, "\"%s\" () () :ws t", LookupId((Graph*)graph));
  TransmitFooter("\"graph\"", NULL, buf, "Change AntiElision Policy");
}

/*
 *  This routine builds up the menu bars that are needed to allow a user
 *  to add an entry into any given display table.
*/
void
TransmitTableMenu(graph_id, table_type, action)
     char *graph_id, *table_type;
     int action;
{
  char values[100], args[100];

  MBtransmit_Cstring(conn, "(menu-bar menu ((button add \"Add Type\" ");
  MBtransmit_Cstring(conn, ":action (shell new-shell (array new-type ");
  MBtransmit_Cstring(conn, "((field id \"Type\" \"\" :key :type)");
  MBtransmit_Cstring(conn, "(field id \"Color\" \"\" :key :color)");
  MBtransmit_Cstring(conn, "(field id \"Font\" \"\" :key :font)");
  MBtransmit_Cstring(conn, "(radio id :noclobber (");
  MBtransmit_Cstring(conn, "(id \"True\" t :select t)");
  MBtransmit_Cstring(conn, "(id \"False\" nil)");
  MBtransmit_Cstring(conn, ") :orientation :horizontal");
  MBtransmit_Cstring(conn, " :title \"Make Entry Noclobber\" )");
  MBtransmit_Cstring(conn, "(prompt id ((transmit \"Transmit\" ");
  MBtransmit_Cstring(conn, ":action (:transmit :delete-top))");
  MBtransmit_Cstring(conn, "(cancel \"Cancel\" :action :delete-top)))))");
  sprintf (values, ":add t :update t :table \"%s\"", table_type);
  sprintf (args, "\"%s\"", graph_id);
  TransmitFooter("\"newdisplayentry\"", values, args,
		 "Add New Table Entry");
  MBtransmit_Cstring(conn, "(button \"remove\" \"Remove Type\" ");
  MBtransmit_Cstring(conn, ":action (shell new-shell (array new-type ");
  MBtransmit_Cstring(conn, "((field id \"Type\" \"\" :key :type)");
  MBtransmit_Cstring(conn, "(prompt id ((transmit \"Transmit\" ");
  MBtransmit_Cstring(conn, ":action (:transmit :delete-top))");
  MBtransmit_Cstring(conn, "(cancel \"Cancel\" :action :delete-top)))))");
  sprintf (values, ":ws t :remove t :table \"%s\"", table_type);
  sprintf (args, "\"%s\"", graph_id);
  TransmitFooter("\"newdisplayentry\"", values, args,
		 "Remove Table Entry");
  MBtransmit_Cstring(conn, ") :special \"remove\" )");
}
/*
 *  A routine to transmit a single display table to the widget server.
 *  This avoids redundant code in all of the routines that generate
 *  these arrays.
*/
void
TransmitDisplayTable(title, id, table, graph_id)
     char *title, *id, *graph_id;
     struct DisplayTable *table;
{
  struct DisplayTable *entry;
  char buf[1000];

  MBtransmit_Cstring(conn, "(label id \"");
  MBtransmit_Cstring(conn, title);
  MBtransmit_Cstring(conn, "\" ");
  MBtransmit_Cstring(conn, ":frame t :align :center)");
  TransmitTableMenu(graph_id, id);
  MBtransmit_Cstring(conn, "(array \"");
  MBtransmit_Cstring(conn, id);
  MBtransmit_Cstring(conn, "\" (");
  for (entry = table; entry; entry = entry->next){
    MBtransmit_Cstring(conn, "(array \"");
    MBtransmit_Cstring(conn, entry->type);
    MBtransmit_Cstring(conn, "\" (");
    MBtransmit_Cstring(conn, "(field id \"Type\" \"");
    MBtransmit_Cstring(conn, entry->type);
    MBtransmit_Cstring(conn, "\" :foreground \"");
    MBtransmit_Cstring(conn, entry->color_name);
    MBtransmit_Cstring(conn, "\" :font \"");
    MBtransmit_Cstring(conn, entry->font_name);
    MBtransmit_Cstring(conn, "\") (field id \"Color\" \"");
    MBtransmit_Cstring(conn, entry->color_name);
    MBtransmit_Cstring(conn, "\" :foreground \"");
    MBtransmit_Cstring(conn, entry->color_name);
    MBtransmit_Cstring(conn, "\" :font \"");
    MBtransmit_Cstring(conn, entry->font_name);
    MBtransmit_Cstring(conn, "\" :key :color) (field id \"Font\" \"");
    MBtransmit_Cstring(conn, entry->font_name);
    MBtransmit_Cstring(conn, "\" :font \"");
    MBtransmit_Cstring(conn, entry->font_name);
    MBtransmit_Cstring(conn, "\" :foreground \"");
    MBtransmit_Cstring(conn, entry->color_name);
    MBtransmit_Cstring(conn, "\" :key :font))");
    MBtransmit_Cstring(conn, " :orientation :horizontal :wrap t");
    sprintf (buf, " :tag \"%s\"", entry->type);
    MBtransmit_Cstring(conn, buf);
    MBtransmit_Cstring(conn, ":values (:add t :update t)");
    MBtransmit_Cstring(conn, " )");
  }
  MBtransmit_Cstring(conn, ") :wrap t :width 550)");
}

/*
 *  This next bit of code will show the display tables in a scrollable
 *  list so the user may get information about what types are actually
 *  set to what value.
*/
void
ShowGraphDisplayTable(widget, graph, call_data)
     Widget *widget;
     caddr_t graph;
     caddr_t call_data;
{
  char buf[100], *graph_id;
  struct DisplayTable *entry, *list;
  Graph *my_graph;

  my_graph = (Graph *)graph;
  graph_id = LookupId(my_graph);
  TransmitHeader();
  MBtransmit_Cstring(conn, "(array display-lists (");
  TransmitDisplayTable("Edge Display Table", "graph-edges",
		       my_graph->display_table->edges, graph_id);
  TransmitDisplayTable("Node Display Table", "graph-nodes",
		       my_graph->display_table->nodes, graph_id);
  TransmitDisplayTable("Common Display Table", "graph-common",
		       my_graph->display_table->common, graph_id);
  TransmitPrompt();
  MBtransmit_Cstring(conn, ") :wrap t )");
  sprintf (buf, "\"%s\"", graph_id);
  TransmitFooter("\"displaytable\"", ":ws t", buf,
		 "Edit Display Table Values");
}

void
ShowBrowserDisplayTable(widget, graph, call_data)
     Widget *widget;
     caddr_t graph;
     caddr_t call_data;
{
  char buf[100], *graph_id;
  struct DisplayTable *entry, *list;
  Graph *my_graph;

  my_graph = (Graph *)graph;
  graph_id = LookupId(my_graph);
  TransmitHeader();
  MBtransmit_Cstring(conn, "(array display-lists (");
  TransmitDisplayTable("Edge Display Table", "browser-edges",
		       BrowserDisplays->edges, graph_id);
  TransmitDisplayTable("Node Display Table", "browser-nodes",
		       BrowserDisplays->nodes, graph_id);
  TransmitDisplayTable("Common Display Table", "browser-common",
		       BrowserDisplays->common, graph_id);
  TransmitPrompt();
  MBtransmit_Cstring(conn, ") :wrap t )");
  TransmitFooter("\"displaytable\"", ":ws t", "\"Browser\"",
		 "Edit Display Table Values");
}

void
AddDisplayEntry(entry, id)
     struct DisplayTable *entry;
     char *id;
{
  extern char *cb_ws_domain, *cb_user, *cb_ui_domain, *cb_base_domain;
  char buf[1000];

  sprintf (buf, "(\"ws\" \"%s.%s.%s.%s\" ",
	   cb_ws_domain, cb_user, cb_ui_domain, cb_base_domain);
  MBtransmit_Cstring(conn, buf);
  MBtransmit_Cstring(conn, "(action \"");
  MBtransmit_Cstring(conn, id);
  MBtransmit_Cstring(conn, "\" :editable t :add ");
  MBtransmit_Cstring(conn, "((array \"");
  MBtransmit_Cstring(conn, entry->type);
  MBtransmit_Cstring(conn, "\" (");
  MBtransmit_Cstring(conn, "(field id \"Type\" \"");
  MBtransmit_Cstring(conn, entry->type);
  MBtransmit_Cstring(conn, "\" :foreground \"");
  MBtransmit_Cstring(conn, entry->color_name);
  MBtransmit_Cstring(conn, "\" :font \"");
  MBtransmit_Cstring(conn, entry->font_name);
  MBtransmit_Cstring(conn, "\") (field id \"Color\" \"");
  MBtransmit_Cstring(conn, entry->color_name);
  MBtransmit_Cstring(conn, "\" :foreground \"");
  MBtransmit_Cstring(conn, entry->color_name);
  MBtransmit_Cstring(conn, "\" :font \"");
  MBtransmit_Cstring(conn, entry->font_name);
  MBtransmit_Cstring(conn, "\" :key :color) (field id \"Font\" \"");
  MBtransmit_Cstring(conn, entry->font_name);
  MBtransmit_Cstring(conn, "\" :font \"");
  MBtransmit_Cstring(conn, entry->font_name);
  MBtransmit_Cstring(conn, "\" :foreground \"");
  MBtransmit_Cstring(conn, entry->color_name);
  MBtransmit_Cstring(conn, "\" :key :font))");
  MBtransmit_Cstring(conn, " :orientation :horizontal :wrap t");
  sprintf (buf, " :tag \"%s\"", entry->type);
  MBtransmit_Cstring(conn, buf);
  MBtransmit_Cstring(conn, ":values (:add t :update t)");
  MBtransmit_Cstring(conn, " ))))");
}

void
RemoveDisplayEntry(table, type)
     char *table, *type;
{
  extern char *cb_ws_domain, *cb_user, *cb_ui_domain, *cb_base_domain;
  char buf[1000];

  sprintf (buf, "(\"ws\" \"%s.%s.%s.%s\" ",
	   cb_ws_domain, cb_user, cb_ui_domain, cb_base_domain);
  MBtransmit_Cstring(conn, buf);
  MBtransmit_Cstring(conn, "(action \"");
  MBtransmit_Cstring(conn, table);
  MBtransmit_Cstring(conn, "\" :delete (\"");
  MBtransmit_Cstring(conn, type);
  MBtransmit_Cstring(conn, "\")))");
}

