/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:	 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
*/

#include <stdio.h>
#include <ctype.h>
#include <X11/X.h>
#include <mbus/api.h>
#include <mbus/keyword.h>
#include "extern.h"
#include "graph.h"
#include "callback.h"
#include "elision.h"

static struct keyword_entry_struct KeyKeywords[] =
{
  { ":add", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":update", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":remove", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":shift", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":ctrl", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":meta", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":up", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":down", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":makeassoc", NULL, KEYWORD_SEXP},
  { ":updateassoc", NULL, KEYWORD_SEXP},
  { ":removeassoc", NULL, KEYWORD_SEXP},
};

static struct keyword_entry_struct FuncKeywords[] =
{
  { ":add", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":update", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":remove", (void *)KEYWORD_FALSE, KEYWORD_FLAG},
  { ":data", (void *)NULL, KEYWORD_RAW},
  { ":tag", (void *)NULL, KEYWORD_RAW},
  { ":domain", (void *)NULL, KEYWORD_RAW},
  { ":function", (void *)NULL, KEYWORD_COOKED},
};

static struct CallbackTable CallbackTable[] =
{
  {"f.redraw", RedrawCallback},
  {"f.randomize", RandomizeCallback},
  {"f.toggle_names", ToggleNamesCallback},
  {"f.send_message", SendMessageCallback},
  {"f.elide_node", ElideNodeCallback},
  {"f.unelide_node", UnelideNodeCallback},
  {"f.elide_swept_nodes", ElideSweptNodesCallback},
  {"f.unelide_swept_nodes", UnelideSweptNodesCallback},
  {"f.elide_selected_nodes", ElideSelectedNodesCallback},
  {"f.unelide_selected_nodes", UnelideSelectedNodesCallback},
  {"f.fix_node", FixNodeCallback},
  {"f.unfix_node", UnfixNodeCallback},
  {"f.fix_swept_nodes", FixSweptNodesCallback},
  {"f.unfix_swept_nodes", UnfixSweptNodesCallback},
  {"f.fix_selected_nodes", FixSelectedNodesCallback},
  {"f.unfix_selected_nodes", UnfixSelectedNodesCallback},
  {"f.fix_all", FixAllCallback},
  {"f.unfix_all", UnfixAllCallback},
  {"f.select_node", SelectNodeCallback},
  {"f.unselect_node", UnselectNodeCallback},
  {"f.select_edge", SelectEdgeCallback},
  {"f.unselect_edge", UnselectEdgeCallback},
  {"f.select_swept_nodes", SelectSweptNodesCallback},
  {"f.unselect_swept_nodes", UnselectSweptNodesCallback},
  {"f.select_all", SelectAllCallback},
  {"f.unselect_all", UnselectAllCallback},
  {"f.move_node", MoveNodeCallback},
  {"f.move_selected", MoveSelectedCallback},
  {"f.choose_active_type", ChooseActiveCallback},
  {"f.select_active_types", SelectActiveCallback},
  {"f.unselect_active_types", UnselectActiveCallback}
};

void
ParseKeyBindings(binding_list, key_list, noclobber)
     struct mb_object *binding_list;
     struct KeyBindings **key_list;
     short noclobber;
{
  struct mb_object *bobject;
  char key, *str;
  short key_mask, ud_mask;
  struct KeyBindings *binding, *new_binding, *prev_binding;
  struct FunctionList *new_funcs, *up_funcs, *rm_funcs;
  extern struct FunctionList *GetAssociations();
    
/*
 *  Scan down the list of key_bindings, and get the appropriate material
 *  for each key binding.
*/
  for (; MBconsp(binding_list); binding_list = MBcdr(binding_list)) {
    bobject = MBcar(binding_list);
    str = MBCstring(MBcar(bobject));
    if (!str)
      continue;
    MBparse_keywords (MBcdr(bobject), KeyKeywords,
		      sizeof(KeyKeywords)/sizeof(struct keyword_entry_struct));
/*
 *  Get the parts of the keywords that we will use often.  Notice that
 *  we store either the button number or the character in the "key"
 *  variable.  We can do this because the buttons will get mapped to
 *  the integer 1, 2, or 3, and that any useable character on the keyboard
 *  will not have this value.
*/
    key_mask = 0;
    ud_mask = 0;

    if (!strncmp(str, "button", 6))
	key = (*(str+6) - '0');
    else if (isupper(*str))
	key = tolower(*str);
    else
	key = *str;
    if ((t_keyword_flag)KeyKeywords[SHIFT].result == KEYWORD_TRUE)
      key_mask = key_mask | ShiftMask;
    if ((t_keyword_flag)KeyKeywords[CTRL].result == KEYWORD_TRUE)
      key_mask = key_mask | ControlMask;
    if ((t_keyword_flag)KeyKeywords[META].result == KEYWORD_TRUE)
      key_mask = key_mask | Mod1Mask;	/* is this right????? */
    if ((t_keyword_flag)KeyKeywords[UP].result == KEYWORD_TRUE)
      ud_mask = ud_mask | UP_MASK;
    if ((t_keyword_flag)KeyKeywords[DOWN].result == KEYWORD_TRUE)
      ud_mask = ud_mask | DOWN_MASK;
/*
 *  Get the association lists that the user has sent.  I guess that I am
 *  going to allow all three to be present at once.  I will just
 *  process them in a given order.
*/
    new_funcs = GetAssociations(KeyKeywords[MAKEASSOC].result);
    up_funcs = GetAssociations(KeyKeywords[UPDATEASSOC].result);
    rm_funcs = GetAssociations(KeyKeywords[REMOVEASSOC].result);
/*
 *  We need to check and see if there is a binding already in the key table.
 *  Then we can decide what to do based on that.
*/
    prev_binding = NULL;
    binding = *key_list;
    while (binding != NULL) {
      if ((binding->key == key) && (binding->keymask & key_mask) &&
	  (binding->udmask && ud_mask))
	break;
      prev_binding = binding;
      binding = binding->next;
    }
/*
 *  Check to see if we can even clobber this key binding.  If not, then
 *  just get the heck out.
*/
    if (binding != NULL) {
      if (binding->noclobber == TRUE)
	return;
/*
 *  We have found a key binding in the keybinding list.  We can only
 *  either update this keybinding or remove it entirely.
*/
      if ((t_keyword_flag)KeyKeywords[UPDATEKEY].result == KEYWORD_TRUE) {
/*
 *  We will deal with the function associations first.  Note that this
 *  is just a list of functions that will activate on the callback.  When
 *  specifying to update the association list, the user is just going
 *  to *add* to the list at this point.  The makeassoc keyword is used
 *  to remove all entries in the table, and add the new ones specified.  From
 *  there, you can remove, and add them (add them with the updateassoc
 *  keyword.  See the programmer's manual for more detail.
*/
/*
 *  deal with updates first.  First, we check to see if there are even
 *  any functions in the list.  If not, then we just put the list of
 *  function names onto the key bindings function-list;
*/
	if (binding->func_list != NULL)
	  FreeAssociations(&binding->func_list);
	binding->func_list = new_funcs;
/*
 *  we have some on the function list already, we need to add these new
 *  associations in at the end.
*/
	if (up_funcs != NULL)
	  AddAssociations(&binding->func_list, up_funcs);
/*
 *  Now, we will check for the for the functions to be removed.  This 
 *  is (just a simple little) list delete routine.
*/
	if (rm_funcs != NULL)
	  RemoveAssociations(&binding->func_list, rm_funcs);
      }
/*
 *  Done with updating keyword.  Now....if the user specifies remove keyword,
 *  then we need to remove keybinding from the key-binding-table.
*/
      else {
	if ((t_keyword_flag)KeyKeywords[REMOVEKEY].result == KEYWORD_TRUE) {
	  if (prev_binding == NULL)
	    *key_list = binding->next;
	  else
	    prev_binding->next = binding->next;
	  FreeAssociations(binding->func_list);
	  free(binding);
	}
      }
    }
    else {
/*
 *  No key binding was found, so we must check to see if the add keyword
 *  was specified.  If it was, then we can add this binding onto the
 *  specifed key binding list.
*/
      if ((t_keyword_flag)KeyKeywords[ADDKEY].result == KEYWORD_TRUE) {
	new_binding = NEW(struct KeyBindings);
	new_binding->keymask = key_mask;
	new_binding->udmask = ud_mask;
	new_binding->key = key;
	new_binding->noclobber = noclobber;
/*
 *  add, update, and remove the associations
*/
	new_binding->func_list = new_funcs;
	if (up_funcs != NULL)
	  AddAssociations(&binding->func_list, up_funcs);
/*
 *  Now, we will check for the for the functions to be removed.  This 
 *  is (just a simple little) list delete routine.
*/
	if (rm_funcs != NULL)
	  RemoveAssociations(&binding->func_list, rm_funcs);
/*
 *  Now, attach this key binding to the key binding list passed in.
*/
	new_binding->next = *key_list;
	*key_list = new_binding;
      }
    }
  }
}

void
ParseFunctionBindings(functions, func_list, noclobber)
     struct mb_object *functions;
     struct FunctionBindings **func_list;
     short noclobber;
{
  struct mb_object *binding;
  char *f_type;				/* predefined type or bus message */
  char *data;				/* possible data message for bus */
  char *tag, *domain, *name;
  struct FunctionBindings *new_binding, *func_binding, *prev_binding;

  for (; MBconsp(functions); functions = MBcdr(functions)) {
    binding = MBcar(functions);
/*
 *  First, let's get all of the neccessary information out of the keywords
*/
    name = MBCstring(MBcar(binding));
    MBparse_keywords(MBcdr(binding), FuncKeywords, sizeof(FuncKeywords) /
		    sizeof (struct keyword_entry_struct));
    f_type = FuncKeywords[FUNCTION].result;
    data = FuncKeywords[DATA].result;
    tag = FuncKeywords[TAG].result;
    domain = FuncKeywords[FDOMAIN].result;
/*
 *  Now, as before, we want to check the update, remove, and add keywords
 *  to really find what actions that we have to perform.
*/
    prev_binding = NULL;
    for (func_binding = *func_list; func_binding != NULL;
	 func_binding = func_binding->next) {
      if (!strcmp(func_binding->name, name)) {
/*
 *  If the names match up, then we have a function-binding already existing in
 *  the database for this callback.  We have the pointer now....break out and
 *  check for some things.
*/
	if (func_binding->noclobber == TRUE)
	  return;
	if ((t_keyword_flag)FuncKeywords[UPFUNCTION].result == KEYWORD_TRUE) {
/*
 *  We must update the information that is stored in the function_binding
 *  table.
*/
	  if (FuncKeywords[FUNCTION].found == 1)
	    func_binding->function = f_type;
	  if (FuncKeywords[DATA].found == 1)
	    func_binding->data = data;
	  if (FuncKeywords[TAG].found == 1)
	    func_binding->tag = tag;
	  if (FuncKeywords[FDOMAIN].found == 1)
	    func_binding->domain = domain;
	  break;
	}
/*
 *  Done with updating keyword.  Now....if the user specifies remove keyword,
 *  then we need to remove keybinding from the key-binding-table.
*/
	else {
	  if ((t_keyword_flag)FuncKeywords[RMFUNCTION].result == KEYWORD_TRUE) {
	    if (prev_binding == NULL)
	      *func_list = func_binding->next;
	    else
	      prev_binding->next = func_binding->next;
	    free(func_binding->name);
	    free(func_binding->data);
	    free(func_binding->function);
	    free(func_binding->tag);
	    free(func_binding->domain);
	    free(func_binding);
	  }
	}
      }
      prev_binding = func_binding;
    }
/*
 *  Now, check and see if a func_binding was found.  If not, and the add 
 *  keyword is true, then we need to add the keybinding to the keybinding
 *  table.
*/
    if ((func_binding == NULL) &&
	((t_keyword_flag)FuncKeywords[ADDFUNCTION].result == KEYWORD_TRUE)) {
      new_binding=NEW(struct FunctionBindings);
      new_binding->name = name;
      new_binding->function = f_type;
      new_binding->data = data;
      new_binding->tag = tag;
      new_binding->domain = domain;
      new_binding->noclobber = noclobber;
      /*
       * Now, attach this to the graph.
       */
      new_binding->next = *func_list;
      *func_list = new_binding;
    }
  }
}

void
ParseCallbacks(callback_list, key_list, func_list, noclobber)
     struct mb_object *callback_list;
     struct KeyBindings **key_list;
     struct FunctionBindings **func_list;
     short noclobber;
{
  struct mb_object *key_bindings, *func_bindings;
  char *f_name;

  key_bindings = MBnth(callback_list, 0);
  func_bindings = MBnth(callback_list, 1);
  ParseKeyBindings(key_bindings, key_list, noclobber);
  ParseFunctionBindings(func_bindings, func_list, noclobber);
}

/*
 *  FindCallbackFunction looks up the callback function and then
 *  returns the function to run and the data to use in the function.
*/

void
FindCallbackFunction (function_name, bindings, CallbackFunc, CallbackData)
  char *function_name;
  struct FunctionBindings *bindings;
  int (**CallbackFunc)();
  struct FunctionBindings **CallbackData;
{
  struct FunctionBindings *fbinding;
  char *function;
  int Count;
  
  *CallbackFunc = NULL;
  *CallbackData = NULL;
  
  for (fbinding = bindings; fbinding; fbinding = fbinding->next) {
    if (!strcmp(function_name, fbinding->name)) {
      for (Count = 0; Count < ARRAY_SIZE(CallbackTable); Count++)
      {
	if (!strcmp(CallbackTable[Count].FunctionName, fbinding->function))
	{
	  *CallbackData = fbinding;
	  *CallbackFunc = CallbackTable[Count].Function;
	  return;
	}
      }
    }
  }
}

/*
 * FindCallback takes a CallbackInfo struct and returns the function
 * list that corresponds to that key/button.
 */
struct FunctionList *
FindCallback (CallbackInfo)
  struct CallbackInfo *CallbackInfo;
{
  struct KeyBindings *key_bindings;

  key_bindings = graph->callback_keys;
  while (key_bindings != NULL) {
    if ((key_bindings->key==(char)CallbackInfo->keysym) &&
	!(key_bindings->keymask ^ CallbackInfo->mod_mask))
	return key_bindings->func_list;
    key_bindings = key_bindings->next; /* continuation of while loop */
  }

  key_bindings = CallbackKeys;

  while (key_bindings != NULL) {
    if ((key_bindings->key==(char)CallbackInfo->keysym) &&
	!(key_bindings->keymask ^ CallbackInfo->mod_mask))
	return key_bindings->func_list;
    key_bindings = key_bindings->next; /* continuation of while loop */
  }
  return NULL;
}

/* InitCallbackInfo will fill in the callback information.  If the
 * key/button press/release is deemed uninteresting, then we will 
 * return FALSE (i.e., we didn't fill in the data.
 */
int
InitCallbackInfo (CallbackInfo, event)
  struct CallbackInfo *CallbackInfo;
  XEvent *event;
{
  char string[31];
  KeySym keysym;
  XComposeStatus composestatus;

  CallbackInfo->data_mask = 0;
  if (event->type == KeyPress)
  {
    CallbackInfo->start_x = event->xkey.x;
    CallbackInfo->start_y = event->xkey.y;
    XLookupString((XKeyEvent *)event, string, 30,
		  &keysym, &composestatus);
    if (isupper((char)keysym))
	CallbackInfo->keysym = (int)tolower((char)keysym);
    else
	CallbackInfo->keysym = (int)keysym;
    CallbackInfo->mod_mask = (event->xkey.state &
			      (ShiftMask | ControlMask | Mod1Mask));
  }
  else if (event->type == ButtonPress)
  {
    CallbackInfo->start_x = event->xbutton.x;
    CallbackInfo->start_y = event->xbutton.y;
    CallbackInfo->keysym = event->xbutton.button;
    CallbackInfo->mod_mask = (event->xkey.state &
			      (ShiftMask | ControlMask | Mod1Mask));
    if (CallbackInfo->keysym == 1)
	CallbackInfo->buttonmask = Button1Mask;
    else if (CallbackInfo->keysym == 2)
	CallbackInfo->buttonmask = Button2Mask;
    else if (CallbackInfo->keysym == 3)
	CallbackInfo->buttonmask = (Button3Mask);
    else
	CallbackInfo->buttonmask = 0;
  }
  else
  {
    return FALSE;
  }
  CallbackInfo->selected_nodes = NULL;
  CallbackInfo->swept_nodes = NULL;
  return TRUE;
}

void
FillCallbackInfo (CallbackInfo, mask)
  struct CallbackInfo *CallbackInfo;
  int mask;
{
  Display *dpy;
  Window win;
  Window root_return, child_return;
  int root_x, root_y, win_x, win_y, x, y;
  unsigned int button_mask;
  
  if (mask & SWEPT_NODES)
      mask |= SWEPT_REGION;	/* If they want the nodes, get region */
  
  if ((mask & CallbackInfo->data_mask) == mask)
  {
    return;			/* Already have all info they want */
  }
  /* Do they want the node that is under the starting location? */
  if ((mask & NODEMASK) && !(CallbackInfo->data_mask & NODEMASK))
  {
    CallbackInfo->node = LocateVertex(CallbackInfo->start_x,
				      CallbackInfo->start_y);
    CallbackInfo->data_mask |= NODEMASK;
  }
  /* Do they want the edge that is under the starting location? */
  if ((mask & EDGEMASK) && !(CallbackInfo->data_mask & EDGEMASK))
  {
    CallbackInfo->edge = LocateEdge(CallbackInfo->start_x,
				    CallbackInfo->start_y);
    CallbackInfo->data_mask |= EDGEMASK;
  }
  /* Do they want the selected nodes? */
  if ((mask & SELECTED_NODES) && !(CallbackInfo->data_mask & SELECTED_NODES))
  {
    CallbackInfo->selected_nodes = SelectedNodes();
    CallbackInfo->data_mask |= SELECTED_NODES;
  }
  /* Do they want the swept region? */
  if ((mask & SWEPT_REGION) && !(CallbackInfo->data_mask & SWEPT_REGION))
  {
    /* This requires us to put up a box that represents their swept
     * area.  When they are done with the region, we will set the end
     * points.
     */
    x = CallbackInfo->start_x;
    y = CallbackInfo->start_y;
    dpy = XtDisplay(graph->viewing_area.widget);
    win = XtWindow(graph->viewing_area.widget);
    XSetForeground(dpy, graph->xor_gc, graph->foreground ^ graph->background);
    DrawSweptBox(graph->xor_gc,
		    CallbackInfo->start_x, CallbackInfo->start_y,
		    x, y);
    do
    {
      XQueryPointer (dpy, win,
                     &root_return, &child_return,
                     &root_x, &root_y, &win_x, &win_y, &button_mask);
      if (x != win_x || y != win_y)
      {
	DrawSweptBox(graph->xor_gc,
			CallbackInfo->start_x, CallbackInfo->start_y,
			x, y);
	x = win_x;
	y = win_y;
	DrawSweptBox(graph->xor_gc,
			CallbackInfo->start_x, CallbackInfo->start_y,
			x, y);
      }
    } while (button_mask & CallbackInfo->buttonmask);
    DrawSweptBox(graph->xor_gc,
		    CallbackInfo->start_x, CallbackInfo->start_y,
		    x, y);
    CallbackInfo->end_x = x;
    CallbackInfo->end_y = y;
    CallbackInfo->data_mask |= SWEPT_REGION;
  }
  /* Do they want the swept nodes? */
  if ((mask & SWEPT_NODES) && !(CallbackInfo->data_mask & SWEPT_NODES))
  {
    CallbackInfo->swept_nodes = SweptNodes(CallbackInfo);
    CallbackInfo->data_mask |= SWEPT_NODES;
  }
}  

void
FreeCallbackInfo (CallbackInfo)
  struct CallbackInfo *CallbackInfo;
{
  FreeNodeList(&(CallbackInfo->swept_nodes));
  FreeNodeList(&(CallbackInfo->selected_nodes));
}
