
;;; widgets.el - handling of the scrollbar and general widget stuff
;;; B<   Brian Kahn   blk@mitre.org   Fri Jul 10 16:55:57 1992
;;; derived from menus.el by Simon Kaplan


(require 'mbus)



(defvar scroll:active t "Scroll:stuff is active.")
(defvar scroll:movement-in-response-to-scrollbars nil "Ignore movement.")

(defvar scroll:buffer nil "Buffer associated with scrollbar.")
(defvar scroll:window nil "Window associated with scrollbar.")
(defvar scroll:window-start 0 "Start pos for the selected buffer.")
(defvar scroll:window-start-line 1 "Line of window-start")
(defvar scroll:buffer-size 0 "Size of selected-buffer")
(defvar scroll:buffer-lines 0 "Lines in selected-buffer")

(defvar scroll:keep-stats t "Keep stats on scroll stuff.")
(defvar scroll:uhoh 0 "Number of bad things that have happened.")



(defun scroll:temp-buffer-p (buffer-name)
  "Is this a temporary buffer (from scrollbar's view)?"
  (or
   (string-match "*Minibuf-" buffer-name)
   (string-match "*Backtrace" buffer-name)
   )
  )





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; NOTE - narrowing/widening has no hook to capture ;;;
;;; Also no hook for resize/split/unsplit of windows ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; the old hook names are honored

(defvar after-update-hook nil)
(defvar window-select-hook nil)
(defvar set-window-buffer-hook nil)
(defvar scroll:saved-after-update-hook after-update-hook)
(defvar scroll:saved-window-select-hook window-select-hook)
(defvar scroll:saved-set-window-buffer-hook set-window-buffer-hook)



(defun scroll:re-vector-hooks ()
  "reset the hook vectors for scrollbars."
  ;; useful when executing in buffer
  (setq window-select-hook
	'scroll:window-select-hook-handler)
  (setq set-window-buffer-hook
	'scroll:set-window-buffer-hook-handler)
  (setq after-update-hook
	'scroll:after-update-hook-handler)
  )



(defun scroll:window-select-hook-handler (&rest args)
  (if scroll:saved-window-select-hook
      (apply scroll:saved-window-select-hook args))
  (apply 'scroll:handle-window-select args)
  (scroll:stats 'select)
  )


(defun scroll:set-window-buffer-hook-handler (&rest args)
  (if scroll:saved-set-window-buffer-hook
      (apply scroll:saved-set-window-buffer-hook args))
  (apply 'scroll:handle-set-window-buffer args)
  (scroll:stats 'set-buffer)
  )


(defun scroll:after-update-hook-handler (&rest args)
  (if scroll:saved-after-update-hook
      (apply scroll:saved-after-update-hook args))
  (when (eq (current-buffer) scroll:buffer)
    (apply 'scroll:handle-after-update args)
    (scroll:stats 'update)
    ))




(defun scroll:handle-window-select (win)
  "The selected window is now WIN."
  (let ((buff (window-buffer win)))
    ;; if win is temporary - keep attachment to other window
    (when (not (scroll:temp-buffer-p (buffer-name buff)))
      (setq scroll:window win)
      (scroll:handle-buffer-change buff)
      )))



(defun scroll:handle-set-window-buffer (win)
  ;; ignore buffer changes in other windows
  (when (eq win scroll:window)
    ;; we have to keep scrollbar on scroll:win, even if in a temp buff
    (scroll:handle-buffer-change (window-buffer win))
    ))



(defun scroll:handle-buffer-change (buff)
  "Set up the vars for BUFF, update scrollbar."
  (when (not (eq scroll:buffer buff))
    (setq scroll:buffer buff)
    ;; update scrollbar for the this buffer
    (scroll:update-scrollbar)
    (scroll:title)
    ))


    
(defun scroll:update-scrollbar ()
  "Update scrollbar."
  (if (and (eq (current-buffer) scroll:buffer)
	   (eq (selected-window) scroll:window))

      (scroll:send-vscroll-update-message)

    ;; else switch there and back
    (let ((buff (current-buffer))
	  (wind (selected-window)))
      (select-window scroll:window)
      (set-buffer scroll:buffer)

      (scroll:send-vscroll-update-message)

      (select-window wind)
      (set-buffer buff)
      ))
  )



(defun scroll:title ()
  "Set the title on this screen to scroll:buffer name."
  (process-send-string
   mbus:*server*
   (format
    "(\"TITLE\" \"%s\" 
      (action \"screen-%s\" :title \"%s\")
      )\n"
    scroll:*ws-address*
    (string-xid-of-screen (current-screen))
    (buffer-name scroll:buffer)
    )))




(defun scroll:handle-after-update (&rest args)
  "Update scroll bar if window start changed, and not by scrollbar move."
  (cond
   (scroll:movement-in-response-to-scrollbars
    (setq scroll:movement-in-response-to-scrollbars nil)
    )
   ((not (= (window-start) scroll:window-start))
    (scroll:update-scrollbar)
    )
   ))


(defun scroll:handle-after-update (&rest args)
  "Update scroll bar after updates."
  (if scroll:movement-in-response-to-scrollbars
      (setq scroll:movement-in-response-to-scrollbars nil)
    (scroll:update-scrollbar)
    ))


;----------------------------------------------------------



(defun scroll:handle-vscrollbar (type data screen)
  "Handle a vertical scrollbar message from the widget< server"
  (if (and (eq (selected-window) scroll:window)
	   (eq (current-buffer) scroll:buffer)
	   )

      (scroll:set-window-start-line
       (string-to-int 
	(resource-to-string 
	 (nth 2 data)
	 )))

    ;; else switch there and back
    (let ((buff (current-buffer))
	  (wind (selected-window))
	  )
      (select-window scroll:window)
      (set-buffer scroll:buffer)

      (scroll:set-window-start-line
       (string-to-int 
	(resource-to-string 
	 (nth 2 data)
	 )))

      (select-window wind)
      (set-buffer buff)
      )))




(defun scroll:set-window-start-line (pos)
  "Set window start line to POS."
  (when (not (= pos scroll:window-start-line))
    (let ((pt (point)))
      (scroll-down (- scroll:window-start-line pos))
      (setq scroll:window-start-line pos)
      ;; semaphore for after-update hook
      (setq scroll:movement-in-response-to-scrollbars t))
      ))




(defun scroll:send-vscroll-update-message ()
  "Update the scroll bar for selected screen."
  (if (not mbus:*server*)
      (error "mbus:*server* is null, no active connection."))
  (when (not (eq scroll:window-start (window-start)))
    (setq scroll:window-start (window-start))
    (setq scroll:window-start-line 
	  (1+ (count-lines (point-min) (window-start))))
    )
  (when (not (eq scroll:buffer-size (buffer-size)))
    (setq scroll:buffer-size (buffer-size))
    (setq scroll:buffer-lines (count-lines (point-min) (point-max)))
    )
  (let* ((maxim (+ 2 scroll:buffer-lines))
	 (size (min (window-height) (1- maxim)))
	 (val (min scroll:window-start-line (- maxim size)))
	 )
      (process-send-string
       mbus:*server*
       (format
	"(\"SCROLL\" \"%s\" 
          (action \"screen-%s.text\" :vscrollbar 
           (:minimum 1 :maximum %d :size %d :value %d)))\n"
	scroll:*ws-address*
	(string-xid-of-screen (current-screen))
	maxim size val)
       )
      ))


;----------------------------------------------------------



(defun scroll:find-screen-with-xid (xid)
  (let ((the-list (screen-list t))
	result)
    (while (and the-list (not result))
      (if (string= xid (resource-to-string (xid-of-screen (car the-list))))
          (setq result (car the-list)))
      ;; this was (car the-list)
      (setq the-list (cdr the-list))
      )
    result
    ))



(defun string-xid-of-screen (&optional screen)
  "Returns string form of Xid for current screen, or SCREEN"
  (resource-to-string 
   (xid-of-screen screen)))



(defun handle-map-of-scrollbarred-screen (scr)
  (epoch::select-screen scr)
  (if (screen-mapped-p scr) (unmap-screen scr))
  (when (and (boundp 'scroll:want-vertical-scrollbars)
	     scroll:want-vertical-scrollbars
	     )
      (process-send-string
	mbus:*server*
	(scroll:screen-def scr)
	)
      (scroll:send-vscroll-update-message)
      ))





;-------------------------------------------------



(defconst scroll:stats '((update.0) 
			 (select.0)
			 (set-buffer.0) 
			 )
  "count of scroll events")




(defun scroll:stats (key)
  "update count for KEY, print message if scroll:vocal-stats"
  (if scroll:keep-stats
      (let ((stat (assoc key scroll:stats)))
	(setcdr stat (1+ (cdr stat)))
	)))



;-------------------------------------------------



(defvar xa-window (intern-atom "WINDOW"))
(defvar scroll:*base-ui-address* "ui.cb")
(defvar scroll:*scroll_text_domain* (the-time))


(defvar scroll:*ws-address* 
  (format "%s.%s.%s" 
	  "ws"
	  mbus:*user-name* 
	  scroll:*base-ui-address*))

(defvar scroll:*text-address*
  (format "%s.%s.%s"
	  scroll:*scroll_text_domain*
	  mbus:*user-name*
	  scroll:*base-ui-address*
	  ))

(defvar scroll:*node-list* nil
  "List of active node screens. 
Each entry is (ID SCREEN NODE-BUFFER LOCAL-VARS)"
)

(mbus-set-dispatch 'area-window 'scroll:reparent-node *mbus-dispatch-dispatch*)
(mbus-set-dispatch 'open-file-prompt 'scroll:open-file *mbus-dispatch-dispatch*)
(mbus-set-dispatch 'quit-epoch 'scroll:quit-epoch *mbus-dispatch-dispatch*)

(setq mbus:*bus-domains*
  (concat scroll:*text-address* mbus:*bus-domains*))



(setq inhibit-initial-screen-mapping t)
(defvar scroll:exit-scrolled-epoch-function 'scrollbarred-exit-from-epoch)
(defvar scroll:scrollbar-system-initialized nil)


;; Should change biscuit to send "_SCROLL_VSCROLLBAR" message
(defvar xa-scroll-vscrollbar (intern-atom "_CB_VSCROLLBAR"))
(pop-message xa-scroll-vscrollbar)
(push-message xa-scroll-vscrollbar 'scroll:handle-vscrollbar)

(defvar scroll:want-vertical-scrollbars t)


;----------------------------------------------------------


(scroll:re-vector-hooks)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; No changes by B< beyond this point ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun create-screen (&optional buff alist)
  "Create a new edit screen. Optional BUFFER indicates the buffer to put in\n\
root window of the screen. Optional ALIST is a list of attributes for the\n\
new screen. ALIST is augmented by searching epoch-mode-alist for mode\n\
dependent attributes."
  (interactive)
  (let*
    (
      (b (and buff (get-buffer buff)))
      (xname
	(if b
	  (concat (buffer-name b) (sys-name))
	  "Edit"
      ))
      (real-alist
	(append
	  alist
	  (get-epoch-mode-alist-contents buff)
	  (list (cons 'title xname) (cons 'icon-name xname))
      ))
      scr
    )
    ;; call the alist adjust hook
    (cond
      ((listp  *create-screen-alist-hook*)
	(dolist (hook *create-screen-alist-hook*)
	  (setq real-alist (funcall hook real-alist))
      ))
      ((functionp *create-screen-alist-hook*)
	(setq real-alist (funcall *create-screen-alist-hook* real-alist))
    ))
    ;; create the screen
    (setq scr (epoch::create-screen buff real-alist))
    (if
      (and
	(boundp 'scroll:want-vertical-scrollbars)
	scroll:want-vertical-scrollbars
	(not (cdr (assq 'no-map real-alist)))
      )
      (handle-map-of-scrollbarred-screen scr)
    )
    ;; set up the buffers and what-not.
    (epoch::set-property xa-wm-protocols wm-protocol-list scr)
    (and buff (get-buffer buff)
      (save-excursion (set-buffer buff)
	(setq allowed-screens (cons scr allowed-screens))
    ))
    (and
      (let ((is (assq 'initial-state real-alist)))
	(or (null is) (cdr is))
      )
      (not (cdr (assq 'no-map real-alist)))
      (if (not scroll:want-vertical-scrollbars)
	(mapraised-screen scr)
    ))
    scr
))



(defun line-of (pt)
  (max 1 (count-lines (point-min) pt))
)




(defun scroll:reparent-node (data)
  "Accepts a data list of the form (id <X-window id string>), and
reparents the screen for the id into the specified X window."
  (let*
    (
      (node-id (nth 0 data))
      (xwin-id (nth 1 data))
      (scroll-screen (not (string-match "screen-" node-id)))
      (record (and scroll-screen (scroll:get-node-record node-id)))
      (is-our-screen
	(scroll:find-screen-with-xid
	  (substring node-id 7)
      ))
      (scr
	(if scroll-screen
	  (nth 1 record)
	  is-our-screen
    )))
    (if is-our-screen
      (progn
	(setq xwin (string-to-resource xwin-id xa-window))
	;;	(on-map-do scr 'cb:adjust-node-width)
	(reparent-screen xwin 0 0 scr)
	))))



(defun scroll:get-node-record (id)
  "Return the node record for the node ID, or create it if it doesn't
exist. A buffer and screen for the node is created if needed also."
  (let ((record (assoc id scroll:*node-list*)))
    (when (null record)
      (setq record (list id nil nil nil))
      (push record scroll:*node-list*)
    )
    ;; verify buffer existence, etc.
    (let
      (
        (buff (nth 2 record))
        (scr (nth 1 record))
      )
      ;; Sigh. It's possible that the buffer got toasted but the screen
      ;; is still around, in which case we have to make sure that the
      ;; regenerated buffer is put into the screen. So first, we'll verify
      ;; the screen existence, and if no screen, set scr to nil as a flag
      (if (nand scr (get-screen scr) (query-tree scr)) (setq scr nil))

      ;; Force buffer into existence.
      (when (or (null buff) (null (buffer-name buff)))
        (setq buff (get-buffer-create (concat "SCROLL " id)))
        (setcar (nthcdr 2 record) buff)
        (if scr (set-window-buffer (epoch::selected-window scr) buff))
      )
      (when (null scr)                  ;checked above
        (setq scr
          (epoch::create-screen buff
            '( ( geometry . "80x8" ) ( ex-border-width . 1 ) )
        ))
        (setcar (nthcdr 1 record) scr)
    ))
    record
))



(defun X-senior-parent (&optional scr)
  (caddr (X-path-to-root scr))
)



(defun X-path-to-root (&optional scr)
  (find-path-from-self-to-root
    (if (epoch::resourcep scr) scr
      (xid-of-screen (or scr (current-screen)))
    )
    (list
      (if (epoch::resourcep scr) scr
	(xid-of-screen (or scr (current-screen)))
))))



(defun find-path-from-self-to-root (xid list)
  (let ((this-xid-struct (query-tree xid)))
    (cond ((null (cadr this-xid-struct)) list)
      (t
	(find-path-from-self-to-root
	  (cadr this-xid-struct)
	  (cons (cadr this-xid-struct) list)
)))))


	  
(defun raise-screen (&optional scr)
  (interactive)
  (epoch::raise-screen (X-senior-parent scr))
)



(defun lower-screen (&optional scr)
  (interactive)
  (epoch::lower-screen (X-senior-parent scr))
)



(defun map-screen (&optional scr)
  (interactive)
  (epoch::map-screen (X-senior-parent scr))
  (mapcar (function (lambda (f) (funcall (cdr f) (car f))))
    (cdr (assq 'map on-event::do-list))
))



(defun mapraised-screen (&optional scr)
  (interactive)
  (epoch::mapraised-screen (X-senior-parent scr))
  (mapcar (function (lambda (f) (funcall (cdr f) (car f))))
    (cdr (assq 'map on-event::do-list))
))



(defun iconify-screen (&optional scr)
  (interactive)
  (epoch::iconify-screen (X-senior-parent scr))
)



(defun move-screen (x y &optional scr)
  (interactive)
  (epoch::move-screen x y (X-senior-parent scr))
)



(setq handle-window-delete-hook 'handle-window-delete)
(setq cleanup-scrollbars-on-exit-hook 'cleanup-scrollbars-on-exit)



(defun delete-screen (&optional scr)
  "Delete SCREEN and remove it from allowed-screens list for all buffers"
  (interactive)
  (let ((the-scr (or scr (current-screen))))
    (mapcar (symbol-function 'remove-screen-from-buffer) (buffer-list))
    (if scroll:want-vertical-scrollbars
      (send-delete-msg-to-scr the-scr)
  ))
  (epoch::delete-screen scr)
)


(defun send-delete-msg-to-scr (scr)
  (process-send-string
    mbus:*server*
    (format "(\"SCROLL\" \"%s\" (action \"screen-%s\" :delete :self))\n"
      scroll:*ws-address*
      (string-xid-of-screen scr)
)))
		       


(defun cleanup-scrollbars-on-exit ()
  (mapcar
    (symbol-function 'send-delete-msg-to-scr)
    (screen-list t)
))



(defun screen-cleanup ()
  "Delete all screens that have had their X window destroyed"
  (dolist (scr (screen-list t))
    (if (not (query-tree scr)) (delete-screen scr))
))



(defun scrollbarred-exit-from-epoch ()
  (interactive )
  (run-hooks 'cleanup-scrollbars-on-exit-hook)
  (save-buffers-kill-emacs)
)



(defun scroll:open-file (data)
  (find-file (cadr data)))



(defun scroll:quit-epoch (data)
  (run-hooks 'cleanup-scrollbars-on-exit-hook)
  (save-buffers-kill-emacs))




(defun scroll-load-elisp ()
  (if
    (and
      (fboundp 'epoch::reparent-screen)
      (not (fboundp 'reparent-screen))
    )
    (fset 'reparent-screen (symbol-function 'epoch::reparent-screen))
  )
  (if
    (and (not scroll:scrollbar-system-initialized)
	 scroll:want-vertical-scrollbars)
    (progn
      (mbus:open)
      (setq scroll:scrollbar-system-initialized t)
      (mapcar
	(symbol-function 'handle-map-of-scrollbarred-screen)
	(epoch::screen-list t)
	)
      (global-set-key "\C-x\C-c" scroll:exit-scrolled-epoch-function)
      )))


